/*
 * Copyright 2016 The Android Open Source Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.dtvkit.inputsource.player;

import android.annotation.IntDef;
import android.annotation.NonNull;
import android.annotation.Nullable;
import android.app.ActivityThread;
import android.content.ContentProvider;
import android.content.ContentResolver;
import android.content.Context;
import android.content.res.AssetFileDescriptor;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.IBinder;
import android.os.Looper;
import android.os.Message;
import android.os.Parcel;
import android.os.Parcelable;
import android.os.PersistableBundle;
import android.os.Process;
import android.os.PowerManager;
import android.os.SystemProperties;
import android.provider.Settings;
import android.system.ErrnoException;
import android.system.OsConstants;
import android.util.Log;
import android.util.Pair;
import android.view.Surface;
import android.view.SurfaceHolder;
import android.widget.VideoView;
import android.graphics.SurfaceTexture;
import android.media.AudioManager;
import android.media.MediaDrm;
import android.media.MediaFormat;
import android.media.MediaPlayer;
import android.media.MediaTimeProvider;
import android.media.PlaybackParams;
import android.media.SubtitleController;
import android.media.SubtitleController.Anchor;
import android.media.SubtitleData;
import android.media.SubtitleTrack.RenderingWidget;
import android.media.SyncParams;

import com.android.internal.util.Preconditions;

import libcore.io.IoBridge;
//import libcore.io.Libcore;
import libcore.io.Streams;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.Runnable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.ref.WeakReference;
import java.net.HttpCookie;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.URL;
import java.nio.ByteOrder;
import java.sql.Timestamp;
import java.util.Arrays;
import java.util.BitSet;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.Set;
import java.util.UUID;
import java.util.Vector;

import org.dtvkit.companionlibrary.TvPlayer;
import org.dtvkit.inputsource.DtvkitGlueClient;
import org.dtvkit.inputsource.player.DemoPlayer;
import org.dtvkit.inputsource.player.RendererBuilderFactory;
import org.dtvkit.inputsource.player.TvMediaPlayer;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class DtvkitTvPlayer extends TvMediaPlayer {
    private final static String TAG = "DtvkitTvPlayer";
    private final static boolean DEBUG = true;
    private TvPlayer mPlayer;
    private DemoPlayer mDemoPlayer;

    private Context mContext;
    private final Handler mHandler;
    private Uri mUri;
    private final List<TvPlayer.Callback> mTvPlayerCallbacks;
    private long mRecordingHandle;
    private long mCurrentPosition;
    private boolean mStarted;
    private float mSpeed;
    private PlaybackParams mPlaybackParams;

    public static DtvkitTvPlayer create (int videoType, Context context, Uri uri) {
        try {
            DtvkitTvPlayer mp = new DtvkitTvPlayer();
            mp.setDataSource(videoType, context, uri);
            mp.prepare();
            return mp;
        } catch (IllegalArgumentException ex) {
            Log.d(TAG, "create failed:", ex);
            // fall through
        } catch (SecurityException ex) {
            Log.d(TAG, "create failed:", ex);
            // fall through
        }

        return null;
    }

    public DtvkitTvPlayer() {
        mPlayer = null;
        mDemoPlayer = null;
        mContext = null;
        mHandler = null;
        mTvPlayerCallbacks = new CopyOnWriteArrayList<>();
        mRecordingHandle = 0;
        mCurrentPosition = 0;
        mStarted = false;
        mSpeed = 1.0f;
        mPlaybackParams = null;
    }

    @Override
    public int getCurrentPosition() {
        if (DEBUG) Log.d(TAG, "getCurrentPosition");
        return (int)getCurrentPosition(0);
    }

    @Override
    public void prepare () {
        if (null != mPlayer) {
            if (this == mPlayer) {
                try {
                    super.prepare();
                }
                catch (IOException ioe) {
                }
            }
            else
                mDemoPlayer.prepare();
            return;
        }
        
        // local handler
    }

    @Override
    public void release () {
        if (null != mPlayer) {
            if (this == mPlayer)
                super.release();
            else {
                mDemoPlayer.stop();
                mDemoPlayer.release();
            }
            return;
        }
        
        // local handler
    }

    @Override
    public void reset () {
        mStarted = false;
        if (null != mPlayer) {
            if (this == mPlayer)
                super.reset();
            else {
                mDemoPlayer.stop();
                mDemoPlayer.release();
            }
            return;
        }
        
        // local handler
    }

    public void setDataSource (int videoType, Context context, Uri uri) {
        mContext = context;
        mUri = uri;
        mCurrentPosition = 0;
        mStarted = false;
        String scheme = uri.getScheme();
        if ("crid".equals(scheme)) {
            mPlayer = null;
            // via Dtvkit & SDK
        }
        else {
            mDemoPlayer = new DemoPlayer(RendererBuilderFactory.createRendererBuilder(
                mContext, videoType, mUri));
            //mDemoPlayer.addListener(this);
            //mDemoPlayer.setCaptionListener(this);
            mPlayer = mDemoPlayer;
        }
    }

    @Override
    public void start () {
        if(DEBUG) Log.i(TAG, "start");
        if (mStarted) {
            resume();
            return;
        }

        if (null != mPlayer) {
            if (this == mPlayer)
                super.start();
            else
                mDemoPlayer.setPlayWhenReady(true);
            
            mStarted = true;
            return;
        }
        
        // local handler
        boolean ret = false;
        String lastSegment = mUri.getLastPathSegment();
        if (false == "timeshift".equals(lastSegment)) {
            try {
                mRecordingHandle = Long.parseLong(lastSegment);
                if(DEBUG) Log.i(TAG, "onPlayRecordedProgram mRecordingHandle " + mRecordingHandle);
            }
            catch (Exception e) {
                Log.w(TAG, "Unable to get recordingHandle for RecordedProgram", e);
            }
        }
        ret = playerPlay(mUri.toString());
        mStarted = true;
    }

    public void resume () {
        if(DEBUG) Log.i(TAG, "resume");
        if (false == mStarted) {
            return;
        }

        if (null != mPlayer) {
            if (this == mPlayer)
                super.start();
            else
                mDemoPlayer.setPlayWhenReady(true);

            return;
        }
        
        // local handler
        boolean ret = false;
        try {
            ret = playerPause(mUri.toString(), false);
        }
        catch (Exception e) {
            Log.w(TAG, "Unable to get recordingHandle for RecordedProgram", e);
        }
    }

    @Override
    public void stop () {
        if(DEBUG) Log.i(TAG, "stop");
        if (null != mPlayer) {
            if (this == mPlayer)
                super.stop();
            else
                mDemoPlayer.stop();
            return;
        }
        
        // local handler
    }

// interface TvPlayer {}
// -------------------------------------------------------------------------
    /**
     * Sets the current position for the current media.
     *
     * @param position The current time in milliseconds to play the media.
     */
    @Override
    public void seekTo(long positionMs) {
        if (null != mPlayer) {
            if (this == mPlayer)
                super.seekTo(positionMs);
            else
                mPlayer.seekTo(positionMs);
            return;
        }
        
        // local handler
        boolean ret = false;
        try {
            ret = playerSeek(mUri.toString(), positionMs);
        }
        catch (Exception e) {
            Log.w(TAG, "Fail to seek ", e);
        }
    }

    /**
     * Sets the playback params for the current media.
     *
     * @param params The new playback params.
     */
    @Override
    public void setPlaybackParams(PlaybackParams params) {
        if (DEBUG) Log.d(TAG, "setPlaybackParams");
        if (null == params) {
            return;
        }
        mPlaybackParams = params;
        float speed = params.getSpeed();
        if (DEBUG) Log.d(TAG, "Set playback speed " + speed);
        if (mSpeed == speed)
           return;

        mSpeed = speed;
        if (0 == speed) {
            pause();
            return;
        }

        if (null != mPlayer) {
            if (this == mPlayer)
                super.setPlaybackParams(params);
            else
                mPlayer.setPlaybackParams(params);
            return;
        }
        
        // local handler
        playerSetSpeed(mUri.toString(), speed);
    }


    /** @return The current time in milliseconds of the media. */
    @Override
    public long getCurrentPosition(int nouse) {
        if (null != mPlayer) {
            if (this == mPlayer)
                mCurrentPosition = super.getCurrentPosition();
            else
                mCurrentPosition = mPlayer.getCurrentPosition(0);
                
            return mCurrentPosition;
        }

        // local handler
        if (null == mUri)
            mCurrentPosition = 0;
        else
            mCurrentPosition = playerGetPosition(mUri.toString());

        return mCurrentPosition;
    }

    /** @return The total length of the currently loaded video in milliseconds. */
    @Override
    public long getDuration(int nouse) {
        if (null != mPlayer) {
            if (this == mPlayer)
                return super.getDuration();
            else
                return mPlayer.getDuration(nouse);
        }

        // local handler
        return 0;
    }

    /**
     * Sets the surface for the current media.
     *
     * @param surface The surface to play media on
     */
    @Override
    public void setSurface(Surface surface) {
        if (null != mPlayer) {
            if (this == mPlayer)
                super.setSurface(surface);
            else
                mPlayer.setSurface(surface);
            return;
        }
        
        // local handler
    }

    /**
     * Sets the volume for the current media.
     *
     * @param volume The volume between 0 and 1 to play the media at.
     */
    @Override
    public void setVolume(float volume) {
        if (null != mPlayer) {
            if (this == mPlayer)
                super.setVolume(volume);
            else
                mPlayer.setVolume(volume);
            return;
        }

        // local handler
    }

    /** Pause the current media. */
    @Override
    public void pause() {
        if (null != mPlayer) {
            if (this == mPlayer)
                super.pause();
            else
                mPlayer.pause();
            return;
        }
        // local handler
        boolean ret = false;
        try {
            ret = playerPause(mUri.toString(), true);
        }
        catch (Exception e) {
            Log.w(TAG, "Unable to get recordingHandle for RecordedProgram", e);
        }
    }

    /** Start playing or resume the current media. */
    @Override
    public void play() {
        start();
    }

    @Override
    public void registerCallback(TvPlayer.Callback callback) {
        if (null != mPlayer) {
            if (this == mPlayer)
                super.registerCallback(callback);
            else
                mPlayer.registerCallback(callback);
            return;
        }

        // local handler
        mTvPlayerCallbacks.add(callback);
    }

    @Override
    public void unregisterCallback(TvPlayer.Callback callback) {
        if (null != mPlayer) {
            if (this == mPlayer)
                super.unregisterCallback(callback);
            else
                mPlayer.unregisterCallback(callback);
            return;
        }

        // local handler
        mTvPlayerCallbacks.remove(callback);
    }
    
    // --------------------------------------------------------------------------
    // DTVKit AIDL
    private long playerGetPosition(String dvbUri) {
        long current = 0;
        long start = 0;
        try {
            JSONArray args = new JSONArray();
            JSONObject response = DtvkitGlueClient.getInstance().request("Player.getPosition", args).getJSONObject("data");
            try {
                current = response.getLong("current"); // second
                current *= 1000;                       // millisecond
                start = response.getLong("start");     // second
                start *= 1000;                         // millisecond
            }
            catch (JSONException e) {
                throw new RuntimeException(e);
            }
        }
        catch (Exception e) {
            Log.e(TAG, e.getMessage());
        }

        return current;
    }

    private boolean playerPause(String dvbUri, boolean pause) {
        if (DEBUG) Log.i(TAG, "playerPause " + dvbUri + " " + (pause ? "pause":"resume"));
        try {
            JSONArray args = new JSONArray();
            args.put(dvbUri);
            args.put(pause);
            DtvkitGlueClient.getInstance().request("Player.pause", args);
            return true;
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
            return false;
        }
    }

    private boolean playerPlay(String dvbUri) {
        if (DEBUG) Log.i(TAG, "playerPlay " + dvbUri);
        try {
            JSONArray args = new JSONArray();
            args.put(dvbUri);
            DtvkitGlueClient.getInstance().request("Player.play", args);
            return true;
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
            return false;
        }
    }

    private boolean playerSeek(String dvbUri, long pos) {
        if (DEBUG) Log.i(TAG, "playerSeek " + dvbUri + " position " + pos);
        Log.i(TAG, "playerSeek pos " + (new Timestamp(pos)).toString());
        try {
            JSONArray args = new JSONArray();
            args.put(dvbUri);
            args.put(Long.valueOf(pos));
            DtvkitGlueClient.getInstance().request("Player.seek", args);
            return true;
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
            return false;
        }
    }

    private boolean playerSetSpeed(String dvbUri, float speed) {
        if (DEBUG) Log.i(TAG, "playerSetSpeed " + dvbUri + "  speed " + speed);
        int dtvkitSpeed = (int)(speed * 100);
        try {
            JSONArray args = new JSONArray();
            args.put(dvbUri);
            args.put(dtvkitSpeed);
            DtvkitGlueClient.getInstance().request("Player.setSpeed", args);
            return true;
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
            return false;
        }
    }

    private void playerStop() {
        if (DEBUG) Log.i(TAG, "playerStop");
        try {
            JSONArray args = new JSONArray();
            DtvkitGlueClient.getInstance().request("Player.stop", args);
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
        }
    }
}
