/*
 * Copyright 2016 The Android Open Source Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.dtvkit.companionlibrary;

import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.database.ContentObserver;
import android.database.Cursor;
import android.media.PlaybackParams;
import android.media.tv.TvContentRating;
import android.media.tv.TvContract;
import android.media.tv.TvInputManager;
import android.media.tv.TvInputService;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.support.annotation.RequiresApi;
import android.util.Log;
import android.util.LongSparseArray;
import android.view.Surface;
import org.dtvkit.companionlibrary.model.Channel;
import org.dtvkit.companionlibrary.model.ModelUtils;
import org.dtvkit.companionlibrary.model.Program;
import org.dtvkit.companionlibrary.model.RecordedProgram;
import org.dtvkit.companionlibrary.utils.Constants;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

/**
 * The BaseTvInputService provides helper methods to make it easier to create a {@link
 * TvInputService} with built-in methods for content blocking and pulling the current program from
 * the Electronic Programming Guide.
 */
public abstract class BaseTvInputService extends TvInputService {
    private static final String TAG = BaseTvInputService.class.getSimpleName();
    private static final boolean DEBUG = false;

    // For database calls
    private static HandlerThread mDbHandlerThread;

    // Map of channel {@link TvContract.Channels#_ID} to Channel objects
    private static LongSparseArray<Channel> mChannelMap;
    private static ContentResolver mContentResolver;
    private static ContentObserver mChannelObserver;

    // For content ratings
    private static final List<Session> mSessions = new ArrayList<>();
    private final BroadcastReceiver mParentalControlsBroadcastReceiver =
            new BroadcastReceiver() {
                @Override
                public void onReceive(Context context, Intent intent) {
                    for (Session session : mSessions) {
                        TvInputManager manager =
                                (TvInputManager) context.getSystemService(Context.TV_INPUT_SERVICE);

                        if (!manager.isParentalControlsEnabled()) {
                            session.onUnblockContent(null);
                        } else {
                            //session.checkCurrentProgramContent();
                        }
                    }
                }
            };

    @Override
    public void onCreate() {
        super.onCreate();
        // Create background thread
        mDbHandlerThread = new HandlerThread(getClass().getSimpleName());
        mDbHandlerThread.start();

        // Initialize the channel map and set observer for changes
        mContentResolver = BaseTvInputService.this.getContentResolver();
        updateChannelMap();
        mChannelObserver =
                new ContentObserver(new Handler(mDbHandlerThread.getLooper())) {
                    @Override
                    public void onChange(boolean selfChange) {
                        updateChannelMap();
                    }
                };
        mContentResolver.registerContentObserver(
                TvContract.Channels.CONTENT_URI, true, mChannelObserver);

        // Setup our BroadcastReceiver
        IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction(TvInputManager.ACTION_BLOCKED_RATINGS_CHANGED);
        intentFilter.addAction(TvInputManager.ACTION_PARENTAL_CONTROLS_ENABLED_CHANGED);
        registerReceiver(mParentalControlsBroadcastReceiver, intentFilter);
    }

    private void updateChannelMap() {
        ComponentName component =
                new ComponentName(
                        BaseTvInputService.this.getPackageName(),
                        BaseTvInputService.this.getClass().getName());
        String inputId = TvContract.buildInputId(component);
        mChannelMap = ModelUtils.buildChannelMap(mContentResolver, inputId);
    }

    /**
     * Adds the Session to the list of currently available sessions.
     *
     * @param session The newly created session.
     * @return The session that was created.
     */
    public Session sessionCreated(Session session) {
        mSessions.add(session);
        return session;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        unregisterReceiver(mParentalControlsBroadcastReceiver);
        mContentResolver.unregisterContentObserver(mChannelObserver);
        mDbHandlerThread.quit();
        mDbHandlerThread = null;
    }

    /**
     * A {@link BaseTvInputService.Session} is called when a user tunes to channel provided by this
     * {@link BaseTvInputService}.
     */
    public abstract static class Session extends TvInputService.Session
            implements Handler.Callback {
        private static final int MSG_PLAY_CONTENT = 1000;
        private static final int MSG_PLAY_AD = 1001;
        private static final int MSG_PLAY_RECORDED_CONTENT = 1002;

        /**
         * Minimum difference between playback time and system time in order for playback to be
         * considered non-live (timeshifted).
         */
        private static final long TIME_SHIFTED_MINIMUM_DIFFERENCE_MILLIS = 3000L;
        /**
         * Buffer around current time for scheduling ads. If an ad will stop within this amount of
         * time relative to the current time, it is considered past and will not load.
         */
        private static final long PAST_AD_BUFFER_MILLIS = 2000L;

        private final Context mContext;
        private final TvInputManager mTvInputManager;
        private Channel mCurrentChannel;
        private Program mCurrentProgram;
        private long mElapsedProgramTime;
        private long mTimeShiftedPlaybackPosition = TvInputManager.TIME_SHIFT_INVALID_TIME;
        private boolean mTimeShiftIsPaused;

        private boolean mNeedToCheckChannelAd;
        private long mElapsedAdsTime;

        private boolean mPlayingRecordedProgram;
        private RecordedProgram mRecordedProgram;
        private long mRecordedPlaybackStartTime = TvInputManager.TIME_SHIFT_INVALID_TIME;

        private TvContentRating mLastBlockedRating;
        private TvContentRating[] mCurrentContentRatingSet;

        private final Set<TvContentRating> mUnblockedRatingSet = new HashSet<>();
        private final Handler mDbHandler;
        private final Handler mHandler;
        private GetCurrentProgramRunnable mGetCurrentProgramRunnable;

        private long mMinimumOnTuneAdInterval = TimeUnit.MINUTES.toMillis(5);
        private Uri mChannelUri;
        private Surface mSurface;
        private float mVolume = 1.0f;

        public Session(Context context, String inputId) {
            super(context);
            this.mContext = context;
            mTvInputManager = (TvInputManager) context.getSystemService(Context.TV_INPUT_SERVICE);
            mLastBlockedRating = null;
            mDbHandler = new Handler(mDbHandlerThread.getLooper());
            mHandler = new Handler(this);
            synchronized(mSessions) {
                mSessions.add(this);
            }
        }

        @Override
        public void onRelease() {
            mDbHandler.removeCallbacksAndMessages(null);
            mHandler.removeCallbacksAndMessages(null);
            synchronized(mSessions) {
                mSessions.remove(this);
            }
        }

        @Override
        public boolean handleMessage(Message msg) {
            switch (msg.what) {
                case MSG_PLAY_CONTENT:
                    mCurrentProgram = (Program) msg.obj;
                    //playCurrentContent();
                    return true;
                case MSG_PLAY_RECORDED_CONTENT:
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                        mPlayingRecordedProgram = true;
                        mRecordedProgram = (RecordedProgram) msg.obj;
                        playRecordedContent();
                    }
                    return true;
            }
            return false;
        }

        @RequiresApi(api = Build.VERSION_CODES.M)
        @Override
        public long onTimeShiftGetStartPosition() {
            if (mCurrentProgram != null) {
                if (mPlayingRecordedProgram) {
                    return mRecordedPlaybackStartTime;
                } else {
                    return mCurrentProgram.getStartTimeUtcMillis();
                }
            }
            return TvInputManager.TIME_SHIFT_INVALID_TIME;
        }

        @RequiresApi(api = Build.VERSION_CODES.M)
        @Override
        public long onTimeShiftGetCurrentPosition() {
            if (getTvPlayer() != null) {
                if (mPlayingRecordedProgram) {
                    long current = getTvPlayer().getCurrentPosition(0);
                    current += mRecordedPlaybackStartTime;
                    return current;
                } else {
                    if (null != mCurrentProgram) {
                        mElapsedProgramTime = getTvPlayer().getCurrentPosition(0);
                        mTimeShiftedPlaybackPosition =
                                mElapsedProgramTime
                                        + mElapsedAdsTime
                                        + mCurrentProgram.getStartTimeUtcMillis();
                        if (DEBUG) {
                            Log.d(TAG, "Time Shift Current Position");
                            Log.d(TAG, "Elapsed program time: " + mElapsedProgramTime);
                            Log.d(TAG, "Elapsed ads time: " + mElapsedAdsTime);
                            Log.d(
                                    TAG,
                                    "Total elapsed time: "
                                            + (mTimeShiftedPlaybackPosition
                                                    - mCurrentProgram.getStartTimeUtcMillis()));
                            Log.d(
                                    TAG,
                                    "Time shift difference: "
                                            + (System.currentTimeMillis()
                                                    - mTimeShiftedPlaybackPosition));
                            Log.d(TAG, "============================");
                        }
                        return getCurrentTime();
                    }
                }
            }
            return TvInputManager.TIME_SHIFT_INVALID_TIME;
        }

        @RequiresApi(api = Build.VERSION_CODES.M)
        @Override
        public void onTimeShiftSetPlaybackParams(PlaybackParams params) {
            if (params.getSpeed() != 1.0f) {
                //mHandler.removeMessages(MSG_PLAY_AD);
                mDbHandler.removeCallbacks(mGetCurrentProgramRunnable);
            }

            if (DEBUG) {
                Log.d(TAG, "Set playback speed to " + params.getSpeed());
            }
            if (getTvPlayer() != null) {
                getTvPlayer().setPlaybackParams(params);
            }
        }

        @RequiresApi(api = Build.VERSION_CODES.M)
        @Override
        public void onTimeShiftPlay(Uri recordedProgramUri) {
            if (DEBUG) {
                Log.d(TAG, "onTimeShiftPlay " + recordedProgramUri);
            }
            GetRecordedProgramRunnable getRecordedProgramRunnable =
                    new GetRecordedProgramRunnable(recordedProgramUri);
            mDbHandler.post(getRecordedProgramRunnable);
        }

        public void onTimeShiftSeekTo(long timeMs) {
            Log.i(TAG, "onTimeShiftSeekTo timeMs " + (new Timestamp(timeMs)).toString());
            if (! mPlayingRecordedProgram) {
                return;
            }
            if (getTvPlayer() != null) {
                long pos = timeMs - mRecordedPlaybackStartTime;
                if (pos < 0)
                    pos = 0;

                getTvPlayer().seekTo(pos);
            }
        }

        private void playRecordedContent() {
            if (DEBUG) {
                Log.d(TAG, "playRecordedContent ");
            }
            if (null != mRecordedProgram) {
                mCurrentProgram = mRecordedProgram.toProgram();
                if (mTvInputManager.isParentalControlsEnabled() && !checkCurrentProgramContent()) {
                    return;
                }
                mRecordedPlaybackStartTime = 0;  //System.currentTimeMillis();
            }
            else {
                // time shifting playback
                mRecordedPlaybackStartTime = 0;
            }

            if (onPlayRecordedProgram(mRecordedProgram)) {
                //setTvPlayerSurface(mSurface);
                //setTvPlayerVolume(mVolume);
            }
        }

        /**
         * This method is called when the currently playing program has been blocked by parental
         * controls. Developers should release their {@link TvPlayer} immediately so unwanted
         * content is not displayed.
         *
         * @param rating The rating for the program that was blocked.
         */
        public void onBlockContent(TvContentRating rating) {}

        private boolean checkCurrentProgramContent() {
            mCurrentContentRatingSet =
                    (mCurrentProgram == null
                                    || mCurrentProgram.getContentRatings() == null
                                    || mCurrentProgram.getContentRatings().length == 0)
                            ? null
                            : mCurrentProgram.getContentRatings();
            return blockContentIfNeeded();
        }

        private long getCurrentTime() {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                long timeShiftedDifference =
                        System.currentTimeMillis() - mTimeShiftedPlaybackPosition;
                if (mTimeShiftedPlaybackPosition != TvInputManager.TIME_SHIFT_INVALID_TIME
                        && timeShiftedDifference > TIME_SHIFTED_MINIMUM_DIFFERENCE_MILLIS) {
                    return mTimeShiftedPlaybackPosition;
                }
            }
            mTimeShiftedPlaybackPosition = TvInputManager.TIME_SHIFT_INVALID_TIME;
            return System.currentTimeMillis();
        }

        /** Return the current {@link TvPlayer}. */
        public abstract TvPlayer getTvPlayer();

        /**
         * This method is called when a particular recorded program is to begin playing. If the
         * program does not exist, the parameter will be {@code null}.
         *
         * @param recordedProgram The program that is set to be playing for a the currently tuned
         *     channel.
         * @return Whether playing this program was successful
         */
        public abstract boolean onPlayRecordedProgram(RecordedProgram recordedProgram);

        public Uri getChannelUri() {
            return mChannelUri;
        }

        private boolean blockContentIfNeeded() {
            if (mCurrentContentRatingSet == null || !mTvInputManager.isParentalControlsEnabled()) {
                // Content rating is invalid so we don't need to block anymore.
                // Unblock content here explicitly to resume playback.
                unblockContent(null);
                return true;
            }
            // Check each content rating that the program has.
            TvContentRating blockedRating = null;
            for (TvContentRating contentRating : mCurrentContentRatingSet) {
                if (mTvInputManager.isRatingBlocked(contentRating)
                        && !mUnblockedRatingSet.contains(contentRating)) {
                    // This should be blocked.
                    blockedRating = contentRating;
                }
            }
            if (blockedRating == null) {
                // Content rating is null so we don't need to block anymore.
                // Unblock content here explicitly to resume playback.
                unblockContent(null);
                return true;
            }
            mLastBlockedRating = blockedRating;
            // Children restricted content might be blocked by TV app as well,
            // but TIS should do its best not to show any single frame of blocked content.
            onBlockContent(blockedRating);
            notifyContentBlocked(blockedRating);
            if (mTimeShiftedPlaybackPosition != TvInputManager.TIME_SHIFT_INVALID_TIME) {
                onTimeShiftPause();
            }
            return false;
        }

        private void unblockContent(TvContentRating rating) {
            // TIS should unblock content only if unblock request is legitimate.
            if (rating == null || mLastBlockedRating == null || rating.equals(mLastBlockedRating)) {
                mLastBlockedRating = null;
                if (rating != null) {
                    mUnblockedRatingSet.add(rating);
                }
                notifyContentAllowed();
            }
        }

        private class GetCurrentProgramRunnable implements Runnable {
            private final Uri mChannelUri;

            GetCurrentProgramRunnable(Uri channelUri) {
                mChannelUri = channelUri;
            }

            @Override
            public void run() {
                ContentResolver resolver = mContext.getContentResolver();
                Program program = null;
                long timeShiftedDifference =
                        System.currentTimeMillis() - mTimeShiftedPlaybackPosition;
                if (mTimeShiftedPlaybackPosition != TvInputManager.TIME_SHIFT_INVALID_TIME
                        && timeShiftedDifference > TIME_SHIFTED_MINIMUM_DIFFERENCE_MILLIS) {
                    program = ModelUtils.getNextProgram(resolver, mChannelUri, mCurrentProgram);
                } else {
                    mTimeShiftedPlaybackPosition = TvInputManager.TIME_SHIFT_INVALID_TIME;
                    program = ModelUtils.getCurrentProgram(resolver, mChannelUri);
                }
                mHandler.removeMessages(MSG_PLAY_CONTENT);
                mHandler.obtainMessage(MSG_PLAY_CONTENT, program).sendToTarget();
            }
        }

        private class GetRecordedProgramRunnable implements Runnable {
            private final Uri mRecordedProgramUri;

            GetRecordedProgramRunnable(Uri recordedProgramUri) {
                mRecordedProgramUri = recordedProgramUri;
            }

            @Override
            public void run() {
                if (null == mRecordedProgramUri) {
                    // time-shift playback
                    mHandler.removeMessages(MSG_PLAY_RECORDED_CONTENT);
                    mHandler.obtainMessage(MSG_PLAY_RECORDED_CONTENT, null).sendToTarget();
                    return;
                }

                ContentResolver contentResolver = null;
                Cursor cursor = null;
                try {
                    contentResolver = mContext.getContentResolver();
                    cursor = contentResolver.query (
                        mRecordedProgramUri,
                        RecordedProgram.PROJECTION,
                        null, null, null
                    );
                    if (cursor == null) {
                        // The recorded program does not exist.
                        notifyVideoUnavailable(TvInputManager.VIDEO_UNAVAILABLE_REASON_UNKNOWN);
                    } else {
                        if (cursor.moveToNext()) {
                            RecordedProgram recordedProgram = RecordedProgram.fromCursor(cursor);
                            if (DEBUG) {
                                Log.d(TAG, "Play program " + recordedProgram.getTitle());
                                Log.d(TAG, recordedProgram.getRecordingDataUri());
                            }
                            if (recordedProgram == null) {
                                Log.e(TAG, "RecordedProgram at " + mRecordedProgramUri + " does not exist");
                                notifyVideoUnavailable(TvInputManager.VIDEO_UNAVAILABLE_REASON_UNKNOWN);
                            }
                            mHandler.removeMessages(MSG_PLAY_RECORDED_CONTENT);
                            mHandler.obtainMessage(MSG_PLAY_RECORDED_CONTENT, recordedProgram)
                                    .sendToTarget();
                        }
                    }
                }
                catch (Exception e) {
                    Log.w(TAG, "Unable to play the program for " + mRecordedProgramUri, e);
                }
                finally {
                    if (cursor != null)
                        cursor.close();
                }
            }
        }
    }

    /**
     * A {@link BaseTvInputService.RecordingSession} is created when a user wants to begin recording
     * a particular channel or program.
     */
    @RequiresApi(api = Build.VERSION_CODES.N)
    public abstract static class RecordingSession extends TvInputService.RecordingSession {
        private Context mContext;
        private String mInputId;
        private Uri mChannelUri;
        private Uri mProgramUri;
        private Handler mDbHandler;

        public RecordingSession(Context context, String inputId) {
            super(context);
            mContext = context;
            mInputId = inputId;
            mDbHandler = new Handler(mDbHandlerThread.getLooper());
        }

        @Override
        public void onTune(Uri uri) {
            mChannelUri = uri;
        }

        @Override
        public void onStartRecording(final Uri uri) {
            mProgramUri = uri;
        }

        @Override
        public void onStopRecording() {
            // Run in the database thread
            mDbHandler.post(
                    new Runnable() {
                        @Override
                        public void run() {
                            // Check if user wanted to record a specific program.
                            if (mProgramUri != null) {
                                Cursor programCursor = null;
                                try {
                                   programCursor = mContext.getContentResolver().query(
                                      mProgramUri,
                                      Program.PROJECTION,
                                      null,
                                      null,
                                      null
                                   );
                                   if (programCursor != null && programCursor.moveToNext()) {
                                      Program programToRecord = Program.fromCursor(programCursor);
                                      onStopRecording(programToRecord);
                                   } else {
                                      Channel recordedChannel = ModelUtils.getChannel(
                                          mContext.getContentResolver(),
                                          mChannelUri
                                      );
                                      onStopRecordingChannel(recordedChannel);
                                   }
                                }
                                catch (Exception e) {
                                   Log.w(TAG, "Unable to get program " + mProgramUri, e);
                                }
                                finally {
                                   if (programCursor != null)
                                       programCursor.close();
                                }
                            } else {
                                // User is recording a channel
                                Channel recordedChannel =
                                        ModelUtils.getChannel(
                                                mContext.getContentResolver(), mChannelUri);
                                onStopRecordingChannel(recordedChannel);
                            }
                        }
                    });
        }

        /**
         * Called when the application requests to stop TV program recording. Recording must stop
         * immediately when this method is called. The session must create a new data entry using
         * {@link #notifyRecordingStopped(RecordedProgram)} that describes the new {@link
         * RecordedProgram} and call {@link #notifyRecordingStopped(Uri)} with the URI to that
         * entry. If the stop request cannot be fulfilled, the session must call {@link
         * #notifyError(int)}.
         *
         * @param programToRecord The program set by the user to be recorded.
         */
        public abstract void onStopRecording(Program programToRecord);

        /**
         * Called when the application requests to stop TV channel recording. Recording must stop
         * immediately when this method is called. The session must create a new data entry using
         * {@link #notifyRecordingStopped(RecordedProgram)} that describes the new {@link
         * RecordedProgram} and call {@link #notifyRecordingStopped(Uri)} with the URI to that
         * entry. If the stop request cannot be fulfilled, the session must call {@link
         * #notifyError(int)}.
         *
         * @param channelToRecord The channel set by the user to be recorded.
         */
        public abstract void onStopRecordingChannel(Channel channelToRecord);

        /**
         * Notify the TV app that the recording has ended.
         *
         * @param recordedProgram The program that was recorded and should be saved.
         */
        public void notifyRecordingStopped(final RecordedProgram recordedProgram) {
            mDbHandler.post(
                    new Runnable() {
                        @Override
                        public void run() {
                            Uri recordedProgramUri = null;
                            if (null != recordedProgram) {
                                recordedProgramUri =
                                        mContext.getContentResolver()
                                                .insert(
                                                        TvContract.RecordedPrograms.CONTENT_URI,
                                                        recordedProgram.toContentValues());
                            }
                            notifyRecordingStopped(recordedProgramUri);
                        }
                    });
        }

        public Uri getChannelUri() {
            return mChannelUri;
        }
    }
}
