#ifndef GLUE_GLUE_H
#define GLUE_GLUE_H

#include "RequestData.h"
#include "ResponseData.h"
#include "RequestDataStub.h"
#include "ResponseDataStub.h"

#include <string>
#include <map>
#include <list>
#include <algorithm>

/**
 * @defgroup metaapi Meta-API
 * The Meta-API is the API that clients of glue services use.
 */

class Glue
{
   public:
   class Interface
   {
      public:
      typedef bool (Interface::*Invokable)(RequestData*, ResponseData*);

      /**
       * @brief Invoke an invokable function.
       * @param name Invokable name.
       * @param request Request data.
       * @param response Response data.
       */
      bool invoke(const std::string& invokable, RequestData* request, ResponseData* response)
      {
         std::map<std::string, Invokable>::iterator it = m_invokable.find(invokable);
         return (it == m_invokable.end()) ? false : (this->*(it->second))(request, response);
      }

      /**
       * @brief Invoke an invokable function without request or response data (default arguments).
       * @param name Invokable name.
       * @param request Request data.
       * @param response Response data.
       */
      bool invoke(const std::string& invokable)
      {
         RequestDataStub request;
         ResponseDataStub response;
         std::map<std::string, Invokable>::iterator it = m_invokable.find(invokable);
         return (it == m_invokable.end()) ? false : (this->*(it->second))(&request, &response);
      }

      protected:
      Interface(){};
      ~Interface(){};

      /**
       * @brief Add an invokable function to this interface.
       * @param name Invokable name.
       * @param invokable Function pointer.
       */
      template <class T>
      void addInvokable(const std::string& name, bool (T::*invokable)(RequestData*, ResponseData*))
      {
         m_invokable[name] = static_cast<Invokable>(invokable);
      }

      /**
       * @brief Send signal to each signal handler added to the glue instance.
       * @param signal Signal name.
       * @param invokable Pointer to an invokable function that populates ResponseData for the signal.
       */
      void signal(const std::string &signal)
      {
         Glue::signal(signal, 0, 0);
      }

      /**
       * @brief Send signal to each signal handler added to the glue instance.
       * @param signal Signal name.
       * @param invokable Pointer to an invokable function that populates ResponseData for the signal.
       */
      template <class T>
      void signal(const std::string &signal, bool (T::*invokable)(RequestData*, ResponseData*))
      {
         Glue::signal(signal, static_cast<Interface::Invokable>(invokable), static_cast<Interface *>(&T::instance()));
      }

      private:
      Interface(Interface const&);
      Interface& operator=(Interface const&);
      std::map<std::string, Invokable> m_invokable;
   };

   typedef void (*SignalHandler)(const std::string &signal, Interface::Invokable invokable, Glue::Interface *invokable_object);

   /**
    * @brief Add an interface to the glue.
    * @param name Interface name.
    */
   template <class T>
   static void addInterface(const std::string &name)
   {
      Glue& glue = instance();
      T& interface = T::instance();
      glue.m_interfaces[name] = &interface;
   }

   /**
    * @brief Add a signal handler to the glue that is called for each signal.
    * @param handler Pointer to a signal handler function.
    */
   static void addSignalHandler(SignalHandler handler)
   {
      Glue& glue = instance();
      if (std::find(glue.m_signalhandlers.begin(), glue.m_signalhandlers.end(), handler) == glue.m_signalhandlers.end())
      {
         glue.m_signalhandlers.push_back(handler);
      }
   }

   /**
    * @brief Invoke an invokable function.
    * @param name Interface name.
    * @param name Invokable name.
    * @param request Request data.
    * @param response Response data.
    */
   static bool invoke(const std::string &interface, const std::string& invokable, RequestData* request, ResponseData* response)
   {
      Glue& glue = instance();
      std::map<std::string, Interface*>::iterator it = glue.m_interfaces.find(interface);
      if (it != glue.m_interfaces.end())
      {
         return it->second->invoke(invokable, request, response);
      }

      return false;
   }

   /**
    * @brief Send signal to each signal handler.
    * @param signal Signal name.
    * @param invokable Pointer to an invokable function that populates ResponseData for the signal.
    * @param invokable_object An object that invokable is a member of.
    */
   static void signal(const std::string &signal, Interface::Invokable invokable, Interface *invokable_object)
   {
      Glue& glue = instance();
      for (std::list<SignalHandler>::iterator it = glue.m_signalhandlers.begin(); it != glue.m_signalhandlers.end(); ++it)
      {
         (*it)(signal, invokable, invokable_object);
      }
   }

   private:
   static Glue& instance()
   {
      static Glue instance;
      return instance;
   }

	Glue() {};
   Glue(Glue const&);
   Glue& operator=(Glue const&);
   std::map<std::string, Interface*> m_interfaces;
   std::list<SignalHandler> m_signalhandlers;
};

#endif

