#ifndef DVBINTERFACE_H
#define DVBINTERFACE_H

#include <iostream>

#include "Glue.h"
#include "RequestData.h"
#include "ResponseData.h"

/**
 * @ingroup metaapi
 * @defgroup dvbmetaapi DVB Meta-API
 * Dvb
 */

class DvbInterface: public Glue::Interface
{
   public:
   static DvbInterface& instance()
   {
      static DvbInterface instance;
      return instance;
   }

   /**
    * @ingroup dvbmetaapi
    * @brief Get frontend.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getFrontend(Int32 path = INVALID_RES_ID)
    * ~~~
    * - __path__ Number (value 0 to UINT8_MAX).
    * @param response Used to write _frontend object_ response (example pseudocode):
    * ~~~{.js}
    * {
    *    "frontendpath": 0,
    *    "frequency": 0,
    *    "integrity": 0,
    *    "strength": 0,
    *    "type": "none"
    * }
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getFrontend(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Get list of all frontends.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getListOfFrontends()
    * ~~~
    * @param response Used to write _array of frontend objects_ response (example pseudocode):
    * ~~~{.js}
    * [
    *    {
    *       "frontendpath": 0,
    *       "frequency": 0,
    *       "integrity": 0,
    *       "strength": 0,
    *       "type": "none"
    *    },
    *    ...
    * ]
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getListOfFrontends(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Add a user-defined service list (e.g. a favourite list) named _name_.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * addServiceList(string name = "")
    * ~~~
    * @param response Used to write _service list object_ response (example pseudocode):
    * ~~~{.js}
    * {
    *    "identifier": "0x08",
    *    "name": "My Favourites",
    *    "size": 0,
    *    "writable": true
    * }
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool addServiceList(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Remove the user-defined service list _identifier_. The list must be user-defined.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * removeServiceList(string identifier = "")
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool removeServiceList(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Get list of all service lists.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getListOfServiceLists()
    * ~~~
    * @param response Used to write _array of service list objects_ response (example pseudocode):
    * ~~~{.js}
    * [
    *    {
    *       "identifier": "all",
    *       "name": "All",
    *       "size": 18,
    *       "type": "default"
    *    },
    *    ...
    * ]
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getListOfServiceLists(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Add service _uri_ to the user-defined service list _identifier_.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * addService(string identifier = "unknown", string uri = NULL_SERVICE_URI)
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool addService(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Remove service _uri_ from the user-defined service list _identifier_.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * removeService(string identifier = "unknown", string uri = NULL_SERVICE_URI)
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool removeService(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Get the service _uri_.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getService(string uri = NULL_SERVICE_URI)
    * ~~~
    * @param response Used to write _service object_ response (example pseudocode):
    * ~~~{.js}
    * {
    *    "blocked": false,
    *    "lcn": 9,
    *    "name": "BBC FOUR",
    *    "radio": false,
    *    "subtitles": false,
    *    "uri": "dvb://233a.4000.41c0"
    * }
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getService(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Set service _uri_ parental control lock to _locked_.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setServiceBlocked(string uri = NULL_SERVICE_URI, bool blocked = true)
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setServiceBlocked(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Get number of services in list _identifier_.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getNumberOfServices(string identifier = "all")
    * ~~~
    * @param response Used to write _number Int32_ response (example pseudocode):
    * ~~~{.js}
    * 18
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getNumberOfServices(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Get list of services in list _identifier_.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getListOfServices(string identifier = "all")
    * ~~~
    * @param response Used to write _array of service objects_ response (example pseudocode):
    * ~~~{.js}
    * [
    *    {
    *       "blocked": true,
    *       "lcn": 9,
    *       "name": "BBC FOUR",
    *       "radio": false,
    *       "subtitles": false,
    *       "uri": "dvb://233a.4000.41c0"
    *    },
    *    ...
    * ]
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getListOfServices(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Get event _uri_.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getEvent(string uri = NULL_EVENT_URI)
    * ~~~
    * @param response Used to write _event object_ response (example pseudocode):
    * ~~~{.js}
    * {
    *    "ad": false,
    *    "duration": 3600,
    *    "endutc": 1205890200,
    *    "guidance": "",
    *    "name": "Auntie's War on Smut",
    *    "rating": 0,
    *    "startutc": 1205886600,
    *    "subtitles": true,
    *    "uri": "dvb://233a.4000.41c0;178c"
    * }
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getEvent(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Get now and next events for service _uri_.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getNowNextEvents(string uri = NULL_EVENT_URI)
    * ~~~
    * @param response Used to write _now next events object_ response (example pseudocode):
    * ~~~{.js}
    * {
    *    "ad": false,
    *    "duration": 3600,
    *    "endutc": 1205890200,
    *    "guidance": "",
    *    "name": "Auntie's War on Smut",
    *    "rating": 0,
    *    "startutc": 1205886600,
    *    "subtitles": true,
    *    "uri": "dvb://233a.4000.41c0;178c"
    * }
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getNowNextEvents(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Get list of events in schedule for service _uri_, that end after _startutc_ and start on or before _endutc_.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getListOfEvents(string uri = NULL_SERVICE_URI, UInt32 startutc = 0, UInt32 endutc = 4294967295)
    * ~~~
    * @param response Used to write _array of event objects_ response (example pseudocode):
    * ~~~{.js}
    * [
    *    {
    *       "ad": false,
    *       "duration": 3600,
    *       "endutc": 1205890200,
    *       "guidance": "",
    *       "name": "Auntie's War on Smut",
    *       "rating": 0,
    *       "startutc": 1205886600,
    *       "subtitles": true,
    *       "uri": "dvb://233a.4000.41c0;178c"
    *    },
    *    ...
    * ]
    * @return Success (on failure, an error string is written to response)
    */
   bool getListOfEvents(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Get parental control on.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getParentalControlOn()
    * ~~~
    * @param response Used to write _on bool_ response (example pseudocode):
    * ~~~{.js}
    * false
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getParentalControlOn(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Set parental control on.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setParentalControlOn(bool on = true)
    * ~~~
    * @param response Used to write _success_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setParentalControlOn(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Get parental control age.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getParentalControlAge()
    * ~~~
    * @param response Used to write _age Int32_ response (example pseudocode):
    * ~~~{.js}
    * 18
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getParentalControlAge(RequestData* request, ResponseData* response);

   /**
    * @ingroup dvbmetaapi
    * @brief Set parental control age.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setParentalControlAge(Int32 age = 18)
    * ~~~
    * - __age__ Number (value 4 to 18).
    * @param response Used to write _success_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setParentalControlAge(RequestData* request, ResponseData* response);

   // TODO comment
   bool getListOfUpdatedEventPeriods(RequestData* request, ResponseData* response);

   private:
	DvbInterface();
   DvbInterface(DvbInterface const&);
   DvbInterface& operator=(DvbInterface const&);

   static void eventHandler(unsigned int code, const void *data, unsigned int data_size);
   static void schedUpdatedLatchTask(void *param);
   static void nowUpdatedLatchTask(void *param);
};

#endif // DVBINTERFACE_H

