#ifndef PLAYERINTERFACE_H
#define PLAYERINTERFACE_H

#include <iostream>
#include <list>
#include <string>

#include "Glue.h"
#include "RequestData.h"
#include "ResponseData.h"

/**
 * @ingroup metaapi
 * @defgroup playermetaapi Player Meta-API
 * Dvb
 */

class PlayerInterface: public Glue::Interface
{
   public:
   static PlayerInterface& instance()
   {
      static PlayerInterface instance;
      return instance;
   }

   static void postEvent(unsigned int code, const void *data, unsigned int data_size);

   /**
    * @ingroup playermetaapi
    * @brief Get the playback position.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getStatus()
    * ~~~
    * @param response Used to write _status object_ response (example pseudocode):
    * ~~~{.js}
    * {
    *    "current": 100    (second)
    *    "start": 0        (second)
    * }
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getPosition(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Get status.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getStatus()
    * ~~~
    * @param response Used to write _status object_ response (example pseudocode):
    * ~~~{.js}
    * {
    *    "frontendpath": 0,
    *    "state": "playing",
    *    "type": "live",
    *    "uri": "dvb://233a.4000.41c0"
    * }
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getStatus(RequestData* request, ResponseData* response);

   /**
    * @ingroup recordermetaapi
    * @brief Get SSU status.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getStatus()
    * ~~~
    * @param response Used to write _status object_ response (example pseudocode):
    * ~~~{.js}
    * {
    *    "release":"HCA5821191021.1",
    *    "uri": null
    * }
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getSsuStatus(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Pause content _uri_. Signals PlayerStatusChanged with getStatus data.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * play(uri = "")
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool pause(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Play content _uri_. Signals PlayerStatusChanged with getStatus data.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * play(uri = "")
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool play(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Stop playing content. Signals PlayerStatusChanged with getStatus data.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * stop()
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * stop
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool stop(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Start decoding.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * startDecoding()
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool startDecoding(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Stop decoding.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * stopDecoding()
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool stopDecoding(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Parental control unblock.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * unblock()
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool unblock(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Get list of audio streams.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getListOfAudioStreams()
    * ~~~
    * @param response Used to write _list of audio stream objects_ response (example pseudocode):
    * ~~~{.js}
    * [
    *    {
    *       "pid": 100,
    *       "ad": false,
    *       "codec": "MP2",
    *       "language": "eng",
    *       "selected": false
    *    },
    *    {
    *       "pid": 101,
    *       "ad": true,
    *       "codec": "MP2",
    *       "language": "eng",
    *       "selected": true
    *    }
    * ]
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getListOfAudioStreams(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Set audio stream.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setAudioStream(int pid = 0xFFFF, string lang_code = "")
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setAudioStream(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Get list of subtitle streams.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getListOfSubtitleStreams()
    * ~~~
    * @param response Used to write _list of audio stream objects_ response (example pseudocode):
    * ~~~{.js}
    * [
    *    {
    *       "pid": 100,
    *       "language": "eng",
    *       "selected": false,
    *       "teletext": true
    *    },
    *    {
    *       "pid": 101,
    *       "language": "eng",
    *       "selected": true,
    *       "teletext": false
    *    }
    * ]
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getListOfSubtitleStreams(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Set subtitle stream.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setSubtitleStream(int pid = 0xFFFF, string lang_code = "")
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setSubtitleStream(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Set player rectangle. The rectangle is scaled from full-HD (1920x1080) to the AV screen size.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setRectangle(x = 0, y = 0, width = 1920, height = 1080)
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setRectangle(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Get mute.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getMute()
    * ~~~
    * @param response Used to write _mute bool_ response (example pseudocode):
    * ~~~{.js}
    * false
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getMute(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Set mute.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setMute(mute = true)
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setMute(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Get volume.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getVolume()
    * ~~~
    * @param response Used to write _volume Int32_ response (example pseudocode):
    * ~~~{.js}
    * 100
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getVolume(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Set volume.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setVolume(volume = 50)
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setVolume(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Get subtitles on.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getSubtitlesOn()
    * ~~~
    * @param response Used to write _on bool_ response (example pseudocode):
    * ~~~{.js}
    * false
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getSubtitlesOn(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Set subtitles on.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setSubtitlesOn(on = true)
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setSubtitlesOn(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Get audio description on.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getAudioDescriptionOn()
    * ~~~
    * @param response Used to write _on bool_ response (example pseudocode):
    * ~~~{.js}
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getAudioDescriptionOn(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Set audio description on.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setAudioDescriptionOn(on = true)
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setAudioDescriptionOn(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Get list of audio languages.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getListOfLanguages()
    * ~~~
    * @param response Used to write _array of language objects_ response (example pseudocode):
    * ~~~{.js}
    * eng
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getListOfLanguages(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Get default audio language.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getDefaultAudioLanguage()
    * ~~~
    * @param response Used to write _ISO 639-2/T string_ response (example pseudocode):
    * ~~~{.js}
    * eng
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getDefaultAudioLanguage(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Set default audio language.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setDefaultAudioLanguage(language = "eng")
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setDefaultAudioLanguage(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Get default subtitles language.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getDefaultSubtitlesLanguage()
    * ~~~
    * @param response Used to write _ISO 639-2/T string_ response (example pseudocode):
    * ~~~{.js}
    * eng
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getDefaultSubtitlesLanguage(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Set default subtitles language.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * setDefaultSubtitlesLanguage(language = eng)
    * ~~~
    * @param response Used to write _success bool_ response (example pseudocode):
    * ~~~{.js}
    * true
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool setDefaultSubtitlesLanguage(RequestData* request, ResponseData* response);

   /**
    * @ingroup playermetaapi
    * @brief Get the video window handle.
    * @param request Used to read arguments for the invokable (pseudocode):
    * ~~~{.java}
    * getVideoWindowHandle()
    * ~~~
    * @param response Used to write _number UInt32_ response (example pseudocode):
    * ~~~{.js}
    * 0
    * ~~~
    * @return Success (on failure, an error string is written to response)
    */
   bool getVideoWindowHandle(RequestData* request, ResponseData* response);

   bool seek(RequestData* request, ResponseData* response);
   
   bool setSpeed(RequestData* request, ResponseData* response);

   private:
	PlayerInterface();
   PlayerInterface(PlayerInterface const&);
   PlayerInterface& operator=(PlayerInterface const&);
   void event(unsigned int code, const void *data, unsigned int data_size);
   
   static void commandHandlerTask(void * param);
   void commandHandler(void);

protected:
   std::list<std::string> zappingList;
   void * mutex;
   void * cmd_queue;
   void * cmd_task;
};

#endif // PLAYERINTERFACE_H

