#include "Glue.h"

#include "ResourceManager.h"
#include "EventManager.h"
#include "Entities.h"
#include "common.h"
#include <cstdio>
#include <sstream>

#include "CasInterface.h"
#include "CDCASS.h"
#include "json/json.h"

static U32BIT BCDToUInt(U32BIT bcd);
static int stringToPin(std::string & str, uint8_t * pin);

static U32BIT BCDToUInt(U32BIT bcd)
{
   FUNCTION_START(BCDToUInt);

   U8BIT * p = (U8BIT *)&bcd;
   U32BIT value = 0;
   U32BIT scale = 1;

   for (int i = 0; i < sizeof(U32BIT); i++) {
      value += (((U32BIT)p[i] & 0xF) * scale);
      scale *= 10;
      value += (((U32BIT)(p[i] >> 4) & 0xF) * scale);
      scale *= 10;   
   }

   FUNCTION_FINISH(BCDToUInt);

   return(value);
}

static int stringToPin(std::string & str, uint8_t * pin)
{
   if ((6 != str.size()) || (NULL == pin))
      return -1;

   memcpy (pin, str.c_str(), 6);
   /* value range: [0,9] */ 
   for (int i=0; i < 6; i++) {
      pin[i] -= '0';
   }
   return 0;
}

CasInterface::CasInterface()
{
   mCasInstance = Rtk_CDCA::getInstance();
   if (NULL != mCasInstance) {
      // register callbacks
      mCasInstance->registerOnActionRequestExtCallback(
         std::bind(&CasInterface::onActionRequestExt, this, std::placeholders::_1, std::placeholders::_2, std::placeholders::_3, std::placeholders::_4)
      );

      mCasInstance->registerOnContinuesWatchLimitCallback(
         std::bind(&CasInterface::onContinuesWatchLimit, this, std::placeholders::_1, std::placeholders::_2, std::placeholders::_3)
      );

      mCasInstance->registerOnDetitleReceivedCallback(
         std::bind(&CasInterface::onDetitleReceived, this, std::placeholders::_1)
      );

      mCasInstance->registerOnEntitleChangedCallback(
         std::bind(&CasInterface::onEntitleChanged, this, std::placeholders::_1)
      );

      mCasInstance->registerOnEmailNotifyIconCallback(
         std::bind(&CasInterface::onEmailNotifyIcon, this, std::placeholders::_1, std::placeholders::_2)
      );

      mCasInstance->registerOnHideIPPVDlgCallback(
         std::bind(&CasInterface::onHideIPPVDlg, this, std::placeholders::_1)
      );

      mCasInstance->registerOnHideOSDMessageCallback(
         std::bind(&CasInterface::onHideOSDMessage, this, std::placeholders::_1)
      );

      mCasInstance->registerOnLockServiceCallback(
         std::bind(&CasInterface::onLockService, this, std::placeholders::_1)
      );

      mCasInstance->registerOnRequestFeedingCallback(
         std::bind(&CasInterface::onRequestFeeding, this, std::placeholders::_1)
      );

      mCasInstance->registerOnShowBuyMessageCallback(
         std::bind(&CasInterface::onShowBuyMessage, this, std::placeholders::_1, std::placeholders::_2)
      );

      mCasInstance->registerOnShowCurtainNotifyCallback(
         std::bind(&CasInterface::onShowCurtainNotify, this, std::placeholders::_1, std::placeholders::_2)
      );

      mCasInstance->registerOnShowFingerInfoCallback(
         std::bind(&CasInterface::onShowFingerInfo, this, std::placeholders::_1, std::placeholders::_2)
      );

      mCasInstance->registerOnShowFingerMessageExtCallback(
         std::bind(&CasInterface::onShowFingerMessageExt, this, std::placeholders::_1, std::placeholders::_2)
      );

      mCasInstance->registerOnShowOSDInfoCallback(
         std::bind(&CasInterface::onShowOSDInfo, this, std::placeholders::_1)
      );

      mCasInstance->registerOnShowOSDMessageCallback(
         std::bind(&CasInterface::onShowOSDMessage, this, std::placeholders::_1, std::placeholders::_2)
      );

      mCasInstance->registerOnShowProgressStripCallback(
         std::bind(&CasInterface::onShowProgressStrip, this, std::placeholders::_1, std::placeholders::_2)
      );

      mCasInstance->registerOnStartIppvBuyDlgCallback(
         std::bind(&CasInterface::onStartIppvBuyDlg, this, std::placeholders::_1, std::placeholders::_2, std::placeholders::_3)
      );

      mCasInstance->registerOnUNLockServiceCallback(
         std::bind(&CasInterface::onUNLockService, this)
      );

      mCasInstance->Init();
   }
   else {
      // output error messages
   }
   
   EventManager::addEventHandler(CasInterface::postEvent);

   addInvokable("changePin", &CasInterface::changePin);
   addInvokable("delDetitleChkNum", &CasInterface::delDetitleChkNum);
   addInvokable("delEmail", &CasInterface::delEmail);
   addInvokable("formatBuffer", &CasInterface::formatBuffer);
   addInvokable("getACList", &CasInterface::getACList);
   addInvokable("getCardFreezeStatus", &CasInterface::getCardFreezeStatus);
   addInvokable("getCardSN", &CasInterface::getCardSN);
   addInvokable("getCardVer", &CasInterface::getCardVer);
   addInvokable("getDetitleChkNums", &CasInterface::getDetitleChkNums);
   addInvokable("getDetitleReaded", &CasInterface::getDetitleReaded);
   addInvokable("getEmailContent", &CasInterface::getEmailContent);
   addInvokable("getEmailHead", &CasInterface::getEmailHead);
   addInvokable("getEmailHeads", &CasInterface::getEmailHeads);
   addInvokable("getEmailSpaceInfo", &CasInterface::getEmailSpaceInfo);
   addInvokable("getEntitleIDs", &CasInterface::getEntitleIDs);
   addInvokable("getOperatorIds", &CasInterface::getOperatorIds);
   addInvokable("getOperatorInfo", &CasInterface::getOperatorInfo);
   addInvokable("getPlatformID", &CasInterface::getPlatformID);
   addInvokable("getRating", &CasInterface::getRating);
   addInvokable("getServiceEntitles", &CasInterface::getServiceEntitles);
   addInvokable("getSlotIDs", &CasInterface::getSlotIDs);
   addInvokable("getSlotInfo", &CasInterface::getSlotInfo);
   addInvokable("getSTBSN", &CasInterface::getSTBSN);
   addInvokable("getTerminalTypeID", &CasInterface::getTerminalTypeID);
   addInvokable("getVer", &CasInterface::getVer);
   addInvokable("getWorkTime", &CasInterface::getWorkTime);
   addInvokable("isPaired", &CasInterface::isPaired);
   addInvokable("setRating", &CasInterface::setRating);
   addInvokable("getDrmKey", &CasInterface::getDrmKey);
   addInvokable("refreshInterface", &CasInterface::refreshInterface);
}

bool CasInterface::changePin(RequestData* request, ResponseData* response)
{
   std::string oldPin = request->shiftString("");
   std::string newPin = request->shiftString("");
   std::string retStr;
   
   CDCA_U8 old_pin[6], new_pin[6];
   memcpy (old_pin, oldPin.c_str(), 6);
   memcpy (new_pin, newPin.c_str(), 6);

   /* value range: [0,9] */ 
   for (int i=0; i < 6; i++) {
      old_pin[i] -= '0';
      new_pin[i] -= '0';
   }
   
   CDCA_U16 ret = CDCASTB_ChangePin((const CDCA_U8*)old_pin, (const CDCA_U8*)new_pin);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_PIN_INVALID:
         retStr = "PIN_INVALID";
         break;
      case CDCA_RC_UNKNOWN:
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   object->finish();

   return true;
}

bool CasInterface::delDetitleChkNum(RequestData* request, ResponseData* response)
{
   CDCA_U16 tvsId = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
   CDCA_U32 detitleChkNum = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
   CDCA_BOOL ret = CDCASTB_DelDetitleChkNum(tvsId, detitleChkNum);
   response->setBool((CDCA_TRUE == ret) ? true : false);

   return true;
}


bool CasInterface::delEmail(RequestData* request, ResponseData* response)
{
   std::string retStr;   
   
   CDCA_U32 emailId = request->shiftUInt32(0, 0, UINT32_MAX);
   CDCASTB_DelEmail(emailId);
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->finish();
      
   return true;
}

bool CasInterface::formatBuffer(RequestData* request, ResponseData* response)
{
   std::string retStr;
   USE_UNWANTED_PARAM(request);  
   
   CDCASTB_FormatBuffer();
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->finish();
      
   return true;
}

bool CasInterface::getACList(RequestData* request, ResponseData* response)
{
   std::string retStr;   
   CDCA_U32 ac_array[CDCA_MAXNUM_ACLIST];
   
   CDCA_U16 tvsid = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
   memset (&ac_array, 0, sizeof(CDCA_U32) * CDCA_MAXNUM_ACLIST);
   CDCA_U16 ret = CDCASTB_GetACList(tvsid, ac_array);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      case CDCA_RC_DATA_NOT_FIND:
         retStr = "DATA_NOT_FIND";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if (NULL != dataObject) {
      ResponseData::Array * acArray = dataObject->insertArray("ac_array", 0);
      if ((NULL != acArray) && (CDCA_RC_OK == ret)) {
         for (int i = 0; i < CDCA_MAXNUM_ACLIST; i++) {
            acArray->appendUInt32((uint32_t)ac_array[i]);
         }
      }
   }
   object->finish();

   return true;
}

bool CasInterface::getCardFreezeStatus(RequestData* request, ResponseData* response)
{
   std::string retStr;
   CDCA_U8 status = 0;   
   
   CDCA_U16 tvsId = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
   CDCA_U16 ret = CDCASTB_GetCardFreezeStatus(tvsId, &status);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if ((NULL != dataObject) && (CDCA_RC_OK == ret)) {
      dataObject->insertUInt32("status", (uint32_t)status);
   }
   object->finish();

   return true;
}

bool CasInterface::getCardSN(RequestData* request, ResponseData* response)
{
   char buf[CDCA_MAXLEN_SN+1];
   USE_UNWANTED_PARAM(request);

   memset(buf, 0, CDCA_MAXLEN_SN+1);
   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (CDCA_RC_OK == CDCASTB_GetCardSN(buf)) {
      response->setString(buf);
   }
   else {
      response->setString(""); 
   }
   object->finish();

   return true;
}

bool CasInterface::getCardVer(RequestData* request, ResponseData* response)
{
   std::string retStr;
   CDCA_U16 cardVer;   
   
   CDCA_U16 tvsId = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
   CDCA_U16 ret = CDCASTB_GetCardVer(tvsId, &cardVer);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if ((NULL != dataObject) && (CDCA_RC_OK == ret)) {
      dataObject->insertUInt32("card_version", (uint32_t)cardVer);
   }
   object->finish();

   return true;
}

bool CasInterface::getDetitleChkNums(RequestData* request, ResponseData* response)
{
   std::string retStr;
   
   CDCA_U16 tvsId = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
   CDCA_BOOL readFlag;
   CDCA_U32 detitleChkNums[CDCA_MAXNUM_DETITLE];                           
   
   CDCA_U16 ret = CDCASTB_GetDetitleChkNums(tvsId, &readFlag, detitleChkNums);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      case CDCA_RC_DATA_NOT_FIND:
         retStr = "DATA_NOT_FIND";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   dataObject->insertBool("read_flag", (CDCA_TRUE == readFlag) ? true : false);
   ResponseData::Array * array = dataObject->insertArray("detitle_check_numbers", CDCA_MAXNUM_DETITLE);
   if (NULL != array) {
      for (int i = 0; i < CDCA_MAXNUM_DETITLE; i++) {
         array->appendUInt32(detitleChkNums[i]);
      }
   }
   object->finish();

   return true;
}

bool CasInterface::getDetitleReaded(RequestData* request, ResponseData* response)
{
   CDCA_U16 tvsId = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
   CDCA_BOOL ret = CDCASTB_GetDetitleReaded(tvsId);
   response->setBool((CDCA_TRUE == ret) ? true : false);

   return true;
}

bool CasInterface::getEmailContent(RequestData* request, ResponseData* response)
{
   std::string retStr;   
   SCDCAEmailContent emailContent;
   
   CDCA_U32 emailId = request->shiftUInt32(0, 0, UINT32_MAX);
   memset (&emailContent, 0, sizeof(SCDCAEmailContent));
   CDCA_U16 ret = CDCASTB_GetEmailContent(emailId, &emailContent);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      case CDCA_RC_DATA_NOT_FIND:
         retStr = "DATA_NOT_FIND";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if (NULL != dataObject) {
      ResponseData::Object * emailHeadObject = dataObject->insertObject("email_content", 1);
      if ((NULL != emailHeadObject) && (CDCA_RC_OK == ret)) {
         std::string email(emailContent.m_szEmail, CDCA_MAXLEN_EMAIL_CONTENT+4);
         emailHeadObject->insertString("email", email);
      }
   }
   object->finish();
      
   return true;
}

bool CasInterface::getEmailHead(RequestData* request, ResponseData* response)
{
   std::string retStr;   
   SCDCAEmailHead emailHead ;
   
   CDCA_U32 emailId = request->shiftUInt32(0, 0, UINT32_MAX);
   memset (&emailHead, 0, sizeof(SCDCAEmailHead));
   CDCA_U16 ret = CDCASTB_GetEmailHead(emailId, &emailHead);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      case CDCA_RC_DATA_NOT_FIND:
         retStr = "DATA_NOT_FIND";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if (NULL != dataObject) {
      ResponseData::Object * emailHeadObject = dataObject->insertObject("email_head", 1);
      if ((NULL != emailHeadObject) && (CDCA_RC_OK == ret)) {
         emailHeadObject->insertUInt32("action_id", emailHead.m_dwActionID);
         emailHeadObject->insertUInt32("create_time", emailHead.m_tCreateTime);
         emailHeadObject->insertUInt32("importance", emailHead.m_wImportance);
         emailHeadObject->insertString("mail_head", emailHead.m_szEmailHead);
         emailHeadObject->insertUInt32("new_email", emailHead.m_bNewEmail);
      }
   }
   object->finish();
      
   return true;
}

bool CasInterface::getEmailHeads(RequestData* request, ResponseData* response)
{
   std::string retStr;   
   SCDCAEmailHead * pEmailHead = NULL;
   CDCA_U8 count = (CDCA_U8)request->shiftUInt32(1, 0, UINT8_MAX);
   CDCA_U8 fromIndex = (CDCA_U8)request->shiftUInt32(0, 0, UINT8_MAX);

   pEmailHead = new SCDCAEmailHead[count];
   if (NULL != pEmailHead) {
      //memset (pEmailHead, 0, sizeof(SCDCAEmailHead) * count);
      CDCA_U16 ret = CDCASTB_GetEmailHeads(pEmailHead, &count, &fromIndex);
      switch (ret) {
         case CDCA_RC_OK:
            retStr = "OK";
            break;
         case CDCA_RC_POINTER_INVALID:
            retStr = "POINTER_INVALID";
            break;
         case CDCA_RC_CARD_INVALID:
            retStr = "CARD_INVALID";
            break;
         case CDCA_RC_DATA_NOT_FIND:
            retStr = "DATA_NOT_FIND";
            break;
         default:
            retStr = "UNKNOWN";
            break;  
      }
      // Write response object
      ResponseData::Object *object = response->setObject(1);   
      object->insertInt32("code", (int32_t)ret);
      object->insertString("message", retStr);
      ResponseData::Object * dataObject = object->insertObject("data", 1);
      if (NULL != dataObject) {
         ResponseData::Array * emailHeadArray = dataObject->insertArray("email_heads", 0);
         if ((NULL != emailHeadArray) && (CDCA_RC_OK == ret)) {
            for (int i = 0; i < count; i++) {
               ResponseData::Object * emailHeadObject = emailHeadArray->appendObject(1);
               if (NULL != emailHeadObject) {
                  emailHeadObject->insertUInt32("action_id", pEmailHead[i].m_dwActionID);
                  emailHeadObject->insertUInt32("create_time", pEmailHead[i].m_tCreateTime);
                  emailHeadObject->insertUInt32("importance", pEmailHead[i].m_wImportance);
                  emailHeadObject->insertString("mail_head", pEmailHead[i].m_szEmailHead);
                  emailHeadObject->insertUInt32("new_email", pEmailHead[i].m_bNewEmail);
               }
            }
         }
      }
      object->finish();
      
      delete[] pEmailHead;
      pEmailHead = NULL;
   }

   return true;
}

bool CasInterface::getEmailSpaceInfo(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   std::string retStr;
   CDCA_U8 emailNum = 0;
   CDCA_U8 emptyNum = 0;
   
   CDCA_U16 ret = CDCASTB_GetEmailSpaceInfo(&emailNum, &emptyNum);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if ((NULL != dataObject) && (CDCA_RC_OK == ret)) {
      dataObject->insertUInt32("email_number", (uint32_t)emailNum);
      dataObject->insertUInt32("empty_number", (uint32_t)emptyNum);
   }
   object->finish();

   return true;
}

bool CasInterface::getEntitleIDs(RequestData* request, ResponseData* response)
{
   std::string retStr;   
   CDCA_U32 * pEntitleIds = new CDCA_U32 [CDCA_MAXNUM_ENTITLE];
   if (NULL !=  pEntitleIds) {  
      CDCA_U16 tvsid = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
      memset (pEntitleIds, 0, sizeof(CDCA_U32) * CDCA_MAXNUM_ENTITLE);
      CDCA_U16 ret = CDCASTB_GetEntitleIDs(tvsid, pEntitleIds);
      switch (ret) {
         case CDCA_RC_OK:
            retStr = "OK";
            break;
         case CDCA_RC_POINTER_INVALID:
            retStr = "POINTER_INVALID";
            break;
         case CDCA_RC_CARD_INVALID:
            retStr = "CARD_INVALID";
            break;
         case CDCA_RC_DATA_NOT_FIND:
            retStr = "DATA_NOT_FIND";
            break;
         default:
            retStr = "UNKNOWN";
            break;  
      }
      // Write response object
      ResponseData::Object *object = response->setObject(1);   
      object->insertInt32("code", (int32_t)ret);
      object->insertString("message", retStr);
      ResponseData::Object * dataObject = object->insertObject("data", 1);
      if (NULL != dataObject) {
         ResponseData::Array * entitleIdArray = dataObject->insertArray("entitle_ids", 0);
         if ((NULL != entitleIdArray) && (CDCA_RC_OK == ret)) {
            for (int i = 0; i < CDCA_MAXNUM_ENTITLE; i++) {
               if (0 == pEntitleIds[i])
                  break;

               entitleIdArray->appendUInt32((uint32_t)pEntitleIds[i]);
            }
         }
      }
      object->finish();

      delete[] pEntitleIds;
      pEntitleIds = NULL;
   }

   return true;
}

bool CasInterface::getOperatorIds(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   std::string retStr;
   
   CDCA_U8 num = 0;
   CDCA_U16 tvsid[CDCA_MAXNUM_OPERATOR];
   
   for (int i=0; i < CDCA_MAXNUM_OPERATOR; i++) {
      tvsid[i] = 0;
   }
   
   CDCA_U16 ret = CDCASTB_GetOperatorIds(tvsid);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   ResponseData::Array * idsArray = dataObject->insertArray("tvsid", 0);
   for (int i = 0; i < CDCA_MAXNUM_OPERATOR; i++) {
      if (0 == tvsid[i])
         break;

      idsArray->appendUInt32(tvsid[i]);
   }
   object->finish();

   return true;
}

bool CasInterface::getOperatorInfo(RequestData* request, ResponseData* response)
{
   std::string retStr;   
   SCDCAOperatorInfo operatorInfo;
   
   CDCA_U16 tvsid = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
   memset (&operatorInfo, 0, sizeof(SCDCAOperatorInfo));
   CDCA_U16 ret = CDCASTB_GetOperatorInfo(tvsid, &operatorInfo);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      case CDCA_RC_DATA_NOT_FIND:
         retStr = "DATA_NOT_FIND";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if (NULL != dataObject) {
      ResponseData::Object * opInfoObject = dataObject->insertObject("operator_info", 1);
      if (NULL != opInfoObject) {
         ResponseData::Array * tvsPriInfoArray = opInfoObject->insertArray("tvs_pri_info", 0);
         if ((NULL != tvsPriInfoArray) && (CDCA_RC_OK == ret)) {
            for (int i = 0; i < (CDCA_MAXLEN_TVSPRIINFO+1); i++) {
               tvsPriInfoArray->appendUInt32((uint32_t)operatorInfo.m_szTVSPriInfo[i]);
            }
         }
      }
   }
   object->finish();

   return true;
}

bool CasInterface::getPlatformID(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object   
   CDCA_U16 id = CDCASTB_GetPlatformID();
   response->setUInt32(id);

   return true;
}

bool CasInterface::getRating(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   std::string retStr;
   CDCA_U8 rating = 4;
   
   CDCA_U16 ret = CDCASTB_GetRating(&rating);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   dataObject->insertUInt32("rating", (uint32_t)rating);
   object->finish();

   return true;
}

bool CasInterface::getServiceEntitles(RequestData* request, ResponseData* response)
{
   CDCA_U16 ret = CDCA_RC_UNKNOWN;
   std::string retStr = "UNKNOWN";   
   SCDCAEntitles* pServiceEntitles = new SCDCAEntitles();
   if (NULL != pServiceEntitles) {
      memset (pServiceEntitles, 0, sizeof(SCDCAEntitles));
      CDCA_U16 tvsid = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
      uint32_t beginIdx = request->shiftUInt32(0, 0, (CDCA_MAXNUM_ENTITLE - 1));
      uint32_t endIdx = request->shiftUInt32(0, 0, (CDCA_MAXNUM_ENTITLE - 1));
      ret = CDCASTB_GetServiceEntitles(tvsid, pServiceEntitles);
      switch (ret) {
         case CDCA_RC_OK:
            retStr = "OK";
            break;
         case CDCA_RC_POINTER_INVALID:
            retStr = "POINTER_INVALID";
            break;
         case CDCA_RC_CARD_INVALID:
            retStr = "CARD_INVALID";
            break;
         case CDCA_RC_DATA_NOT_FIND:
            retStr = "DATA_NOT_FIND";
            break;  
      }
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if ((NULL != dataObject) && (CDCA_RC_OK == ret)) {
      ResponseData::Object * serviceEntitlesObject = dataObject->insertObject("service_entitles", 1);
      if (NULL != serviceEntitlesObject) {
         CDCA_U16 productCount = pServiceEntitles->m_wProductCount;
         serviceEntitlesObject->insertUInt32("product_count", (uint32_t)productCount);
         ResponseData::Array * entitlesArray = serviceEntitlesObject->insertArray("entitles", 0);
         if ((NULL != entitlesArray)) {
            for (int i = 0; i < productCount; i++) {
               if (i > 20)
                  break;

               SCDCAEntitle  * pEntitle = &pServiceEntitles->m_Entitles[i];
               ResponseData::Object* entitleObject = entitlesArray->appendObject(1);
               if (NULL != entitleObject) {
                  entitleObject->insertUInt32("product_id", pEntitle->m_dwProductID);
                  entitleObject->insertUInt32("begin_date", pEntitle->m_tBeginDate);
                  entitleObject->insertUInt32("expire_date", pEntitle->m_tExpireDate);
                  entitleObject->insertUInt32("can_tape", pEntitle->m_bCanTape);
               }
            }
         }
      }
   }

   object->finish();   
   if (NULL != pServiceEntitles) {
      delete pServiceEntitles;
      pServiceEntitles = NULL;
   }

   return true;
}

bool CasInterface::getSlotIDs(RequestData* request, ResponseData* response)
{
   std::string retStr;
   CDCA_U8 slotID[CDCA_MAXNUM_SLOT];   
   
   CDCA_U16 tvsid = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
   memset (slotID, 0, sizeof(CDCA_U8) * CDCA_MAXNUM_SLOT);
   CDCA_U16 ret = CDCASTB_GetSlotIDs(tvsid, slotID);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      case CDCA_RC_DATA_NOT_FIND:
         retStr = "DATA_NOT_FIND";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if (NULL != dataObject) {
      ResponseData::Array * slotIdsArray = dataObject->insertArray("slot_ids", 0);
      if ((NULL != slotIdsArray) && (CDCA_RC_OK == ret)) {
         for (int i = 0; i < CDCA_MAXNUM_SLOT; i++) {
            if (0 == slotID[i])
               break;

            slotIdsArray->appendUInt32((uint32_t)slotID[i]);
         }
      }
   }
   object->finish();

   return true;
}

bool CasInterface::getSlotInfo(RequestData* request, ResponseData* response)
{
   std::string retStr;
   SCDCATVSSlotInfo slotInfo;      
   
   CDCA_U16 tvsId = (CDCA_U16)request->shiftUInt32(0, 0, UINT16_MAX);
   CDCA_U8 slotId = (CDCA_U8)request->shiftUInt32(0, 0, UINT8_MAX);

   CDCA_U16 ret = CDCASTB_GetSlotInfo(tvsId, slotId, &slotInfo);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      case CDCA_RC_DATA_NOT_FIND:
         retStr = "DATA_NOT_FIND";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if (NULL != dataObject) {
      ResponseData::Object * slotInfoObject = dataObject->insertObject("slot_info", 1);
      if ((NULL != slotInfoObject) && (CDCA_RC_OK == ret)) {
         slotInfoObject->insertUInt32("credit_limit", (int32_t)slotInfo.m_wCreditLimit);
         slotInfoObject->insertUInt32("balance", (int32_t)slotInfo.m_wBalance);
      }
   }
   object->finish();

   return true;
}

/**
 * @ingroup casmetaapi
 * @brief Get the STB serial number.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getSTBSN()
 * ~~~
 * @param response Used to write _status object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "sn": "0x83830000008E"
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool CasInterface::getSTBSN(RequestData* request, ResponseData* response)
{
   char buf[32];  // must not less than 16 bytes
   USE_UNWANTED_PARAM(request);

   memset(buf, 0, 32);
   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (CDCA_RC_OK == CDCASTB_GetSTBSN(buf)) {
      response->setString(buf);
   }
   else {
      response->setString(""); 
   }
   object->finish();

   return true;
}

bool CasInterface::getTerminalTypeID(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object   
   CDCA_U16 id = CDCASTB_GetTerminalTypeID();
   response->setUInt32(id);

   return true;
}

bool CasInterface::getVer(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object   
   CDCA_U32 ver = CDCASTB_GetVer();
   response->setUInt32(ver);

   return true;
}

bool CasInterface::getWorkTime(RequestData* request, ResponseData* response)
{
   std::string retStr;
   CDCA_U8 startHour, startMin, startSec;
   CDCA_U8 endHour, endMin, endSec;

   CDCA_U16 ret = CDCASTB_GetWorkTime(
      &startHour, &startMin, &startSec,
      &endHour, &endMin, &endSec
   );
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      case CDCA_RC_DATA_NOT_FIND:
         retStr = "DATA_NOT_FIND";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if ((NULL != dataObject) && (CDCA_RC_OK == ret)) {
      dataObject->insertUInt32("start_hour", (uint32_t)startHour);
      dataObject->insertUInt32("start_minute", (uint32_t)startMin);
      dataObject->insertUInt32("start_second", (uint32_t)startSec);

      dataObject->insertUInt32("end_hour", (uint32_t)endHour);
      dataObject->insertUInt32("end_minute", (uint32_t)endMin);
      dataObject->insertUInt32("end_second", (uint32_t)endSec);
   }
   object->finish();

   return true;
}

bool CasInterface::isPaired(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   std::string retStr;
   
   CDCA_U8 num = 0;
   CDCA_U8 stb_id_list[5][6];
   
   CDCA_U16 ret = CDCASTB_IsPaired(&num, stb_id_list[0]);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_CARD_PAIROTHER:
         retStr = "CARD_PAIROTHER";
         break;
      case CDCA_RC_CARD_NOPAIR:
         retStr = "CARD_NOPAIR";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   dataObject->insertInt32("number", num);
   ResponseData::Array * idsArray = dataObject->insertArray("stbid", 0);
   for (int i = 0; i < num; i++) {
      uint64_t stbId = 0;
      for (int j=0; j<6; j++) {
         // network byte order
         stbId += (((uint64_t)stb_id_list[i][j]) << (8 * (5 - j)));
      }
      idsArray->appendString(std::to_string(stbId));
   }
   object->finish();

   return true;
}

bool CasInterface::setRating(RequestData* request, ResponseData* response)
{
   std::string retStr;   
   std::string pinStr = request->shiftString("");
   CDCA_U8 rating = (CDCA_U8)request->shiftUInt32(4, 4, 18);
   CDCA_U8 pin[6];
   stringToPin(pinStr, pin);   

   CDCA_U16 ret = CDCASTB_SetRating(pin, rating);
   switch (ret) {
      case CDCA_RC_OK:
         retStr = "OK";
         break;
      case CDCA_RC_POINTER_INVALID:
         retStr = "POINTER_INVALID";
         break;
      case CDCA_RC_CARD_INVALID:
         retStr = "CARD_INVALID";
         break;
      case CDCA_RC_PIN_INVALID:
         retStr = "PIN_INVALID";
         break;
      case CDCA_RC_WATCHRATING_INVALID:
         retStr = "WATCHRATING_INVALID";
         break;
      default:
         retStr = "UNKNOWN";
         break;  
   }
   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   object->finish();

   return true;
}

#define CDCA_DRM_KEY_LEN 16
CDCA_U16 CDCASTB_GetDrmKey(CDCA_U8 *url, CDCA_U8 *key_array)
{
    CDCA_U8 key_default[CDCA_DRM_KEY_LEN] = {
	0xFB, 0x90, 0x6B, 0xF6, 0x6C, 0xC4, 0x19, 0x49,
	0x5D, 0x38, 0x22, 0xB7, 0x34, 0x43, 0x4D, 0xC4
    };

    memcpy(key_array, key_default, 16);
    return CDCA_DRM_KEY_LEN;
}

bool CasInterface::getDrmKey(RequestData* request, ResponseData* response)
{
   std::string retStr;   
   std::string drmUrlStr = request->shiftString("");
   CDCA_U8 drmUrl[1024];
   CDCA_U8 ac_array[CDCA_DRM_KEY_LEN];
   stringToPin(drmUrlStr, drmUrl);   

   CDCA_U16 ret = CDCASTB_GetDrmKey(drmUrl, ac_array);
   if (ret == CDCA_DRM_KEY_LEN)
       retStr = "OK";
   else
       retStr = "DRM_FAILURE";

   // Write response object
   ResponseData::Object *object = response->setObject(1);   
   object->insertInt32("code", (int32_t)ret);
   object->insertString("message", retStr);
   ResponseData::Object * dataObject = object->insertObject("data", 1);
   if (NULL != dataObject) {
      ResponseData::Array * acArray = dataObject->insertArray("drm_key", 0);
      if ((NULL != acArray) && (CDCA_DRM_KEY_LEN == ret)) {
         for (int i = 0; i < CDCA_DRM_KEY_LEN; i++) {
            acArray->appendUInt32((uint32_t)ac_array[i]);
         }
      }
   }
   object->finish();

   return true;
}

bool CasInterface::refreshInterface(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   
   CDCASTB_RefreshInterface();
   // Write response object
   ResponseData::Object *object = response->setObject(1);
   object->finish();

   return true;
}

// -------------------------------------------------------------------------------------------
// callback data

bool CasInterface::getActionRequestExt(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mActionRequestExt.wTVSID);
         array->appendUInt32(mActionRequestExt.byActionType);
         array->appendUInt32(mActionRequestExt.byLen);
         ResponseData::Array * dataArray = array->appendArray(mActionRequestExt.byLen);
         if (NULL != dataArray && NULL != mActionRequestExt.pbyData) {
			for (int i=0; i < mActionRequestExt.byLen; i++) {
				dataArray->appendUInt32(mActionRequestExt.pbyData[i]);
            }
         }
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getContinuesWatchLimit(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mContinuesWatchLimit.byType);
         array->appendUInt32(mContinuesWatchLimit.wWorkTime);
         array->appendUInt32(mContinuesWatchLimit.wStopTime);
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getDetitleReceived(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mDetitleReceived.bstatus);
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getEntitleChanged(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mEntitleChanged.wTvsID);
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getEmailNotifyIcon(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mEmailNotifyIcon.byShow);
         array->appendUInt32(mEmailNotifyIcon.dwEmailID);
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getHideIPPVDlg(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mHideIPPVDlg.wEcmPid);
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getHideOSDMessage(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mHideOSDMessage.byStyle);
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getLockService(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         ResponseData::Object * lockServiceObject = array->appendObject(1);
         if (NULL != lockServiceObject && NULL != mLockService.pLockService) {
            const SCDCALockService * pLockService = mLockService.pLockService;
            U32BIT freq = BCDToUInt(pLockService->m_dwFrequency) * 100;
            U32BIT symbolRate = BCDToUInt(pLockService->m_symbol_rate) * 100;

            S_ACTL_SERVICE_PARAMS serviceParams;
            memcpy (&serviceParams, pLockService, sizeof(S_ACTL_SERVICE_PARAMS));
            serviceParams.frequency = freq;
            serviceParams.symbol_rate = symbolRate;
            ACTL_SetLockedService (&serviceParams);

            lockServiceObject->insertUInt32("frequency", freq);
            lockServiceObject->insertUInt32("symbo_rate", symbolRate);
            lockServiceObject->insertUInt32("pcr_pid", pLockService->m_wPcrPid);
            lockServiceObject->insertUInt32("modulation", pLockService->m_Modulation);
            lockServiceObject->insertUInt32("component_number", pLockService->m_ComponentNum);
      	    ResponseData::Array * componentArray = lockServiceObject->insertArray("components", pLockService->m_ComponentNum);
            if (NULL != componentArray) {
               for (int i=0; i < pLockService->m_ComponentNum; i++) {
                  ResponseData::Object * componentObject = componentArray->appendObject(1);
                  if (NULL != componentObject) {
                     const SCDCAComponent & component = pLockService->m_CompArr[i];
                     componentObject->insertUInt32("component_pid", component.m_wCompPID);
                     componentObject->insertUInt32("ecm_pid", component.m_wECMPID);
                     componentObject->insertUInt32("component_type", component.m_CompType);
                  }
               }
            }
            lockServiceObject->insertUInt32("fec_outer", pLockService->m_fec_outer);
            lockServiceObject->insertUInt32("fec_inner", pLockService->m_fec_inner);
         }
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getRequestFeeding(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendBool((CDCA_TRUE == mRequestFeeding.bReadStatus) ? true : false);
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getShowBuyMessage(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mShowBuyMessage.wEcmPID);
         array->appendUInt32(mShowBuyMessage.byMessageType);
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getShowCurtainNotify(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mShowCurtainNotify.wEcmPID);
         array->appendUInt32(mShowCurtainNotify.wCurtainCode);
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getShowFingerInfo(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mShowFingerInfo.wEcmPID);
         ResponseData::Object * fingerInfoObject = array->appendObject(1);
         const SCDCAFingerInfo * pFingerInfo = mShowFingerInfo.pFingerInfo;
         if (NULL != fingerInfoObject && NULL != pFingerInfo) {
            fingerInfoObject->insertUInt32("show_type", pFingerInfo->byShowType);
            fingerInfoObject->insertUInt32("content_type", pFingerInfo->byContentType);
            fingerInfoObject->insertBool("is_visible", (CDCA_TRUE == pFingerInfo->bIsVisible) ? true : false);
            fingerInfoObject->insertBool("is_force_show", (CDCA_TRUE == pFingerInfo->bIsForcedShow) ? true : false);
            fingerInfoObject->insertUInt32("encryption", pFingerInfo->byEncryption);
            fingerInfoObject->insertUInt32("flashing_each_show_time", pFingerInfo->wFlashing_EachShowTime);
            fingerInfoObject->insertUInt32("flashing_each_hide_time", pFingerInfo->wFlashing_EachHideTime);
            fingerInfoObject->insertUInt32("font_size", pFingerInfo->byFontSize);
            fingerInfoObject->insertUInt32("font_color", pFingerInfo->dwFontColor);
            fingerInfoObject->insertUInt32("background_color", pFingerInfo->dwBackgroundColor);
            fingerInfoObject->insertUInt32("x_position", pFingerInfo->byX_position);
            fingerInfoObject->insertUInt32("y_position", pFingerInfo->byY_position);
      	    ResponseData::Array * matrixArray = fingerInfoObject->insertArray("matrix_inner_pos", 16);
            if (NULL != matrixArray) {
               for (int i=0; i < 16; i++) {
                  matrixArray->appendUInt32(pFingerInfo->pbyMatrixInnerPos[i]);
               }
            }
            fingerInfoObject->insertString("content", pFingerInfo->szContent);
         }
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getShowFingerMessageExt(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mShowFingerMessageExt.wEcmPID);
         if (NULL != mShowFingerMessageExt.fingerMsg)
            array->appendString(mShowFingerMessageExt.fingerMsg);
         else
            array->appendString("");
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getShowOSDInfo(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         ResponseData::Object * osdInfoObject = array->appendObject(1);
         SCDCAOSDInfo * pOSDInfo = mShowOSDInfo.pOSDInfo;
         if (NULL != osdInfoObject && NULL != pOSDInfo) {
            osdInfoObject->insertUInt32("display_mode", pOSDInfo->byDisplayMode);
            osdInfoObject->insertUInt32("show_type", pOSDInfo->byShowType);
            osdInfoObject->insertUInt32("font_size", pOSDInfo->byFontSize);
            osdInfoObject->insertUInt32("background_area", pOSDInfo->byBackgroundArea);
            osdInfoObject->insertBool("show_card_sn", (CDCA_TRUE == pOSDInfo->bShowCardSN) ? true : false);
            osdInfoObject->insertString("card_sn", pOSDInfo->szCardSN);
            osdInfoObject->insertUInt32("font_color", pOSDInfo->dwFontColor);
            osdInfoObject->insertUInt32("background_color", pOSDInfo->dwBackgroundColor);
            osdInfoObject->insertString("content", pOSDInfo->szContent);
         }
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getShowOSDMessage(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mShowOSDMessage.byStyle);
         if (NULL != mShowOSDMessage.szMessage)
            array->appendString(mShowOSDMessage.szMessage);
         else
            array->appendString("");
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getShowProgressStrip(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mShowProgressStrip.byProgress);
         array->appendUInt32(mShowProgressStrip.byMark);
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getStartIppvBuyDlg(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      if (NULL != array) {
         array->appendUInt32(mStartIppvBuyDlg.byMessageType);
         array->appendUInt32(mStartIppvBuyDlg.wEcmPid);
         ResponseData::Object * startIppvBuyDlgObject = array->appendObject(1);
         const SCDCAIppvBuyInfo * pIppvProgram = mStartIppvBuyDlg.pIppvProgram;
         if (NULL != startIppvBuyDlgObject && NULL != pIppvProgram) {
            startIppvBuyDlgObject->insertUInt32("product_id", pIppvProgram->m_dwProductID);
            startIppvBuyDlgObject->insertUInt32("tvsid", pIppvProgram->m_wTvsID);
            startIppvBuyDlgObject->insertUInt32("slot_id", pIppvProgram->m_bySlotID);
            startIppvBuyDlgObject->insertUInt32("price_number", pIppvProgram->m_byPriceNum);
      	    ResponseData::Array * priceArray = startIppvBuyDlgObject->insertArray("price", pIppvProgram->m_byPriceNum);
            if (NULL != priceArray) {
               for (int i=0; i < pIppvProgram->m_byPriceNum; i++) {
                  ResponseData::Object * priceObject = priceArray->appendObject(1);
                  if (NULL != priceObject) {
                     const SCDCAIPPVPrice & price = pIppvProgram->m_Price[i];
                     priceObject->insertUInt32("price", price.m_wPrice);
                     priceObject->insertUInt32("price_code", price.m_byPriceCode);
                  }
               }
            }
            ResponseData::Object * ippvTimeObject = startIppvBuyDlgObject->insertObject("ippv_time", 2);
            if (NULL != ippvTimeObject) {
				ippvTimeObject->insertUInt32("expired_date", pIppvProgram->m_wIPPVTime.m_wExpiredDate);
				ippvTimeObject->insertUInt32("interval_minute", pIppvProgram->m_wIPPVTime.m_wIntervalMin);
            }
         }
      }
      object->finish();
   }

   return true;
}

bool CasInterface::getUNLockService(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   // Write response object
   ResponseData::Object *object = response->setObject(1);
   if (NULL != object) {
      ResponseData::Array * array = object->insertArray("params", 0);
      object->finish();
   }

   //FIXME: workaround for the immediate lock->unlock service condition
   usleep(1200000);
   ACTL_SetLockedService(NULL);

   return true;
}

// -------------------------------------------------------------------------------------------
// callback

void CasInterface::onActionRequestExt(CDCA_U16 wTVSID, CDCA_U8 byActionType, CDCA_U8 byLen, CDCA_U8* pbyData)
{
   mActionRequestExt.wTVSID = wTVSID;
   mActionRequestExt.byActionType = byActionType;
   mActionRequestExt.byLen = byLen;
   mActionRequestExt.pbyData = pbyData;
   signal("CasActionRequestExt", &CasInterface::getActionRequestExt);
}

void CasInterface::onContinuesWatchLimit(CDCA_U8 byType, CDCA_U16 wWorkTime, CDCA_U16 wStopTime)
{
   mContinuesWatchLimit.byType = byType;
   mContinuesWatchLimit.wWorkTime = wWorkTime;
   mContinuesWatchLimit.wStopTime = wStopTime;
   signal("CasContinuesWatchLimit", &CasInterface::getContinuesWatchLimit);
}

void CasInterface::onDetitleReceived(CDCA_U8 bstatus)
{
   mDetitleReceived.bstatus = bstatus;
   signal("CasDetitleReceived", &CasInterface::getDetitleReceived);
}

void CasInterface::onEntitleChanged(CDCA_U16 wTvsID)
{
   mEntitleChanged.wTvsID = wTvsID;
   signal("CasEntitleChanged", &CasInterface::getEntitleChanged);
}

void CasInterface::onEmailNotifyIcon(CDCA_U8 byShow, CDCA_U32 dwEmailID)
{
   mEmailNotifyIcon.byShow = byShow;
   mEmailNotifyIcon.dwEmailID = dwEmailID;
   signal("CasEmailNotifyIcon", &CasInterface::getEmailNotifyIcon);
}

void CasInterface::onHideIPPVDlg(CDCA_U16 wEcmPid)
{
   mHideIPPVDlg.wEcmPid = wEcmPid;
   signal("CasHideIPPVDlg", &CasInterface::getHideIPPVDlg);
}

void CasInterface::onHideOSDMessage(CDCA_U8 byStyle)
{
   mHideOSDMessage.byStyle = byStyle;
   signal("CasHideOSDMessage", &CasInterface::getHideOSDMessage);
}

void CasInterface::onLockService(const SCDCALockService* pLockService)
{
   mLockService.pLockService = pLockService;
   signal("CasLockService", &CasInterface::getLockService);
}

void CasInterface::onRequestFeeding(CDCA_BOOL bReadStatus)
{
   mRequestFeeding.bReadStatus = bReadStatus;
   signal("CasRequestFeeding", &CasInterface::getRequestFeeding);
}

void CasInterface::onShowBuyMessage (CDCA_U16 wEcmPID, CDCA_U8 byMessageType)
{
   mShowBuyMessage.wEcmPID = wEcmPID;
   mShowBuyMessage.byMessageType = byMessageType;
   signal("CasShowBuyMessage", &CasInterface::getShowBuyMessage);
}

void CasInterface::onShowCurtainNotify (CDCA_U16 wEcmPID, CDCA_U16 wCurtainCode)
{
   mShowCurtainNotify.wEcmPID = wEcmPID;
   mShowCurtainNotify.wCurtainCode = wCurtainCode;
   signal("CasShowCurtainNotify", &CasInterface::getShowCurtainNotify);
}

void CasInterface::onShowFingerInfo (CDCA_U16 wEcmPID, const SCDCAFingerInfo* pFingerInfo)
{
   mShowFingerInfo.wEcmPID = wEcmPID;
   mShowFingerInfo.pFingerInfo = pFingerInfo;
   signal("CasShowFingerInfo", &CasInterface::getShowFingerInfo);
}

void CasInterface::onShowFingerMessageExt(CDCA_U16 wEcmPID, char* fingerMsg)
{
   mShowFingerMessageExt.wEcmPID = wEcmPID;
   mShowFingerMessageExt.fingerMsg = fingerMsg;
   signal("CasShowFingerMessageExt", &CasInterface::getShowFingerMessageExt);
}

void CasInterface::onShowOSDInfo(SCDCAOSDInfo *pOSDInfo)
{
   mShowOSDInfo.pOSDInfo = pOSDInfo;
   signal("CasShowOSDInfo", &CasInterface::getShowOSDInfo);
}

void CasInterface::onShowOSDMessage(CDCA_U8 byStyle, const char * szMessage)
{
   mShowOSDMessage.byStyle = byStyle;
   mShowOSDMessage.szMessage = szMessage;
   signal("CasShowOSDMessage", &CasInterface::getShowOSDMessage);
}

void CasInterface::onShowProgressStrip(CDCA_U8 byProgress, CDCA_U8 byMark)
{
   mShowProgressStrip.byProgress = byProgress;
   mShowProgressStrip.byMark = byMark;
   signal("CasShowProgressStrip", &CasInterface::getShowProgressStrip);
}

void CasInterface::onStartIppvBuyDlg(CDCA_U8 byMessageType, CDCA_U16 wEcmPid, const SCDCAIppvBuyInfo * pIppvProgram)
{
   mStartIppvBuyDlg.byMessageType = byMessageType;
   mStartIppvBuyDlg.wEcmPid = wEcmPid;
   mStartIppvBuyDlg.pIppvProgram = pIppvProgram;
   signal("CasStartIppvBuyDlg", &CasInterface::getStartIppvBuyDlg);
}

void CasInterface::onUNLockService(void)
{
   signal("CasUNLockService", &CasInterface::getUNLockService);
}

//---------------------------------------------------------------------------------------------------

void CasInterface::event(unsigned int code, const void *data, unsigned int data_size)
{
   USE_UNWANTED_PARAM(data_size);

  switch (code)
  {
#if 0
     case STB_EVENT_TUNE_NOTLOCKED:
          if (diff_state != g_recorder_state)
          {
             signal("OSDMessage", &CasInterface::getOSDMessage);
          }
          break;
#endif
  }
}

