#include "Glue.h"

#include "ResourceManager.h"
#include "EventManager.h"
#include "Entities.h"
#include "common.h"

extern "C"
{
   #include "stbhwtun.h"
}

#include <cstdio>
#include <iostream>
#include <cutils/properties.h>

#include "DvbInterface.h"

#define FRONTEND_OBJECT_SIZE 5
#define SERVICE_OBJECT_SIZE 6
#define EVENT_OBJECT_SIZE 11
#define LIST_OBJECT_SIZE 4

#define SCHED_UPDATE_TIMEOUT_PERIOD 15000 // 15s
#define SCHED_UPDATE_TRICKLE_PERIOD 30000 // 30s
#define NOW_UPDATE_TIMEOUT_PERIOD 10000 // 15s
#define NOW_UPDATE_TRICKLE_PERIOD 20000 // 30s

static BOOLEAN UIInfo(S_ACB_UI_INFO *info);
static void ScheduleUpdateHandler(S_APP_SI_EIT_SCHED_UPDATE *update);
static std::string GetGenreName(ADB_EVENT_CONTENT content);
static ADB_SERVICE_LIST_TYPE GetServiceListType(std::string identifier);
static void WriteFrontend(ResponseData::Object *object, U8BIT path);
static void WriteService(ResponseData::Object *object, const void *handle);
static void WriteEvent(ResponseData::Object *object, const void *handle);
static void WriteList(ResponseData::Object *object, ADB_SERVICE_LIST_TYPE type);

static void *g_eit_dirty_regions_mutex = NULL;
static std::map<std::string, std::pair<uint32_t, uint32_t> > *g_eit_dirty_regions = NULL;
static void *g_sched_updated_latch_queue = NULL;
static void *g_sched_updated_latch_task = NULL;
static void *g_now_updated_latch_queue = NULL;
static void *g_now_updated_latch_task = NULL;

DvbInterface::DvbInterface()
{
   if (ResourceManager::acquire(ResourceManager::InitialiseDvb, this, NULL, 100))
   {
      APP_InitialiseDVB(EventManager::event, CONFIG_TELETEXT ? DVB_INIT_TELETEXT_AND_SUBTITLES : DVB_INIT_SUBTITLES_ONLY);

      APP_RegisterUIInfoCallback(&UIInfo);

      if ((BOOLEAN)APP_NvmRead(FIRST_BOOT_NVM))
      {
         ACFG_SetCountry(CONFIG_COUNTRY);
      }

      U8BIT region_id = ACFG_GetRegionId();
      //FIXME: workaround for selinux failure
      /*
      char value[PROPERTY_VALUE_MAX];
      if (property_get("persist.vendor.dvb.regionid", value, NULL) > 0) {
         region_id =(U8BIT)atoi(value);
      }
      */
      ACFG_SetCountryIds(CONFIG_COUNTRY, region_id, ACFG_GetPrimaryAudioLangId(),
         ACFG_GetPrimaryTextLangId());

      U8BIT code_ptr;
      BOOLEAN valid = ACFG_GetRegionCode(CONFIG_COUNTRY, region_id, &code_ptr);
      printf("DVB region ID: 0x%02X %s\n", region_id, (TRUE == valid) ? "" : "(invalid)");
   }

   EventManager::addEventHandler(DvbInterface::eventHandler);

   if (!g_eit_dirty_regions_mutex)
   {
      g_eit_dirty_regions_mutex = STB_OSCreateMutex();
   }

   if (!g_sched_updated_latch_queue)
   {
      g_sched_updated_latch_queue = (void *)STB_OSCreateQueue(sizeof(U8BIT), 100);
   }

   if (!g_sched_updated_latch_task)
   {
      g_sched_updated_latch_task = STB_OSCreateTask(DvbInterface::schedUpdatedLatchTask, NULL, 4096, 7, (U8BIT *)"SchedUpdatedLatchTask");
   }

   if (!g_now_updated_latch_queue)
   {
      g_now_updated_latch_queue = (void *)STB_OSCreateQueue(sizeof(U8BIT), 100);
   }

   if (!g_now_updated_latch_task)
   {
      g_now_updated_latch_task = STB_OSCreateTask(DvbInterface::nowUpdatedLatchTask, NULL, 4096, 7, (U8BIT *)"NowUpdatedLatchTask");
   }

   addInvokable("getFrontend", &DvbInterface::getFrontend);
   addInvokable("getListOfFrontends", &DvbInterface::getListOfFrontends);
   addInvokable("addServiceList", &DvbInterface::addServiceList);
   addInvokable("removeServiceList", &DvbInterface::removeServiceList);
   addInvokable("getListOfServiceLists", &DvbInterface::getListOfServiceLists);
   addInvokable("addService", &DvbInterface::addService);
   addInvokable("removeService", &DvbInterface::removeService);
   addInvokable("getService", &DvbInterface::getService);
   addInvokable("setServiceBlocked", &DvbInterface::setServiceBlocked);
   addInvokable("getNumberOfServices", &DvbInterface::getNumberOfServices);
   addInvokable("getListOfServices", &DvbInterface::getListOfServices);
   addInvokable("getEvent", &DvbInterface::getEvent);
   addInvokable("getNowNextEvents", &DvbInterface::getNowNextEvents);
   addInvokable("getListOfEvents", &DvbInterface::getListOfEvents);
   addInvokable("getParentalControlOn", &DvbInterface::getParentalControlOn);
   addInvokable("setParentalControlOn", &DvbInterface::setParentalControlOn);
   addInvokable("getParentalControlAge", &DvbInterface::getParentalControlAge);
   addInvokable("setParentalControlAge", &DvbInterface::setParentalControlAge);
   addInvokable("getListOfUpdatedEventPeriods", &DvbInterface::getListOfUpdatedEventPeriods);

   #if CONFIG_EIT_MONITOR
   ASI_RegisterEitSchedUpdateCallback(ScheduleUpdateHandler);
   #endif

   // ASI_SetUpdateBatFunction(F_BatTableUpdate update_func);
   // FIXME: configured by TV input service
   U16BIT bouquet_ids[1]= {25149 /*CNS default bouguet ID*/};
   ASI_EnableBatCollection(TRUE, bouquet_ids, 1);
}

/**
 * @ingroup dvbmetaapi
 * @brief Get frontend.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getFrontend(Int32 path = INVALID_RES_ID)
 * ~~~
 * - __path__ Number (value 0 to UINT8_MAX).
 * @param response Used to write _frontend object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "frontendpath": 0,
 *    "frequency": 0,
 *    "integrity": 0,
 *    "strength": 0,
 *    "type": "none"
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::getFrontend(RequestData* request, ResponseData* response)
{
   bool ok = false;

   U8BIT path = request->shiftInt32(INVALID_RES_ID, 0, UINT8_MAX);
   if (path < STB_HWGetTunerPaths())
   {
      WriteFrontend(response->setObject(FRONTEND_OBJECT_SIZE), path);
      ok = true;
   }
   else
   {
      response->setString("Invalid path");
   }

   return ok;
}

/**
 * @ingroup dvbmetaapi
 * @brief Get list of all frontends.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getListOfFrontends()
 * ~~~
 * @param response Used to write _array of frontend objects_ response (example pseudocode):
 * ~~~{.js}
 * [
 *    {
 *       "frontendpath": 0,
 *       "frequency": 0,
 *       "integrity": 0,
 *       "strength": 0,
 *       "type": "none"
 *    },
 *    ...
 * ]
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::getListOfFrontends(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   U8BIT length = STB_HWGetTunerPaths();
   ResponseData::Array* array = response->setArray(length);
   for (U8BIT path = 0; path < length; path++)
   {
      WriteFrontend(array->appendObject(FRONTEND_OBJECT_SIZE), path);
   }
   array->finish();

   return true;
}

/**
 * @ingroup dvbmetaapi
 * @brief Add a user-defined service list (e.g. a favourite list) named _name_.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * addServiceList(string name = "")
 * ~~~
 * @param response Used to write _service list object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "identifier": "0x08",
 *    "name": "My Favourites",
 *    "size": 0,
 *    "writable": true
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::addServiceList(RequestData* request, ResponseData* response)
{
   bool ok = false;

   U8BIT favlist;
   U32BIT user_data = 0;
   std::string name = request->shiftString("");
   if (name != "" && ADB_AddFavouriteList((U8BIT*)name.c_str(), user_data, FAVLIST_INSERT_AT_END, &favlist))
   {
      WriteList(response->setObject(LIST_OBJECT_SIZE), static_cast<ADB_SERVICE_LIST_TYPE>(
         ADB_LIST_TYPE_FROM_FAVLIST(favlist)));
      ok = true;
   }
   else
   {
      response->setString("Failed");
   }

   ADB_SaveDatabase();

   return ok;
}

/**
 * @ingroup dvbmetaapi
 * @brief Remove the user-defined service list _identifier_. The list must be user-defined.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * removeServiceList(string identifier = "")
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::removeServiceList(RequestData* request, ResponseData* response)
{
   bool ok = false;

   ADB_SERVICE_LIST_TYPE type = GetServiceListType(request->shiftString(""));
   if (type & ADB_SERVICE_LIST_FAV_LIST)
   {
      ADB_DeleteFavouriteList(ADB_FAVLIST_FROM_LIST_TYPE(type));
      response->setBool(true);
      ok = true;
   }
   else
   {
      response->setString("Invalid identifier");
   }

   ADB_SaveDatabase();

   return ok;
}

/**
 * @ingroup dvbmetaapi
 * @brief Get list of all service lists.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getListOfServiceLists()
 * ~~~
 * @param response Used to write _array of service list objects_ response (example pseudocode):
 * ~~~{.js}
 * [
 *    {
 *       "identifier": "all",
 *       "name": "All",
 *       "size": 18,
 *       "type": "default"
 *    },
 *    ...
 * ]
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::getListOfServiceLists(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   U16BIT length = ADB_GetNumFavouriteLists();
   ResponseData::Array* array = response->setArray(length + 4);
   WriteList(array->appendObject(LIST_OBJECT_SIZE), ADB_SERVICE_LIST_ALL);
   WriteList(array->appendObject(LIST_OBJECT_SIZE), ADB_SERVICE_LIST_TV);
   WriteList(array->appendObject(LIST_OBJECT_SIZE), ADB_SERVICE_LIST_RADIO);
   WriteList(array->appendObject(LIST_OBJECT_SIZE), ADB_SERVICE_LIST_DATA);
   for (U16BIT i = 0; i < length; i++)
   {
      WriteList(array->appendObject(LIST_OBJECT_SIZE), static_cast<ADB_SERVICE_LIST_TYPE>(
         ADB_LIST_TYPE_FROM_FAVLIST(ADB_GetFavouriteListIdByIndex(i))));
   }
   array->finish();

   return true;
}

/**
 * @ingroup dvbmetaapi
 * @brief Add service _uri_ to the user-defined service list _identifier_.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * addService(string identifier = "unknown", string uri = NULL_SERVICE_URI)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::addService(RequestData* request, ResponseData* response)
{
   bool ok = false;

   ADB_SERVICE_LIST_TYPE list = GetServiceListType(request->shiftString("unknown"));
   void *service = Entities::getServiceHandle(request->shiftString(NULL_SERVICE_URI));
   if (list & ADB_SERVICE_LIST_FAV_LIST)
   {
      if (service)
      {
         ok = ADB_AddServiceToFavouriteList(ADB_FAVLIST_FROM_LIST_TYPE(list), service, FAVLIST_INSERT_AT_END);
         response->setBool(ok);
      }
      else
      {
         response->setString("Invalid uri");
      }
   }
   else
   {
      response->setString("Invalid identifier");
   }

   ADB_SaveDatabase();

   return ok;
}

/**
 * @ingroup dvbmetaapi
 * @brief Remove service _uri_ from the user-defined service list _identifier_.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * removeService(string identifier = "unknown", string uri = NULL_SERVICE_URI)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::removeService(RequestData* request, ResponseData* response)
{
   bool ok = false;

   ADB_SERVICE_LIST_TYPE list = GetServiceListType(request->shiftString("unknown"));
   void *service = Entities::getServiceHandle(request->shiftString(NULL_SERVICE_URI));
   if (list & ADB_SERVICE_LIST_FAV_LIST)
   {
      if (service)
      {
         ADB_DeleteServiceFromFavouriteList(ADB_FAVLIST_FROM_LIST_TYPE(list), service);
         ok = true;
         response->setBool(ok);
      }
      else
      {
         response->setString("Invalid uri");
      }
   }
   else
   {
      response->setString("Invalid identifier");
   }

   ADB_SaveDatabase();

   return ok;
}

/**
 * @ingroup dvbmetaapi
 * @brief Get the service _uri_.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getService(string uri = NULL_SERVICE_URI)
 * ~~~
 * @param response Used to write _service object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "blocked": false,
 *    "lcn": 9,
 *    "name": "BBC FOUR",
 *    "radio": false,
 *    "subtitles": false,
 *    "uri": "dvb://233a.4000.41c0"
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::getService(RequestData* request, ResponseData* response)
{
   bool ok = false;

   void *service = Entities::getServiceHandle(request->shiftString(NULL_SERVICE_URI));
   if (service)
   {
      WriteService(response->setObject(SERVICE_OBJECT_SIZE), service);
      ok = true;
   }
   else
   {
      response->setString("Invalid uri");
   }

   return ok;
}

/**
 * @ingroup dvbmetaapi
 * @brief Set service _uri_ parental control lock to _locked_.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setServiceBlocked(string uri = NULL_SERVICE_URI, bool blocked = true)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::setServiceBlocked(RequestData* request, ResponseData* response)
{
   bool ok = false;
   void *service = Entities::getServiceHandle(request->shiftString(NULL_SERVICE_URI));
   if (service)
   {
      ADB_SetServiceLockedFlag(service, request->shiftBool(true));
      response->setBool(true);
      ok = true;
   }
   else
   {
      response->setString("Invalid uri");
   }

   return ok;
}

/**
 * @ingroup dvbmetaapi
 * @brief Get number of services in list _identifier_.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getNumberOfServices(string identifier = "all")
 * ~~~
 * @param response Used to write _number Int32_ response (example pseudocode):
 * ~~~{.js}
 * 18
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::getNumberOfServices(RequestData* request, ResponseData* response)
{
   BOOLEAN include_hidden = FALSE;
   response->setInt32(ADB_GetNumServicesInList(GetServiceListType(request->shiftString("all")), include_hidden));
   return true;
}

/**
 * @ingroup dvbmetaapi
 * @brief Get list of services in list _identifier_.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getListOfServices(string identifier = "all")
 * ~~~
 * @param response Used to write _array of service objects_ response (example pseudocode):
 * ~~~{.js}
 * [
 *    {
 *       "blocked": true,
 *       "lcn": 9,
 *       "name": "BBC FOUR",
 *       "radio": false,
 *       "subtitles": false,
 *       "uri": "dvb://233a.4000.41c0"
 *    },
 *    ...
 * ]
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::getListOfServices(RequestData* request, ResponseData* response)
{
   void **list = NULL;
   U16BIT length = 0;
   ADB_GetServiceListIncludingHidden(GetServiceListType(request->shiftString("all")), &list, &length, FALSE);
   ResponseData::Array* array = response->setArray(length);
   for (U16BIT i = 0; i < length; i++)
   {
      WriteService(array->appendObject(SERVICE_OBJECT_SIZE), list[i]);
   }
   array->finish();

   if (list)
   {
      ADB_ReleaseServiceList(list, length);
   }

   return true;
}

/**
 * @ingroup dvbmetaapi
 * @brief Get event _uri_.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getEvent(string uri = NULL_EVENT_URI)
 * ~~~
 * @param response Used to write _event object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "ad": false,
 *    "duration": 3600,
 *    "endutc": 1205890200,
 *    "guidance": "",
 *    "name": "Auntie's War on Smut",
 *    "rating": 0,
 *    "startutc": 1205886600,
 *    "subtitles": true,
 *    "uri": "dvb://233a.4000.41c0;178c"
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::getEvent(RequestData* request, ResponseData* response)
{
   bool ok = false;

   void *event = Entities::getEventHandle(request->shiftString(NULL_EVENT_URI));
   if (event)
   {
      WriteEvent(response->setObject(EVENT_OBJECT_SIZE), event);
      Entities::deleteEventHandle(event);
      ok = true;
   }
   else
   {
      response->setString("Invalid uri");
   }

   return ok;
}

/**
 * @ingroup dvbmetaapi
 * @brief Get now and next events for service _uri_.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getNowNextEvents(string uri = NULL_EVENT_URI)
 * ~~~
 * @param response Used to write _now next events object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "ad": false,
 *    "duration": 3600,
 *    "endutc": 1205890200,
 *    "guidance": "",
 *    "name": "Auntie's War on Smut",
 *    "rating": 0,
 *    "startutc": 1205886600,
 *    "subtitles": true,
 *    "uri": "dvb://233a.4000.41c0;178c"
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::getNowNextEvents(RequestData* request, ResponseData* response)
{
   bool ok = false;

   void *service = Entities::getServiceHandle(request->shiftString(NULL_SERVICE_URI));
   if (service)
   {
      ResponseData::Object *object = response->setObject(2);

      void *now, *next;
      ADB_GetNowNextEvents(service, &now, &next);

      if (now != NULL)
      {
         WriteEvent(object->insertObject("now", EVENT_OBJECT_SIZE), now);
         ADB_ReleaseEventData(now);
      }
      else
      {
         object->insertBool("now", false);
      }

      if (next != NULL)
      {
         WriteEvent(object->insertObject("next", EVENT_OBJECT_SIZE), next);
         ADB_ReleaseEventData(next);
      }
      else
      {
         object->insertBool("next", false);
      }

      ok = true;
      object->finish();
   }
   else
   {
      response->setString("Invalid uri");
   }

   return ok;
}

/**
 * @ingroup dvbmetaapi
 * @brief Get list of events in schedule for service _uri_, that end after _startutc_ and start on or before _endutc_.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getListOfEvents(string uri = NULL_SERVICE_URI, UInt32 startutc = 0, UInt32 endutc = 4294967295)
 * ~~~
 * @param response Used to write _array of event objects_ response (example pseudocode):
 * ~~~{.js}
 * [
 *    {
 *       "ad": false,
 *       "duration": 3600,
 *       "endutc": 1205890200,
 *       "guidance": "",
 *       "name": "Auntie's War on Smut",
 *       "rating": 0,
 *       "startutc": 1205886600,
 *       "subtitles": true,
 *       "uri": "dvb://233a.4000.41c0;178c"
 *    },
 *    ...
 * ]
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::getListOfEvents(RequestData* request, ResponseData* response)
{
   bool ok = false;

   void *service = Entities::getServiceHandle(request->shiftString(NULL_SERVICE_URI));

   U32BIT startutc = request->shiftUInt32(0);
   U32BIT endutc = request->shiftUInt32(UINT32_MAX);

   if (service != NULL)
   {
      void **list = NULL;
      U16BIT length;
      BOOLEAN include_old = FALSE;

      ADB_GetEventSchedule(include_old, service, &list, &length);

      U16BIT selectionindex = 0;
      U16BIT selectionlength = length;

      if (startutc != 0 || endutc != UINT32_MAX)
      {
         selectionlength = 0;
         for (U16BIT i = 0; i < length; i++)
         {
            if (STB_GCConvertToTimestamp(ADB_GetEventStartDateTime(list[i])) > endutc)
            {
               break;
            }
            else if (STB_GCConvertToTimestamp(ADB_GetEventEndDateTime(list[i])) > startutc)
            {
               if (++selectionlength == 1)
               {
                  selectionindex = i;
               }
            }
         }
      }

      ResponseData::Array* array = response->setArray(selectionlength);
      for (U16BIT i = selectionindex; i < selectionindex + selectionlength; i++)
      {
         WriteEvent(array->appendObject(EVENT_OBJECT_SIZE), list[i]);
      }
      array->finish();

      ADB_ReleaseEventList(list, length);

      ok = true;
   }
   else
   {
      response->setString("Invalid uri");
   }

   return ok;
}

/**
 * @ingroup dvbmetaapi
 * @brief Get parental control on.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getParentalControlOn()
 * ~~~
 * @param response Used to write _on bool_ response (example pseudocode):
 * ~~~{.js}
 * false
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::getParentalControlOn(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   response->setBool(ACTL_ParentalControlEnabled());
   return true;
}

/**
 * @ingroup dvbmetaapi
 * @brief Set parental control on.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setParentalControlOn(bool on = true)
 * ~~~
 * @param response Used to write _success_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::setParentalControlOn(RequestData* request, ResponseData* response)
{
   ACTL_SetParentalControl(request->shiftBool(true));
   response->setBool(true);
   return true;
}

/**
 * @ingroup dvbmetaapi
 * @brief Get parental control age.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getParentalControlAge()
 * ~~~
 * @param response Used to write _age Int32_ response (example pseudocode):
 * ~~~{.js}
 * 18
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::getParentalControlAge(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   response->setInt32(ACTL_GetParentalControlAge());
   return true;
}

/**
 * @ingroup dvbmetaapi
 * @brief Set parental control age.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setParentalControlAge(Int32 age = 18)
 * ~~~
 * - __age__ Number (value 4 to 18).
 * @param response Used to write _success_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbInterface::setParentalControlAge(RequestData* request, ResponseData* response)
{
   ACTL_SetParentalControlAge(request->shiftInt32(4, 4, 18));
   response->setBool(true);
   return true;
}

// TODO comment
bool DvbInterface::getListOfUpdatedEventPeriods(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   STB_OSMutexLock(g_eit_dirty_regions_mutex);
   std::map<std::string, std::pair<uint32_t, uint32_t> > *eit_dirty_regions = g_eit_dirty_regions;
   g_eit_dirty_regions = NULL;
   STB_OSMutexUnlock(g_eit_dirty_regions_mutex);

   if (eit_dirty_regions)
   {
      ResponseData::Array* array = response->setArray(eit_dirty_regions->size());

      for (std::map<std::string, std::pair<uint32_t, uint32_t> >::iterator it = eit_dirty_regions->begin(); it != eit_dirty_regions->end(); ++it)
      {
         ResponseData::Object *object = array->appendObject(4);
         object->insertString("uri", it->first.substr(0, 20));
         object->insertInt32("startutc", it->second.first);
         object->insertInt32("endutc", it->second.second);
         object->insertInt32("duration", it->second.second - it->second.first);
         object->finish();
      }

      array->finish();

      delete eit_dirty_regions;
   }
   else
   {
      response->setArray(0)->finish();
   }

   return true;
}

void DvbInterface::eventHandler(unsigned int code, const void *data, unsigned int data_size)
{
   USE_UNWANTED_PARAM(data);
   USE_UNWANTED_PARAM(data_size);

   static DvbInterface& instance = DvbInterface::instance();

   if (code == APP_EVENT_SERVICE_EIT_SCHED_UPDATE)
   {
      U8BIT foo = 0;
      STB_OSWriteQueue(g_sched_updated_latch_queue, (void *)&foo, sizeof(foo), TIMEOUT_NOW);
   }
   else if (code == APP_EVENT_SERVICE_EIT_NOW_UPDATE)
   {
      U8BIT foo = 0;
      STB_OSWriteQueue(g_now_updated_latch_queue, (void *)&foo, sizeof(foo), TIMEOUT_NOW);
   }
   else
   {
      switch (code)
      {
         case APP_EVENT_SERVICE_DELETED:
            instance.signal("ServiceDeleted");
            break;

         case APP_EVENT_SERVICE_ADDED:
            instance.signal("ServiceAdded");
            break;

         case APP_EVENT_SERVICE_UPDATED:
            instance.signal("ServiceUpdated");
            break;
      }
   }
}

void DvbInterface::schedUpdatedLatchTask(void *param)
{
   static DvbInterface& instance = DvbInterface::instance();
   U32BIT updated = 0;

   USE_UNWANTED_PARAM(param);

   while (1)
   {
      U8BIT foo;
      if (STB_OSReadQueue(g_sched_updated_latch_queue, &foo, sizeof(foo), SCHED_UPDATE_TIMEOUT_PERIOD))
      {
         /* The schedule was updated */
         U32BIT now = STB_OSGetClockMilliseconds();
         if (updated == 0)
         {
            updated = now;
         }
         else if ((updated + SCHED_UPDATE_TRICKLE_PERIOD) < now)
         {
            /* This sequence of latched events started SCHED_UPDATE_TRICKLE_PERIOD ago, signal update */
            instance.signal("DvbUpdatedEventPeriods");
            updated = 0;
         }
      }
      else if (updated > 0)
      {
         /* The schedule has not been updated for SCHED_UPDATE_TIMEOUT_PERIOD, signal update */
         instance.signal("DvbUpdatedEventPeriods");
         updated = 0;
      }
   }
}

void DvbInterface::nowUpdatedLatchTask(void *param)
{
   static DvbInterface& instance = DvbInterface::instance();
   U32BIT updated = 0;

   USE_UNWANTED_PARAM(param);

   while (1)
   {
      U8BIT foo;
      if (STB_OSReadQueue(g_now_updated_latch_queue, &foo, sizeof(foo), NOW_UPDATE_TIMEOUT_PERIOD))
      {
         /* The schedule was updated */
         U32BIT now = STB_OSGetClockMilliseconds();
         if (updated == 0)
         {
            updated = now;
         }
         else if ((updated + NOW_UPDATE_TRICKLE_PERIOD) < now)
         {
            /* This sequence of latched events started NOW_UPDATE_TRICKLE_PERIOD ago, signal update */
            instance.signal("DvbUpdatedEventNow");
            updated = 0;
         }
      }
      else if (updated > 0)
      {
         /* The schedule has not been updated for NOW_UPDATE_TIMEOUT_PERIOD, signal update */
         instance.signal("DvbUpdatedEventNow");
         updated = 0;
      }
   }
}

static BOOLEAN UIInfo(S_ACB_UI_INFO *info)
{
   BOOLEAN result = TRUE;

   if (info)
   {
      switch (info->type)
      {
         case ACB_GET_UI_LANG_PREF:
            info->u.ui_lang_id = ACFG_DB_LANG_ENGLISH;
            break;
         case ACB_GET_SUBTITLE_PREF:
            info->u.subtitle.pref = FALSE; // TODO
            break;
         case ACB_GET_AD_PREF:
            info->u.ad.pref = FALSE; // TODO
            break;
         default:
            result = FALSE;
            break;
      }
   }

   return result;
}

static void ScheduleUpdateHandler(S_APP_SI_EIT_SCHED_UPDATE *update)
{
   STB_OSMutexLock(g_eit_dirty_regions_mutex);

   if (g_eit_dirty_regions == NULL)
   {
      g_eit_dirty_regions = new std::map<std::string, std::pair<uint32_t, uint32_t> >();
   }

   void *service;

   if ((service = ADB_FindServiceByIds(update->orig_net_id, update->tran_id, update->serv_id)) != NULL)
   {
      void *event;

      if (update->type == APP_SI_EIT_JOURNAL_TYPE_CLEAR)
      {
         char key[64];
         for (int i = 0; i < CONFIG_EIT_MONITOR_CHUNK_NUMBER; i++)
         {
            snprintf(key, sizeof(key), "dvb://%04hx.%04hx.%04hx/%d", update->orig_net_id, update->tran_id, update->serv_id, i);
            g_eit_dirty_regions->erase(key);
         }
      }
      else if ((event = ADB_GetEvent(service, update->event_id)) != NULL)
      {
         U32DHMS start = ADB_GetEventStartDateTime(event);
         uint32_t value = STB_GCConvertToTimestamp(start);
         char key[64];

         snprintf(key, sizeof(key), "dvb://%04hx.%04hx.%04hx/%d", update->orig_net_id, update->tran_id, update->serv_id,
            (value / (CONFIG_EIT_MONITOR_CHUNK_WINDOW / CONFIG_EIT_MONITOR_CHUNK_NUMBER)) % CONFIG_EIT_MONITOR_CHUNK_NUMBER);

         std::map<std::string, std::pair<uint32_t, uint32_t> >::iterator it = g_eit_dirty_regions->find(key);
         if (it != g_eit_dirty_regions->end())
         {

            if (value < it->second.first)
            {
               it->second = std::make_pair(value, it->second.second);
            }
            else if ((value = STB_GCConvertToTimestamp(start + ADB_GetEventDuration(event))) > it->second.second)
            {
               it->second = std::make_pair(it->second.first, value);
            }
         }
         else
         {
            (*g_eit_dirty_regions)[key] = std::make_pair(STB_GCConvertToTimestamp(start),
               STB_GCConvertToTimestamp(start + ADB_GetEventDuration(event)));
         }

         ADB_ReleaseEventData(event);
      }
   }

   STB_OSMutexUnlock(g_eit_dirty_regions_mutex);
}

static std::string GetGenreName(ADB_EVENT_CONTENT content)
{
   std::string name;

   switch (content)
   {
      case ADB_EVENT_CONTENT_MOVIE:
         name = "movies";
         break;
      case ADB_EVENT_CONTENT_NEWS:
         name = "news";
         break;
      case ADB_EVENT_CONTENT_ENTERTAINMENT:
         name = "entertainment";
         break;
      case ADB_EVENT_CONTENT_SPORT:
         name = "sport";
         break;
      case ADB_EVENT_CONTENT_CHILD:
         name = "childrens";
         break;
      case ADB_EVENT_CONTENT_MUSIC:
         name = "music";
         break;
      case ADB_EVENT_CONTENT_ARTS:
         name = "arts";
         break;
      case ADB_EVENT_CONTENT_SOCIAL:
         name = "social";
         break;
      case ADB_EVENT_CONTENT_EDUCATION:
         name = "education";
         break;
      case ADB_EVENT_CONTENT_LEISURE:
         name = "leisure";
         break;
      default:
         name = "unknown";
         break;
   }

   return name;
}

static ADB_SERVICE_LIST_TYPE GetServiceListType(std::string identifier)
{
   U8BIT favlist;
   if (identifier == "all")
   {
      return ADB_SERVICE_LIST_ALL;
   }
   if (std::sscanf(identifier.c_str(), "0x%02hhx", &favlist) == 1)
   {
      return (ADB_SERVICE_LIST_TYPE)ADB_LIST_TYPE_FROM_FAVLIST(favlist);
   }
   if (identifier == "tv")
   {
      return ADB_SERVICE_LIST_TV;
   }
   if (identifier == "radio")
   {
      return ADB_SERVICE_LIST_RADIO;
   }
   if (identifier == "data")
   {
      return ADB_SERVICE_LIST_DATA;
   }

   return ADB_SERVICE_LIST_UNKNOWN;
}

static void WriteFrontend(ResponseData::Object *object, U8BIT path)
{
   // If changing object size, change FRONTEND_OBJECT_SIZE

   std::string type("none");
   switch (STB_TuneGetSignalType(path))
   {
      case TUNE_SIGNAL_COFDM:
         type = "dvbt";
         break;
      case TUNE_SIGNAL_QAM:
         type = "dvbc";
         break;
      case TUNE_SIGNAL_QPSK:
         type = "dvbs";
         break;
      default:
         break;
   }

   U32BIT frequency = 0;
   U8BIT decoderpath;
   if ((decoderpath = STB_DPPathForTuner(INVALID_RES_ID, path)) != INVALID_RES_ID)
   {
      frequency = STB_DPGetFrequency(decoderpath);
   }

   object->insertInt32("frontendpath", path);
   object->insertString("type", type);
   object->insertInt32("frequency", frequency);
   object->insertInt32("strength", STB_TuneGetSignalStrength(path));
   object->insertInt32("integrity", STB_TuneGetDataIntegrity(path));
   object->finish();
}

static void WriteService(ResponseData::Object *object, const void *chandle)
{
   // If changing object size, change SERVICE_OBJECT_SIZE

   ASSERT(chandle);

   void *handle = const_cast<void *>(chandle);

   BOOLEAN use_pref_name = TRUE;
   U8BIT *name = ADB_GetServiceFullName(handle, use_pref_name);
   U16BIT lcn = ADB_GetServiceLcn(handle);
   ADB_SERVICE_TYPE service_type = ADB_GetServiceType(handle);
   BOOLEAN radio = (service_type == ADB_SERVICE_TYPE_RADIO || service_type == ADB_SERVICE_TYPE_AVC_RADIO);
   BOOLEAN notused;
   BOOLEAN subtitles = ADB_ServiceHasSubtitles(handle, &notused);
   BOOLEAN blocked = ADB_GetServiceLockedFlag(handle);
   bool hidden = (TRUE == ADB_GetServiceHiddenFlag(handle));
   bool selectable = (TRUE == ADB_GetServiceSelectableFlag(handle));
   U32BIT group_mask = ADB_GetServiceGroupMask(handle);
   BOOLEAN is_signal2 = false;
   E_STB_DP_SIGNAL_TYPE signal_type = ADB_GetServiceSignalType(handle, &is_signal2);
   std::string type("none");
   switch (signal_type) {
      case SIGNAL_COFDM:
         type = "dvbt";
         break;
      case SIGNAL_QAM:
         type = "dvbc";
         break;
      case SIGNAL_QPSK:
         type = "dvbs";
         break;
      default:
         break;
   }
   if (TRUE == is_signal2)
        type += "2";

   // FIXME: hide the data broadcast service (0x0C) and
   //        the teletext service (0x03)
   if ((ADB_SERVICE_TYPE_DATA == service_type) ||
        (ADB_SERVICE_TYPE_TELETEXT == service_type))
   {
        hidden = true;
   }

   object->insertString("uri", Entities::getServiceUri(handle));
   object->insertString("name", Entities::parseDvbString(name));
   object->insertInt32("lcn", lcn);
   object->insertBool("radio", radio);  // FIXME: replaced by "servicetype"
   object->insertBool("subtitles", subtitles);
   object->insertBool("blocked", blocked);
   object->insertInt32("servicetype", service_type);
   object->insertString("type", type);
   object->insertBool("hidden", hidden);
   object->insertBool("selectable", selectable);

   /* CNS channel groups */
   {
      const U32BIT group[] = {
         0x100000,
         0x08000,
         0x00100,
         0x10000,
         0x01000,
         0x00800,
         0x00200,
         0x00400,
         0x04000,
         0x80000,
         0x02000,
         0x40000
      };
      const char * group_name_chi[] = {
         "多螢",
         "購物",
         "公益 / 宗教",
         "外語 / 學習",
         "兒童 / 卡通",
         "休閒 / 新知",
         "戲劇 / 音樂",
         "新聞 / 財經",
         "綜合",
         "成人",
         "電影 / 影集",
         "運動 / 其他"
      };
      const char * group_name_eng[] = {
         "at Home",
		   "Home Shopping",
		   "Public Welfare / Religion",
		   "Foreign Language / Learning",
		   "Chidren / Animation",
		   "Leisure / Knowedge",
		   "Drama / Music",
		   "News / Finance",
		   "Wariety",
		   "Adult",
		   "Films / Series",
		   "Sport / Others"
      };
      int group_size = sizeof(group) / sizeof(U32BIT); 
      int count = 0;
	   ResponseData::Array *array = NULL;
      int i;
	   	   
      if(group_mask != 0) {
   	   for (i = 0; i < group_size; i++) {
            if ((group_mask & group[i]) != 0)
		         count ++;
         }
      }
      array = object->insertArray("groups", count);
      if ((count != 0) && (array != NULL)) {
         for (i = 0; i < group_size; i++) {
            if ((group_mask & group[i]) != 0) {
               ResponseData::Object* groupObject = array->appendObject(2);
               if (NULL != groupObject) {
                  groupObject->insertUInt32("id", group[i]);
                  ResponseData::Object* nameObject = groupObject->insertObject("name", 2);
                  if (NULL != nameObject) {
                        nameObject->insertString("chi", group_name_chi[i]);
                        nameObject->insertString("eng", group_name_eng[i]);
                        nameObject->finish();
                  }
                  groupObject->finish();
               }
            }
         }
         array->finish(); 
      }
   }

   if (name)
   {
      STB_ReleaseUnicodeString(name);
   }

   object->finish();
}

static void WriteEvent(ResponseData::Object *object, const void *chandle)
{
   // If changing object size, change EVENT_OBJECT_SIZE

   ASSERT(chandle);

   void *handle = const_cast<void *>(chandle);
   U8BIT *name = ADB_GetEventName(handle);
   U8BIT *description = ADB_GetEventDescription(handle);
   U8BIT *long_description = ADB_GetEventExtendedDescription(handle);
   ADB_EVENT_CONTENT content = ADB_GetEventContentDesc(handle);
   U8BIT notused;
   U8BIT *guidance = ADB_GetEventGuidance(handle, ADB_GetEventService(handle), &notused, &notused);
   U32BIT startutc = STB_GCConvertToTimestamp(ADB_GetEventStartDateTime(handle));
   U32BIT endutc = STB_GCConvertToTimestamp(ADB_GetEventEndDateTime(handle));
   U32DHMS dhms = ADB_GetEventDuration(handle);
   U32BIT duration = ((DHMS_DAYS(dhms) * 24 + DHMS_HOUR(dhms)) * 60 + DHMS_MINS(dhms)) * 60 + DHMS_SECS(dhms);
   BOOLEAN subtitles = ADB_GetEventSubtitlesAvailFlag(handle);
   BOOLEAN ad = ADB_GetEventAudioDescriptionFlag(handle);
   U32BIT rating = ADB_GetEventParentalAge(handle);

   U8BIT *episode_name = NULL;
   S32BIT epispode_number = 0;
   U8BIT content_status = 0;
   BOOLEAN last_episode = TRUE;
   BOOLEAN has_series_info;
   has_series_info = ADB_GetEventSeriesInfo(handle, &episode_name, &epispode_number, &content_status, &last_episode);

   object->insertString("uri", Entities::getEventUri(handle));
   object->insertString("name", Entities::parseDvbString(name));
   object->insertString("description", Entities::parseDvbString(description));
   object->insertString("long_description", Entities::parseDvbString(long_description));
   object->insertString("genre", GetGenreName(content));
   object->insertString("guidance", Entities::parseDvbString(guidance));
   object->insertUInt32("startutc", startutc);
   object->insertUInt32("endutc", endutc);
   object->insertInt32("duration", duration);
   object->insertBool("subtitles", subtitles);
   object->insertBool("ad", ad);
   object->insertInt32("rating", rating);
   if (TRUE == has_series_info) {
      object->insertString("episode_title", Entities::parseDvbString(episode_name));
      if (TRUE == last_episode)
      {
         if (epispode_number > 0)
            epispode_number = -epispode_number; //  a negative value to indicate the last episode number
      }
      object->insertInt32("episode_number", epispode_number);
      if (NULL != episode_name)
      {
         STB_ReleaseUnicodeString(episode_name);
         episode_name = NULL;
      }
   }
   object->finish();

   if (name)
   {
      STB_ReleaseUnicodeString(name);
   }

   if (description)
   {
      STB_ReleaseUnicodeString(description);
   }

   if (long_description)
   {
      STB_ReleaseUnicodeString(long_description);
   }

   if (guidance)
   {
      STB_ReleaseUnicodeString(guidance);
   }
}

static void WriteList(ResponseData::Object *object, ADB_SERVICE_LIST_TYPE type)
{
   // If changing object size, change LIST_OBJECT_SIZE

   std::string name;
   std::string identifier;
   switch (type & 0xFFFF)
   {
      case ADB_SERVICE_LIST_ALL:
         name = "All";
         identifier = "all";
         break;
      case ADB_SERVICE_LIST_TV:
         name = "TV";
         identifier = "tv";
         break;
      case ADB_SERVICE_LIST_RADIO:
         name = "Radio";
         identifier = "radio";
         break;
      case ADB_SERVICE_LIST_DATA:
         name = "Data";
         identifier = "data";
         break;
      case ADB_SERVICE_LIST_FAV_LIST:
         U8BIT *tmp_name;
         if ((tmp_name = ADB_GetFavouriteListName(ADB_FAVLIST_FROM_LIST_TYPE(type))))
         {
            name = Entities::parseDvbString(tmp_name);
            STB_ReleaseUnicodeString(tmp_name);
         }
         char tmp_identifier[5];
         if (snprintf(tmp_identifier, sizeof(tmp_identifier), "0x%02hhx", ADB_FAVLIST_FROM_LIST_TYPE(type)) == 4)
         {
            identifier = tmp_identifier;
         }
         break;
   }

   BOOLEAN include_hidden = FALSE;
   object->insertString("name", name);
   object->insertString("identifier", identifier);
   object->insertBool("writable", type & ADB_SERVICE_LIST_FAV_LIST);
   object->insertInt32("size", ADB_GetNumServicesInList(type, include_hidden));
   object->finish();
}

