#include "Glue.h"

#include "ResourceManager.h"
#include "EventManager.h"
#include "Entities.h"
#include "common.h"
#include <cstdio>
#include <cstring>
#include <sstream>

#include "DvbcInterface.h"

static void ReleaseActivePath(ResourceManager::Resource resource);
static E_STB_DP_CMODE GetModulation(const std::string &modulation);
static E_STB_DP_TBWIDTH GetBandwidth(const std::string &str);

static BOOLEAN g_started = FALSE;
static BOOLEAN g_clear_new_flags = FALSE;
static U8BIT g_frontendid = INVALID_RES_ID;
static U32BIT g_bit_error_rate = 0;  /*1.0E-7*/
static U32BIT g_signal_noise_ratio = 0;  /*dB*/
static S16BIT g_signal_strength = 0;  /*dBuV*/
static U8BIT g_progress = 0;  /*[0 - 100]*/
static BOOLEAN g_manual = FALSE;

DvbcInterface::DvbcInterface()
{
   EventManager::addEventHandler(DvbcInterface::eventHandler);

   addInvokable("getStatus", &DvbcInterface::getStatus);
   addInvokable("getSignalStatus", &DvbcInterface::getSignalStatus);
   addInvokable("startSearch", &DvbcInterface::startSearch);
   addInvokable("startManualSearch", &DvbcInterface::startManualSearch);
   addInvokable("finishSearch", &DvbcInterface::finishSearch);
   addInvokable("startSignalDetection", &DvbcInterface::startSignalDetection);
   addInvokable("stopSignalDetection", &DvbcInterface::stopSignalDetection);
}

/**
 * @ingroup dvbcmetaapi
 * @brief Get status.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getStatus()
 * ~~~
 * @param response Used to write _status object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "frontendpath": 0,
 *    "progress": 70,
 *    "started": true
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbcInterface::getStatus(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   ResponseData::Object *object = response->setObject(6);
   object->insertBool("started", g_started);
   object->insertInt32("frontendpath", g_frontendid);
   object->insertInt32("progress", g_progress);
   object->insertInt32("bit-error-rate", (int32_t)g_bit_error_rate);
   object->insertInt32("signal-noise-ratio", (int32_t)g_signal_noise_ratio);
   object->insertInt32("signal-strength", (int32_t)g_signal_strength);
   object->finish();

   return true;
}

/**
 * @ingroup dvbcmetaapi
* @brief Get signal quality.
* @param request Used to read arguments for the invokable (pseudocode):
* ~~~{.java}
* getSignalStatus()
* ~~~
* @param response Used to write _status object_ response (example pseudocode):
* ~~~{.js}
* {
*    "frontendpath": 0,
*    "progress": 70,
*    "started": true
* }
* ~~~
* @return Success (on failure, an error string is written to response)
*/
bool DvbcInterface::getSignalStatus(RequestData* request, ResponseData* response)
{
   U8BIT path = ACTL_GetSearchPath();
   g_bit_error_rate = ACTL_GetSignalBitErrorRate(path);
   g_signal_noise_ratio = ACTL_GetSignalNoiseRatio(path);
   g_signal_strength = ACTL_GetSignalStrength(path);
   
   return getStatus(request, response);
}

/**
 * @ingroup dvbcmetaapi
 * @brief Start search. Signals DvbcStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * startSearch(clear_old_search = false)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbcInterface::startSearch(RequestData* request, ResponseData* response)
{
   bool ok = false;

   // arg1 clear_old_search
   bool clear_old_search = request->shiftBool(false);
   
   if (ResourceManager::acquire(ResourceManager::ActivePath, this, ReleaseActivePath, 100))
   {
      g_frontendid = INVALID_RES_ID;
      g_progress = 0;
      g_clear_new_flags = clear_old_search;
      g_started = TRUE;
      
      ADB_PrepareDatabaseForSearch(SIGNAL_QAM, NULL, clear_old_search, FALSE);
      
      if (ACTL_StartServiceSearch(SIGNAL_QAM, ACTL_FREQ_SEARCH))
      {
         response->setBool(true);
         ok = true;
      }
      else
      {
         // Don't save non-completed search
         ADB_FinaliseDatabaseAfterSearch(FALSE, SIGNAL_QAM, NULL, FALSE, g_clear_new_flags, FALSE);
         g_started = FALSE;
         ResourceManager::released(ResourceManager::ActivePath, this);
         response->setString("Failed");
      }
   }
   else
   {
      response->setString("Conflict");
   }

   return ok;
}

/**
 * @ingroup dvbcmetaapi
 * @brief Start manual search. Signals DvbcStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * startManualSearch(network = false, frequency = 0, modulation = "auto", symbol_rate = 0, clear_old_search = false)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbcInterface::startManualSearch(RequestData* request, ResponseData* response)
{
   bool ok = false;

   bool network = request->shiftBool(false); // arg1 network

   S_MANUAL_TUNING_PARAMS params;
   params.freq = request->shiftUInt32(0); // arg2 frequency
   params.u.cab.mode = GetModulation(request->shiftString("auto")); // arg3 modulation
   params.u.cab.symbol_rate = request->shiftInt32(0, 0, UINT16_MAX); // arg4 symbol rate   
   
   bool clear_old_search = request->shiftBool(false); // arg5 clear_old_search
      
   params.end_freq = request->shiftUInt32(0); // arg6 end frequency
   params.u.cab.bwidth = GetBandwidth(request->shiftString("6MHz")); // arg7 bandwidth

   U16BIT bouquet_ids[1]= {0};   
   bouquet_ids[0] = (U16BIT)request->shiftUInt32(25149); // arg8 bouquet id
   ASI_EnableBatCollection(TRUE, bouquet_ids, 1);
      
   std::stringstream ss;
   ss  << "Manual Search\n" 
       << "network: " << (bool)network << "\n"
       << "params.freq: " << params.freq << "\n"
       << "params.end_freq: " << params.end_freq << "\n"
       << "params.u.cab.mode: " << params.u.cab.mode << "\n"
       << "params.u.cab.symbol_rate: " << params.u.cab.symbol_rate << "\n"
       << "params.u.cab.bwidth: " << params.u.cab.bwidth << "\n"
       << "clear_old_search: " << (bool)clear_old_search << "\n"
       << "bouquet id: " << bouquet_ids[0] << std::endl;
   __android_log_print(ANDROID_LOG_INFO, "dtvkit", "%s", ss.str().c_str());

   if (ResourceManager::acquire(ResourceManager::ActivePath, this, ReleaseActivePath, 100))
   {
      g_frontendid = INVALID_RES_ID;
      g_progress = 0;
      g_clear_new_flags = clear_old_search;
      g_started = TRUE;

      ADB_PrepareDatabaseForSearch(SIGNAL_QAM, NULL, clear_old_search, FALSE);
      if (ACTL_StartManualSearch(SIGNAL_QAM, &params, network ? ACTL_NETWORK_SEARCH : ACTL_FREQ_SEARCH))
      {
         response->setBool(true);
         ok = true;
      }
      else
      {
         // Don't save non-completed search
         ADB_FinaliseDatabaseAfterSearch(FALSE, SIGNAL_QAM, NULL, FALSE, g_clear_new_flags, g_manual);
         g_started = FALSE;
         ResourceManager::released(ResourceManager::ActivePath, this);
         response->setString("Failed");
      }
   }
   else
   {
      response->setString("Conflict");
   }

   return ok;
}

/**
 * @ingroup dvbcmetaapi
 * @brief Finish search. Signals DvbcStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * finishSearch(commit = true)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbcInterface::finishSearch(RequestData* request, ResponseData* response)
{
   // arg1 commit
   bool commit = request->shiftBool(true);

   if (g_started)
   {
      g_started = FALSE;

      if (!ACTL_IsSearchComplete())
      {
         if (g_manual)
         {
            ACTL_FinishManualSearch();
         }
         else
         {
            ACTL_StopServiceSearch();
         }
      }

      ADB_FinaliseDatabaseAfterSearch(commit, SIGNAL_QAM, NULL, (g_progress == 100), g_clear_new_flags, g_manual);

      ResourceManager::released(ResourceManager::ActivePath, this);
   }

   response->setBool(true);

   return true;
}

/**
* @ingroup dvbcmetaapi
* @brief Start signal detection. Signals DvbcStatusChanged with getStatus data.
* @param request Used to read arguments for the invokable (pseudocode):
* ~~~{.java}
* startSignalDetection(frequency = 0, modulation = "auto", symbol_rate = 0)
* ~~~
* @param response Used to write _success bool_ response (example pseudocode):
* ~~~{.js}
* true
* ~~~
* @return Success (on failure, an error string is written to response)
*/
bool DvbcInterface::startSignalDetection(RequestData* request, ResponseData* response)
{
   bool ok = false;

   S_MANUAL_TUNING_PARAMS params;
   params.freq = request->shiftUInt32(0); // arg1 frequency
   params.u.cab.mode = GetModulation(request->shiftString("auto")); // arg2 modulation
   params.u.cab.symbol_rate = request->shiftInt32(0, 0, UINT16_MAX); // arg3 symbol rate   
   params.end_freq = params.freq;
   params.u.cab.bwidth = GetBandwidth(request->shiftString("6MHz")); // arg4 bandwidth
       
   std::stringstream ss;
   ss  << "Signal Detection\n" 
       << "params.freq: " << params.freq << "\n"
       << "params.u.cab.mode: " << params.u.cab.mode << "\n"
       << "params.u.cab.symbol_rate: " << params.u.cab.symbol_rate << "\n"
       << "params.u.cab.bwidth: " << params.u.cab.bwidth << "\n";
   __android_log_print(ANDROID_LOG_INFO, "dtvkit", "%s", ss.str().c_str());

   if (ResourceManager::acquire(ResourceManager::MonitorPath, this, NULL, 100))
   {
      g_frontendid = INVALID_RES_ID;
      g_progress = 0;
      g_started = TRUE;

      if (TRUE == ACTL_StartSignalDetection(SIGNAL_QAM, &params))
      {
         response->setBool(true);
         ok = true;
      }
      else
      {
         // Don't save non-completed search
         //ADB_FinaliseDatabaseAfterSearch(FALSE, SIGNAL_QAM, NULL, FALSE, g_clear_new_flags, g_manual);
         g_started = FALSE;
         ResourceManager::released(ResourceManager::MonitorPath, this);
         response->setString("Failed");
      }
   }
   else
   {
      response->setString("Conflict");
   }

    response->setBool(true);
   
    return true;
}

/**
* @ingroup dvbcmetaapi
* @brief stop signal detection. Signals DvbcStatusChanged with getStatus data.
* @param request Used to read arguments for the invokable (pseudocode):
* ~~~{.java}
* stopSignalDetection(commit = true)
* ~~~
* @param response Used to write _success bool_ response (example pseudocode):
* ~~~{.js}
* true
* ~~~
* @return Success (on failure, an error string is written to response)
*/
bool DvbcInterface::stopSignalDetection(RequestData* request, ResponseData* response)
{
    ACTL_StopSignalDetection();
    ResourceManager::released(ResourceManager::MonitorPath, this);

    response->setBool(true);
   
    return true;
}

void DvbcInterface::eventHandler(unsigned int code, const void *data, unsigned int data_size)
{
   USE_UNWANTED_PARAM(data_size);

   static DvbcInterface& instance = DvbcInterface::instance();

   U8BIT diff_frontendid = g_frontendid;
   U8BIT diff_progress;
   if (g_started)
   {
      if (code == STB_EVENT_TUNE_LOCKED || code == STB_EVENT_TUNE_NOTLOCKED ||
         code == STB_EVENT_SEARCH_FAIL || code == STB_EVENT_SEARCH_SUCCESS)
      {
         g_bit_error_rate = 1;
         g_signal_noise_ratio = 0;
         g_signal_strength = 0;

         if (data != NULL)
         {
            diff_frontendid = *((U8BIT *)data);

            g_bit_error_rate = STB_TuneGetBer(diff_frontendid);
            g_signal_noise_ratio = STB_TuneGetSnr(diff_frontendid);
            g_signal_strength = STB_TuneGetSignalStrength(diff_frontendid);
         }

         diff_progress = (ACTL_IsSearchComplete() ? 100 : MIN(ACTL_GetSearchProgress(), 99));

         if (diff_frontendid != g_frontendid || diff_progress != g_progress)
         {
            g_frontendid = diff_frontendid;
            g_progress = diff_progress;
            instance.signal("DvbcStatusChanged", &DvbcInterface::getStatus);
         }
      }
   }
}

static void ReleaseActivePath(ResourceManager::Resource resource)
{
   USE_UNWANTED_PARAM(resource);
   DvbcInterface::instance().invoke("finishSearch");
}

static void ReleaseMonitorPath(ResourceManager::Resource resource)
{
   USE_UNWANTED_PARAM(resource);
   //DvbcInterface::instance().invoke("finishMonitor");
}

static E_STB_DP_CMODE GetModulation(const std::string &modulation)
{
   if (modulation == "qam4")
   {
      return MODE_QAM_4;
   }
   if (modulation == "qam8")
   {
      return MODE_QAM_8;
   }
   if (modulation == "qam16")
   {
      return MODE_QAM_16;
   }
   if (modulation == "qam32")
   {
      return MODE_QAM_32;
   }
   if (modulation == "qam64")
   {
      return MODE_QAM_64;
   }
   if (modulation == "qam128")
   {
      return MODE_QAM_128;
   }
   if (modulation == "qam256")
   {
      return MODE_QAM_256;
   }
   
   return MODE_QAM_AUTO;
}

static E_STB_DP_TBWIDTH GetBandwidth(const std::string &str)
{
    E_STB_DP_TBWIDTH bandwidth = TBWIDTH_UNDEFINED;

   if (str == "6MHz")
   {
      bandwidth = TBWIDTH_6MHZ;
   }
   else if (str == "7MHz")
   {
      bandwidth = TBWIDTH_7MHZ;
   }
   else if (str == "8MHz")
   {
      bandwidth = TBWIDTH_8MHZ;
   }
   
   return bandwidth;
}
