#include "Glue.h"

#include "ResourceManager.h"
#include "EventManager.h"
#include "Entities.h"
#include "common.h"
#include <cstdio>
#include <cstring>

#include "DvbsInterface.h"

#define DEFAULT_SATELLITE "Astra 28.2"

static void ReleaseActivePath(ResourceManager::Resource resource);
static void * FindSatellite(std::string name);
static void SetupDefaultSatellites(void);
static E_STB_DP_POLARITY GetPolarity(const std::string &polarity);
static E_STB_DP_FEC GetFec(const std::string &fec);
static E_STB_DP_MODULATION GetModulation(const std::string &modulation);

static BOOLEAN g_started = FALSE;
static BOOLEAN g_clear_new_flags = FALSE;
static U8BIT g_frontendid = INVALID_RES_ID;
static U8BIT g_progress = 0;
static void *g_satellite = NULL;
static BOOLEAN g_manual = FALSE;

DvbsInterface::DvbsInterface()
{
   EventManager::addEventHandler(DvbsInterface::eventHandler);

   addInvokable("getStatus", &DvbsInterface::getStatus);
   addInvokable("startSearch", &DvbsInterface::startSearch);
   addInvokable("startManualSearch", &DvbsInterface::startManualSearch);
   addInvokable("finishSearch", &DvbsInterface::finishSearch);
}

/**
 * @ingroup dvbsmetaapi
 * @brief Get status.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getStatus()
 * ~~~
 * @param response Used to write _status object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "frontendpath": 0,
 *    "progress": 70,
 *    "started": true
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbsInterface::getStatus(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   ResponseData::Object *object = response->setObject(3);
   object->insertBool("started", g_started);
   object->insertInt32("frontendpath", g_frontendid);
   object->insertInt32("progress", g_progress);
   object->finish();

   return true;
}

/**
 * @ingroup dvbsmetaapi
 * @brief Start search. Signals DvbsStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * startSearch(clear_old_search = false)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbsInterface::startSearch(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   USE_UNWANTED_PARAM(response);
   bool ok = false;

   // TODO

   return ok;
}

/**
 * @ingroup dvbsmetaapi
 * @brief Start manual search. Signals DvbsStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * startManualSearch(network = false, frequency = 0, satellite = "default", polarity = "horizontal", symbol_rate = 0,
 *    fec = "auto", dvbs2 = true, modulation = "auto", clear_old_search = false)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbsInterface::startManualSearch(RequestData* request, ResponseData* response)
{
   bool ok = false;

   SetupDefaultSatellites();

   bool network = request->shiftBool(false); // arg1 network
   S_MANUAL_TUNING_PARAMS params;
   params.freq = request->shiftUInt32(0); // arg2 frequency
   params.u.sat.satellite = FindSatellite(request->shiftString("default")); // arg3 satellite
   params.u.sat.polarity = GetPolarity(request->shiftString("horizontal")); // arg4 polarity
   params.u.sat.symbol_rate = request->shiftInt32(0, 0, UINT16_MAX); // arg5 symbol rate
   params.u.sat.fec = GetFec(request->shiftString("auto")); // arg6 fec
   params.u.sat.dvb_s2 = request->shiftBool(true); // arg7 dvbs2
   params.u.sat.modulation = GetModulation(request->shiftString("auto")); // arg8 modulation
   bool clear_old_search = request->shiftBool(false); // arg9 clear_old_search

   std::cout
      << "Manual Search\n"
      << "network: " << (bool)network << "\n"
      << "params.freq: " << params.freq << "\n"
      << "params.u.sat.satellite: " << (params.u.sat.satellite ? reinterpret_cast<char *>(ADB_GetSatelliteName(params.u.sat.satellite)) : "None") << "\n"
      << "params.u.sat.polarity: " << params.u.sat.polarity << "\n"
      << "params.u.sat.symbol_rate: " << params.u.sat.symbol_rate << "\n"
      << "params.u.sat.fec: " << params.u.sat.fec << "\n"
      << "params.u.sat.dvb_s2: " << (bool)params.u.sat.dvb_s2 << "\n"
      << "params.u.sat.modulation: " << params.u.sat.modulation << "\n" 
      << "clear_old_search: " << (bool)clear_old_search << std::endl;

   if (ResourceManager::acquire(ResourceManager::ActivePath, this, ReleaseActivePath, 100))
   {
      g_frontendid = INVALID_RES_ID;
      g_progress = 0;
      g_clear_new_flags = clear_old_search;
      g_started = TRUE;
      g_manual = TRUE;
      g_satellite = params.u.sat.satellite;

      ADB_PrepareDatabaseForSearch(SIGNAL_QPSK, g_satellite, clear_old_search, g_manual);
      if (ACTL_StartManualSearch(SIGNAL_QPSK, &params, network ? ACTL_NETWORK_SEARCH : ACTL_FREQ_SEARCH))
      {
         response->setBool(true);
         ok = true;
      }
      else
      {
         // Don't save non-completed search
         ADB_FinaliseDatabaseAfterSearch(FALSE, SIGNAL_QPSK, g_satellite, FALSE, g_clear_new_flags, g_manual);
         g_started = FALSE;
         ResourceManager::released(ResourceManager::ActivePath, this);
         response->setString("Failed");
      }
   }
   else
   {
      response->setString("Conflict");
   }

   return ok;
}

/**
 * @ingroup dvbsmetaapi
 * @brief Finish search. Signals DvbsStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * finishSearch(commit = true)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbsInterface::finishSearch(RequestData* request, ResponseData* response)
{
   // arg1 commit
   bool commit = request->shiftBool(true);

   if (g_started)
   {
      g_started = FALSE;

      if (!ACTL_IsSearchComplete())
      {
         if (g_manual)
         {
            ACTL_FinishManualSearch();
         }
         else
         {
            ACTL_StopServiceSearch();
         }
      }

      ADB_FinaliseDatabaseAfterSearch(commit, SIGNAL_QPSK, g_satellite, (g_progress == 100), g_clear_new_flags, g_manual);

      ResourceManager::released(ResourceManager::ActivePath, this);
   }

   response->setBool(true);

   return true;
}

void DvbsInterface::eventHandler(unsigned int code, const void *data, unsigned int data_size)
{
   USE_UNWANTED_PARAM(data_size);

   static DvbsInterface& instance = DvbsInterface::instance();

   U8BIT diff_frontendid = g_frontendid;
   U8BIT diff_progress;
   if (g_started)
   {
      if (code == STB_EVENT_TUNE_LOCKED || code == STB_EVENT_TUNE_NOTLOCKED ||
         code == STB_EVENT_SEARCH_FAIL || code == STB_EVENT_SEARCH_SUCCESS ||
         code == UI_EVENT_UPDATE)
      {
         if (data != NULL)
         {
            diff_frontendid = *((U8BIT *)data);
         }

         diff_progress = (ACTL_IsSearchComplete() ? 100 : MIN(ACTL_GetSearchProgress(), 99));

         if (diff_frontendid != g_frontendid || diff_progress != g_progress)
         {
            g_frontendid = diff_frontendid;
            g_progress = diff_progress;
            instance.signal("DvbsStatusChanged", &DvbsInterface::getStatus);
         }
      }
   }
}

static void ReleaseActivePath(ResourceManager::Resource resource)
{
   USE_UNWANTED_PARAM(resource);
   DvbsInterface::instance().invoke("finishSearch");
}

static void * FindSatellite(std::string name)
{
   if (name == "default")
   {
      name = DEFAULT_SATELLITE;
   }

   void *satellite = ADB_GetNextSatellite(NULL);
   while (satellite)
   {
      char *tmp = reinterpret_cast<char *>(ADB_GetSatelliteName(satellite));
      if (tmp && strcmp(name.c_str(), tmp) == 0)
      {
         break;
      }

      satellite = ADB_GetNextSatellite(satellite);
   }

   return satellite;
}

static void SetupDefaultSatellites(void)
{
   if (!FindSatellite("Astra 28.2"))
   {
      ADB_LNB_SETTINGS settings;
      settings.type = LNB_TYPE_UNIVERSAL;
      settings.power = LNB_POWER_AUTO;
      settings.is_22k = FALSE;
      settings.is_12v = FALSE;
      settings.is_pulse_posn = FALSE;
      settings.is_diseqc_posn = FALSE;
      settings.diseqc_tone = DISEQC_TONE_OFF;
      settings.c_switch = DISEQC_CSWITCH_OFF;
      settings.u_switch = 0;
      settings.is_smatv = FALSE;
      settings.diseqc_repeats = 0;
      settings.name = (U8BIT *)"Default";

      void *lnb = ADB_AddLNB(&settings);
      if (lnb)
      {
         ADB_AddSatellite((U8BIT *)"Astra 28.2", 0, 282, FALSE, lnb);
      }
   }
}

static E_STB_DP_POLARITY GetPolarity(const std::string &polarity)
{
   if (polarity == "left")
   {
      return POLARITY_LEFT;
   }
   if (polarity == "right")
   {
      return POLARITY_RIGHT;
   }
   if (polarity == "vertical")
   {
      return POLARITY_VERTICAL;
   }

   return POLARITY_HORIZONTAL;
}

static E_STB_DP_FEC GetFec(const std::string &fec)
{
   if (fec == "1/2")
   {
      return FEC_1_2;
   }
   if (fec == "2/3")
   {
      return FEC_2_3;
   }
   if (fec == "3/4")
   {
      return FEC_3_4;
   }
   if (fec == "5/6")
   {
      return FEC_5_6;
   }
   if (fec == "7/8")
   {
      return FEC_7_8;
   }
   if (fec == "1/4")
   {
      return FEC_1_4;
   }
   if (fec == "1/3")
   {
      return FEC_1_3;
   }
   if (fec == "2/5")
   {
      return FEC_2_5;
   }
   if (fec == "8/9")
   {
      return FEC_8_9;
   }
   if (fec == "9/10")
   {
      return FEC_9_10;
   }

   return FEC_AUTOMATIC;
}

static E_STB_DP_MODULATION GetModulation(const std::string &modulation)
{
   if (modulation == "qpsk")
   {
      return MOD_QPSK;
   }
   if (modulation == "8psk")
   {
      return MOD_8PSK;
   }
   if (modulation == "16qam")
   {
      return MOD_16QAM;
   }

   return MOD_AUTO;
}

