#include "Glue.h"

#include "ResourceManager.h"
#include "EventManager.h"
#include "Entities.h"
#include "common.h"
#include <cstdio>

#include "DvbtInterface.h"

static void ReleaseActivePath(ResourceManager::Resource resource);

static BOOLEAN started = FALSE;
static BOOLEAN clear_new_flags = FALSE;
static U8BIT frontendid = INVALID_RES_ID;
static U8BIT progress = 0;

DvbtInterface::DvbtInterface()
{
   EventManager::addEventHandler(DvbtInterface::eventHandler);

   addInvokable("getStatus", &DvbtInterface::getStatus);
   addInvokable("startSearch", &DvbtInterface::startSearch);
   addInvokable("finishSearch", &DvbtInterface::finishSearch);
}

/**
 * @ingroup dvbtmetaapi
 * @brief Get status.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getStatus()
 * ~~~
 * @param response Used to write _status object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "frontendpath": 0,
 *    "progress": 70,
 *    "started": true
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbtInterface::getStatus(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   // Write response object
   ResponseData::Object *object = response->setObject(3);
   object->insertBool("started", started);
   object->insertInt32("frontendpath", frontendid);
   object->insertInt32("progress", progress);
   object->finish();

   return true;
}

/**
 * @ingroup dvbtmetaapi
 * @brief Start search. Signals DvbtStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * startSearch(clear_old_search = false)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbtInterface::startSearch(RequestData* request, ResponseData* response)
{
   bool ok = false;

   // arg1 clear_old_search
   bool clear_old_search = request->shiftBool(false);

   if (ResourceManager::acquire(ResourceManager::ActivePath, this, ReleaseActivePath, 100))
   {
      frontendid = INVALID_RES_ID;
      progress = 0;
      started = TRUE;
      clear_new_flags = clear_old_search;
      // Prepare for non-satellite, automatic search
      ADB_PrepareDatabaseForSearch(SIGNAL_COFDM, NULL, clear_old_search, FALSE);
      if (ACTL_StartServiceSearch(SIGNAL_COFDM, ACTL_FREQ_SEARCH))
      {
         response->setBool(true);
         ok = true;
      }
      else
      {
         // Don't save non-satellite, non-completed, automatic search
         ADB_FinaliseDatabaseAfterSearch(FALSE, SIGNAL_COFDM, NULL, FALSE, clear_new_flags, FALSE);
         started = FALSE;
         ResourceManager::released(ResourceManager::ActivePath, this);
         response->setString("Failed");
      }
   }
   else
   {
      response->setString("Conflict");
   }

   return ok;
}

/**
 * @ingroup dvbtmetaapi
 * @brief Finish search. Signals DvbtStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * finishSearch(save_changes = true)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool DvbtInterface::finishSearch(RequestData* request, ResponseData* response)
{
   bool save_changes = request->shiftBool(true);

   if (started)
   {
      started = FALSE;

      if (!ACTL_IsSearchComplete())
      {
         ACTL_StopServiceSearch();
      }
      // Finalise non-satellite, automatic search
      ADB_FinaliseDatabaseAfterSearch(save_changes, SIGNAL_COFDM, NULL, (progress == 100), clear_new_flags, FALSE);
      ResourceManager::released(ResourceManager::ActivePath, this);
   }

   response->setBool(true);

   return true;
}

void DvbtInterface::eventHandler(unsigned int code, const void *data, unsigned int data_size)
{
   USE_UNWANTED_PARAM(data_size);

   static DvbtInterface& instance = DvbtInterface::instance();

   U8BIT diff_frontendid = frontendid;
   U8BIT diff_progress;
   if (started)
   {
      if (code == STB_EVENT_TUNE_LOCKED || code == STB_EVENT_TUNE_NOTLOCKED ||
         code == STB_EVENT_SEARCH_FAIL || code == STB_EVENT_SEARCH_SUCCESS)
      {
         if (data != NULL)
         {
            diff_frontendid = *((U8BIT *)data);
         }
         diff_progress = (ACTL_IsSearchComplete() ? 100 : MIN(ACTL_GetSearchProgress(), 99));
         if (diff_frontendid != frontendid || diff_progress != progress)
         {
            frontendid = diff_frontendid;
            progress = diff_progress;
            instance.signal("DvbtStatusChanged", &DvbtInterface::getStatus);
         }
      }
   }
}

static void ReleaseActivePath(ResourceManager::Resource resource)
{
   USE_UNWANTED_PARAM(resource);
   DvbtInterface::instance().invoke("finishSearch");
}

