#include "Glue.h"

#include "ResourceManager.h"
#include "EventManager.h"
#include "Entities.h"
#include "common.h"
#include <unistd.h>
#include <cstdio>
#include <map>

#include "PlayerInterface.h"

extern "C"
{
#ifdef PLATFORM_HISILICON
   #include <platform_public.h>
#endif
}

#define ENABLE_FCC

#define LANGUAGE_OBJECT_SIZE 2

typedef enum e_player_state
{
   PLAYER_STATE_OFF,
   PLAYER_STATE_STARTING,
   PLAYER_STATE_PLAYING,
   PLAYER_STATE_BLOCKED,
   PLAYER_STATE_BAD_SIGNAL
} E_PLAYER_STATE;

typedef enum
{
   COMMAND_TRIGGER,
   COMMAND_EXIT,
   COMMAND_INVALID
} E_PLAYER_COMMAND;

static void WriteLanguage(ResponseData::Object *object, U8BIT index);
static void ReleaseActivePath(ResourceManager::Resource resource);
static std::string GetLanguageName(U8BIT index);
static U8BIT GetLanguageIndex(const std::string &language);
static std::string GetLanguageIsoCode(U8BIT index);
static U32BIT GetLanguageIsoCodeInteger(const std::string &language);
static void * GetStream(void *service, ADB_STREAM_LIST_TYPE stream_list_type, U16BIT pid);
static void ReleaseStream(void *stream);

static E_PLAYER_STATE g_player_state = PLAYER_STATE_OFF;
static Entities::ContentSource g_content_source = Entities::ContentSourceNull;
static bool g_time_shift = false;
static bool subtitles_on = false;
static bool g_beginning_of_file = false;
static bool g_end_of_file = false;

// paths for FCC preloading channels
static std::map<std::string, U8BIT> monitoringPaths;

void PlayerInterface::postEvent(unsigned int code, const void *data, unsigned int data_size)
{
   switch (code)
   {
#if 0
      case APP_EVENT_SERVICE_DELETED:
         instance.signal("ServiceDeleted");
         break;
#endif

      default:
         break;
   }
   instance().event(code, data, data_size);
}

PlayerInterface::PlayerInterface()
{
   zappingList.clear();
   mutex = STB_OSCreateMutex();
   cmd_queue = (void *)STB_OSCreateQueue(sizeof(void *), 100);
   cmd_task = STB_OSCreateTask(PlayerInterface::commandHandlerTask, this, 4096, 7, (U8BIT *)"command handler");

   EventManager::addEventHandler(PlayerInterface::postEvent);

   addInvokable("getPosition", &PlayerInterface::getPosition);
   addInvokable("getStatus", &PlayerInterface::getStatus);
   addInvokable("pause", &PlayerInterface::pause);
   addInvokable("play", &PlayerInterface::play);
   addInvokable("stop", &PlayerInterface::stop);
   addInvokable("startDecoding", &PlayerInterface::startDecoding);
   addInvokable("stopDecoding", &PlayerInterface::stopDecoding);
   addInvokable("unblock", &PlayerInterface::unblock);
   addInvokable("getListOfAudioStreams", &PlayerInterface::getListOfAudioStreams);
   addInvokable("setAudioStream", &PlayerInterface::setAudioStream);
   addInvokable("getListOfSubtitleStreams", &PlayerInterface::getListOfSubtitleStreams);
   addInvokable("setSubtitleStream", &PlayerInterface::setSubtitleStream);
   addInvokable("setRectangle", &PlayerInterface::setRectangle);
   addInvokable("getMute", &PlayerInterface::getMute);
   addInvokable("setMute", &PlayerInterface::setMute);
   addInvokable("getVolume", &PlayerInterface::getVolume);
   addInvokable("setVolume", &PlayerInterface::setVolume);
   addInvokable("getSubtitlesOn", &PlayerInterface::getSubtitlesOn);
   addInvokable("setSubtitlesOn", &PlayerInterface::setSubtitlesOn);
   addInvokable("getAudioDescriptionOn", &PlayerInterface::getAudioDescriptionOn);
   addInvokable("setAudioDescriptionOn", &PlayerInterface::setAudioDescriptionOn);
   addInvokable("getListOfLanguages", &PlayerInterface::getListOfLanguages);
   addInvokable("getDefaultAudioLanguage", &PlayerInterface::getDefaultAudioLanguage);
   addInvokable("setDefaultAudioLanguage", &PlayerInterface::setDefaultAudioLanguage);
   addInvokable("getDefaultSubtitlesLanguage", &PlayerInterface::getDefaultSubtitlesLanguage);
   addInvokable("setDefaultSubtitlesLanguage", &PlayerInterface::setDefaultSubtitlesLanguage);
   addInvokable("getVideoWindowHandle", &PlayerInterface::getVideoWindowHandle);
   addInvokable("seek", &PlayerInterface::seek);
   addInvokable("setSpeed", &PlayerInterface::setSpeed);
}

bool PlayerInterface::getPosition(RequestData* request, ResponseData* response)
{
    USE_UNWANTED_PARAM(request);
    U32BIT current = 0;
    U32BIT start = 0;
    U32BIT duration = 0;
    U8BIT length_hours;
    U8BIT length_mins;
    U8BIT length_secs;
    U32BIT rec_size_kb;
    BOOLEAN ret;

    ret = APVR_RecordingGetLength(
       APVR_GetPlaybackHandle(),
       &length_hours,
       &length_mins,
       &length_secs,
       &rec_size_kb
    );
    if (TRUE != ret)
       goto end;


    duration = length_hours * 3600 + length_mins * 60 + length_secs;
    if (TRUE == APVR_IsPlaying()) {
        current = APVR_GetPlaybackTime();
    }

    if ((current < start) || (current > duration)) {
        ret = FALSE;
        goto end;
    }

    if (g_beginning_of_file) {
        current = start;
    }
    if (g_end_of_file) {
        if (g_time_shift) {
             current += APVR_GetPauseProgress();
             current += 3;  // extra ticks since the time-shift recording is on going.
        }
        else {
             current = duration;
        }
    }

end:
    // Write response object
    ResponseData::Object *object = response->setObject(2);
    object->insertUInt32("current", current);
    object->insertUInt32("start", start);
    object->finish();

    return (TRUE == ret) ? true : false;
}

/**
 * @ingroup playermetaapi
 * @brief Get status.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getStatus()
 * ~~~
 * @param response Used to write _status object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "frontendpath": 0,
 *    "state": "playing",
 *    "type": "live",
 *    "uri": "dvb://233a.4000.41c0"
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::getStatus(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   std::string type("none");
   std::string uri("");
   U8BIT frontendpath = INVALID_RES_ID;
   U8BIT livedecodepath = STB_DPGetLivePath();

   // Live dvb
   if (g_content_source == Entities::ContentSourceDvbLive && livedecodepath != INVALID_RES_ID)
   {
      type = "dvblive";//LC-NAMING?
      void *tunedservice = ADB_GetTunedService(livedecodepath);
      uri = Entities::getServiceUri(tunedservice);
      frontendpath = STB_DPGetPathTuner(livedecodepath);
   }
   else if (g_content_source == Entities::ContentSourceTimeShift) {
      U8BIT path = STB_DPGetPlaybackPath();
      if (INVALID_RES_ID != path) {
         type = "timeshift";
      }
   }

   // Write response object
   ResponseData::Object *object = response->setObject(4);
   object->insertString("type", type);
   object->insertString("uri", uri);
   object->insertInt32("frontendpath", frontendpath);
   switch (g_player_state)
   {
      case PLAYER_STATE_OFF:
         object->insertString("state", "off");
         break;
      case PLAYER_STATE_STARTING:
         object->insertString("state", "starting");
         break;
      case PLAYER_STATE_PLAYING:
         object->insertString("state", "playing");
         break;
      case PLAYER_STATE_BLOCKED:
         object->insertString("state", "blocked");
         break;
      case PLAYER_STATE_BAD_SIGNAL:
         object->insertString("state", "badsignal");
         break;
   }
   object->finish();

   return true;
}

/**
 * @ingroup recordermetaapi
 * @brief Get SSU status.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getSsuStatus()
 * ~~~
 * @param response Used to write _status object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "release":"HCA5821191021.1",
 *    "uri": null
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
*/
bool PlayerInterface::getSsuStatus(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   U8BIT livedecodepath = STB_DPGetLivePath();

   // Live dvb
   if (g_content_source == Entities::ContentSourceDvbLive && livedecodepath != INVALID_RES_ID)
   {
      const char * ssu_info = (const char *)ACTL_GetSSUInfo();
      if (NULL != ssu_info) {
          std::string ssuInfo(ssu_info);
          int32_t percent = ACTL_GetSSUDownloadProgress();
          // Write response object
          ResponseData::Object *object = response->setObject(2);
          object->insertString("ssu", ssuInfo);
          object->insertInt32("progress", percent);
          object->finish();
       }
   }

   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Play content _uri_. Signals PlayerStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * play(uri = "")
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::pause(RequestData* request, ResponseData* response)
{
   bool ok = false;
   std::string uri = request->shiftString(""); // arg1 uri
   bool pause = request->shiftBool(true);
   Entities::ContentSource type = Entities::getContentSource(uri);

   if (type == Entities::ContentSourceDvbLive)
   {
      return stop(request, response);
   }
   else if (type == Entities::ContentSourceTimeShift)
   {
        E_PVR_PLAY_STATUS mode = APVR_GetPlayMode();
        if (pause) {
            switch (mode) {
                case PVR_PLAY_STATUS_NULL:
                case PVR_PLAY_STATUS_STOP:
                case PVR_PLAY_STATUS_PAUSE:
                    response->setString("Pause, invalid state");
                    break;
                default:
                    APVR_PausePlay();
                    signal("PlayerStatusChanged", &PlayerInterface::getStatus);
                    response->setBool(true);
                    ok = true;
                    break;
            }
        }
        else {
            switch (mode) {
                case PVR_PLAY_STATUS_PAUSE:
                    APVR_NormalPlay();
                    g_player_state = PLAYER_STATE_PLAYING;
                    signal("PlayerStatusChanged", &PlayerInterface::getStatus);
                    response->setBool(true);
                    ok = true;
                    break;
                default:
                    response->setString("Resume, invalid state");
                    break;
            }
        }      
   }
   else if (type == Entities::ContentSourceNull)
   {
      response->setString("Unsupported content");
   }

   return ok;
}

/**
 * @ingroup playermetaapi
 * @brief Play content _uri_. Signals PlayerStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * play(uri = "")
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::play(RequestData* request, ResponseData* response)
{
   bool ok = false;
   std::string uri = request->shiftString(""); // arg1 uri
   
   Entities::ContentSource type = Entities::getContentSource(uri);

   if (type == Entities::ContentSourceDvbLive)
   {
      U8BIT preloadPath = INVALID_RES_ID;
      std::string uri2 = request->shiftString(""); // arg2 uri FCC
      std::string uri3 = request->shiftString(""); // arg3 uri FCC

      // Live dvb
      void *service = Entities::getServiceHandle(uri);
      if (service)
      {
         STB_OSMutexLock(mutex);
         zappingList.push_back(uri);
         zappingList.push_back(uri2);
         zappingList.push_back(uri3);
         STB_OSMutexUnlock(mutex);
         U8BIT cmd = COMMAND_TRIGGER;
         STB_OSWriteQueue(cmd_queue, (void *)&cmd, sizeof(U8BIT), TIMEOUT_NEVER);

         signal("PlayerStatusChanged", &PlayerInterface::getStatus);
         response->setBool(true);
         ok = true;
      }
      else
      {
         response->setString("Service not found");
      }
   }
   else if (type == Entities::ContentSourceTimeShift)
   {
      g_beginning_of_file = false;
      g_end_of_file = false;

      size_t found = uri.find_last_of('/');
      std::string str = uri.substr(found + 1);
      if ("timeshift" == str) {
          // time shifting
          g_time_shift = true;
          U8BIT path = APVR_StartPausePlay();
          if (INVALID_RES_ID != path) {
             //TODO: seek
             APVR_NormalPlay();
             ok = true;
          }
      }
      else {
          g_time_shift = false;
          U32BIT recordingHandle = (U32BIT)std::stoi(str);
          g_player_state = PLAYER_STATE_STARTING;
          g_content_source = Entities::ContentSourceTimeShift;
          signal("PlayerStatusChanged", &PlayerInterface::getStatus);
          ok = APVR_PlayRecording(recordingHandle , false, NULL);
      }
      if (ok) {
         response->setBool(true);
      }
      else {
         response->setString("Failed");
      }
   }
   else if (type == Entities::ContentSourceNull)
   {
      response->setString("Unsupported content");
   }

   return ok;
}

/**
 * @ingroup playermetaapi
 * @brief Stop playing content. Signals PlayerStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * stop()
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * stop
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::stop(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   U8BIT livedecodepath;

#ifdef ENABLE_FCC
   // stop all preloading channels
   for (auto it = monitoringPaths.begin(); it != monitoringPaths.end(); it++) {
      U8BIT path = it->second;
      STB_DPStopSI(path);
      ACTL_TuneOff(path);
      STB_DPReleasePath(path, RES_OWNER_NONE);
      usleep(10000);
   }
   monitoringPaths.clear();
#endif

   if (g_player_state != PLAYER_STATE_OFF)
   {
      if (g_content_source == Entities::ContentSourceDvbLive)
      {
         ACTL_StopSubtitles();
         if ((livedecodepath = STB_DPGetLivePath()) != INVALID_RES_ID)
         {
            STB_DPStopSI(livedecodepath);
            ACTL_TuneOff(livedecodepath);
            STB_DPReleasePath(livedecodepath, RES_OWNER_NONE);
         }
      }
      else if (g_content_source == Entities::ContentSourceTimeShift)
      {
         APVR_StopPlay(FALSE);
      }

      g_player_state = PLAYER_STATE_OFF;
      g_content_source = Entities::ContentSourceNull;
      ResourceManager::released(ResourceManager::ActivePath, this);
      signal("PlayerStatusChanged", &PlayerInterface::getStatus);
   }

   response->setBool(true);

   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Start decoding.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * startDecoding()
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::startDecoding(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   U8BIT decodepath = STB_DPGetLivePath();

   if (g_content_source == Entities::ContentSourceDvbLive && decodepath != INVALID_RES_ID)
   {
      void *tunedservice = ADB_GetTunedService(decodepath);
      ACTL_StartDecoding(decodepath, tunedservice);
   }

   response->setBool(true);
   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Stop decoding.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * stopDecoding()
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::stopDecoding(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   U8BIT decodepath = STB_DPGetLivePath();

   if (g_content_source == Entities::ContentSourceDvbLive && decodepath != INVALID_RES_ID)
   {
      ACTL_DecodeOff(decodepath);
   }

   response->setBool(true);
   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Parental control unblock.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * unblock()
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::unblock(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   if (g_player_state == PLAYER_STATE_BLOCKED)
   {
      ACTL_ReleaseChannelLock();
   }

   response->setBool(true);

   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Get list of audio streams.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getListOfAudioStreams()
 * ~~~
 * @param response Used to write _list of audio stream objects_ response (example pseudocode):
 * ~~~{.js}
 * [
 *    {
 *       "pid": 100,
 *       "ad": false,
 *       "codec": "MP2",
 *       "language": "eng",
 *       "selected": false
 *    },
 *    {
 *       "pid": 101,
 *       "ad": true,
 *       "codec": "MP2",
 *       "language": "eng",
 *       "selected": true
 *    }
 * ]
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::getListOfAudioStreams(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   bool ok = false;
   void *service = NULL;
   if (g_content_source == Entities::ContentSourceDvbLive)
   {
      service = Entities::getServiceHandle("dvb://current");
   }

   if (service != NULL)
   {
      void **list;
      U16BIT length = 0;
      U16BIT i;
      ADB_GetStreamList(service, ADB_AUDIO_LIST_STREAM, &list, &length);
      // Write response array
      ResponseData::Array* array = response->setArray(length);
      for (i = 0; i < length; i++)
      {
         // Write object for response array
         ResponseData::Object *object = array->appendObject(5);
         object->insertUInt32("pid", ADB_GetStreamPID(list[i]));
         object->insertBool("selected", ADB_GetStreamInUse(list[i]));
         object->insertString("language", Entities::parseDvbString(STB_GCGetLangCodeString(ADB_GetAudioStreamLangCode(list[i]))));
         object->insertBool("ad", ADB_GetAudioStreamType(list[i]) == ADB_AUDIO_TYPE_FOR_VISUALLY_IMPAIRED);
         switch (ADB_GetStreamType(list[i]))
         {
            case ADB_AAC_AUDIO_STREAM:
               object->insertString("codec", "AAC");
               break;
            case ADB_HEAAC_AUDIO_STREAM:
               object->insertString("codec", "HE-AAC");
               break;
            case ADB_AC3_AUDIO_STREAM:
               object->insertString("codec", "AC3");
               break;
            case ADB_EAC3_AUDIO_STREAM:
               object->insertString("codec", "E-AC3");
               break;
            case ADB_AUDIO_STREAM:
               object->insertString("codec", "MP2");
               break;
            default:
               object->insertString("codec", "");
               break;
         }
         object->finish();
      }
      array->finish();
      ADB_ReleaseStreamList(list, length);
      ok = true;
   }
   else
   {
      response->setString("Service not found");
   }

   return ok;
}

/**
 * @ingroup playermetaapi
 * @brief Set audio stream.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setAudioStream(int pid = 0xFFFF, string lang_code = "")
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::setAudioStream(RequestData* request, ResponseData* response)
{
   U16BIT pid = request->shiftUInt32(0xFFFF, 0, UINT16_MAX);
   U32BIT lang_code = GetLanguageIsoCodeInteger(request->shiftString(""));
   void *service = NULL;
   if (g_content_source == Entities::ContentSourceDvbLive)
   {
      service = Entities::getServiceHandle("dvb://current");
   }

   if (service != NULL)
   {
      void *stream = GetStream(service, ADB_AUDIO_LIST_STREAM, pid);
      if (stream != NULL)
      {
         if (lang_code != 0)
         {
            ADB_AUDIO_TYPE audio_type = ADB_GetAudioStreamType(stream);
            ADB_STREAM_TYPE stream_type = ADB_GetStreamType(stream);
            // FIXME: wo'nt work if audio tracks without language code descriptors
            //        or with the same language codes
            //ADB_SetReqdAudioStreamSettings(service, TRUE, lang_code, audio_type, stream_type);
            ADB_SetReqdAudioStreamPID(service, TRUE, pid);
         }
         else
         {
            ADB_SetReqdAudioStreamPID(service, TRUE, pid);
         }

         U8BIT path = ACTL_GetActivePath();
         if (path != INVALID_RES_ID && ADB_GetTunedService(path) == service && STB_DPIsDecodingPath(path))
         {
            ACTL_ReTuneAudio();
         }
         response->setBool(true);
         ReleaseStream(stream);
      }
      else
      {
         response->setString("Stream not found");
      }
   }
   else
   {
      response->setString("Service not found");
   }

   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Get list of subtitle streams.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getListOfSubtitleStreams()
 * ~~~
 * @param response Used to write _list of audio stream objects_ response (example pseudocode):
 * ~~~{.js}
 * [
 *    {
 *       "pid": 100,
 *       "language": "eng",
 *       "selected": false,
 *       "teletext": true
 *    },
 *    {
 *       "pid": 101,
 *       "language": "eng",
 *       "selected": true,
 *       "teletext": false
 *    }
 * ]
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::getListOfSubtitleStreams(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   bool ok = false;
   void *service = NULL;
   if (g_content_source == Entities::ContentSourceDvbLive)
   {
      service = Entities::getServiceHandle("dvb://current");
   }

   if (service != NULL)
   {
      void **list;
      U16BIT length = 0;
      U16BIT i;
      ADB_GetStreamList(service, (ADB_STREAM_LIST_TYPE)(ADB_SUBTITLE_LIST_STREAM | ADB_TTEXT_SUBT_LIST_STREAM), &list, &length);
      // Write response array
      ResponseData::Array* array = response->setArray(length);
      for (i = 0; i < length; i++)
      {
         // Write object for response array
         ResponseData::Object *object = array->appendObject(4);
         object->insertUInt32("pid", ADB_GetStreamPID(list[i]));
         object->insertBool("selected", ADB_GetStreamInUse(list[i]));
         object->insertString("language", Entities::parseDvbString(STB_GCGetLangCodeString(ADB_GetAudioStreamLangCode(list[i]))));
         object->insertBool("teletext", ADB_GetStreamType(list[i]) == ADB_TTEXT_STREAM);
         object->finish();
      }
      array->finish();
      ADB_ReleaseStreamList(list, length);
      ok = true;
   }
   else
   {
      response->setString("Service not found");
   }

   return ok;
}

/**
 * @ingroup playermetaapi
 * @brief Set subtitle stream.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setSubtitleStream(int pid = 0xFFFF, string lang_code = "")
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::setSubtitleStream(RequestData* request, ResponseData* response)
{
   U16BIT pid = request->shiftUInt32(0xFFFF, 0, UINT16_MAX);
   U32BIT lang_code = GetLanguageIsoCodeInteger(request->shiftString(""));

   void *service = NULL;
   if (g_content_source == Entities::ContentSourceDvbLive)
   {
      service = Entities::getServiceHandle("dvb://current");
   }

   if (service != NULL)
   {
      void *stream = GetStream(service, (ADB_STREAM_LIST_TYPE)(ADB_SUBTITLE_LIST_STREAM | ADB_TTEXT_SUBT_LIST_STREAM), pid);
      if (stream != NULL)
      {
         if (ADB_GetStreamType(stream) == ADB_TTEXT_STREAM)
         {
            ADB_TELETEXT_TYPE ttext_type = (ADB_TELETEXT_TYPE)ADB_GetTtextStreamType(stream);
            ADB_SetReqdTeletextStreamSettings(service, TRUE, lang_code, ttext_type);
         }
         else
         {
            ADB_SUBTITLE_TYPE subtitle_type = ADB_GetSubtitleStreamType(stream);
            ADB_SetReqdSubtitleStreamSettings(service, TRUE, lang_code, subtitle_type);
         }
         if (ACTL_AreSubtitlesStarted())
         {
            ACTL_ReTuneSubtitles();
         }
         response->setBool(true);
         ReleaseStream(stream);
      }
      else
      {
         response->setString("Stream not found");
      }
   }
   else
   {
      response->setString("Service not found");
   }

   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Set player rectangle. The rectangle is scaled from full-HD (1920x1080) to the AV screen size.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setRectangle(x = 0, y = 0, width = 1920, height = 1080)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::setRectangle(RequestData* request, ResponseData* response)
{
   U16BIT screen_width;
   U16BIT screen_height;
   S16BIT player_x = request->shiftInt32(0, INT16_MIN, INT16_MAX);
   S16BIT player_y = request->shiftInt32(0, INT16_MIN, INT16_MAX);
   U16BIT player_width = request->shiftInt32(1920, 0, UINT16_MAX);
   U16BIT player_height = request->shiftInt32(1080, 0, UINT16_MAX);
   STB_AVGetScreenSize(0, &screen_width, &screen_height);
   ACTL_SetVideoWindow((player_x * (screen_width / 1920)) + 0.5, (player_y * (screen_height / 1080)) + 0.5,
      (player_width * (screen_width / 1920)) + 0.5, (player_height * (screen_height / 1080)) + 0.5);
   response->setBool(true);
   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Get mute.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getMute()
 * ~~~
 * @param response Used to write _mute bool_ response (example pseudocode):
 * ~~~{.js}
 * false
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::getMute(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   response->setBool(ACTL_IsMuted());
   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Set mute.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setMute(mute = true)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::setMute(RequestData* request, ResponseData* response)
{
   ACTL_SetMute(request->shiftBool(true));
   response->setBool(true);
   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Get volume.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getVolume()
 * ~~~
 * @param response Used to write _volume Int32_ response (example pseudocode):
 * ~~~{.js}
 * 100
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::getVolume(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   response->setInt32(ACTL_GetVolume());
   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Set volume.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setVolume(volume = 50)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::setVolume(RequestData* request, ResponseData* response)
{
   ACTL_SetVolume(request->shiftInt32(50, 0, 100));
   response->setBool(true);
   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Get subtitles on.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getSubtitlesOn()
 * ~~~
 * @param response Used to write _on bool_ response (example pseudocode):
 * ~~~{.js}
 * false
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::getSubtitlesOn(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   response->setBool(subtitles_on);
   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Set subtitles on.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setSubtitlesOn(on = true)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::setSubtitlesOn(RequestData* request, ResponseData* response)
{
   subtitles_on = request->shiftBool(true);
   if (subtitles_on)
   {
      if (ACTL_AreSubtitlesStarted()) {
         ACTL_ReTuneSubtitles();
      }
      else {
         ACTL_StartSubtitles();
      }
   }
   else
   {
      ACTL_StopSubtitles();
   }
   response->setBool(true);
   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Get audio description on.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getAudioDescriptionOn()
 * ~~~
 * @param response Used to write _on bool_ response (example pseudocode):
 * ~~~{.js}
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::getAudioDescriptionOn(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   response->setBool(ACTL_IsAudioDescriptionOn());
   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Set audio description on.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setAudioDescriptionOn(on = true)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::setAudioDescriptionOn(RequestData* request, ResponseData* response)/*MONoverall/comphbbtv/compapp*/
{
   // TODO Keep on
   U8BIT path;
   if ((path = ACTL_GetActivePath()) != INVALID_RES_ID)
   {
      if (request->shiftBool(true))
      {
         ACTL_StartAudioDescription(path);
      }
      else
      {
         ACTL_StopAudioDescription(path);
      }

      response->setBool(true);
   }

   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Get list of audio languages.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getListOfLanguages()
 * ~~~
 * @param response Used to write _array of language objects_ response (example pseudocode):
 * ~~~{.js}
 * eng
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::getListOfLanguages(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   bool ok = false;

   U8BIT **languages;
   U8BIT length;
   if (ACFG_GetDbLangList(ACFG_GetCountry(), &languages, &length))
   {
      // Write response array
      ResponseData::Array* array = response->setArray(length);
      for (U16BIT i = 0; i < length; i++)
      {
         // Write object for response array
         WriteLanguage(array->appendObject(LANGUAGE_OBJECT_SIZE), i);
      }
      array->finish();
      ACFG_ReleaseDbLangList(languages, length);

      ok = true;
   }

   return ok;
}

/**
 * @ingroup playermetaapi
 * @brief Get default audio language.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getDefaultAudioLanguage()
 * ~~~
 * @param response Used to write _ISO 639-2/T string_ response (example pseudocode):
 * ~~~{.js}
 * eng
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::getDefaultAudioLanguage(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   WriteLanguage(response->setObject(LANGUAGE_OBJECT_SIZE), ACFG_GetPrimaryAudioLangId());

   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Set default audio language.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setDefaultAudioLanguage(language = "eng")
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::setDefaultAudioLanguage(RequestData* request, ResponseData* response)
{
   bool ok = false;

   U8BIT index = GetLanguageIndex(request->shiftString(""));

   if (index != ACFG_INVALID_LANG)
   {
      if (index != ACFG_GetPrimaryAudioLangId())
      {
         U8BIT path;
         ACFG_SetPrimaryAudioLangId(index);
         if ((path = ACTL_GetActivePath()) != INVALID_RES_ID && STB_DPIsDecodingPath(path))
         {
            ACTL_ReTuneAudio();
         }
         response->setBool(true);
      }
      else
      {
         response->setBool(false);
      }
      ok = true;
   }
   else
   {
      response->setString("Failed");
   }

   return ok;
}

/**
 * @ingroup playermetaapi
 * @brief Get default subtitles language.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getDefaultSubtitlesLanguage()
 * ~~~
 * @param response Used to write _ISO 639-2/T string_ response (example pseudocode):
 * ~~~{.js}
 * eng
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::getDefaultSubtitlesLanguage(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

   WriteLanguage(response->setObject(LANGUAGE_OBJECT_SIZE), ACFG_GetPrimaryTextLangId());

   return true;
}

/**
 * @ingroup playermetaapi
 * @brief Set default subtitles language.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * setDefaultSubtitlesLanguage(language = eng)
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::setDefaultSubtitlesLanguage(RequestData* request, ResponseData* response)
{
   bool ok = false;

   U8BIT index = GetLanguageIndex(request->shiftString(""));

   if (index != ACFG_INVALID_LANG)
   {
      if (index != ACFG_GetPrimaryTextLangId())
      {
         ACFG_SetPrimaryTextLangId(index);
         if (ACTL_AreSubtitlesStarted())
         {
            ACTL_ReTuneSubtitles();
         }
         response->setBool(true);
      }
      else
      {
         response->setBool(false);
      }
      ok = true;
   }
   else
   {
      response->setString("Failed");
   }

   return ok;
}

/**
 * @ingroup playermetaapi
 * @brief Get the video window handle.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getVideoWindowHandle()
 * ~~~
 * @param response Used to write _number UInt32_ response (example pseudocode):
 * ~~~{.js}
 * 0
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::getVideoWindowHandle(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);

#ifdef PLATFORM_HISILICON
   U32BIT handle;

   handle = STB_GetVideoWindowHandle();
   response->setUInt32(handle);
#else
   response->setUInt32(0);
#endif

   return true;
}

void PlayerInterface::event(unsigned int code, const void *data, unsigned int data_size)
{
   USE_UNWANTED_PARAM(data_size);
   U8BIT dp;
   E_PLAYER_STATE diff_state = g_player_state;
   void *tuned_service;

   if (g_content_source == Entities::ContentSourceDvbLive)
   {
      switch (code)
      {
         case STB_EVENT_TUNE_NOTLOCKED:
            dp = INVALID_RES_ID;
            do {
               dp = STB_DPPathForTuner(dp, *(U8BIT *)data /*tuner id*/);
               if (INVALID_RES_ID == dp)
                  break;

               if (STB_DPGetLivePath() == dp) {
                  g_player_state = PLAYER_STATE_BAD_SIGNAL;
                  signal("PlayerStatusChanged", &PlayerInterface::getStatus);
               }             
            } while(1);
            break;
         case STB_EVENT_TUNE_LOCKED:
         case STB_EVENT_AUDIO_DECODE_STARTED:
         case STB_EVENT_VIDEO_DECODE_STARTED:
         case STB_EVENT_DECODE_LOCKED:
            if ((dp = STB_DPGetLivePath()) != INVALID_RES_ID)
            {
               if (data != NULL && *((U8BIT *)data) != INVALID_RES_ID)
               {
                  if (code == STB_EVENT_TUNE_NOTLOCKED)
                  {
                     if (*((U8BIT *)data) == STB_DPGetPathTuner(dp))
                     {
                        g_player_state = PLAYER_STATE_BAD_SIGNAL;
                     }
                  }
                  else if (code == STB_EVENT_TUNE_LOCKED)
                  {
                     if (*((U8BIT *)data) == STB_DPGetPathTuner(dp))
                     {
                        g_player_state = PLAYER_STATE_STARTING;
                     }
                  }
                  else if (code == STB_EVENT_VIDEO_DECODE_STARTED)
                  {
                     if (*((U8BIT *)data) == STB_DPGetPathAudioDecoder(dp))
                     {
                        g_player_state = PLAYER_STATE_PLAYING;
                     }
                  }
                  else if (code == STB_EVENT_AUDIO_DECODE_STARTED)
                  {
                     if (*((U8BIT *)data) == STB_DPGetPathVideoDecoder(dp))
                     {
                        g_player_state = PLAYER_STATE_PLAYING;
                     }
                  }
                  else if (code == STB_EVENT_DECODE_LOCKED)
                  {
                     if (*((U8BIT *)data) == STB_DPGetPathAudioDecoder(dp) || *((U8BIT *)data) == STB_DPGetPathVideoDecoder(dp))
                     {
                        g_player_state = PLAYER_STATE_BLOCKED;
                     }
                  }

                  if (diff_state != g_player_state)
                  {
                     signal("PlayerStatusChanged", &PlayerInterface::getStatus);
                  }
               }
            }
            break;

         case APP_EVENT_SERVICE_NOT_RUNNING:
            signal("PlayerStatusChanged", &PlayerInterface::getStatus);
            break;

         case APP_EVENT_SERVICE_CHANGED:
            if (subtitles_on)
            {
               // FIXME: do'nt automatically start the subtitle
               // ACTL_StartSubtitles();
            }

            if (data != NULL)
            {
               /* void *service = *((void **)data); */
            }

            signal("PlayerStatusChanged", &PlayerInterface::getStatus);
            break;

         case APP_EVENT_SERVICE_MOVED:
         case APP_EVENT_DELETE_SERVICE:
         {
            dp = STB_DPGetLivePath();
            if ((dp != INVALID_RES_ID) && (data != NULL))
            {
               void *service = *(void **)data;

               tuned_service = ADB_GetTunedService(dp);
               if ((tuned_service != NULL) && (tuned_service == service))
               {
                  if (code == APP_EVENT_SERVICE_MOVED)
                  {
                     /* The tuned service has changed transport so retune to stay on the service */
                     ACTL_TuneToService(INVALID_RES_ID, NULL, service, FALSE, TRUE);
                     signal("CurrentServiceMoved", &PlayerInterface::getStatus);
                  }
                  else if (code == APP_EVENT_DELETE_SERVICE)
                  {
                     /* The tuned service needs to be deleted. The application is expected to 
                      * handle the behaviour when the tuned service is deleted */
                     PlayerInterface::instance().invoke("stop");
                     ADB_DeleteServiceRec(service);
                     signal("CurrentServiceDeleted");
                  }
               }
            }

            break;
         }

         case APP_EVENT_SSU_UPDATE_DETECTED:
            signal("SsuUpdate", &PlayerInterface::getSsuStatus);
            break;

      }
   }
   else if (g_content_source == Entities::ContentSourceTimeShift)
   {
#if 0
      U8BIT path = INVALID_RES_ID;
      if (NULL != data) {
          path = *(U8BIT *)data;
      }
#endif
      switch (code) {
         case APP_EVENT_PVR_PLAY_STARTED:
            g_player_state = PLAYER_STATE_PLAYING;
            break;

         case APP_EVENT_PVR_PLAY_STOPPED:
            g_player_state = PLAYER_STATE_OFF;
            break;

         case APP_EVENT_PVR_PLAY_BOF:
            g_beginning_of_file = true;
            g_end_of_file = false;
            break;

         case APP_EVENT_PVR_PLAY_EOF:
            g_beginning_of_file = false;
            g_end_of_file = true;
            break;

         default:
            break;

         //signal("PlayerStatusChanged", &PlayerInterface::getStatus);
      }
   }
}

void PlayerInterface::commandHandlerTask(void * param)
{
   PlayerInterface * pPlayerInterface = (PlayerInterface *)param;
   if (NULL != pPlayerInterface)
      pPlayerInterface->commandHandler();
}

void PlayerInterface::commandHandler(void)
{
   while (1) {
      U8BIT cmd = COMMAND_INVALID;
      BOOLEAN ret = STB_OSReadQueue(cmd_queue, &cmd, sizeof(cmd), 10000 /*ms*/);
      if (TRUE != ret) {
      
      }

      switch (cmd) {
         case COMMAND_TRIGGER:
            std::string uri, uri2, uri3;
            STB_OSMutexLock(mutex);
            int size = zappingList.size();
            if (size < 3) {
               STB_OSMutexUnlock(mutex);
               break;
            }

            do {
               uri = zappingList.front();
               zappingList.pop_front();
               
               uri2 = zappingList.front();
               zappingList.pop_front();
               
               uri3 = zappingList.front();
               zappingList.pop_front();

               size = zappingList.size();
            } while (size >= 3);
            STB_OSMutexUnlock(mutex);

            void * service = ACTL_GetLockedService();
            if (NULL != service) {
               // lock service
               //uri = "dvb://current";
               uri2.clear();
               uri3.clear();
            }
            else {
               service = Entities::getServiceHandle(uri);
               if (NULL == service)
                  break;
            }

#ifdef ENABLE_FCC
            U8BIT preloadPath = INVALID_RES_ID;
            U8BIT livePath = STB_DPGetLivePath();
            if (INVALID_RES_ID != livePath) {
               // find if the channel is preloaded
               auto it = monitoringPaths.find(uri);
               if (it != monitoringPaths.end()) {
                  preloadPath = it->second;
                  monitoringPaths.erase (it);  // remove 
               }
            }

            BOOLEAN switchViaFcc = FALSE;
            if ((INVALID_RES_ID != preloadPath) && (g_player_state == PLAYER_STATE_PLAYING)) {
               ACTL_StopSubtitles();
               STB_DPStopSI(livePath);
               STB_DPStopSI(preloadPath);
               switchViaFcc = ACTL_SwitchLivePath(livePath, preloadPath); 
            }

            if (TRUE == switchViaFcc) {
               usleep(10000);
               // stop last path
               ACTL_TuneOff(livePath);
               STB_DPReleasePath(livePath, RES_OWNER_NONE);

               g_player_state = PLAYER_STATE_STARTING;
               g_content_source = Entities::ContentSourceDvbLive;
               signal("PlayerStatusChanged", &PlayerInterface::getStatus);
   
               STB_DPStartSI(preloadPath);
               usleep(10000);
            }
            else
#endif  // #ifdef ENABLE_FCC
            if (ACTL_CanServiceBeViewed(service) && ResourceManager::acquire(ResourceManager::ActivePath, this, ReleaseActivePath, 50))
            {
#ifdef ENABLE_FCC
               if (INVALID_RES_ID != preloadPath) {
                  // FCC switching failure
                  usleep(10000);
                  // stop the preload path
                  ACTL_TuneOff(preloadPath);
                  STB_DPReleasePath(preloadPath, RES_OWNER_NONE);
                  preloadPath = INVALID_RES_ID;
               }
#endif               
               ACTL_StopSubtitles();
#if 1
               if (INVALID_RES_ID != livePath) {
                  STB_DPStopSI(livePath);
                  usleep(10000);
                  ACTL_TuneOff(livePath);
                  STB_DPReleasePath(livePath, RES_OWNER_NONE);
                  livePath = INVALID_RES_ID;
               }

               // stop all preloading channels
               for (auto it = monitoringPaths.begin(); it != monitoringPaths.end(); it++) {
                  U8BIT path = it->second;
                  if (INVALID_RES_ID != path) {
                     STB_DPStopSI(path);
                     usleep(10000);
                     ACTL_TuneOff(path);
                     STB_DPReleasePath(path, RES_OWNER_NONE);
                     usleep(10000);
                  }
               }
               monitoringPaths.clear();
               usleep(100000);
#endif
               S_ACTL_OWNER_INFO *owner_info = NULL;
               BOOLEAN override_lock = FALSE;
               BOOLEAN for_live = TRUE;
               if (ACTL_TuneToService(INVALID_RES_ID, owner_info, service, override_lock, for_live) != INVALID_RES_ID)
               {
                  g_player_state = PLAYER_STATE_STARTING;
                  g_content_source = Entities::ContentSourceDvbLive;
                  signal("PlayerStatusChanged", &PlayerInterface::getStatus);
               }
               else
               {
                  ResourceManager::released(ResourceManager::ActivePath, this);
               }
            }
            else
            {

            }

#ifdef ENABLE_FCC
            // stop all preloading channels
            for (auto it = monitoringPaths.begin(); it != monitoringPaths.end(); it++) {
               U8BIT path = it->second;
               STB_DPStopSI(path);
               ACTL_TuneOff(path);
               STB_DPReleasePath(path, RES_OWNER_NONE);
               usleep(10000);
            }
            monitoringPaths.clear();

//FIXME: disable FCC for CDCAS
#if 1
            // process FCC monitoring services
            Entities::ContentSource type;
            if (! uri2.empty()) {
               type = Entities::getContentSource(uri2);
               if (type == Entities::ContentSourceDvbLive) {
                  service = Entities::getServiceHandle(uri2);
                  if (NULL != service) {
                     /* Acquire a new path to monitor SI data */
                     U8BIT path = ACTL_AcquirePathForService(service, FALSE, FALSE, NULL);
                     //printf("\nPath for monitoring 1: %u", path);
                     if (path != INVALID_RES_ID) {
                        ACTL_TuneToService(path, NULL, service, TRUE, FALSE);
                        monitoringPaths[uri2] = path;
                     }
                  } 
               }
            }

            if (! uri3.empty()) {
               type = Entities::getContentSource(uri3);
               if (type == Entities::ContentSourceDvbLive) {
                  service = Entities::getServiceHandle(uri3);
                  if (NULL != service) {
                     /* Acquire a new path to monitor SI data */
                     U8BIT path = ACTL_AcquirePathForService(service, FALSE, FALSE, NULL);
                     //printf("\nPath for monitoring 2: %u", path);
                     if (path != INVALID_RES_ID) {
                        ACTL_TuneToService(path, NULL, service, TRUE, FALSE);
                        monitoringPaths[uri3] = path;
                     }
                  }
               }
            }
#endif
            // flow control
            usleep(500000);
#endif  // #ifdef ENABLE_FCC                                             
            break;    
      }

      if (COMMAND_EXIT == cmd)
         break;


      {
         //instance.signal("DvbUpdatedEventPeriods");
      }
   }
}

static void WriteLanguage(ResponseData::Object *object, U8BIT index)
{
   // If changing object size, change LANGUAGE_OBJECT_SIZE

   object->insertString("code", GetLanguageIsoCode(index));
   object->insertString("language", GetLanguageName(index));
   object->finish();
}

/**
 * @ingroup playermetaapi
 * @brief Seek to the playback position.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * play(uri = "")
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::seek(RequestData* request, ResponseData* response)
{
   bool ok = false;
   std::string uri = request->shiftString(""); // arg1 uri
   U32BIT pos = request->shiftUInt32(0 /*BOS*/, 0, UINT32_MAX);  // millisecond
   pos /= 1000;  // second

   Entities::ContentSource type = Entities::getContentSource(uri);

   if (type == Entities::ContentSourceDvbLive)
   {
      // Live dvb
      response->setString("Set speed for the live stream !!");
   }
   else if (type == Entities::ContentSourceTimeShift)
   {
		E_PVR_PLAY_STATUS playMode = APVR_GetPlayMode();
        switch (playMode) {
        case PVR_PLAY_STATUS_PAUSE:
            APVR_SetPlaybackTime(pos);
            break;
        case PVR_PLAY_STATUS_NORMAL:
            APVR_SetPlaybackTime(pos);
            break;
        case PVR_PLAY_STATUS_FF:
            APVR_SetPlaybackTime(pos);
            break;
        case PVR_PLAY_STATUS_FR:
            APVR_SetPlaybackTime(pos);
            break;
        default:
            break;		
		}
        response->setBool(true);
		ok = true;
   }
   else if (type == Entities::ContentSourceNull)
   {
      response->setString("Unsupported content");
   }

   return ok;
}

/**
 * @ingroup playermetaapi
 * @brief Set the playback speed.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * play(uri = "")
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool PlayerInterface::setSpeed(RequestData* request, ResponseData* response)
{
   bool ok = false;
   std::string uri = request->shiftString(""); // arg1 uri
   // DTVKit defined speed value: 100 == 1x
   S16BIT speed = (S16BIT)request->shiftInt32(100 /*1x*/, INT16_MIN, INT16_MAX);
   if ((speed > -100) && (speed < 100)) {
        // ignore slow motion
        return false;
   }
   Entities::ContentSource type = Entities::getContentSource(uri);

   if (type == Entities::ContentSourceDvbLive)
   {
      // Live dvb
      response->setString("Set speed for the live stream !!");
   }
   else if (type == Entities::ContentSourceTimeShift)
   {
		E_PVR_PLAY_STATUS playMode = APVR_GetPlayMode();
   		S16BIT currentSpeed = APVR_GetPlaySpeed();
        switch (playMode) {
        case PVR_PLAY_STATUS_PAUSE:
            APVR_SetPlaySpeed(speed);
            break;
        case PVR_PLAY_STATUS_NORMAL:
            APVR_SetPlaySpeed(speed);
            break;
        case PVR_PLAY_STATUS_FF:
            APVR_SetPlaySpeed(speed);
            break;
        case PVR_PLAY_STATUS_FR:
            APVR_SetPlaySpeed(speed);
            break;
        default:
            break;		
		}
        g_beginning_of_file = false;
        g_end_of_file = false;
        response->setBool(true);
		ok = true;
   }
   else if (type == Entities::ContentSourceNull)
   {
      response->setString("Unsupported content");
   }

   return ok;
}

static void ReleaseActivePath(ResourceManager::Resource resource)
{
   USE_UNWANTED_PARAM(resource);
   PlayerInterface::instance().invoke("stop");
}

static std::string GetLanguageName(U8BIT index)
{
   std::string language;

   U8BIT **languages;
   U8BIT length;
   if (ACFG_GetDbLangList(ACFG_GetCountry(), &languages, &length))
   {
      language = (index < length) ? Entities::parseDvbString(languages[index]) : "";
      ACFG_ReleaseDbLangList(languages, length);
   }

   return language;
}

static U8BIT GetLanguageIndex(const std::string &language)
{
   U8BIT index = ACFG_INVALID_LANG;

   if (language.length() == 3)   // Language ISO code
   {
      U32BIT lang_code = GetLanguageIsoCodeInteger(language);
      U8BIT lang_code_id = ACFG_ConvertLangCodeToId(lang_code);

      U8BIT **languages;
      U8BIT length;
      if (ACFG_GetDbLangList(ACFG_GetCountry(), &languages, &length))
      {
         // Check if language ISO code is for a particular language in the array of language names
         for (U8BIT i = 0; (i < length) && (index == ACFG_INVALID_LANG); i++)
         {
            U8BIT *lang_ids = ACFG_GetDbLangId(ACFG_GetCountry(), i);

            // Check if language ISO code is in the array of language ids
            for (U8BIT j = 0; (lang_ids[j] != ACFG_INVALID_LANG) && (index == ACFG_INVALID_LANG); j++)
            {
               if (lang_ids[j] == lang_code_id)
               {
                  index = i;
               }
            }
         }

         ACFG_ReleaseDbLangList(languages, length);
      }
   }

   return index;
}

static std::string GetLanguageIsoCode(U8BIT index)
{
   std::string iso_code ("");

   U8BIT *lang_ids = ACFG_GetDbLangId(ACFG_GetCountry(), index);
   if (lang_ids != NULL && (*lang_ids) != ACFG_INVALID_LANG)
   {
      U32BIT lang_code = ACFG_ConvertLangIdToCode(*lang_ids);
      if (lang_code != 0)
      {
         char code[3 + 1];
         code[0] = lang_code >> 16;
         code[1] = lang_code >> 8;
         code[2] = lang_code;
         code[3] = '\0';

         iso_code = code;
      }
   }

   return iso_code;
}

static U32BIT GetLanguageIsoCodeInteger(const std::string &language)
{
   U32BIT lang_code;

   if (language.length() == 3)
   {
      lang_code = ((language[0] << 16) | (language[1] << 8) | language[2]);
   }
   else
   {
      lang_code = 0;
   }

   return lang_code;
}

static void * GetStream(void *service, ADB_STREAM_LIST_TYPE stream_list_type, U16BIT pid)
{
   void *stream = NULL;
   void **list;
   U16BIT length = 0;
   ADB_GetStreamList(service, stream_list_type, &list, &length);
   for (U16BIT i = 0; i < length; i++)
   {
      if (ADB_GetStreamPID(list[i]) == pid)
      {
         stream = list[i];
         list[i] = list[--length]; // Return value to be released with ReleaseStream
         break;
      }
   }
   ADB_ReleaseStreamList(list, length);

   return stream;
}

static void ReleaseStream(void *stream)
{
   void **list = (void**)STB_AppGetMemory(sizeof(void *)); // TODO Public API?
   list[0] = stream;
   ADB_ReleaseStreamList(list, 1);
}

