#include "Glue.h"

#include "ResourceManager.h"
#include "EventManager.h"
#include "Entities.h"
#include "common.h"
#include <cstdio>
#include <sstream>

#include "RecorderInterface.h"

extern "C"
{
#ifdef PLATFORM_HISILICON
   #include <platform_public.h>
#endif
}

#define LANGUAGE_OBJECT_SIZE 2

typedef enum e_player_state
{
   RECORDER_STATE_OFF,
   RECORDER_STATE_TUNED,

   RECORDER_STATE_STARTING,
   RECORDER_STATE_RECORDINGING,
   RECORDER_STATE_BLOCKED,
   RECORDER_STATE_BAD_SIGNAL
} E_RECORDER_STATE;

static void ReleasePausePath(ResourceManager::Resource resource);
static void ReleaseRecordPath(ResourceManager::Resource resource);

static E_RECORDER_STATE g_recorder_state = RECORDER_STATE_OFF;
static Entities::ContentSource g_content_source = Entities::ContentSourceNull;
static U8BIT pausePath = INVALID_RES_ID;
static U8BIT recordPath = INVALID_RES_ID;
static U32BIT pauseRecordingHandle = 0;
static U32BIT recordingHandle = 0; /* invalid handle, HANDLE_BASE = 1*/
static long recycleTime = 0;  // millisecond

RecorderInterface::RecorderInterface()
{
   EventManager::addEventHandler(RecorderInterface::postEvent);

   addInvokable("getSsuStatus", &RecorderInterface::getSsuStatus);
   addInvokable("getStatus", &RecorderInterface::getStatus);
   addInvokable("tune", &RecorderInterface::tune);
   addInvokable("startRecording", &RecorderInterface::startRecording);
   addInvokable("stopRecording", &RecorderInterface::stopRecording);
   addInvokable("release", &RecorderInterface::release);
}

/**
 * @ingroup recordermetaapi
 * @brief Get status.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getStatus()
 * ~~~
 * @param response Used to write _status object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "frontendpath": 0,
 *    "state": "playing",
 *    "type": "live",
 *    "uri": "dvb://233a.4000.41c0"
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool RecorderInterface::getStatus(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   std::string type("none");
   std::string uri("");
   U8BIT frontendpath = INVALID_RES_ID;
   U8BIT livedecodepath = STB_DPGetLivePath();

   // Live dvb
   if (g_content_source == Entities::ContentSourceDvbLive && livedecodepath == INVALID_RES_ID)
   {
      type = "dvblive";//LC-NAMING?
      void *tunedservice = ADB_GetTunedService(livedecodepath);
      uri = Entities::getServiceUri(tunedservice);
      frontendpath = STB_DPGetPathTuner(livedecodepath);
   }

   // Write response object
   ResponseData::Object *object = response->setObject(4);
   object->insertString("type", type);
   object->insertString("uri", uri);
   object->insertInt32("frontendpath", frontendpath);
   switch (g_recorder_state)
   {
      case RECORDER_STATE_OFF:
         object->insertString("state", "off");
         break;
      case RECORDER_STATE_STARTING:
         object->insertString("state", "starting");
         break;
      case RECORDER_STATE_RECORDINGING:
         object->insertString("state", "playing");
         break;
      case RECORDER_STATE_BLOCKED:
         object->insertString("state", "blocked");
         break;
      case RECORDER_STATE_BAD_SIGNAL:
         object->insertString("state", "badsignal");
         break;
   }
   object->finish();

   return true;
}

/**
 * @ingroup recordermetaapi
 * @brief Get SSU status.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * getSsuStatus()
 * ~~~
 * @param response Used to write _status object_ response (example pseudocode):
 * ~~~{.js}
 * {
 *    "release":"HCA5821191021.1",
 *    "uri": null
 * }
 * ~~~
 * @return Success (on failure, an error string is written to response)
*/
bool RecorderInterface::getSsuStatus(RequestData* request, ResponseData* response)
{
   USE_UNWANTED_PARAM(request);
   U8BIT livedecodepath = STB_DPGetLivePath();

   // Live dvb
   //if (g_content_source == Entities::ContentSourceDvbLive && livedecodepath == INVALID_RES_ID)
   {
      const char * ssu_info = (const char *)ACTL_GetSSUInfo();
      if (NULL != ssu_info) {
          std::string ssuInfo(ssu_info);
          int32_t percent = ACTL_GetSSUDownloadProgress();
          // Write response object
          ResponseData::Object *object = response->setObject(2);
          object->insertString("ssu", ssuInfo);
          object->insertInt32("progress", percent);
          object->finish();
       }
   }

   return true;
}

/**
 * @ingroup recordermetaapi
 * @brief Play content _uri_. Signals PlayerStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * play(uri = "")
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool RecorderInterface::tune(RequestData* request, ResponseData* response)
{
   bool ok = false;
   std::string uri = request->shiftString(""); // arg1 uri
   Entities::ContentSource type = Entities::getContentSource(uri);
   if (recordPath != INVALID_RES_ID) {
      response->setString("Resource conflict");
   }

   if (type == Entities::ContentSourceDvbLive)
   {
      // Live dvb
      void *s_ptr = Entities::getServiceHandle(uri);
      if (NULL != s_ptr) {
         void *t_ptr = ADB_GetServiceTransportPtr(s_ptr);
         if (t_ptr != NULL) {
             BOOLEAN new_tuned_service;
             U16BIT onid = ADB_GetTransportOriginalNetworkId(t_ptr);
             U16BIT tsid = ADB_GetTransportTid(t_ptr);
             U16BIT sid = ADB_GetServiceId(s_ptr);
             recordPath = APVR_PrepareNewRecording(onid, tsid, sid, &new_tuned_service);
             if (recordPath == INVALID_RES_ID) {
                response->setString("Resource conflict");
                return false;
             }

             if (ResourceManager::acquire(ResourceManager::RecordPath, this, ReleaseRecordPath, 50)) {
                 g_recorder_state = RECORDER_STATE_TUNED;
                 g_content_source = Entities::ContentSourceDvbLive;
                 signal("RecorderStatusChanged", &RecorderInterface::getStatus);
                 response->setBool(true);
                 ok = true;
             }
             else {
                 response->setString("Resource conflict");
             }
         }
      }
      else
      {
         response->setString("Service not found");
      }
   }
   else if (type == Entities::ContentSourceNull)
   {
      response->setString("Unsupported content");
   }

   return ok;
}

/**
 * @ingroup recordermetaapi
 * @brief Play content _uri_. Signals PlayerStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * play(uri = "")
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool RecorderInterface::startRecording(RequestData* request, ResponseData* response)
{
    bool ok = false;
    std::string uri = request->shiftString(""); // arg1 DVB uri
    long recycleTime = request->shiftInt32(0, INT32_MIN, INT32_MAX); // arg2 the recording recycle time, millisecond
    std::string ssu = request->shiftString("");  // arg3 the SSU action: "search", "download"

    // check SSU
    if(ssu == "download") {
        ACTL_PrepareSSUDownload();
        ResponseData::Object *object = response->setObject(1);
        if (NULL != object) {
            object->insertUInt32("handle", 0);
            object->finish();
        }
        return true;
    }

    // check time-shift
    if (recycleTime > 0) {
        // time-shifting recording
        APVR_SetTimeshiftBufferSize(recycleTime / (1000 * 60));
        pausePath = APVR_StartPauseRecord();
        if (INVALID_RES_ID == pausePath) {
            ResponseData::Object *object = response->setObject(1);
            if (NULL != object) {
                object->insertInt32("error", -1);
                object->finish();
            }
        }
        else {
            BOOLEAN ret = APVR_GetRecordingHandle(pausePath, &recordingHandle);
            if (FALSE == ret) {
                recordingHandle = 0;  // a invalid hadle
            }
            ResponseData::Object *object = response->setObject(1);
            if (NULL != object) {
                object->insertUInt32("handle", recordingHandle);
                object->finish();
            }
            ok = true;
        }
        return ok;
    }
    else {
        APVR_SetTimeshiftBufferSize(0);
    }

    Entities::ContentSource type = Entities::getContentSource(uri);
    if (INVALID_RES_ID == recordPath)
        return false;

    BOOLEAN ret = APVR_StartNewRecording(
        STB_PVRGetDefaultDisk(),
        recordPath,
        NULL,
        0,
        NULL,
        NULL,
        &recordingHandle
    );
    if (ret == FALSE) {
        ResponseData::Object *object = response->setObject(1);
        if (NULL != object) {
            object->insertInt32("error", -1);
            object->finish();
        }
    }
    else {
        ResponseData::Object *object = response->setObject(1);
        if (NULL != object) {
            object->insertUInt32("handle", recordingHandle);
            object->finish();
        }
        ok = true;
    }

   return ok;
}

/**
 * @ingroup recordermetaapi
 * @brief Stop playing content. Signals PlayerStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * stop()
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * stop
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool RecorderInterface::stopRecording(RequestData* request, ResponseData* response)
{
    USE_UNWANTED_PARAM(request);
    bool ok = false;

    std::string uri = request->shiftString(""); // arg1 DVB uri
    long recycleTime = request->shiftInt32(0, INT32_MIN, INT32_MAX);; // arg2 the recording recycle time
    std::string ssu = request->shiftString("");  // arg3 the SSU action: "search", "download"

    if(ssu == "download") {
        ACTL_StopSSUDownload();
        ResponseData::Object *object = response->setObject(2);
        if (NULL != object) {
            object->insertUInt32("duration", 0);
            object->insertUInt32("size", 0);
            object->finish();
        }
        return true;
    }

    if (recycleTime > 0) {
        APVR_StopPauseRecord(FALSE);
        signal("RecorderStatusChanged", &RecorderInterface::getStatus);
        // Write response object
        ResponseData::Object *object = response->setObject(2);
        object->insertUInt32("duration", recycleTime);
        object->insertUInt32("size", 0);
        object->finish();
        return true;
    }

    //if (g_recorder_state != RECORDER_STATE_OFF)
    {
        BOOLEAN ret;
        U8BIT length_hours = 0;
        U8BIT length_mins = 0;
        U8BIT length_secs = 0;
        U32BIT rec_size_kb = 0;
        U32BIT duration = 0;
        ret = APVR_StopRecording(recordingHandle);
        ret = STB_PVRRecordingGetLength(recordingHandle, &length_hours, &length_mins,
            &length_secs, &rec_size_kb);
        if (TRUE == ret) {
            duration = (length_hours * 60 * 60) + (length_mins * 60) + length_secs;
            ok = true;
        }

        g_content_source = Entities::ContentSourceNull;
        g_recorder_state = RECORDER_STATE_OFF;
        ResourceManager::released(ResourceManager::RecordPath, this);
        signal("RecorderStatusChanged", &RecorderInterface::getStatus);
        // Write response object
        ResponseData::Object *object = response->setObject(2);
        object->insertUInt32("duration", duration);
        object->insertUInt32("size", rec_size_kb);
        object->finish();
    }
    recordPath = INVALID_RES_ID;

    return ok;
}

/**
 * @ingroup recordermetaapi
 * @brief Play content _uri_. Signals PlayerStatusChanged with getStatus data.
 * @param request Used to read arguments for the invokable (pseudocode):
 * ~~~{.java}
 * play(uri = "")
 * ~~~
 * @param response Used to write _success bool_ response (example pseudocode):
 * ~~~{.js}
 * true
 * ~~~
 * @return Success (on failure, an error string is written to response)
 */
bool RecorderInterface::release(RequestData* request, ResponseData* response)
{
   bool ok = true;

   return ok;
}

void RecorderInterface::event(unsigned int code, const void *data, unsigned int data_size)
{
   USE_UNWANTED_PARAM(data_size);
   U8BIT dp;
   E_RECORDER_STATE diff_state = g_recorder_state;

   if (g_content_source == Entities::ContentSourceDvbLive)
   {
      switch (code)
      {
         case STB_EVENT_TUNE_NOTLOCKED:
         case STB_EVENT_TUNE_LOCKED:
         case STB_EVENT_AUDIO_DECODE_STARTED:
         case STB_EVENT_VIDEO_DECODE_STARTED:
         case STB_EVENT_DECODE_LOCKED:
            if ((dp = STB_DPGetLivePath()) != INVALID_RES_ID)
            {
               if (data != NULL && *((U8BIT *)data) != INVALID_RES_ID)
               {
                  if (code == STB_EVENT_TUNE_NOTLOCKED)
                  {
                     if (*((U8BIT *)data) == STB_DPGetPathTuner(dp))
                     {
                        g_recorder_state = RECORDER_STATE_BAD_SIGNAL;
                     }
                  }
                  else if (code == STB_EVENT_TUNE_LOCKED)
                  {
                     if (*((U8BIT *)data) == STB_DPGetPathTuner(dp))
                     {
                        g_recorder_state = RECORDER_STATE_STARTING;
                     }
                  }
                  else if (code == STB_EVENT_VIDEO_DECODE_STARTED)
                  {
                     if (*((U8BIT *)data) == STB_DPGetPathAudioDecoder(dp))
                     {
                        g_recorder_state = RECORDER_STATE_RECORDINGING;
                     }
                  }
                  else if (code == STB_EVENT_AUDIO_DECODE_STARTED)
                  {
                     if (*((U8BIT *)data) == STB_DPGetPathVideoDecoder(dp))
                     {
                        g_recorder_state = RECORDER_STATE_RECORDINGING;
                     }
                  }
                  else if (code == STB_EVENT_DECODE_LOCKED)
                  {
                     if (*((U8BIT *)data) == STB_DPGetPathAudioDecoder(dp) || *((U8BIT *)data) == STB_DPGetPathVideoDecoder(dp))
                     {
                        g_recorder_state = RECORDER_STATE_BLOCKED;
                     }
                  }

                  if (diff_state != g_recorder_state)
                  {
                     signal("RecorderStatusChanged", &RecorderInterface::getStatus);
                  }
               }
            }
            break;

         case APP_EVENT_SSU_UPDATE_DETECTED:
            std::string str((const char *)data, (size_t)data_size);
            signal("SsuUpdate", &RecorderInterface::getSsuStatus);
            break;
      }
   }
}

static void ReleasePausePath(ResourceManager::Resource resource)
{
   USE_UNWANTED_PARAM(resource);
   // indicate that this is time-shifting
   //RecorderInterface::instance().invoke("stopRecording");
}

static void ReleaseRecordPath(ResourceManager::Resource resource)
{
   USE_UNWANTED_PARAM(resource);
   //RecorderInterface::instance().invoke("stopRecording");
}
