#ifndef RESOURCEMANAGER_H
#define RESOURCEMANAGER_H

#include <map>

class ResourceManager
{
   public:
   enum Resource
   {
      InitialiseDvb,
      ActivePath,
      PausePath,
      RecordPath,
      MonitorPath
   };

   typedef void (*ReleaseResource)(Resource resource);

   /**
    * @brief Acquire resource.
    * @param resource The resource to acquire.
    * @param acquirer A pointer that uniquely identifies the acquirer.
    * @param release_callback A function that when called must release the resource before returning.
    */
   static bool acquire(Resource resource, const void *acquirer, ReleaseResource release_callback, int priority = 50)
   {
      ResourceManager& manager = instance();

      bool acquired = true;
      std::map<Resource, Acquisition>::iterator it = manager.m_resources.find(resource);
      if (it != manager.m_resources.end())
      {
         if (it->second.acquirer != acquirer)
         {
            if (it->second.priority < priority)
            {
               if (it->second.release_callback)
               {
                  it->second.release_callback(resource);
               }
            }
            else
            {
               acquired = false;
            }
         }
      }

      if (acquired)
      {
         manager.m_resources[resource] = Acquisition(acquirer, release_callback, priority);
      }

      return acquired;
   }

   /**
    * @brief Once the resource is released, this method must be called.
    * @param resource The resource that was released.
    * @param acquirer The pointer that was used to uniquely identify the acquirer.
    */
   static void released(Resource resource, const void *acquirer)
   {
      ResourceManager& manager = instance();

      std::map<Resource, Acquisition>::iterator it = manager.m_resources.find(resource);
      if (it != manager.m_resources.end())
      {
         if (it->second.acquirer == acquirer)
         {
            manager.m_resources.erase(it);
         }
      }
   }

   private:
   struct Acquisition
   {
      Acquisition() :
         acquirer(0), release_callback(0), priority(0) {}
      Acquisition(const void *acquirer, ReleaseResource release_callback, int priority) :
         acquirer(acquirer), release_callback(release_callback), priority(priority) {}
      const void *acquirer;
      ReleaseResource release_callback;
      int priority;
   };

	ResourceManager() {};
   ResourceManager(ResourceManager const&);
   ResourceManager& operator=(ResourceManager const&);

   static ResourceManager& instance()
   {
      static ResourceManager instance;
      return instance;
   }

   std::map<Resource, Acquisition> m_resources;
};

#endif // RESOURCEMANAGER_H

