/*******************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/

#ifndef _INTERNAL_H
#define _INTERNAL_H


/*--- Includes ----------------------------------------------------------------*/

#include "techtype.h"

/*--- Preprocessor definitions ------------------------------------------------*/

#define INVALID_AV      0xFF
#define INVALID_DEMUX   0xFF
#define INVALID_TUNER   0xFF
#define INVALID_PID     0xFFFF

#define MAX_AUDIO_TRACK_NUM 	32
typedef struct multi_audio_info
{
   U16BIT audio_index_default; 			 /* The default audio index, the PID is in array audio_pid, CODEC is in audio_codec */
   U16BIT audio_num;					 /* audio track number, to compatible with multi-audio channels */
   U16BIT audio_pid[MAX_AUDIO_TRACK_NUM];
   U16BIT audio_codec[MAX_AUDIO_TRACK_NUM];
}S_MULTI_AUDIO_INFO;

typedef struct recording_user_data
{
   U16BIT video_pid;
   U16BIT video_codec;
   U16BIT pcr_pid;
   S_MULTI_AUDIO_INFO audio;
} S_RECORDING_USER_DATA;

/*--- Global types definitions ------------------------------------------------*/

/*--- Global variables --------------------------------------------------------*/

/*--- Global function declarations --------------------------------------------*/

/*!**************************************************************************
 * @fn      TUN_Terminate
 * @brief   Deinit tuner module
 * @param   paths - not used
 ****************************************************************************/
void TUN_Terminate(int paths);

/*!**************************************************************************
 * @fn      DMX_Terminate
 * @brief   Deinit demux module
 * @param   paths - not used
 ****************************************************************************/
void DMX_Terminate(int paths);

/*!**************************************************************************
 * @fn      DMX_GetDecodePIDs
 * @brief   Get PIDs set for decoding by demux path
 * @param   path - demux path
 * @param   pcr_pid - (out) PCR PID
 * @param   video_pid - (out) video PID
 * @param   audio_pid - (out) audio PID
 * @param   text_pid - (out) text PID
 * @param   data_pid - (out) data PID
 * @param   ad_pid - (out) audio description PID
 * @return  TRUE if successfull, FALSE otherwise
 ****************************************************************************/
BOOLEAN DMX_GetDecodePIDs(U8BIT path, U16BIT *pcr_pid, U16BIT *video_pid, U16BIT *audio_pid,
                          U16BIT *text_pid, U16BIT *data_pid, U16BIT *ad_pid);

/*!**************************************************************************
 * @fn      DMX_GetEcmPIDs
 * @brief   Get ECM PIDs set for decoding by demux path
 * @param   path - demux path
 * @param   ecm_pid - (out) program ECM PID
 * @param   video_pid - (out) video ECM PID
 * @param   audio_pid - (out) audio ECM PID
 * @param   text_pid - (out) text ECM PID
 * @param   data_pid - (out) data ECM PID
 * @param   ad_pid - (out) audio description ECM PID
 * @return  TRUE if successfull, FALSE otherwise
 ****************************************************************************/
BOOLEAN DMX_GetEcmPIDs(U8BIT path, U16BIT *ecm_pid, U16BIT *video_ecm_pid, U16BIT *audio_ecm_pid,
                          U16BIT *text_ecm_pid, U16BIT *data_ecm_pid, U16BIT *ad_ecm_pid);

/*!**************************************************************************
 * @fn      DMX_GetLiveAudioPID
 * @brief
 * @return  the audio pid for current LIVE TV
 ****************************************************************************/
U16BIT DMX_GetLiveAudioPID(void);

/*!**************************************************************************
 * @fn      DMX_GetRecordPath
 * @brief
 * @path	the record path
 * @return  the demux path for ts recording
 ****************************************************************************/
U8BIT DMX_GetRecordPath(U8BIT path);

/*!**************************************************************************
 * @fn      AV_IsVideoDecoding
 * @brief   Get video decoder status
 * @param   path - AV path ID
 * @return  TRUE if video decoder is started, FALSE otherwise
 ****************************************************************************/
BOOLEAN AV_IsVideoDecoding(U8BIT path);



/*!**************************************************************************
 * @fn      AV_IsAudioDecoding
 * @brief   Get audio decoder status
 * @param   path - AV path ID
 * @return  TRUE if audio decoder is started, FALSE otherwise
 ****************************************************************************/
BOOLEAN AV_IsAudioDecoding(U8BIT path);

/*!**************************************************************************
 * @fn      PVR_DeinitRecording
 * @brief
 * @param
 * @return
 ****************************************************************************/
void PVR_DeinitRecording(void);

/*!**************************************************************************
 * @fn      STB_AVRestartAV
 * @brief   Restart the Audio and Video decodings
 * @param   video_pid - video pid
 * @param   video_codec - video codec
 * @param   audio_pid - audio pid
 * @param   audio_codec - audio codec
 * @return
 ****************************************************************************/
void STB_AVRestartAV(U16BIT video_pid, U16BIT video_codec, U16BIT audio_pid, U16BIT audio_codec);

/*!**************************************************************************
 * @fn      STB_AVSetAudioTrack
 * @brief	switch multi-audio track
 * @param	audio_pid the audio pid of the track to be switched to.
 * @return	HI_SUCCESS if successful, others is failure.
 ****************************************************************************/
S32BIT STB_AVSetAudioTrack(U16BIT audio_pid);

#endif /*  _INTERNAL_H */

/****************************************************************************
** End of file
*****************************************************************************/


