#include "stbhwav.h"
#include <stdio.h>

/* STB header files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbhwmem.h"
#include "stbhwdmx.h"
#include "stbhwav.h"
#include "stbhwsdk.h"

extern "C" {
#include "stbdpc.h"
}

#include "internal_generic.h"
#include "internal.h"
#include "board_config.h"
#include "platform_public.h"

/*--- Preprocessor definitions ------------------------------------------------*/

#define VIDEO_FORMAT_NUMBER   (VIDEO_FORMAT_2160P60UHD + 1)
//#define DEFAULT_VIDEO_FORMAT  HI_UNF_ENC_FMT_720P_50

#define AV_DEBUGS
#define AV_WARNINGS
#define AV_ERRORS

#ifdef AV_DEBUGS
#define AV_DEBUG(format, args...) STB_DEBUG_LOG("AV", format, ##args)
#else
#define AV_DEBUG(format, args...)
#endif

#ifdef AV_WARNINGS
#define AV_WARNING(format, args...) STB_WARNING_LOG("AV", format, ##args)
#else
#define AV_WARNING(format, args...)
#endif

#ifdef AV_ERRORS
#define AV_ERROR(format, args...) STB_ERROR_LOG("AV", format, ##args)
#else
#define AV_ERROR(format, args...)
#endif

/*--- Local types definitions -------------------------------------------------*/

typedef struct av_status
{
   BOOLEAN is_audio_decoding;
   BOOLEAN is_video_decoding;
   BOOLEAN is_iframe_displayed;

   U8BIT demux;
   E_STB_AV_DECODE_SOURCE source;
   E_STB_AV_VIDEO_CODEC video_codec;
   E_STB_AV_AUDIO_CODEC audio_codec;

   /* Callback */
   S_STB_AV_VIDEO_INFO cb_info;
   void *cb_user_data;
   void (*callback)(S_STB_AV_VIDEO_INFO *, void *);

   /* If PVR playback need to be stopped STB_PVRPlayStop should be called first.
    * But DVBCore calls STB_AVStopAudioDecording/STB_AVStopVideoDecording
    * first which currently leads to errors. So this flag is a hack to prevent
    * this behaviour. It can be set from outside by AV_PreventStop */
   BOOLEAN prevent_stop;
} S_AV_STATUS;

/*--- Local (static) variable declarations ------------------------------------*/

/* Based on HI_UNF_PVR_PLAY_SPEED_E */
const static S16BIT SPEEDS_TABLE[] =
{
   -6400, -3200, -1600, -800, -400, -200, -100, -50, -25, -12, -6, -3, -1,
   1, 3, 6, 12, 25, 50, 100, 200, 400, 800, 1600, 3200, 6400
};

const static U32BIT SPEEDS_NUM = sizeof(SPEEDS_TABLE) / sizeof(S16BIT);


static BOOLEAN is_av_initialised = FALSE;
static U8BIT avs_num = 0;
static S_AV_STATUS *av_status = NULL;
static void *av_mutex = NULL;

/*--- Local function prototypes -----------------------------------------------*/

static void SetAVSource(U8BIT path, E_STB_AV_DECODE_SOURCE source, U32BIT param);
static BOOLEAN ChangeDemux(U8BIT path, U8BIT demux);
static void UpdateCallbackInfo(U8BIT path, S_STB_AV_VIDEO_INFO *info);

/* Conversion functions */
static int ConvertAudioCodecToPlatformAPI(E_STB_AV_AUDIO_CODEC audio_codec);
static int ConvertVideoCodecToPlatformAPI(E_STB_AV_VIDEO_CODEC video_codec);
//static void GetResolutionFromVideoFormat(E_STB_AV_VIDEO_FORMAT format, U16BIT *width, U16BIT *height);

/* Init functions */
static BOOLEAN InitAudioOutput(void);
static void DeinitAudioOutput(void);
static BOOLEAN InitVideoOutput(void);
static void DeinitVideoOutput(void);

void AV_Terminate(void);

/*--- Global function definitions ---------------------------------------------*/

/**
 * @brief   Initialises the AV components
 * @param   audio_paths number of audio decoders
 * @param   video_paths number of video decoders
 */
void STB_AVInitialise(U8BIT audio_paths, U8BIT video_paths)
{
   U8BIT i;

   FUNCTION_START(STB_AVInitialise);

   if (is_av_initialised)
   {
      AV_ERROR("AV is already initialised");
   }
   else
   {
      av_mutex = STB_OSCreateMutex();
      if (av_mutex == NULL)
      {
         AV_ERROR("Failed to create AV mutex");
      }
      else
      {
         if (!InitAudioOutput())
         {
            AV_ERROR("InitAudioOutput failed");
         }
         else if (!InitVideoOutput())
         {
            AV_ERROR("InitVideoOutput failed");
         }
         else
         {
            av_status = (S_AV_STATUS*) STB_MEMGetSysRAM(sizeof(S_AV_STATUS) * video_paths);
            if (av_status == NULL)
            {
               AV_ERROR("Failed to allocate memory for av_status");
            }
            else
            {
               {
                  avs_num = video_paths;
                  for (i = 0; i < video_paths; i++)
                  {
                     av_status[i].video_codec = AV_VIDEO_CODEC_AUTO;
                     av_status[i].audio_codec = AV_AUDIO_CODEC_AUTO;
                     av_status[i].source = AV_DEMUX;
                     av_status[i].demux = 0xFF;
                     av_status[i].is_audio_decoding = FALSE;
                     av_status[i].is_video_decoding = FALSE;
                     av_status[i].is_iframe_displayed = FALSE;
                     av_status[i].prevent_stop = FALSE;
                     av_status[i].callback = NULL;
                     av_status[i].cb_user_data = NULL;

                     /* Create AVPLAY */

                     /* Create window */

                     /* Create audiotrack */

                  }

                  if (avs_num == video_paths)
                  {
                     is_av_initialised = TRUE;
                  }
               }
            }
         }
      }
   }

   if (!is_av_initialised)
   {
      AV_Terminate();
      AV_ERROR("Failed to initialise AV");
   }

   FUNCTION_FINISH(STB_AVInitialise);
}

/**
 * @brief   Blanks or unblanks the video display
 * @param   path video path
 * @param   blank TRUE to blank, FALSE to unblank
 */
void STB_AVBlankVideo(U8BIT path, BOOLEAN blank)
{
   FUNCTION_START(STB_AVBlankVideo);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      STB_OSMutexLock(av_mutex);
      {
         AV_DEBUG("Video blank %s", (blank) ? "ON" : "OFF");
      }
      STB_OSMutexUnlock(av_mutex);
   }

   FUNCTION_FINISH(STB_AVBlankVideo);
}

/**
 * @brief   Sets the source of the input to the given video decoder path
 * @param   path video path to configure
 * @param   source the source device to use
 * @param   param parameter set with the source, typically the tuner or demux number
 */
void STB_AVSetVideoSource(U8BIT path, E_STB_AV_DECODE_SOURCE source, U32BIT param)
{
   FUNCTION_START(STB_AVSetVideoSource);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      SetAVSource(path, source, param);
   }
   FUNCTION_FINISH(STB_AVSetVideoSource);
}

/**
 * @brief   Sets the video codec to be used when decoding video with the given video decoder path
 * @param   path video path
 * @param   codec codec to be used
 * @return  TRUE if the codec is supported and is set correctly, FALSE otherwise
 */
BOOLEAN STB_AVSetVideoCodec(U8BIT path, E_STB_AV_VIDEO_CODEC codec)
{
   FUNCTION_START(STB_AVSetVideoCodec);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      av_status[path].video_codec = codec;
   }

   FUNCTION_FINISH(STB_AVSetVideoCodec);
   return TRUE;
}

/**
 * @brief   Returns the video codec previously set for the given video path.
 *          This function is currently unused within DVBCore.
 * @param   path video path
 * @return  codec previously set
 */
E_STB_AV_VIDEO_CODEC STB_AVGetVideoCodec(U8BIT path)
{
   E_STB_AV_VIDEO_CODEC retval = AV_VIDEO_CODEC_AUTO;

   FUNCTION_START(STB_AVGetVideoCodec);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      retval = av_status[path].video_codec;
   }

   FUNCTION_FINISH(STB_AVGetVideoCodec);
   return retval;
}

/**
 * @brief   Starts video decoding on the given video path
 * @param   path video decoder path to be started
 */
void STB_AVStartVideoDecoding(U8BIT path)
{
   BOOLEAN set_attr_failed;
   BOOLEAN successfull;
   BOOLEAN need_close_channel;
   BOOLEAN need_detach_window;
   U16BIT pcr_pid;
   U16BIT video_pid;
   E_STB_DMX_DEMUX_SOURCE dmx_source;
   U16BIT u16stub;
   U8BIT u8stub;

   FUNCTION_START(STB_AVStartVideoDecoding);

   successfull = FALSE;
   need_close_channel = FALSE;
   need_detach_window = FALSE;

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      STB_OSMutexLock(av_mutex);
      if (av_status[path].demux == INVALID_DEMUX)
      {
         AV_ERROR("Invalid demux");
      }
      else if (av_status[path].is_video_decoding)
      {
         AV_WARNING("Video decoding is already started");
      }
      else if (av_status[path].is_iframe_displayed)
      {
         AV_WARNING("I-frame is displayed. Need to release it first");
      }
      else
      {
         need_close_channel = TRUE;
         set_attr_failed = FALSE;

         if (!DMX_GetDecodePIDs(av_status[path].demux, &pcr_pid, &video_pid,
                                &u16stub, &u16stub, &u16stub, &u16stub))
         {
            AV_ERROR("Failed to fetch decoding PIDs from demux");
            set_attr_failed = TRUE;
         }
         else if (video_pid == 0 || video_pid > 0x1FFF || pcr_pid == 0 || pcr_pid >= 0x1FFF)
         {
            /* DVBCore often makes STB_AVStartVideoDecoding calls with wrong PIDs,
               It doesn't make sense to try to start it */
            AV_ERROR("Invalid Input: video_pid =%d, pcr_pid =%d",video_pid, pcr_pid);
            //set_attr_failed = TRUE;
            need_close_channel = FALSE;
            STB_OSMutexUnlock(av_mutex);
            return;
         }
         else
         {
            /* Set video PID */

            /* Set PCR PID */
         }

         /* Set video codec */
         RTK_Codec codec = (RTK_Codec) ConvertVideoCodecToPlatformAPI(av_status[path].video_codec);
         {
            /* Set syncronization attributes */

            // Detect framerate

            if (!set_attr_failed)
            {
               //attach video output window
               need_detach_window = TRUE;

               STB_DMXGetDemuxSource(av_status[path].demux, &dmx_source, &u8stub);
               {
                  RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(path);
                  if (NULL != pRealtekPlayer) {
                     if((pRealtekPlayer->RT_PlayerStart()) != RTK_OK) {

                     }
                     RTK_VideoConfig videoConfig;
                     memset(&videoConfig, 0, sizeof(RTK_VideoConfig));
                     videoConfig.pid = video_pid;
                     videoConfig.codec = codec;
                     videoConfig.output = RTK_WIN_MAIN;
                     videoConfig.encrypted = RTK_FALSE;
                     pRealtekPlayer->RT_Player_ChangeVideo(&videoConfig);

                     AV_DEBUG("Video decoding started");
                     av_status[path].is_video_decoding = TRUE;
                     successfull = TRUE;
                     if((!av_status[path].is_video_decoding)||(dmx_source == DMX_MEMORY)) {
                        /* Set syncronization attributes */
                     }

                     AV_DEBUG("Video decoding started");
                  }

                  //start
                  {
                     AV_DEBUG("Video decoding started");
                     av_status[path].is_video_decoding = TRUE;
                     //av_status[path].avplay_handle = avplay;
                     // STB_AVBlankVideo(path, FALSE);  /* Don't force unblank video here, Removed by Simon Zhou */
                     successfull = TRUE;
                  }
               }  // if (dmx_source == DMX_MEMORY)
            }
         }
      }

      if (!successfull)
      {
         if (need_detach_window)
         {
            // detatch the video output window
         }
         if (need_close_channel)
         {
            // close channel
         }
      }
      STB_OSMutexUnlock(av_mutex);
   }

   FUNCTION_FINISH(STB_AVStartVideoDecoding);
}

/**
 * @brief   Pauses video decoding on the given video path.
 *          The video should not be blanked.
 * @param   path video decoder path to be paused
 */
void STB_AVPauseVideoDecoding(U8BIT path)
{

}

/**
 * @brief   Resumes video decoding on the given video path that has previously been paused
 *          The video should not be unblanked.
 * @param   path video decoder path to be resumed
 */
void STB_AVResumeVideoDecoding(U8BIT path)
{

}

/**
 * @brief   Stops video decoding on the given video path.
 *          The video is not expected to be blanked.
 * @param   path video decoder path to be stopped
 */
void STB_AVStopVideoDecoding(U8BIT decoder)
{
   E_STB_DMX_DEMUX_SOURCE dmx_source;
   U8BIT u8stub;
   S_STB_AV_VIDEO_INFO info;

   FUNCTION_START(STB_AVStopVideoDecoding);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (decoder >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", decoder, avs_num - 1);
   }
   else
   {
      STB_OSMutexLock(av_mutex);
      if (!av_status[decoder].is_video_decoding)
      {
         AV_WARNING("Nothing to stop. Video decoding was not started yet");
      }
      else if (!av_status[decoder].prevent_stop)
      {
         U8BIT decode_path = STB_DPGetPathOfVideoDecoder(decoder);
         RTK_Player hRTK_Player = STB_SdkGetPlayer(decode_path);
         if (NULL != hRTK_Player) {
            RTK_VideoConfig videoConfig;
            memset(&videoConfig, 0, sizeof(RTK_VideoConfig));
            videoConfig.codec = RTK_CODEC_NONE;
            RTK_Player_ChangeVideo(hRTK_Player, &videoConfig);

            AV_DEBUG("Video decoding stoped");
         }

         av_status[decoder].is_video_decoding = FALSE;

         STB_DMXGetDemuxSource(av_status[decoder].demux, &dmx_source, &u8stub);
         if (dmx_source == DMX_TUNER)
         {
            STB_OSSendEvent(FALSE, HW_EV_CLASS_DECODE, HW_EV_TYPE_VIDEO_STOPPED,
                            &decoder, sizeof(U8BIT));

            info.flags = VIDEO_INFO_DECODER_STATUS;
            info.status = DECODER_STATUS_NONE;
            UpdateCallbackInfo(decoder, &info);
         }
      }
      STB_OSMutexUnlock(av_mutex);
   }

   FUNCTION_FINISH(STB_AVStopVideoDecoding);
}

/**
 * @brief   Sets the source of the input for the main audio on the given audio decoder path
 * @param   path audio path to configure
 * @param   source the source device to use
 * @param   param parameter set with the source, typically the tuner or demux number
 */
void STB_AVSetAudioSource(U8BIT path, E_STB_AV_DECODE_SOURCE source, U32BIT param)
{
   FUNCTION_START(STB_AVSetAudioSource);
   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      SetAVSource(path, source, param);
   }
   FUNCTION_FINISH(STB_AVSetAudioSource);
}

/**
 * @brief   Sets the audio codec to be used when decoding audio with the given audio decoder path
 * @param   path audio path
 * @param   codec codec to be used
 * @return  TRUE if the codec is supported and is set correctly, FALSE otherwise
 */
BOOLEAN STB_AVSetAudioCodec(U8BIT path, E_STB_AV_AUDIO_CODEC codec)
{
   FUNCTION_START(STB_AVSetAudioCodec);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      av_status[path].audio_codec = codec;
   }

   FUNCTION_FINISH(STB_AVSetAudioCodec);
   return TRUE;
}

/**
 * @brief   Returns the audio codec previously set for the given audio path.
 *          This function is currently unused within DVBCore.
 * @param   path audio path
 * @return  codec previously set
 */
E_STB_AV_AUDIO_CODEC STB_AVGetAudioCodec(U8BIT path)
{
   E_STB_AV_AUDIO_CODEC retval = AV_AUDIO_CODEC_AUTO;

   FUNCTION_START(STB_AVGetAudioCodec);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      retval = av_status[path].audio_codec;
   }

   FUNCTION_FINISH(STB_AVGetAudioCodec);
   return retval;
}

/**
 * @brief   Configures the main audio channel mode (stereo/left/right) in the case where
 *          dual-mono audio is used, such that only the audio from one channel is heard.
 * @param   path audio path to be configured
 * @param   mode audio mode to use
 */
void STB_AVChangeAudioMode(U8BIT path, E_STB_AV_AUDIO_MODE mode)
{
   FUNCTION_START(STB_AVChangeAudioMode);
   USE_UNWANTED_PARAM(path);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else
   {
      switch (mode)
      {
         case AV_AUDIO_STEREO:        break;
         case AV_AUDIO_LEFT:          break;
         case AV_AUDIO_RIGHT:         break;
         case AV_AUDIO_MONO:          break;
         case AV_AUDIO_MULTICHANNEL:  break;
      }

   }

   FUNCTION_FINISH(STB_AVChangeAudioMode);
}

/**
 * @brief   Starts audio decoding on the given audio path
 * @param   path audio decoder path to be started
 */
void STB_AVStartAudioDecoding(U8BIT decoder)
{
   BOOLEAN set_attr_failed;
   U16BIT audio_pid;
   U16BIT pcr_pid;

   /* Failure destructure logic */
   BOOLEAN successfull;
   BOOLEAN need_close_channel;
   BOOLEAN need_detach_audiotrack;

   E_STB_DMX_DEMUX_SOURCE dmx_source;
   U16BIT u16stub;
   U8BIT u8stub;

   FUNCTION_START(STB_AVStartAudioDecoding);

   successfull = FALSE;
   need_close_channel = FALSE;
   need_detach_audiotrack = FALSE;

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (decoder >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", decoder, avs_num - 1);
   }
   else
   {
      STB_OSMutexLock(av_mutex);
      if (av_status[decoder].demux == INVALID_DEMUX)
      {
         AV_ERROR("Invalid demux");
      }
      else if (av_status[decoder].is_audio_decoding)
      {
         AV_WARNING("Audio decording is already started");
      }
      else
      {
         {
            need_close_channel = TRUE;
            set_attr_failed = FALSE;

            if (!DMX_GetDecodePIDs(av_status[decoder].demux, &pcr_pid, &u16stub,
                                   &audio_pid, &u16stub, &u16stub, &u16stub))
            {
               AV_ERROR("Failed to fetch decoding PIDs from demux");
               set_attr_failed = TRUE;
               STB_OSMutexUnlock(av_mutex);
               return;
            }
            else if (audio_pid == 0 || audio_pid > 0x1FFF)
            {
               /* DVBCore often makes STB_AVStartAudioDecoding calls with wrong PIDs,
                  It doesn't make sense to try to start it */
               set_attr_failed = TRUE;
               STB_OSMutexUnlock(av_mutex);
               return;
            }
            else
            {
               /* Set audio PID */
               AV_DEBUG("Setting audio PID: %d", audio_pid);

            }

            /* Set audio codec */
            RTK_Codec codec = (RTK_Codec) ConvertAudioCodecToPlatformAPI(av_status[decoder].audio_codec);
            need_detach_audiotrack = TRUE;
            STB_DMXGetDemuxSource(av_status[decoder].demux, &dmx_source, &u8stub);
            {
               U8BIT path = av_status[decoder].demux;
               U8BIT decode_path = STB_DPGetPathOfDemux(path);
               RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(decode_path);
               if (NULL != pRealtekPlayer) {
               	  if((pRealtekPlayer->RT_PlayerStart()) != RTK_OK) {

                  }

                  if (pRealtekPlayer->RT_SetAudioMute(RTK_FALSE) != RTK_OK) {

                  }

                  RTK_AudioConfig audioConfig;            
                  memset(&audioConfig, 0, sizeof(RTK_AudioConfig));
                  audioConfig.pid = audio_pid;
                  audioConfig.codec = codec;
                  audioConfig.encrypted = RTK_FALSE;
                  audioConfig.enable_raw_output_mode = RTK_FALSE;
                  audioConfig.focused = RTK_TRUE;
                  pRealtekPlayer->RT_Player_ChangeAudio(&audioConfig);

                  AV_DEBUG("Audio decoding started");
                  av_status[decoder].is_audio_decoding = TRUE;
                  successfull = TRUE;

                  if((!av_status[decoder].is_video_decoding)||(dmx_source == DMX_MEMORY)) {
                     /* Set syncronization attributes */
                  }

                  RTK_Player hRTK_Player = pRealtekPlayer->getDemux();
                  if (NULL != hRTK_Player) {
                     if(RTK_Player_ChangePCR(hRTK_Player, pcr_pid) != RTK_OK) {
	                     RTK_PLAYER_ERR("Failed to change pcr.\n");
                     }
                  }

                  AV_DEBUG("Audio decoding started");
               }
            }
         }
      }

      if (!successfull)
      {
         if (need_detach_audiotrack)
         {
            // detatch audio output;

         }
         if (need_close_channel)
         {
            // close channel
         }
      }
      STB_OSMutexUnlock(av_mutex);
   }

   FUNCTION_FINISH(STB_AVStartAudioDecoding);
}

void STB_AVRestartAV(U16BIT video_pid, U16BIT video_codec, U16BIT audio_pid, U16BIT audio_codec)
{
	/* Set audio PID */
	AV_DEBUG("Setting audio PID: %d", audio_pid);

	/* Set audio codec */

	/* Set video PID */

	/* Set video codec */

	/* Set syncronization attributes */
}

/**
 * @brief   Stops audio decoding on the given audio path.
 * @param   path audio decoder path to be stopped
 */
void STB_AVStopAudioDecoding(U8BIT decoder)
{
   U8BIT decode_path = STB_DPGetPathOfAudioDecoder(decoder);
   RTK_Player hRTK_Player = STB_SdkGetPlayer(decode_path);
   if (NULL != hRTK_Player) {
	   //if(RTK_Player_Start(hRTK_Player) == RTK_OK) {
         RTK_AudioConfig audioConfig;
         
         memset(&audioConfig, 0, sizeof(RTK_AudioConfig));
         audioConfig.codec = RTK_CODEC_NONE;
	      RTK_Player_ChangeAudio(hRTK_Player, &audioConfig);

         AV_DEBUG("Audio decoding stoped");
         av_status[decoder].is_audio_decoding = FALSE;
	   //}
   }
}

/**
 * @brief   Sets the volume of the main audio output
 * @param   path audio path to be configured
 * @param   vol audio volume (0-100%)
 */
void STB_AVSetAudioVolume(U8BIT path, U8BIT vol)
{
   FUNCTION_START(STB_AVSetAudioVolume);
   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      if (vol > 100)
      {
         vol = 100;
      }
   }
   FUNCTION_FINISH(STB_AVSetAudioVolume);
}

/**
 * @brief   Returns the current volume of the main audio output
 * @param   path audio path
 * @return  audio volume (0-100%)
 */
U8BIT STB_AVGetAudioVolume(U8BIT path)
{
   U8BIT retval;
   FUNCTION_START(STB_AVGetAudioVolume);

   retval = 100;
   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {

   }

   FUNCTION_FINISH(STB_AVGetAudioVolume);
   return retval;
}

/**
 * @brief   Mute or unmute the audio output on the given audio decoder path
 * @param   path audio path to be configured
 * @param   mute TRUE to mute, FALSE to unmute
 */
void STB_AVSetAudioMute(U8BIT path, BOOLEAN mute)
{
   FUNCTION_START(STB_AVSetAudioMute);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {

   }

   FUNCTION_FINISH(STB_AVSetAudioMute);
}

/**
 * @brief   Returns the current mute setting of the audio output on the given path
 * @param   path audio path
 * @return  TRUE if audio is muted, FALSE otherwise
 */
BOOLEAN STB_AVGetAudioMute(U8BIT path)
{
   BOOLEAN retval;

   FUNCTION_START(STB_AVGetAudioMute);

   retval = FALSE;
   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {

   }

   FUNCTION_FINISH(STB_AVGetAudioMute);
   return retval;
}

/**
 * @brief   Sets the source of the input for the audio description audio on the
 *          given audio decoder path
 * @param   path audio path to configure
 * @param   source the source device to use
 * @param   param parameter set with the source, typically the tuner or demux number
 */
void STB_AVSetADSource(U8BIT path, E_STB_AV_DECODE_SOURCE source, U32BIT param)
{
   FUNCTION_START(STB_AVSetADSource);
   USE_UNWANTED_PARAM(path);
   USE_UNWANTED_PARAM(source);
   USE_UNWANTED_PARAM(param);
   FUNCTION_FINISH(STB_AVSetADSource);
}

/**
 * @brief   Sets the codec to be used for audio description when decoding audio
 *          with the given audio decoder path
 * @param   path audio path
 * @param   codec codec to be used
 * @return  TRUE if the codec is supported and is set correctly, FALSE otherwise
 */
BOOLEAN STB_AVSetADCodec(U8BIT path, E_STB_AV_AUDIO_CODEC codec)
{
   FUNCTION_START(STB_AVSetADCodec);
   USE_UNWANTED_PARAM(path);
   USE_UNWANTED_PARAM(codec);
   FUNCTION_FINISH(STB_AVSetADCodec);
   return FALSE;
}

/**
 * @brief   Returns the codec previously set for audio description on the given audio path.
 *          This function is currently unused within DVBCore.
 * @param   path audio path
 * @return  codec previously set
 */
E_STB_AV_AUDIO_CODEC STB_AVGetADCodec(U8BIT path)
{
   FUNCTION_START(STB_AVGetADCodec);
   USE_UNWANTED_PARAM(path);
   FUNCTION_FINISH(STB_AVGetADCodec);
   return AV_AUDIO_CODEC_AUTO;
}

/**
 * @brief   Configures the audio description channel mode (stereo/left/right) in the case where
 *          dual-mono audio is used, such that only the audio from one channel is heard.
 * @param   path audio path to be configured
 * @param   mode audio mode to use
 */
void STB_AVChangeADMode(U8BIT path, E_STB_AV_AUDIO_MODE mode)
{
   FUNCTION_START(STB_AVChangeADMode);
   USE_UNWANTED_PARAM(path);
   USE_UNWANTED_PARAM(mode);
   FUNCTION_FINISH(STB_AVChangeADMode);
}

/**
 * @brief   Starts decoding audio description on the given audio path
 * @param   path audio decoder path to be started
 */
void STB_AVStartADDecoding(U8BIT decoder)
{
   FUNCTION_START(STB_AVStartADDecoding);
   USE_UNWANTED_PARAM(decoder);
   FUNCTION_FINISH(STB_AVStartADDecoding);
}

/**
 * @brief   Stops decoding audio description on the given audio path.
 * @param   path audio decoder path to be stopped
 */
void STB_AVStopADDecoding(U8BIT decoder)
{
   FUNCTION_START(STB_AVStopADDecoding);
   USE_UNWANTED_PARAM(decoder);
   FUNCTION_FINISH(STB_AVStopADDecoding);
}

/**
 * @brief   Sets the volume of the audio description output
 * @param   path audio path to be configured
 * @param   vol audio volume (0-100%)
 */
void STB_AVSetADVolume(U8BIT path, U8BIT vol)
{
   FUNCTION_START(STB_AVSetADVolume);
   USE_UNWANTED_PARAM(path);
   USE_UNWANTED_PARAM(vol);
   FUNCTION_FINISH(STB_AVSetADVolume);
}

/**
 * @brief   Returns the current volume of the audio description output
 * @param   path audio path
 * @return  audio volume (0-100%)
 */
U8BIT STB_AVGetADVolume(U8BIT path)
{
   FUNCTION_START(STB_AVGetADVolume);
   USE_UNWANTED_PARAM(path);
   FUNCTION_FINISH(STB_AVGetADVolume);
   return 0;
}

/**
 * @brief   Returns the current 33-bit System Time Clock from the PCR PES.
 *          On some systems, this information may need to be obtained from the associated demux,
 *          which will be contained in the 'param' value when STB_AVSetVideoSource is called.
 * @param   path video path
 * @param   stc an array in which the STC will be returned, ordered such that
 *                stc[0] contains the MS bit (33) of the STC value and stc[4]
 *                contains the LS bits (0-7).
 */
void STB_AVGetSTC(U8BIT path, U8BIT stc[5])
{
   uint64_t pts = 0;

   FUNCTION_START(STB_AVGetSTC);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else if (NULL == stc) {
      AV_ERROR("NULL pointer of the STC buffer");   
   }
   else
   {
      U8BIT decode_path = STB_DPGetPathOfVideoDecoder(path);
      RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(decode_path);
      if (NULL != pRealtekPlayer) {
         RTK_PTSInfo pts_info;
         RTK_Error ret;

         ret = pRealtekPlayer->getCurrentDisplayingPTSInfo (&pts_info);
         if(RTK_OK == ret) {
            pts = pts_info.presentVideoPts;	
            //pts = pts_info.presentAudioPts;
            pts += 63000;  // FIXME: 700ms ahead of PTS, 90k == 1s 
            //AV_DEBUG("getCurrentDisplayingPTSInfo  presentVideoPts=%llu, presentAudioPts=%llu\n", pts_info.presentVideoPts, pts_info.presentAudioPts);

            stc[4] = pts         & 0xFF;
            stc[3] = (pts >> 8)  & 0xFF;
            stc[2] = (pts >> 16) & 0xFF;
            stc[1] = (pts >> 24) & 0xFF;
            stc[0] = (pts >> 32) & 0x01;
         }
      }
   }

   FUNCTION_FINISH(STB_AVGetSTC);
}

/**
 * @brief   Sets the aspect ratio and signal format for the connected television
 * @param   path The video path to be set
 * @param   ratio The aspect ratio of the tv
 * @param   format The signal format of the tv
 */
void STB_AVSetTVType(U8BIT path, E_STB_AV_ASPECT_RATIO ratio, E_STB_AV_VIDEO_FORMAT format)
{

}

/**
 * @brief   Returns the current size of the screen in pixels
 * @param   path 0
 * @param   width returned width
 * @param   height returned height
 */
void STB_AVGetScreenSize(U8BIT path, U16BIT *width, U16BIT *height)
{
    *width = 1920;
    *height = 1080;
}

/**
 * @brief   Routes a specified AV source to a specified AV output
 * @param   output The output to be connected
 * @param   source The source signal to be connected
 * @param   param parameter associated with the source
 */
void STB_AVSetAVOutputSource(E_STB_AV_OUTPUTS output, E_STB_AV_SOURCES source, U32BIT param)
{

}

/**
 * @brief   Turns on/off all AV outputs (e.g. for standby mode)
 * @param   av_on TRUE to turn on, FALSE to turn off
 */
void STB_AVSetAVOutput(BOOLEAN av_on)
{

}

/**
 * @brief   Sets the standby state of the HDMI output
 * @param   standby TRUE to put the HDMI in standby, FALSE to come out of standby
 */
void STB_AVSetHDMIStandby(BOOLEAN standby)
{

}

/**
 * @brief   Returns the resolutions supported by the HDMI
 * @param   modes array of supported modes
 * @return  number of supported modes
 */
U8BIT STB_AVGetHDMISupportedModes(E_STB_AV_VIDEO_FORMAT **modes)

{
    return 1;
}

/**
 * @brief   Returns the native resolution, i.e. the resolution that is
 *          set with a call to STB_AVSetTVType(path, ratio, VIDEO_FORMAT_AUTO);
 * @param   width pointer to the variable where to store the width
 * @param   height pointer to the variable where to store the height
 */
void STB_AVGetHDMINativeResolution(U16BIT *width, U16BIT *height)
{
    *width = 1920;
    *height = 1080;
}

/**
 * @brief   Enables AV output to HDMI
 */
void STB_AVEnableHDMIDecoding(void)
{
   FUNCTION_START(STB_AVEnableHDMIDecoding);

   FUNCTION_FINISH(STB_AVEnableHDMIDecoding);
}

/**
 * @brief   Disables AV output to HDMI
 */
void STB_AVDisableHDMIDecoding(void)
{
   FUNCTION_START(STB_AVDisableHDMIDecoding);

   FUNCTION_FINISH(STB_AVDisableHDMIDecoding);
}

/**
 * @brief   Returns whether HDCP has authenticated
 * @return  TRUE if authenticated, FALSE otherwise
 */
BOOLEAN STB_AVIsHDCPAuthenticated(void)
{
	return FALSE;
}

/**
 * @brief   Sets the codec to be used when decoding the next i-frame from memory
 * @param   path video path
 * @param   codec codec to be used
 * @return  TRUE if the codec is supported and is set correctly, FALSE otherwise
 */
BOOLEAN STB_AVSetIFrameCodec(U8BIT path, E_STB_AV_VIDEO_CODEC codec)
{
    return TRUE;
}

/**
 * @brief   Provides the video data for an i-frame for subsequent decode and display
 * @param   path video decode path to be used
 * @param   data i-frame data to be saved for decoding
 * @param   size size of the data in bytes
 */
void STB_AVLoadIFrame(U8BIT path, U8BIT *data, U32BIT size)
{

}

/**
 * @brief   Decodes and displays the previously loaded i-frame data
 * @param   path video path to use
 */
void STB_AVShowIFrame(U8BIT path)
{

}

/**
 * @brief   Hides the i-frame currently being displayed
 * @param   path video path to use
 */
void STB_AVHideIFrame(U8BIT path)
{

}

/**
 * @brief   Plays back a previously loaded audio sample
 * @param   path the audio path to use for playback
 * @param   loop_count the number of times to play the sample, 0=forever
 * @return  HW_OK on success, error code otherwise
 */
E_HW_STATUS STB_AVPlayAudioSample(U8BIT path, U32BIT loop_count)
{
    return HW_OK;
}

/**
 * @brief   Loads an audio sample for subsequent playback
 * @param   path the decoder path to use for playback
 * @param   data the audio sample data to be loaded
 * @param   size the size of the audio sample in bytes
 * @return  HW_OK on success, error code otherwise
 */
E_HW_STATUS STB_AVLoadAudioSample(U8BIT path, U8BIT *data, U32BIT size)
{
    return HW_OK;
}

/**
 * @brief   Pauses playback of an audio sample
 * @param   path Audio path on which to pause
 * @return  HW_OK on success, error code otherwise
 */
E_HW_STATUS STB_AVPauseAudioSample(U8BIT path)
{
    return HW_OK;
}

/**
 * @brief   Resumes playback of an audio sample
 * @param   path Audio path on which to resume
 * @return  HW_OK on success, error code otherwise
 */
E_HW_STATUS STB_AVResumeAudioSample(U8BIT path)
{
    return HW_OK;
}

/**
 * @brief   Stops playback of an audio sample
 * @param   path Audio path on which to stop
 */
void STB_AVStopAudioSample(U8BIT path)
{

}

/**
 * @brief   Sets the SPDIF output mode, PCM or compressed audio
 * @param   path decoder path
 * @param   spdif_type PCM or compressed
 */
void STB_AVSetSpdifMode(U8BIT path, E_STB_DIGITAL_AUDIO_TYPE audio_type)
{
   FUNCTION_START(STB_AVSetSpdifMode);
   USE_UNWANTED_PARAM(path);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else
   {
      switch(audio_type)
      {
         case DIGITAL_AUDIO_AUTO:
         [[fallthrough]];
         case DIGITAL_AUDIO_PCM:
            break;
         case DIGITAL_AUDIO_COMPRESSED:
            break;
      }
   }

   FUNCTION_FINISH(STB_AVSetSpdifMode);
}

/**
 * @brief   Sets the HDMI audio output mode, PCM or compressed
 * @param   path decoder path
 * @param   audio_type PCM or compressed
 */
void STB_AVSetHDMIAudioMode(U8BIT path, E_STB_DIGITAL_AUDIO_TYPE audio_type)
{
   FUNCTION_START(STB_AVSetHDMIAudioMode);
   USE_UNWANTED_PARAM(path);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else
   {
      switch(audio_type)
      {
         case DIGITAL_AUDIO_AUTO:
            break;
         case DIGITAL_AUDIO_PCM:
            break;
         case DIGITAL_AUDIO_COMPRESSED:
            break;
      }
   }
   FUNCTION_FINISH(STB_AVSetHDMIAudioMode);
}

/**
 * @brief   Sets the audio delay on the given path
 * @param   path decoder path
 * @param   millisecond audio delay to be applied
 */
void STB_AVSetAudioDelay(U8BIT path, U16BIT millisecond)
{
   FUNCTION_START(STB_AVSetAudioDelay);

   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else
   {

   }

   FUNCTION_FINISH(STB_AVSetAudioDelay);
}

/**
 * @brief   Register callback for updated video information
 * @param   path decoder path
 * @param   callback the callback to call when video information is updated
 * @param   user_data user data to pass to the callback
 */
void STB_AVSetVideoCallback(U8BIT path, void (*callback)(S_STB_AV_VIDEO_INFO *, void *), void *user_data)
{

}

/**
 * @brief   Apply video transformation
 * @param   path decoder path
 * @param   input input video rectangle
 * @param   output output video rectangle
 */
void STB_AVApplyVideoTransformation(U8BIT path, S_RECTANGLE *input, S_RECTANGLE *output)
{

}

/**
 * @brief   Returns minimum video play speed as a percentage.
 * @param   video_decoder video decoder path
 * @return  Minimum play speed.
 */
S16BIT STB_AVGetMinPlaySpeed(U8BIT video_decoder)
{
    return 100;
}

/**
 * @brief   Returns maximum video play speed as a percentage.
 * @param   video_decoder video decoder path
 * @return  Maximum play speed.
 */
S16BIT STB_AVGetMaxPlaySpeed(U8BIT video_decoder)
{
    return 3200;
}

/**
 * @brief   Returns the next valid speed that is +/- inc above or below the
 *          given speed. Slow motion speeds (>-100% and < 100%) can be included.
 * @param   path Decode path
 * @param   speed Percentage speed above/below which the new speed is calculated
 * @param   inc number of speeds above that specified to return
 * @param   include_slow_speeds selects whether speeds >-100% and <100% are included
 * @return  Speed as a percentage
 */
S16BIT STB_AVGetNextPlaySpeed(U8BIT video_decoder, S16BIT speed, S16BIT inc,
   BOOLEAN include_slow_speeds)
{
    return 100;
}

/**
 * @brief Apply the specified copy protection. This function is used for CI+
 * @param copy_protection
 */
void STB_AVSetCopyProtection(S_STB_AV_COPY_PROTECTION *copy_protection)
{

}

/**
 * @brief   Sets the output channel of the RF Modulator
 * @param   chan the UHF channel number
 */
void STB_AVSetUhfModulatorChannel(U8BIT chan)
{

}

/**
 * @brief   Gets the current RF modulator channel
 * @return  The RF frequency as a UHF channel number
 */
U8BIT STB_AVGetUhfModulatorChannel(void)
{
    return 0;
}

/**
 * @brief   Returns the frame rate of the video being decoded
 * @param   path video path
 * @return  video frame rate in frame per seconds
 */
U8BIT STB_AVGetVideoFrameRate(U8BIT path)
{
    return 30;
}

/**
 * @brief   Apply System Renewability Message (SRM) to HDCP function
 * @param   path output path
 * @param   data SRM data
 * @param   len length of SRM data in bytes
 * @return  SRM_OK, SRM_BUSY or SRM_NOT_REQUIRED
 */
E_STB_AV_SRM_REPLY STB_AVApplySRM(U8BIT path, U8BIT *data, U32BIT len)
{
    return SRM_OK;
}

void STB_AVCECOneTouchPlay(void)
{

}

E_STB_AV_VIDEO_FORMAT STB_AVGetHDMINativeFormat(void)
{
    return VIDEO_FORMAT_AUTO;
}

void STB_AVPlayMp3(U8BIT path, U8BIT *buffer, U32BIT buffer_size)
{

}

void STB_AVStopMp3(U8BIT path)
{

}



/*!**************************************************************************
 * @fn      AV_Terminate
 * @brief
 * @param
 * @return
 ****************************************************************************/
void AV_Terminate(void)
{
   U8BIT i;

   FUNCTION_START(AV_Terminate);

   is_av_initialised = FALSE;
   for (i = 0; i < avs_num; i++)
   {
      if (av_status[i].is_video_decoding)
      {
         STB_AVStopVideoDecoding(i);
      }
      if (av_status[i].is_audio_decoding)
      {
         STB_AVStopAudioDecoding(i);
      }
   }

   if (av_status != NULL)
   {
      STB_MEMFreeSysRAM(av_status);
      av_status = NULL;
   }
   avs_num = 0;

   DeinitVideoOutput();
   DeinitAudioOutput();
   STB_OSDeleteMutex(av_mutex);

   FUNCTION_FINISH(AV_Terminate);
}



/*!**************************************************************************
 * @fn      AV_IsVideoDecoding
 * @brief   Get video decoder status
 * @param   path - AV path ID
 * @return  TRUE if video decoder is started, FALSE otherwise
 ****************************************************************************/
BOOLEAN AV_IsVideoDecoding(U8BIT path)
{
   BOOLEAN retval;

   FUNCTION_START(AV_IsVideoDecoding);

   retval = FALSE;
   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      retval = av_status[path].is_video_decoding;
   }

   FUNCTION_FINISH(AV_IsVideoDecoding);
   return retval;
}



/*!**************************************************************************
 * @fn      AV_IsAudioDecoding
 * @brief   Get audio decoder status
 * @param   path - AV path ID
 * @return  TRUE if audio decoder is started, FALSE otherwise
 ****************************************************************************/
BOOLEAN AV_IsAudioDecoding(U8BIT path)
{
   BOOLEAN retval;

   FUNCTION_START(AV_IsAudioDecoding);

   retval = FALSE;
   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      retval = av_status[path].is_audio_decoding;
   }

   FUNCTION_FINISH(AV_IsAudioDecoding);
   return retval;
}



/*!**************************************************************************
 * @fn      AV_PreventStop
 * @brief   Prevent audio/video decording from stopping by STB_AVStop*Decoding.
 *          See more details in S_AV_STATUS struct description
 * @param   prevent - if TRUE, STB_AVStop*Decoding will return early.
 ****************************************************************************/
void AV_PreventStop(U8BIT path, BOOLEAN prevent_stop)
{
   FUNCTION_START(AV_PreventStop);
   if (!is_av_initialised)
   {
      AV_ERROR("AV is not initialised");
   }
   else if (path >= avs_num)
   {
      AV_ERROR("Wrong AV ID: %d (max is %d)", path, avs_num - 1);
   }
   else
   {
      av_status[path].prevent_stop = prevent_stop;
   }
   FUNCTION_FINISH(AV_PreventStop);
}

void STB_AVStartDecoding(U8BIT path)
{
   if (!is_av_initialised)
      return;
   //if (path >= avs_num)
     // return;

   RTK_VideoConfig videoConfig;
   RTK_AudioConfig audioConfig;
   U8BIT decode_path = STB_DPGetPathOfVideoDecoder(path);
   RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(decode_path);
   if (NULL == pRealtekPlayer)
      return;

  // if((pRealtekPlayer->RT_PlayerStart()) != RTK_OK)
  //    return;

   // FCC switch decode
   U16BIT pcr_pid;
   U16BIT video_pid;
   U16BIT audio_pid;
   U16BIT text_pid;
   U16BIT data_pid;
   U16BIT ad_pid;

   U16BIT ecm_pid;
   U16BIT video_ecm_pid;
   U16BIT audio_ecm_pid;
   U16BIT text_ecm_pid;
   U16BIT data_ecm_pid;
   U16BIT ad_ecm_pid;

   RTK_Codec codec;
   E_STB_DMX_DEMUX_SOURCE dmx_source;
   U16BIT u16stub;
   U8BIT u8stub;

   FUNCTION_START(STB_AVStartDecoding);


   STB_OSMutexLock(av_mutex);
   if (av_status[path].demux == INVALID_DEMUX) {
      av_status[path].demux = av_status[0].demux;
      av_status[0].demux = INVALID_DEMUX;
   }
   if (av_status[path].demux == INVALID_DEMUX) {
      STB_OSMutexUnlock(av_mutex);
      return;
   }
   if (!DMX_GetDecodePIDs(av_status[path].demux, &pcr_pid, &video_pid,
                          &audio_pid, &text_pid, &data_pid, &ad_pid))
   {
      STB_OSMutexUnlock(av_mutex);
      return;
   }
   if (video_pid == 0 || video_pid > 0x1FFF || pcr_pid == 0 || pcr_pid >= 0x1FFF)
   {
      STB_OSMutexUnlock(av_mutex);
      return;
   }

   if(!DMX_GetEcmPIDs(av_status[path].demux, &ecm_pid, &video_ecm_pid, &audio_ecm_pid,
                          &text_ecm_pid, &data_ecm_pid, &ad_ecm_pid))
   {
      STB_OSMutexUnlock(av_mutex);
      return;   
   }

   /* Set video codec */
   codec = (RTK_Codec) ConvertVideoCodecToPlatformAPI(av_status[path].video_codec);
   memset(&videoConfig, 0, sizeof(RTK_VideoConfig));
   videoConfig.pid = video_pid;
   videoConfig.codec = codec;
   //videoConfig.output = RTK_WIN_MAIN;
   //videoConfig.encrypted = RTK_FALSE;
   av_status[path].is_video_decoding = TRUE;

   /* Set audio codec */
   U8BIT decoder = 0;  // av_status[path].source
   codec = (RTK_Codec) ConvertAudioCodecToPlatformAPI(av_status[decoder].audio_codec); 
   memset(&audioConfig, 0, sizeof(RTK_AudioConfig));
   audioConfig.pid = audio_pid;
   audioConfig.codec = codec;
   //audioConfig.encrypted = RTK_FALSE;
   //audioConfig.enable_raw_output_mode = RTK_FALSE;
   audioConfig.focused = RTK_TRUE;
   av_status[decoder].is_audio_decoding = TRUE;

   STB_OSMutexUnlock(av_mutex);

   pRealtekPlayer->enableKeepLastFrameForChangeChannel(RTK_TRUE);
   pRealtekPlayer->RT_Player_ChangeChannel(&videoConfig, &audioConfig);

   FUNCTION_FINISH(STB_AVStartDecoding);
}


/*--- Local function definitions ----------------------------------------------*/

/*!**************************************************************************
 * @fn      SetAVSource
 * @brief   Sets the source of the input to the given decoder path
 * @param   path - AV path to configure
 * @param   source - the source device to use
 * @param   param - parameter set with the source, typically the tuner or
 *          demux number
 ****************************************************************************/
static void SetAVSource(U8BIT path, E_STB_AV_DECODE_SOURCE source, U32BIT param)
{
   FUNCTION_START(SetAVSource);

   STB_OSMutexLock(av_mutex);

   if (av_status[path].is_video_decoding || av_status[path].is_audio_decoding)
   {
      AV_ERROR("AV decoding active: need to stop decoding first");
   }
   else if (av_status[path].source != source ||
            (source == AV_DEMUX && av_status[path].demux != (U8BIT) param))
   {
      /* Update source */
      av_status[path].source = source;
      if (source == AV_DEMUX)
      {
         ChangeDemux(path, param);
      }
   }

   STB_OSMutexUnlock(av_mutex);
   FUNCTION_FINISH(SetAVSource);
}


/*!**************************************************************************
 * @fn      ChangeDemux
 * @brief   Change AVPLAY source demux
 * @param   path - decoder path
 * @param   demux - demux number
 * @return  TRUE if successfull, FALSE otherwise
 ****************************************************************************/
static BOOLEAN ChangeDemux(U8BIT path, U8BIT demux)
{
   BOOLEAN retval;
   FUNCTION_START(ChangeDemux);

   retval = FALSE;

   STB_OSMutexLock(av_mutex);
   if (av_status[path].demux == demux)
   {
      retval = TRUE;
   }
   else
   {
      retval = TRUE;
      av_status[path].demux = demux;
   }
   STB_OSMutexUnlock(av_mutex);

   FUNCTION_FINISH(ChangeDemux);
   return retval;
}

/*!**************************************************************************
 * @fn      UpdateCallbackInfo
 * @brief   Check if the video callback info is changed compared to cached value,
 *          and call user callback if so.
 * @param   path - decoder path
 * @param   info - updated callback info
 ****************************************************************************/
static void UpdateCallbackInfo(U8BIT path, S_STB_AV_VIDEO_INFO *info)
{
   FUNCTION_START(UpdateCallbackInfo);
   ASSERT(is_av_initialised);
   ASSERT(path < avs_num);

   if (av_status[path].callback != NULL)
   {
      if ((info->flags & VIDEO_INFO_VIDEO_RESOLUTION) != 0)
      {
         if (av_status[path].cb_info.video_width == info->video_width &&
             av_status[path].cb_info.video_height == info->video_height)
         {
            info->flags = (E_STB_AV_VIDEO_INFO_TYPE)(info->flags ^ VIDEO_INFO_VIDEO_RESOLUTION);
         }
         else
         {
            av_status[path].cb_info.video_width = info->video_width;
            av_status[path].cb_info.video_height = info->video_height;
            AV_DEBUG("VIDEO RESOLUTION: %dx%d", info->video_width, info->video_height);
         }
      }

      if ((info->flags & VIDEO_INFO_SCREEN_RESOLUTION) != 0)
      {
         if (av_status[path].cb_info.screen_width == info->screen_width &&
             av_status[path].cb_info.screen_height == info->screen_height)
         {
            info->flags = (E_STB_AV_VIDEO_INFO_TYPE)(info->flags ^ VIDEO_INFO_SCREEN_RESOLUTION);
         }
         else
         {
            av_status[path].cb_info.screen_width = info->screen_width;
            av_status[path].cb_info.screen_height = info->screen_height;
            AV_DEBUG("DISPLAY RESOLUTION: %dx%d", info->screen_width, info->screen_height);
         }
      }

      if ((info->flags & VIDEO_INFO_VIDEO_ASPECT_RATIO) != 0)
      {
         if (av_status[path].cb_info.video_aspect_ratio == info->video_aspect_ratio)
         {
            info->flags = (E_STB_AV_VIDEO_INFO_TYPE)(info->flags ^ VIDEO_INFO_VIDEO_ASPECT_RATIO);
         }
         else
         {
            av_status[path].cb_info.video_aspect_ratio = info->video_aspect_ratio;
            AV_DEBUG("VIDEO ASPECT RATIO: %d", info->video_aspect_ratio);
         }
      }

      if ((info->flags & VIDEO_INFO_DISPLAY_ASPECT_RATIO) != 0)
      {
         if (av_status[path].cb_info.display_aspect_ratio == info->display_aspect_ratio)
         {
            info->flags = (E_STB_AV_VIDEO_INFO_TYPE)(info->flags ^ VIDEO_INFO_DISPLAY_ASPECT_RATIO);
         }
         else
         {
            av_status[path].cb_info.display_aspect_ratio = info->display_aspect_ratio;
            AV_DEBUG("DISPLAY ASPECT RATIO: %d", info->display_aspect_ratio);
         }
      }
      if ((info->flags & VIDEO_INFO_AFD) != 0)
      {
         if (av_status[path].cb_info.afd == info->afd)
         {
            info->flags = (E_STB_AV_VIDEO_INFO_TYPE)(info->flags ^ VIDEO_INFO_AFD);
         }
         else
         {
            av_status[path].cb_info.afd = info->afd;
            AV_DEBUG("AFD: %d", info->afd);
         }
      }

      if ((info->flags & VIDEO_INFO_DECODER_STATUS) != 0)
      {
         if (av_status[path].cb_info.status == info->status)
         {
            info->flags = (E_STB_AV_VIDEO_INFO_TYPE)(info->flags ^ VIDEO_INFO_DECODER_STATUS);
         }
         else
         {
            av_status[path].cb_info.status = info->status;
            AV_DEBUG("DECODER STATUS: %d", info->status);
         }
      }

      if (info->flags != 0)
      {
         av_status[path].cb_info.flags = info->flags;
         (av_status[path].callback)(&av_status[path].cb_info, av_status[path].cb_user_data);
      }
   }

   FUNCTION_FINISH(UpdateCallbackInfo);
}

/*!**************************************************************************
 * @fn      ConvertAudioCodecToPlatformAPI
 * @brief
 * @param
 * @return
 ****************************************************************************/
static int ConvertAudioCodecToPlatformAPI(E_STB_AV_AUDIO_CODEC audio_codec)
{
   int retval;

   FUNCTION_START(ConvertAudioCodecToPlatformAPI);

   switch (audio_codec)
   {
      case AV_AUDIO_CODEC_MP2:      retval = RTK_CODEC_AUDIO_MPEG2;         break;
      case AV_AUDIO_CODEC_MP3:      retval = RTK_CODEC_AUDIO_MP3;           break;
      case AV_AUDIO_CODEC_AC3:      retval = RTK_CODEC_AUDIO_AC3_PLUS;      break;
      case AV_AUDIO_CODEC_EAC3:     retval = RTK_CODEC_AUDIO_EAC3;          break;
      case AV_AUDIO_CODEC_HEAAC:    retval = RTK_CODEC_AUDIO_MP4;           break;
      case AV_AUDIO_CODEC_AAC:      retval = RTK_CODEC_AUDIO_AAC;           break;
      case AV_AUDIO_CODEC_AAC_ADTS: retval = RTK_CODEC_AUDIO_AAC;           break;
      case AV_AUDIO_CODEC_AUTO:     retval = RTK_CODEC_AUDIO_MP3;           break;
      default:
         AV_WARNING("Failed to convert DTVKit audio codec #%d to SDK representation", audio_codec);
         retval = RTK_CODEC_NONE;
         break;
   }

   FUNCTION_FINISH(ConvertAudioCodecToPlatformAPI);
   return retval;
}

/*!**************************************************************************
 * @fn      ConvertVideoCodecToPlatformAPI
 * @brief
 * @param
 * @return
 ****************************************************************************/
static int ConvertVideoCodecToPlatformAPI(E_STB_AV_VIDEO_CODEC video_codec)
{
   int retval;

   FUNCTION_START(ConvertVideoCodecToPlatformAPI);

   switch (video_codec)
   {
      case AV_VIDEO_CODEC_MPEG2: retval = RTK_CODEC_VIDEO_MPEG2; break;
      case AV_VIDEO_CODEC_H264:  retval = RTK_CODEC_VIDEO_H264;  break;
      case AV_VIDEO_CODEC_H265:  retval = RTK_CODEC_VIDEO_H265;  break;
#ifdef AV_VIDEO_CODEC_AVS
      case AV_VIDEO_CODEC_AVS:   retval = RTK_CODEC_NONE;   break;
#endif
      case AV_VIDEO_CODEC_AUTO:  retval = RTK_CODEC_VIDEO_MPEG2; break;
      default:
         AV_WARNING("Failed to convert DTVKit video codec #%d to SDK representation", video_codec);
         retval = RTK_CODEC_NONE;
         break;
   }

   FUNCTION_FINISH(ConvertVideoCodecToPlatformAPI);
   return retval;
}

/*!**************************************************************************
 * @fn      STB_AVSetAudioTrack
 * @brief	switch multi-audio track
 * @param	audio_pid the audio pid of the track to be switched to.
 * @return	HI_SUCCESS if successful, others is failure.
 ****************************************************************************/
S32BIT STB_AVSetAudioTrack(U16BIT audio_pid)
{
   FUNCTION_START(STB_AVSetAudioTrack);

   STB_OSMutexLock(av_mutex);

   STB_OSMutexUnlock(av_mutex);

   FUNCTION_FINISH(STB_AVSetAudioTrack);

   return 0;
}

void STB_AVSwitchLivePath(U8BIT decoder, U8BIT demux)
{
   av_status[decoder].demux = demux;
   av_status[decoder].is_audio_decoding = FALSE;
   av_status[decoder].is_video_decoding = FALSE;
   av_status[decoder].is_iframe_displayed = FALSE;
}

/*!**************************************************************************
 * @fn      InitAudioOutput
 * @brief   Initialise sound device
 * @return  TRUE if successfull, FALSE otherwise
 ****************************************************************************/
static BOOLEAN InitAudioOutput(void)
{
   BOOLEAN retval;

   FUNCTION_START(InitAudioOutput);
   STB_OSMutexLock(av_mutex);

   retval = TRUE;
   STB_OSMutexUnlock(av_mutex);
   FUNCTION_FINISH(InitAudioOutput);
   return retval;
}


/*!**************************************************************************
 * @fn      DeinitAudioOutput
 * @brief   Deinitialise sound device
 ****************************************************************************/
static void DeinitAudioOutput(void)
{
   FUNCTION_START(DeinitAudioOutput);
   STB_OSMutexLock(av_mutex);

   STB_OSMutexUnlock(av_mutex);
   FUNCTION_FINISH(DeinitAudioOutput);
}


/*!**************************************************************************
 * @fn      InitVideoOutput
 * @brief   Initialise display device
 * @return  TRUE if successfull, FALSE otherwise
 ****************************************************************************/
static BOOLEAN InitVideoOutput(void)
{
   BOOLEAN hdmi_initialised;
   BOOLEAN disp_initialised;
   BOOLEAN video_output_initialised;
   BOOLEAN disp1_attached;
   BOOLEAN disp0_attached;

   U32BIT i;

   U32BIT hd_interfaces_num = 0;
   U32BIT sd_interfaces_num = 0;

   FUNCTION_START(InitVideoOutput);
   STB_OSMutexLock(av_mutex);

   // Initialise HDMI
   hdmi_initialised = TRUE;

   // Choose video output formats
   //format = DEFAULT_VIDEO_FORMAT;
   if (hdmi_initialised)
   {

   }


   // Initialise Display
   disp_initialised = FALSE;
   if (hdmi_initialised)
   {
      {
         disp1_attached = TRUE;
         disp0_attached = TRUE;
         disp_initialised = TRUE;
      }
   }

   // Init window
   video_output_initialised = FALSE;
   if (disp_initialised)
   {
      {
         video_output_initialised = TRUE;
      }
   }

   STB_OSMutexUnlock(av_mutex);
   FUNCTION_FINISH(InitVideoOutput);
   return video_output_initialised;
}


/*!**************************************************************************
 * @fn      DeinitVideoOutput
 * @brief   Deinitialise display device
 ****************************************************************************/
static void DeinitVideoOutput(void)
{
   FUNCTION_START(DeinitVideoOutput);
   STB_OSMutexLock(av_mutex);

   STB_OSMutexUnlock(av_mutex);
   FUNCTION_FINISH(DeinitVideoOutput);
}


