/*******************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/

#ifndef _STBHWCAS_H
#define _STBHWCAS_H


/*--- Includes ----------------------------------------------------------------*/

#include "stbhwcas.h"
#include "stbhwsdk.h"
#include "stbhwdmx.h"
#include "internal.h"

extern "C" {
#include "stbdpc.h"
}

extern "C" {
/*--- Preprocessor definitions ------------------------------------------------*/

/*--- Global types definitions ------------------------------------------------*/

/*--- Global function declarations --------------------------------------------*/

/**
 * @brief   Initialises encryption/decryption
 * @return  TRUE if initialisation is successful, FALSE otherwise
 */
BOOLEAN STB_CASInitialise(void)
{
    return TRUE;
}


/*!**************************************************************************
 * @brief   When there's an update to the CAT for a service, the updated CAT
 *          will be reported to the CA system using this function.
 * @param   path - decode path
 * @param   handle - CA descrambler handle
 * @param   ca_system_id - CA system ID
 * @param   pid - the EMM PID
 ****************************************************************************/
void STB_CASSetEmmPid(U8BIT path, U32BIT handle, U16BIT ca_system_id, U16BIT pid)
{
    
    RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(path);
    if (NULL == pRealtekPlayer)
        return;
        
    pRealtekPlayer->RT_Player_setEmmPid(ca_system_id, pid);
}

BOOLEAN STB_CASStartDescrambler(U32BIT handle)
{
   U16BIT sid = (U16BIT)((handle >> 16) & 0xFFFF);
   U8BIT demux_num = (U8BIT)(handle & 0xFF);
   if (INVALID_RES_ID == demux_num)
       return FALSE;

   U8BIT path = STB_DPGetPathOfDescrambler(handle);
   if (INVALID_RES_ID == path)
       return FALSE;
   
   RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(path);
   if (NULL == pRealtekPlayer)
      return FALSE;
  
   std::vector<std::pair<uint16_t,uint16_t>> ecmPidList;
   U16BIT ecm_pid[6], stream_pid[6]; 
   DMX_GetEcmPIDs(
      path,
      &ecm_pid[0],
      &ecm_pid[1],
      &ecm_pid[2],
      &ecm_pid[3],
      &ecm_pid[4],
      &ecm_pid[5]
   );
   
   for (int i=0; i< 6; i++) {
      if (0x1FFF <= ecm_pid[i])
          ecm_pid[i] = 0;
   }
   
   DMX_GetDecodePIDs(
      path,
      &stream_pid[0],
      &stream_pid[1],
      &stream_pid[2],
      &stream_pid[3],
      &stream_pid[4],
      &stream_pid[5]
   );
   
   U16BIT videoEcmPid = 0;
   U16BIT audioEcmPid = 0;
 
   if (ecm_pid[0] > 0 && ecm_pid[0] < 0x1FFF) {
      videoEcmPid = ecm_pid[0];
      audioEcmPid = ecm_pid[0];
   }
   if (ecm_pid[1] > 0 && ecm_pid[1] < 0x1FFF) {
      videoEcmPid = ecm_pid[1];
   }
   if (ecm_pid[2] > 0 && ecm_pid[2] < 0x1FFF) {
      audioEcmPid = ecm_pid[2];
   }

   if (videoEcmPid > 0 && videoEcmPid < 0x1FFF) {
      ecmPidList.push_back(std::make_pair(videoEcmPid,stream_pid[1]));  // video ECM PID
   }
   if (audioEcmPid > 0 && audioEcmPid < 0x1FFF) {
      ecmPidList.push_back(std::make_pair(audioEcmPid,stream_pid[2]));  // audio ECM PID
   }   
   
   pRealtekPlayer->RT_Player_StartDescrambler(sid, ecmPidList);

   return TRUE;
}

BOOLEAN STB_CASStopDescrambler(U32BIT handle)
{
printf("\n===> %s  %d  %s\n", __FILE__, __LINE__, __func__);
   U8BIT demux_num = (U8BIT)(handle & 0xFF);
   if (INVALID_RES_ID == demux_num)
       return FALSE;

   U8BIT path = STB_DPGetPathOfDescrambler(handle);
   if (INVALID_RES_ID == path)
       return FALSE;
   
   RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(path);
   if (NULL == pRealtekPlayer)
      return FALSE;
       
   pRealtekPlayer->RT_Player_StopDescrambler();

   return TRUE;
}

}
#endif /*  _STBHWCAS_H */

/****************************************************************************
** End of file
*****************************************************************************/

