#include "stbhwdmx.h"
#include "stbhwmem.h"
#include "stbhwos.h"
#include "dbgfuncs.h"
#include "internal_generic.h"
#include "internal.h"

extern "C" {
#include "stbdpc.h"
}

#include "stbhwsdk.h"
#include <stdio.h>
#include <list>
#include <utility>
//#include <vector>

#ifdef ENABLE_LIBDVBPSI
#include <dvbpsi/dvbpsi.h>
#include <dvbpsi/descriptor.h>
#include <dvbpsi/dr.h>
#include <dvbpsi/nit.h>
#endif

static void rtk_player_callback (RTK_PLAYER_CB_TYPE_e type, void * data, void * user_param);

/*--- Preprocessor definitions ------------------------------------------------*/

/* DemuxTask attributes */
#define DMX_TASK_PRIORITY           12
#define DMX_TASK_STACK_SIZE         4096

/* Path type */
#define LIVE_PATH                   0
#define PLAYBACK_PATH               1		/* playback path */
#define RECORD_PATH                 2		/* record path */
#define MONITOR_PATH                3

#define DEMUX_TASK_TIMEOUT          100

#define SUBT_FLTR_SECTION_NUM		(8)
#define MAX_CHANNEL_BUF_SIZE		(64 * 1024)
#define TEXT_BUFFER_SIZE			(3008 * 1024)

#define PES_FILTERS_A_CHANNEL       1
#define SECTION_FILTERS_A_CHANNEL   8
#define DEMUX_FILTER_NOT_ALLOCATED  0xFFFF

/* Debug logs */
//#define DEMUX_DEBUGS
#define DEMUX_WARNINGS
#define DEMUX_ERRORS

#ifdef DEMUX_DEBUGS
#define DMX_DEBUG(format, args...) STB_DEBUG_LOG("DMX", format, ##args)
#else
#define DMX_DEBUG(format, args...)
#endif

#ifdef DEMUX_WARNINGS
#define DMX_WARNING(format, args...) STB_WARNING_LOG("DMX", format, ##args)
#else
#define DMX_WARNING(format, args...)
#endif

#ifdef DEMUX_ERRORS
#define DMX_ERROR(format, args...) STB_ERROR_LOG("DMX", format, ##args)
#else
#define DMX_ERROR(format, args...)
#endif


/*--- Local types definitions -------------------------------------------------*/

typedef enum demux_cmd
{
   DEMUX_CMD_TRIGGER,
   DEMUX_CMD_TASK_TERMINATE,
   DEMUX_CMD_DATA_READY,
} E_DEMUX_CMD;

typedef enum
{
   CHANNEL_TYPE_SECTION,
   CHANNEL_TYPE_PES
} E_CHANNEL_TYPE;

typedef struct sect_filter
{
   RTK_SectionFilter filter_handle;
   RTK_SectionFilterMask filter_mask;
   //
   U16BIT pidFilterId;
   U16BIT sectionFilterId;
   //
   U8BIT * buffer;
   U16BIT length;
   
   FILTER_CALLBACK callback;
   BOOLEAN is_valid;
   BOOLEAN is_active;
   U32BIT u32FilterDepth;
   U8BIT au8Match[16];
   U8BIT au8Mask[16];
   U8BIT au8Negate[16];
} S_SECT_FILTER;

typedef struct pes_filter
{
	RTK_PESFilterPattern pes_pattern;
   //
   U16BIT pidFilterId;
   //
   U8BIT * buffer;
   U16BIT length;
   
   FILTER_CALLBACK callback;
} S_PES_FILTER;

typedef struct pid_filter
{
   U8BIT demux_path;
   BOOLEAN is_valid;
   BOOLEAN is_active;
   E_CHANNEL_TYPE chan_type;
   U16BIT pid;
   //
   S_PES_FILTER pes_filter;
   FILTER_CALLBACK pes_callback;
   S_SECT_FILTER section_filters[SECTION_FILTERS_A_CHANNEL];
   FILTER_CALLBACK section_callback[SECTION_FILTERS_A_CHANNEL];
} S_PID_FILTER;

typedef struct demux
{
   U16BIT video_pid;
   U16BIT audio_pid;
   U16BIT pcr_pid;
   U16BIT text_pid;
   U16BIT data_pid;
   U16BIT ad_pid;
   E_STB_DMX_DEMUX_SOURCE source;
   U8BIT source_param;

   U16BIT text_channel;
   U8BIT *text_buffer;
   U32BIT text_datalen;

   U16BIT ecm_pid;
   U16BIT video_ecm_pid;
   U16BIT audio_ecm_pid;
   U16BIT text_ecm_pid;
   U16BIT data_ecm_pid;
   U16BIT ad_ecm_pid;
} S_DEMUX;

/*--- Local (static) variable declarations ------------------------------------*/

static void *dmx_mutex = NULL;
static void *dmx_queue = NULL;
static void *dmx_task_termination_mutex = NULL;
static void *dmx_task = NULL;

static U8BIT dmxs_number = 0;
static S_DEMUX *demuxes = NULL;

static BOOLEAN is_dmx_initialised = FALSE;

static U32BIT active_pid_filters_num = 0;
#ifdef DEMUX_DEBUGS
static U32BIT sect_filters_num = 0;
#endif
static U32BIT pid_filters_num = 0;
static U32BIT max_pid_filters_num = 0;
static U32BIT max_sect_filters_num = 0;
static S_PID_FILTER *pid_filters = NULL;

static std::list<std::pair<void *,uint32_t>> textBufferList;
static std::list<void *> sectionBufferLists[16];  //FIXME: hardcode

/*--- Local function prototypes -----------------------------------------------*/

static void DemuxTask(void *param);
static BOOLEAN AcquireTextPESFilter(U8BIT path, U16BIT pid);
static void ReleaseTextPESFilter(U8BIT path);
static U16BIT DMX_GrabChannel(U8BIT path, U8BIT chan_type, U16BIT pid, FILTER_CALLBACK func_ptr);
static void rtk_pes_callback(uint32_t filterID , void * pes, uint32_t length, void * userParam);
static void rtk_section_callback (
    const RTK_SectionFilter handle,
    RTK_HAL_SECTION_CB_STATUS enStatus,
    const unsigned char * section,
    const unsigned int length,
    void * userParam
);

/*--- Global function definitions ---------------------------------------------*/

/**
 * @brief   Initialises the demux / programmable transport interface
 * @param   paths Number of demux paths to be initialised
 * @param   inc_pes_collection Not used
 */
void STB_DMXInitialise(U8BIT paths, BOOLEAN inc_pes_collection)
{
   S32BIT ret;
   U32BIT i;

   FUNCTION_START(STB_DMXInitialise);
   USE_UNWANTED_PARAM(inc_pes_collection);

   if (is_dmx_initialised)
   {
      DMX_WARNING("Demux is already initialised!");
   }
   else
   {

      {

         {

            {
               dmxs_number = paths;
               DMX_DEBUG("%d demux will be used.", dmxs_number);
            }

            demuxes = (S_DEMUX*) STB_MEMGetSysRAM(sizeof(S_DEMUX) * dmxs_number);

            max_pid_filters_num = 128;  //FIXME: hardcode
            pid_filters = (S_PID_FILTER*) STB_MEMGetSysRAM(sizeof(S_PID_FILTER) * max_pid_filters_num);

            if (demuxes == NULL || pid_filters == NULL)
            {
               DMX_ERROR("Failed to allocate memory for demux status data");
            }
            else
            {
               textBufferList.clear();
               for (i = 0; i < dmxs_number; i++)
               {
                  demuxes[i].video_pid = INVALID_PID;
                  demuxes[i].audio_pid = INVALID_PID;
                  demuxes[i].pcr_pid = INVALID_PID;
                  demuxes[i].text_pid = INVALID_PID;
                  demuxes[i].data_pid = INVALID_PID;
                  demuxes[i].ad_pid = INVALID_PID;
                  demuxes[i].source = DMX_MEMORY;
                  demuxes[i].source_param = 0;
                  demuxes[i].text_buffer = NULL;
                  demuxes[i].text_datalen = 0;
                  demuxes[i].text_channel = 0xFFFF;
                  sectionBufferLists[i].clear();
               }

               memset(pid_filters, 0, sizeof(S_PID_FILTER) * max_pid_filters_num);
               for(i=0; i<max_pid_filters_num; i++)
               {
	               memset(pid_filters[i].section_filters, 0, sizeof(S_SECT_FILTER) * SECTION_FILTERS_A_CHANNEL);
                  S_SECT_FILTER * sectionFilters = pid_filters[i].section_filters;
                  for (int j = 0; j < SECTION_FILTERS_A_CHANNEL; j++) {
                     sectionFilters[j].pidFilterId = i;
                     sectionFilters[j].sectionFilterId = j;
                  }
                  // PES filter
                  pid_filters[i].pes_filter.pidFilterId = i;
               }

               dmx_mutex = STB_OSCreateMutex();
               dmx_queue = STB_OSCreateQueue(sizeof(E_DEMUX_CMD), 256);
               dmx_task_termination_mutex = STB_OSCreateMutex();
               if (dmx_mutex == NULL || dmx_queue == NULL ||
                   dmx_task_termination_mutex == NULL)
               {
                  DMX_ERROR("Failed to prepare resources for DemuxTask");
               }
               else
               {
                  dmx_task = STB_OSCreateTask(DemuxTask, NULL, DMX_TASK_STACK_SIZE,
                                              DMX_TASK_PRIORITY, (U8BIT*)"DemuxTask");
                  if (dmx_task == NULL)
                  {
                     DMX_ERROR("Failed to create DemuxTask");
                  }
                  else
                  {
                     is_dmx_initialised = TRUE;
                  }
               }
            }
         }
      }
   }

   if (!is_dmx_initialised)
   {
      DMX_Terminate(paths);
      DMX_ERROR("Failed to initialize demux");
   }

   FUNCTION_FINISH(STB_DMXInitialise);
}

/**
 * @brief   Returns the capability flags of the given demux
 * @param   path - demux
 * @return  Capability flags
 */
U16BIT STB_DMXGetCapabilities(U8BIT path)
{
   U16BIT demux_capability_flag;

   FUNCTION_START(STB_DMXGetCapabilities);

   demux_capability_flag = 0xFFFF;
   if (0 == path) {
      // demux[0] is reserved for playback
      demux_capability_flag = DMX_CAPS_LIVE;
   }
   else {
      demux_capability_flag = ~DMX_CAPS_LIVE;   
   }

   FUNCTION_FINISH(STB_DMXGetCapabilities);

   return demux_capability_flag;
}

/**
 * @brief   Returns the maximum number of section filters available on this hw
 * @return  The number of filters
 */
U8BIT STB_DMXGetMaxSectionFilters(void)
{
   FUNCTION_START(STB_DMXGetMaxSectionFilters);
   FUNCTION_FINISH(STB_DMXGetMaxSectionFilters);

   return max_sect_filters_num;
}

/**
 * @brief   Configures the source of the demux
 * @param   path the demux path to configure
 * @param   source the source to use
 * @param   param source specific parameters (e.g. tuner number)
 */
void STB_DMXSetDemuxSource(U8BIT path, E_STB_DMX_DEMUX_SOURCE source, U8BIT param)
{
   BOOLEAN ready_to_connect = FALSE;
   RealtekPlayer * pRealtekPlayer = NULL;
   const char * sourceName = NULL;
   bool preload = (path & (1<<7)) ? true : false;
   path &= 0x7F;
   U8BIT decode_path = STB_DPGetPathOfDemux(path);
   if (INVALID_RES_ID == decode_path) {
      return;
   }

   FUNCTION_START(STB_DMXSetDemuxSource);

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
      pRealtekPlayer = STB_SdkGetRealtekPlayer(decode_path);
      if (source != DMX_INVALID) {
         RTKPLAYER_ARCH type;
         
         if (TRUE == STB_DPIsDecodingPath(decode_path))
            type = ARCH_PLAY;  //ARCH_DEMUX_AND_PLAY;
         else
            type = ARCH_DEMUX;

         if (NULL == pRealtekPlayer)
            pRealtekPlayer = new RealtekPlayer(type, param /*tuner number*/, preload /*preload A/V streams*/);

         STB_SdkSetRealtekPlayer(decode_path, pRealtekPlayer);
         if (NULL != pRealtekPlayer) {
            pRealtekPlayer->setUserData(decode_path);
            if (ARCH_DEMUX == type) {
               pRealtekPlayer->RT_Player_SetAudioFocused(RTK_FALSE);
               //pRealtekPlayer->RT_PlayerSetVideoOutputMode(RTK_WIN_PIP);
            }
            else {
               pRealtekPlayer->RT_Player_SetAudioFocused(RTK_FALSE);
               //pRealtekPlayer->RT_PlayerSetVideoOutputMode(RTK_WIN_MAIN);
            }
            pRealtekPlayer->RT_PlayerCallbackRegister(rtk_player_callback, pRealtekPlayer);
         }
      }
      else {
         if (NULL != pRealtekPlayer) {
            pRealtekPlayer->removeAllSectionFilters();
            pRealtekPlayer->RT_PlayerStop(RTK_TRUE);
            pRealtekPlayer->RT_PlayerDestory();      
            pRealtekPlayer->RT_PlayerCallbackRegister(NULL, NULL);
            STB_SdkSetRealtekPlayer(decode_path, NULL);
            usleep(10000);
            delete pRealtekPlayer;
            pRealtekPlayer = NULL;

            if (TRUE == STB_DPIsDecodingPath(decode_path))
               RT_setVideoSurface(RTK_WIN_MAIN, NULL);
            //else
              // RT_setVideoSurface(RTK_WIN_PIP, NULL);
         }
      }
      
      if (source == DMX_TUNER)
      {
         ready_to_connect = TRUE;
         sourceName = "DMX_TUNER";
      }
      else if (source == DMX_MEMORY)
      {
         ready_to_connect = TRUE;
         sourceName = "DMX_MEMORY";
      }
      else if (source == DMX_INVALID)
      {
         ready_to_connect = FALSE;
         sourceName = "DMX_INVALID";
      }
      else
      {
         ready_to_connect = TRUE;
         sourceName = "DMX_INVALID";
      }

      {
         demuxes[path].source = source;
         demuxes[path].source_param = param;
         DMX_DEBUG("Demux source %s:%d is set successfully", sourceName, param);
      }
   }

   FUNCTION_FINISH(STB_DMXSetDemuxSource);
}

/**
 * @brief   Gets the current source of a given demux
 * @param   path the demux path to query
 * @param   source the source of the demux
 * @param   param the source specific parameter (e.g. tuner number)
 */
void STB_DMXGetDemuxSource(U8BIT path, E_STB_DMX_DEMUX_SOURCE *source, U8BIT *param)
{
   FUNCTION_START(STB_DMXGetDemuxSource);

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
      *source = demuxes[path].source;
      *param = demuxes[path].source_param;
   }

   FUNCTION_FINISH(STB_DMXGetDemuxSource);
}

/**
 * @brief   Reads Teletext PES data from the demux
 * @param   path the demux path to read
 * @param   buffer pointer to PES data
 * @param   num_bytes the number of bytes of data
 */
void STB_DMXReadTextPES(U8BIT path, U8BIT **buffer, U32BIT *num_bytes)
{
   S32BIT ret;
   U32BIT offset = 0;
   U16BIT pfilt_id;

   FUNCTION_START(STB_DMXReadTextPES);

   *num_bytes = 0;
   *buffer = NULL;
   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else if ((demuxes[path].text_channel >> 8) < max_pid_filters_num)
   {
      pfilt_id = (demuxes[path].text_channel >> 8) & 0xFF;
      //channel = pid_filters[pfilt_id].channel_handle;
      
      U8BIT * pData = NULL;
      uint32_t length = 0;
         
      STB_OSMutexLock(dmx_mutex);
      int size = textBufferList.size();
      if (size > 0) {
         std::pair<void *, uint32_t> p = textBufferList.front();
         pData = (U8BIT *)p.first;
         length = p.second;
         textBufferList.pop_front();
      }
      STB_OSMutexUnlock(dmx_mutex);
      if ((NULL != pData) && (length > 0)) {
         demuxes[path].text_datalen = length;
         if (NULL != demuxes[path].text_buffer) {
            memcpy (demuxes[path].text_buffer, pData, length);
         }
         STB_MEMFreeSysRAM(pData);  // consumer
      }
      else {
         demuxes[path].text_datalen = 0;
      }

      if(demuxes[path].text_buffer)
      {
         *buffer = demuxes[path].text_buffer;
         *num_bytes = demuxes[path].text_datalen;
      }
   }

   FUNCTION_FINISH(STB_DMXReadTextPES);
}

/**
 * @brief   Changes the packet IDs for the PCR Video, Audio, Text and Data
 * @param   path The demux path to be configured
 * @param   pcr_pid The PID to use for the Program Clock Reference
 * @param   video_pid The PID to use for the Video PES
 * @param   audio_pid The PID to use for the Audio PES
 * @param   text_pid The PID to use for the Teletext data
 * @param   data_pid The PID to use for the data
 */
void STB_DMXChangeDecodePIDs(U8BIT path, U16BIT pcr_pid, U16BIT video_pid, U16BIT audio_pid,
   U16BIT text_pid, U16BIT data_pid, U16BIT ad_pid)
{
   int i;
   BOOLEAN preload = (0 != (path & (1 << 7))) ? TRUE : FALSE;
 
   FUNCTION_START(STB_DMXChangeDecodePIDs);

   path &= 0x7F;  // clear bit[8]: preloading bit

   U8BIT decode_path = STB_DPGetPathOfDemux(path);
   if (INVALID_RES_ID == decode_path) {
      return;
   }

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
      if(pcr_pid == 0)
         pcr_pid = INVALID_PID;
      if(video_pid == 0)
         video_pid = INVALID_PID;
      if(audio_pid == 0)
         audio_pid = INVALID_PID;
      if(text_pid == 0)
         text_pid = INVALID_PID;
      if(data_pid == 0)
         data_pid = INVALID_PID;
      if(ad_pid == 0)
         ad_pid = INVALID_PID;

#if 0  //FIXME
      if (text_pid != demuxes[path].text_pid)
      {
		 /* Release all existed Text PES filter */
         for (i = 0; i < dmxs_number; i++)
         {
            ReleaseTextPESFilter(i);
         }
         if (text_pid != 0 && text_pid != INVALID_PID)
         {
			/* Acquire a new Text PES Filter */
            AcquireTextPESFilter(path, text_pid);
         }
      }
#endif

#if 0
      if(path == PLAYBACK_PATH)
      {
         if(((demuxes[path].video_pid != video_pid) || (demuxes[path].audio_pid != audio_pid)) &&
		 	((INVALID_PID != demuxes[path].video_pid) || (INVALID_PID != demuxes[path].audio_pid)) &&
		 	((INVALID_PID != video_pid) || (INVALID_PID != audio_pid))) 
         {
         	if((demuxes[path].video_pid == demuxes[path].video_pid)&&(INVALID_PID != video_pid))
			   {
			      /* multi-audio channel, switch audio track */	
			      STB_AVSetAudioTrack(audio_pid);
			   }
			   else
			   {
			      //STB_AVRestartAV(video_pid, (U16BIT)STB_AVGetVideoCodec(0), audio_pid, (U16BIT)STB_AVGetAudioCodec(0));
			   }
         }
      }
#endif

      demuxes[path].pcr_pid = pcr_pid;
      demuxes[path].video_pid = video_pid;
      demuxes[path].audio_pid = audio_pid;
      demuxes[path].text_pid = text_pid;
      demuxes[path].data_pid = data_pid;
      demuxes[path].ad_pid = ad_pid;

      if (TRUE == preload) {
         // start RealtekPlayer FCC preload
         RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(decode_path);
         if (NULL != pRealtekPlayer) {
            if (TRUE == STB_DPIsMonitoringPath(decode_path)) {
               uint16_t pids[2] = { video_pid, audio_pid };
               int ret = pRealtekPlayer->preloadPidStreams(pids, 2);
               if (0 !=ret) {

               }
            }      
         }
      }
   }

   FUNCTION_FINISH(STB_DMXChangeDecodePIDs);
}

/**
 * @brief   Changes the ECM PIDs for the Video, Audio, Text and Data
 * @param   path The demux path to be configured
 * @param   ecm_pid The program ECM PID to use for the Audio/Video PES
 * @param   video_ecm_pid The ECM PID to use for the Video PES
 * @param   audio_ecm_pid The ECM PID to use for the Audio PES
 * @param   text_ecm_pid The ECM PID to use for the Teletext data
 * @param   data_ecm_pid The ECM PID to use for the data
 */
void STB_DMXChangeEcmPIDs(U8BIT path, U16BIT ecm_pid, U16BIT video_ecm_pid, U16BIT audio_ecm_pid,
   U16BIT text_ecm_pid, U16BIT data_ecm_pid, U16BIT ad_ecm_pid)
{
   int i;
   BOOLEAN preload = (0 != (path & (1 << 7))) ? TRUE : FALSE;
 
   FUNCTION_START(STB_DMXChangeEcmPIDs);

   path &= 0x7F;  // clear bit[8]: preloading bit

   U8BIT decode_path = STB_DPGetPathOfDemux(path);
   if (INVALID_RES_ID == decode_path) {
      return;
   }

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
      if(ecm_pid == 0)
         ecm_pid = INVALID_PID;
      if(video_ecm_pid == 0)
         video_ecm_pid = INVALID_PID;
      if(audio_ecm_pid == 0)
         audio_ecm_pid = INVALID_PID;
      if(text_ecm_pid == 0)
         text_ecm_pid = INVALID_PID;
      if(data_ecm_pid == 0)
         data_ecm_pid = INVALID_PID;
      if(ad_ecm_pid == 0)
         ad_ecm_pid = INVALID_PID;

      demuxes[path].ecm_pid = ecm_pid;
      demuxes[path].video_ecm_pid = video_ecm_pid;
      demuxes[path].audio_ecm_pid = audio_ecm_pid;
      demuxes[path].text_ecm_pid = text_ecm_pid;
      demuxes[path].data_ecm_pid = data_ecm_pid;
      demuxes[path].ad_ecm_pid = ad_ecm_pid;

#if 0
      if (TRUE == preload) {
         // start RealtekPlayer FCC preload
         RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(decode_path);
         if (NULL != pRealtekPlayer) {
            if (TRUE == STB_DPIsMonitoringPath(decode_path)) {
               uint16_t pids[2] = { video_ecm_pid, audio_ecm_pid };
               int ret = pRealtekPlayer->preloadPidStreams(pids, 2);
               if (0 !=ret) {

               }
            }      
         }
      }
#endif
   }

   FUNCTION_FINISH(STB_DMXChangeEcmPIDs);
}

/**
 * @brief   Changes just the teletext PID
 * @param   path The demux path to configure
 * @param   text_pid The PID to use for the teletext data
 */
void STB_DMXChangeTextPID(U8BIT path, U16BIT text_pid)
{
   int i;

   FUNCTION_START(STB_DMXChangeTextPID);

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
      if (text_pid != demuxes[path].text_pid)
      {
         /* Release existed Text PES filter */
         if (demuxes[path].text_pid >0 && demuxes[path].text_pid < 0x1FFF) {
            ReleaseTextPESFilter(path);
         }

         if ((text_pid > 0) && (text_pid < 0x1FFF)) {
            if (!AcquireTextPESFilter(path, text_pid))
               DMX_ERROR("Failed to create text PES filter");
         }
      }
   }

   FUNCTION_FINISH(STB_DMXChangeTextPID);
}

/**
 * @brief   Get a New PID Filter & Setup Associated Buffer and Callback
 *          Function Address.
 * @param   path Required Decode Path Number.
 * @param   pid Required PID to Demux.
 * @param   func_ptr User's Interrupt Procedure Function Address.
 * @return  New PID filter identifier or invalid id.
 */
U16BIT STB_DMXGrabPIDFilter(U8BIT path, U16BIT pid, FILTER_CALLBACK func)
{
   U16BIT retval;

   FUNCTION_START(STB_DMXGrabPIDFilter);

   retval = DMX_GrabChannel(path, CHANNEL_TYPE_SECTION, pid, func);

   FUNCTION_FINISH(STB_DMXGrabPIDFilter);

   return retval;
}

/**
 * @brief   Copies a filtered section to caller's buffer
 * @param   path the demux path to use
 * @param   buffer the caller's buffer
 * @param   size the size of the caller's buffer
 * @param   pfilt_id the handle of the PID filter to read from
 * @return  TRUE copied ok
 * @return  FALSE no data to copy
 */
BOOLEAN STB_DMXCopyPIDFilterSect(U8BIT path, U8BIT *buffer, U16BIT size, U16BIT pfilt_id)
{
    U16BIT dmx_channel_index;
    U16BIT dmx_filter_index;
    S_PID_FILTER* dmx_channel_ptr;
    S_SECT_FILTER* dmx_filter_ptr;

    /* Section filter id includes the PID filter id */
    dmx_channel_index = pfilt_id >> 8;
    dmx_filter_index = pfilt_id & 0x0f;

	dmx_channel_ptr = (S_PID_FILTER *)&pid_filters[dmx_channel_index];
	dmx_filter_ptr = (S_SECT_FILTER *)&dmx_channel_ptr->section_filters[dmx_filter_index];
    U16BIT len = dmx_filter_ptr->length;
    if (len > size)
        len = size;
        
    memcpy(buffer, dmx_filter_ptr->buffer, len);

    return TRUE;
}

/**
 * @brief   Skips (discards) a section in the PID filter buffer
 * @param   path the demux path of the filter
 * @param   pfilt_id the PID filter handle
 */
void STB_DMXSkipPIDFilterSect(U8BIT path, U16BIT pfilt_id)
{

}

/**
 * @brief   Allocated a new section filter on the specified PID filter
 * @param   path the demux path to use
 * @param   pfilt_id the PID filter to assign the section filter to
 * @return  The section filter handle
 */
U16BIT STB_DMXGrabSectFilter(U8BIT path, U16BIT pfilt_id)
{
   U16BIT dmx_channel_index;
   U16BIT dmx_filter_index;
   S_PID_FILTER* dmx_channel_ptr;
   S_SECT_FILTER* dmx_filter_ptr;
   U16BIT sfilt_id;

   FUNCTION_START(STB_DMXGrabSectFilter);

   dmx_channel_index = (pfilt_id >> 8) & 0xff;
   dmx_filter_index = pfilt_id & 0x0f;

   sfilt_id = DEMUX_FILTER_NOT_ALLOCATED;
   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
      if((dmx_channel_index < max_pid_filters_num)&&(dmx_filter_index < SECTION_FILTERS_A_CHANNEL))
      {
         dmx_channel_ptr = (S_PID_FILTER *)&pid_filters[dmx_channel_index];
         dmx_filter_ptr = (S_SECT_FILTER *)&dmx_channel_ptr->section_filters[dmx_filter_index];
         if(dmx_filter_ptr->is_valid)
         {
	         DMX_ERROR("This filter is not free");
         }
         else
         {
            /* Clear the filter's match/mask data */
            memset(dmx_filter_ptr->au8Match, 0, 16);
            memset(dmx_filter_ptr->au8Mask, 0, 16);
            memset(dmx_filter_ptr->au8Negate, 0, 16);
            dmx_filter_ptr->callback = dmx_channel_ptr->section_callback[dmx_filter_index];
            dmx_filter_ptr->is_valid = TRUE;
            dmx_filter_ptr->is_active = FALSE;
            sfilt_id = pfilt_id;
            dmx_filter_ptr->pidFilterId = dmx_channel_index;
            dmx_filter_ptr->sectionFilterId = dmx_filter_index;                        
#ifdef DEMUX_DEBUGS
            sect_filters_num++;
#endif
         }
      }
   }

   FUNCTION_FINISH(STB_DMXGrabSectFilter);

   return(sfilt_id);
}

/**
 * @brief   Configures a match and mask for a specified section filter
 * @param   path the demux path of the section filter
 * @param   sfilt_id the handle of the section filter
 * @param   match_ptr pointer to the match bytes
 * @param   mask_ptr pointer to the mask bytes
 * @param   not_equal_byte_index the byte position for a not equal compare
 * @param   crc TRUE to use CRC checking FALSE to ignore
 */
void STB_DMXSetupSectFilter(U8BIT path, U16BIT sfilt_id, U8BIT *match_ptr, U8BIT *mask_ptr,
   U8BIT not_equal_byte_index, BOOLEAN crc)
{
   S32BIT i;
   U32BIT depth;
   U16BIT dmx_channel_index;
   U16BIT dmx_filter_index;
   S_PID_FILTER* dmx_channel_ptr;
   S_SECT_FILTER* dmx_filter_ptr;

   FUNCTION_START(STB_DMXSetupSectFilter);

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
	   /* Section filter id includes the PID filter id */
	   dmx_channel_index = sfilt_id >> 8;
	   dmx_filter_index = sfilt_id & 0x0f;
	   
	   if (dmx_channel_index >= max_pid_filters_num || dmx_filter_index >= SECTION_FILTERS_A_CHANNEL)
	   {
		  DMX_ERROR("Wrong section filter");
	   }
	   else
	   {
		   dmx_channel_ptr = (S_PID_FILTER *)&pid_filters[dmx_channel_index];
		   dmx_filter_ptr = (S_SECT_FILTER *)&dmx_channel_ptr->section_filters[dmx_filter_index];
		   if(!dmx_channel_ptr->is_valid || !dmx_filter_ptr->is_valid)
		   {
			   DMX_ERROR(("This filter is not used"));
		   }
		   else
		   {
                RTK_SectionFilterMask * pSectionFilterMask = &dmx_filter_ptr->filter_mask;
                memset(pSectionFilterMask, 0, sizeof(RTK_SectionFilterMask));
                pSectionFilterMask->pid = dmx_channel_ptr->pid;
                pSectionFilterMask->monitoring_mode = RTK_SECTION_MODE_CONTINOUS;
                pSectionFilterMask->callback_01 = rtk_section_callback;
                pSectionFilterMask->u32byWaitTimeoutMs = 0;
                pSectionFilterMask->section_user_param = dmx_filter_ptr;

			   depth = MAX_HW_SECT_FILT_LEN;
			   while (depth > 1 && mask_ptr[depth - 1] == 0)
			   {
				  depth--;
			   }
			   memset(dmx_filter_ptr->au8Match, 0, 16);
			   memset(dmx_filter_ptr->au8Mask, 0, 16);
			   memset(dmx_filter_ptr->au8Negate, 0, 16);
			   for (i = 0; i < depth; i++)
			   {
				  dmx_filter_ptr->au8Match[i] = match_ptr[i];
				  dmx_filter_ptr->au8Mask[i] = mask_ptr[i];

                  pSectionFilterMask->mask[i] = mask_ptr[i];    /// inclusion mask for comp[]. See Description.
                  //pSectionFilterMask->mode[i] = 0x00;      	/// exclusion mask for comp[]. See Description.
                  pSectionFilterMask->comp[i] = match_ptr[i];   /// bits to be included or excluded, depending on mask[] and mode[]..
			   }

               // FIXME: workaround for EIT
               if (pSectionFilterMask->comp[0] == 0x70) {
                   pSectionFilterMask->comp[0] = 0x40;
                   pSectionFilterMask->mask[0] = 0xC0;
               }
               else if ((pSectionFilterMask->comp[0] >= 0x51) && (pSectionFilterMask->comp[0] <= 0x5F))
                   pSectionFilterMask->comp[0] = 0x50;
               else if ((pSectionFilterMask->comp[0] >= 0x61) && (pSectionFilterMask->comp[0] <= 0x6F))
                   pSectionFilterMask->comp[0] = 0x60;
               
			   if ((not_equal_byte_index > 0) && (not_equal_byte_index < depth))
			   {
				  dmx_filter_ptr->au8Negate[not_equal_byte_index] = 1;
			   }
			   {
				  dmx_filter_ptr->u32FilterDepth = depth;
			   }
		   }
	   }
   }

   FUNCTION_FINISH(STB_DMXSetupSectFilter);
}

/**
 * @brief   Start Specified PID Filter Collecting Data.
 * @param   path Required Decode Path Number.
 * @param   pfilter_id Required PID Filter Identifier.
 */
void STB_DMXStartPIDFilter(U8BIT path, U16BIT pfilt_id)
{
   E_DEMUX_CMD cmd;
   U16BIT dmx_channel_index;
   U16BIT dmx_filter_index;
   S_PID_FILTER* dmx_channel_ptr;
   S_SECT_FILTER* dmx_filter_ptr;
   S_PES_FILTER * pes_filter_ptr;

   U8BIT decode_path = STB_DPGetPathOfDemux(path);
   if (INVALID_RES_ID == decode_path) {
      return;
   }

   FUNCTION_START(STB_DMXStartPIDFilter);

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
	   /* Section filter id includes the PID filter id */
	   dmx_channel_index = (pfilt_id >> 8) & 0xFF;
	   dmx_filter_index = pfilt_id & 0x0f;
	   
	   if ((dmx_channel_index >= max_pid_filters_num) || (dmx_filter_index >= SECTION_FILTERS_A_CHANNEL))
	   {
		  DMX_ERROR("Wrong PID filter");
	   }
	   else
	   {
		   dmx_channel_ptr = (S_PID_FILTER *)&pid_filters[dmx_channel_index];
		   if(!dmx_channel_ptr->is_valid)
		   {
			   DMX_ERROR("This filter is not used, pfilt_id=0x%x, dmx_channel_index=%d, dmx_filter_index=%d, dmx_channel_ptr->is_valid=0x%x, dmx_filter_ptr->is_valid=0x%x", 
			   	pfilt_id, dmx_channel_index, dmx_filter_index, dmx_channel_ptr->is_valid, dmx_filter_ptr->is_valid);
		   }
		   else
		   {
		   	if(dmx_channel_ptr->chan_type == CHANNEL_TYPE_SECTION)
		   	{
		         dmx_filter_ptr = (S_SECT_FILTER *)&dmx_channel_ptr->section_filters[dmx_filter_index];
				   if (!dmx_filter_ptr->is_active)
				   {
                  RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(decode_path);
                  if (NULL != pRealtekPlayer) {
                     RTK_Error ret;
                     ret = pRealtekPlayer->addSectionFilter(&(dmx_filter_ptr->filter_handle) , &dmx_filter_ptr->filter_mask);
                     if (RTK_OK == ret) {
                        dmx_filter_ptr->is_active = TRUE;
                     }     
					   }
				   }
		   	}
            else if (dmx_channel_ptr->chan_type == CHANNEL_TYPE_PES) {
               pes_filter_ptr = (S_PES_FILTER *)&dmx_channel_ptr->pes_filter;
               if (!dmx_channel_ptr->is_active)
               {
                  RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(decode_path);
                  if (NULL != pRealtekPlayer) {
                     RTK_Error ret;

                     pes_filter_ptr->pes_pattern.u8filterID = 0;
                     pes_filter_ptr->pes_pattern.callback = rtk_pes_callback;
                     pes_filter_ptr->pes_pattern.userParam = dmx_channel_ptr;	 // FIXME
                     pes_filter_ptr->pes_pattern.pid = dmx_channel_ptr->pid;

                     ret = pRealtekPlayer->addPESFilter(&(pes_filter_ptr->pes_pattern));
                     if (RTK_OK == ret) {
                        dmx_channel_ptr->is_active = TRUE;
                     }
                  }
               }
            }
            if (!dmx_channel_ptr->is_active)
            {
               dmx_channel_ptr->is_active = TRUE;
               active_pid_filters_num++;
               cmd = DEMUX_CMD_TRIGGER;
               STB_OSWriteQueue(dmx_queue, &cmd, sizeof(E_DEMUX_CMD), TIMEOUT_NOW);
            }
         }
      }
   }

   FUNCTION_FINISH(STB_DMXStartPIDFilter);
}

/**
 * @brief   Stop Specified PID Filter Collecting Data.
 * @param   path Required Decode Path Number.
 * @param   pfilter_id Required PID Filter Identifier.
 */
void STB_DMXStopPIDFilter(U8BIT path, U16BIT pfilt_id)
{
   S32BIT i;
   U16BIT dmx_channel_index;
   U16BIT dmx_filter_index;
   S_PID_FILTER* dmx_channel_ptr;
   S_SECT_FILTER* dmx_filter_ptr;
   S_PES_FILTER * pes_filter_ptr;
   BOOLEAN no_filter_open;
   U8BIT decode_path = STB_DPGetPathOfDemux(path);
   if (INVALID_RES_ID == decode_path) {
      return;
   }

   FUNCTION_START(STB_DMXStopPIDFilter);

   if (!is_dmx_initialised)
   {
	  DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
	  DMX_ERROR("Wrong demux path");
   }
   else
   {
	   /* Section filter id includes the PID filter id */
	   dmx_channel_index = (pfilt_id >> 8) & 0xFF;
	   dmx_filter_index = pfilt_id & 0x0f;
	   
	   if ((dmx_channel_index >= max_pid_filters_num) || (dmx_filter_index >= SECTION_FILTERS_A_CHANNEL))
	   {
		  DMX_ERROR("Wrong PID filter");
	   }
	   else
	   {
		   dmx_channel_ptr = (S_PID_FILTER *)&pid_filters[dmx_channel_index];
		   dmx_filter_ptr = (S_SECT_FILTER *)&dmx_channel_ptr->section_filters[dmx_filter_index];
		   if(!dmx_channel_ptr->is_valid)
		   {
			   DMX_ERROR("This filter is not used, pfilt_id=0x%x, dmx_channel_index=%d, dmx_filter_index=%d, dmx_channel_ptr->is_valid=0x%x, dmx_filter_ptr->is_valid=0x%x", 
			   	pfilt_id, dmx_channel_index, dmx_filter_index, dmx_channel_ptr->is_valid, dmx_filter_ptr->is_valid);
		   }
		   else
		   {
			   if(dmx_channel_ptr->chan_type == CHANNEL_TYPE_SECTION)
			   {
					if (dmx_filter_ptr->is_active)
					{
                  RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(decode_path);
                  if (NULL != pRealtekPlayer) {
                     RTK_Error ret;
                     ret = pRealtekPlayer->removeSectionFilter(dmx_filter_ptr->filter_handle);
                     if (RTK_OK != ret) {
                        DMX_ERROR("RealtekPlayer::removeSectionFilter failed  decode_path %u", decode_path);
                     }
                     dmx_filter_ptr->is_active = FALSE;
					   }
					}
			   }
            else if (dmx_channel_ptr->chan_type == CHANNEL_TYPE_PES) {
        		   pes_filter_ptr = (S_PES_FILTER *)&dmx_channel_ptr->pes_filter;
				   if (!dmx_filter_ptr->is_active)
				   {
                  RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(decode_path);
                  if (NULL != pRealtekPlayer) {
                     RTK_Error ret;

                     ret = pRealtekPlayer->removePESFilter(0);
                     if (RTK_OK == ret) {
                        dmx_filter_ptr->is_active = FALSE;
                     }
                  }
               }
            }

            if (dmx_channel_ptr->is_active)
            {
               no_filter_open = TRUE;
               /* Check to see whether any filters are still running */
               for(i=0; i<SECTION_FILTERS_A_CHANNEL; i++)
               {
                  if(dmx_channel_ptr->section_filters[i].is_active)
                  {
	                  no_filter_open = FALSE;
	                  break;
                  }
               }
               if (no_filter_open)
               {
                  active_pid_filters_num--;
                  //cmd = DEMUX_CMD_TRIGGER;
                  //STB_OSWriteQueue(dmx_queue, &cmd, sizeof(E_DEMUX_CMD), TIMEOUT_NOW);
                  dmx_channel_ptr->is_active = FALSE;
               }
            }
         }
      }
   }
   FUNCTION_FINISH(STB_DMXStopPIDFilter);
}

/**
 * @brief   Releases a previously allocated PID filter
 * @param   path the demux path of the filter
 * @param   pfilt_id the handle of the filter
 */
void STB_DMXReleasePIDFilter(U8BIT path, U16BIT pfilt_id)
{
   U16BIT dmx_channel_index;
   U16BIT dmx_filter_index;
   S_PID_FILTER* dmx_channel_ptr;
   BOOLEAN no_filter_open;
   S32BIT i;

   FUNCTION_START(STB_DMXReleasePIDFilter);

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
	   /* Section filter id includes the PID filter id */
	   dmx_channel_index = (pfilt_id >> 8) & 0xFF;
	   dmx_filter_index = pfilt_id & 0x0f;
	   
	   if (dmx_channel_index >= max_pid_filters_num || dmx_filter_index >= SECTION_FILTERS_A_CHANNEL)
	   {
		  DMX_ERROR("Wrong PID filter");
	   }
	   else
	   {
		   dmx_channel_ptr = (S_PID_FILTER *)&pid_filters[dmx_channel_index];
		   if(!dmx_channel_ptr->is_valid)
		   {
			   DMX_ERROR(("This filter is not used"));
		   }
		   else
		   {
			   no_filter_open = TRUE;
			   if(dmx_channel_ptr->chan_type == CHANNEL_TYPE_SECTION)
			   {
				   /* Check to see whether any filters are still running */
				   for(i=0; i<SECTION_FILTERS_A_CHANNEL; i++)
				   {
					   if(dmx_channel_ptr->section_filters[i].is_valid)
					   {
						   no_filter_open = FALSE;
						   break;
					   }
				   }
			   }			   
			   if (no_filter_open)
			   {
				   if (dmx_channel_ptr->is_active)
				   {
					  /* This is deadlock-safe, because stbos_mutex implementation
					   * allows to lock mutex by the same thread multiple times */
					  STB_DMXStopPIDFilter(path, pfilt_id);
				   }

				   {
					  DMX_DEBUG("PID filter [%d] is released", pfilt_id);
					  memset(dmx_channel_ptr, 0, sizeof(S_PID_FILTER));
					  if (pid_filters_num == 0)
					  {
						 DMX_WARNING("Something is wrong with PID filters counter");
					  }
					  else
					  {
						 pid_filters_num--;
					  }
				   }
			   }
		   }
	   	}
   	}
   FUNCTION_FINISH(STB_DMXReleasePIDFilter);
}

/**
 * @brief   Releases a previously allocated section filter
 * @param   path the demux path of the filter
 * @param   sfilt_id the handle of the section filter
 */
void STB_DMXReleaseSectFilter(U8BIT path, U16BIT sfilt_id)
{
	U16BIT dmx_channel_index;
	U16BIT dmx_filter_index;
	S_PID_FILTER* dmx_channel_ptr;
	S_SECT_FILTER* dmx_filter_ptr;

   FUNCTION_START(STB_DMXReleaseSectFilter);

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
	   /* Section filter id includes the PID filter id */
	   dmx_channel_index = (sfilt_id >> 8) & 0xFF;
	   dmx_filter_index = sfilt_id & 0x0f;
	   
	   if (dmx_channel_index >= max_pid_filters_num || dmx_filter_index >= SECTION_FILTERS_A_CHANNEL)
	   {
		  DMX_ERROR("Wrong PID filter");
	   }
	   else
	   {
		   dmx_channel_ptr = (S_PID_FILTER *)&pid_filters[dmx_channel_index];
		   dmx_filter_ptr = (S_SECT_FILTER *)&dmx_channel_ptr->section_filters[dmx_filter_index];
		   if(!dmx_channel_ptr->is_valid)
		   {
			   DMX_ERROR(("This filter is not used"));
		   }
		   else
		   {
			   if(dmx_filter_ptr->is_active)
			   {
				   {
					  dmx_filter_ptr->is_active = FALSE;
				   }
			   }
			   if(dmx_filter_ptr->is_valid)
			   {
				  {
					 dmx_filter_ptr->is_valid = FALSE;
#ifdef DEMUX_DEBUGS
					 if (sect_filters_num == 0)
					 {
						DMX_WARNING("Something is wrong with section filters counter");
					 }
					 else
					 {
						sect_filters_num--;
					 }
					 
					 DMX_DEBUG("sect_filters_num = %d", sect_filters_num);
#endif					 
					 memset(dmx_filter_ptr, 0, sizeof(S_SECT_FILTER));
				  }
			   }

		   }
	   	}
   	}
   FUNCTION_FINISH(STB_DMXReleaseSectFilter);
}

/**
 * @brief   Acquires a descrambler for the specified track on this path
 * @param   path the demux path for which the descrambler is acquired
 * @param   track enum representing audio, video or subtitles PES
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN STB_DMXGetDescramblerKey(U8BIT path, E_STB_DMX_DESC_TRACK track)
{
    return FALSE;
}

/**
 * @brief   Frees the descrambler for the specified track on this path
 * @param   path the demux path for which the descrambler is freed
 * @param   track enum representing audio, video or subtitles PES
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN STB_DMXFreeDescramblerKey(U8BIT path, E_STB_DMX_DESC_TRACK track)
{
    return FALSE;
}

/**
 * @brief   Set the descrambler key data for the specified track on this path
 * @param   path the demux path for which the descrambler key data is set
 * @param   track enum representing audio, video or subtitles PES
 * @param   parity even or odd
 * @param   data pointer to the key data, its length depends on the descrambler
 *          type (see STB_DMXSetDescramblerType)
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN STB_DMXSetDescramblerKeyData(U8BIT path, E_STB_DMX_DESC_TRACK track, E_STB_DMX_DESC_KEY_PARITY parity, U8BIT *data)
{
    return FALSE;
}

/**
 * @brief   Get the descrambler key usage for the specified track on this path as set by
 *          STB_DMXSetKeyUsage
 * @param   path the demux path that the descrambler key usage refers to
 * @param   track enum representing audio, video or subtitles PES
 * @param   key_usage whether the descrambler has been set to operate at PES level, transport
 *          level or all.
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN STB_DMXGetKeyUsage(U8BIT path, E_STB_DMX_DESC_TRACK track, E_STB_DMX_KEY_USAGE *key_usage)
{
    return FALSE;
}

/**
 * @brief   Set the descrambler key usage for the specified track on this path
 * @param   path the demux path that the descrambler key usage refers to
 * @param   track enum representing audio, video or subtitles PES
 * @param   key_usage whether the descrambler operates at PES level, transport
 *          level or all.
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN STB_DMXSetKeyUsage(U8BIT path, E_STB_DMX_DESC_TRACK track, E_STB_DMX_KEY_USAGE key_usage)
{
    return FALSE;
}

/**
 * @brief   Get the descrambler type for the specified track on this path, as set by
 *          STB_DMXSetDescramblerType
 * @param   path the demux path that the descrambler type refers to
 * @param   type descrambler type (DES, AES, etc...)
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN STB_DMXGetDescramblerType(U8BIT path, E_STB_DMX_DESC_TRACK track, E_STB_DMX_DESC_TYPE *type)
{
    return FALSE;
}

/**
 * @brief   Set the descrambler type for the specified track on this path
 * @param   path the demux path that the descrambler type refers to
 * @param   type descrambler type (DES, AES, etc...)
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN STB_DMXSetDescramblerType(U8BIT path, E_STB_DMX_DESC_TRACK track, E_STB_DMX_DESC_TYPE type)
{
    return FALSE;
}

/**
 * @brief   Writes data to the demux from memory
 * @param   path the demux path to be written
 * @param   data the data to be written
 * @param   size the number of bytes to be written
 */
void STB_DMXWriteDemux(U8BIT path, U8BIT *data, U32BIT size)
{

}

/*--- Global functions not declared in stbhwdmx.h -----------------------------*/

/*!**************************************************************************
 * @fn      DMX_Terminate
 * @brief
 * @param
 ****************************************************************************/
void DMX_Terminate(int paths)
{
   U32BIT i;
   E_DEMUX_CMD cmd;

   FUNCTION_START(DMX_Terminate);
   USE_UNWANTED_PARAM(paths);
   for (i = 0; i < dmxs_number; i++)
   {
      ReleaseTextPESFilter(i);
   }
   if (dmx_task != NULL)
   {
      cmd = DEMUX_CMD_TASK_TERMINATE;
      STB_OSWriteQueue(dmx_queue, &cmd, sizeof(E_DEMUX_CMD), TIMEOUT_NEVER);
      STB_OSMutexLock(dmx_task_termination_mutex); // Wait for DemuxTask to return
      STB_OSDestroyTask(dmx_task);
      STB_OSMutexUnlock(dmx_task_termination_mutex);
      dmx_task = NULL;
   }
   if (dmx_task_termination_mutex != NULL)
   {
      STB_OSDeleteMutex(dmx_task_termination_mutex);
      dmx_task_termination_mutex = NULL;
   }
   if (dmx_queue != NULL)
   {
      STB_OSDestroyQueue(dmx_queue);
      dmx_queue = NULL;
   }
   if (dmx_mutex != NULL)
   {
      STB_OSDeleteMutex(dmx_mutex);
      dmx_mutex = NULL;
   }
   if (pid_filters != NULL)
   {
      STB_MEMFreeSysRAM(pid_filters);
      pid_filters = NULL;
   }
   if (demuxes != NULL)
   {
      STB_MEMFreeSysRAM(demuxes);
      demuxes = NULL;
   }
#ifdef DEMUX_DEBUGS
   sect_filters_num = 0;
#endif
   pid_filters_num = 0;
   max_sect_filters_num = 0;
   max_pid_filters_num = 0;
   dmxs_number = 0;

   is_dmx_initialised = FALSE;
   DMX_DEBUG("Terminated");

   FUNCTION_FINISH(DMX_Terminate);
}

/*!**************************************************************************
 * @fn      DMX_GetDecodePIDs
 * @brief
 * @param
 * @return  TRUE if successfull, FALSE otherwise
 ****************************************************************************/
BOOLEAN DMX_GetDecodePIDs(U8BIT path, U16BIT *pcr_pid, U16BIT *video_pid, U16BIT *audio_pid,
                          U16BIT *text_pid, U16BIT *data_pid, U16BIT *ad_pid)
{
   BOOLEAN retval;

   FUNCTION_START(DMX_GetDecodePIDs);
   retval = FALSE;

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
      *pcr_pid = demuxes[path].pcr_pid;
      *video_pid = demuxes[path].video_pid;
      *audio_pid = demuxes[path].audio_pid;
      *text_pid = demuxes[path].text_pid;
      *data_pid = demuxes[path].data_pid;
      *ad_pid = demuxes[path].ad_pid;
      retval = TRUE;
   }

   FUNCTION_FINISH(DMX_GetDecodePIDs);
   return retval;
}

/*!**************************************************************************
 * @fn      DMX_GetEcmPIDs
 * @brief   Get ECM PIDs set for decoding by demux path
 * @param   path - demux path
 * @param   ecm_pid - (out) program ECM PID
 * @param   video_pid - (out) video ECM PID
 * @param   audio_pid - (out) audio ECM PID
 * @param   text_pid - (out) text ECM PID
 * @param   data_pid - (out) data ECM PID
 * @param   ad_pid - (out) audio description ECM PID
 * @return  TRUE if successfull, FALSE otherwise
 ****************************************************************************/
BOOLEAN DMX_GetEcmPIDs(U8BIT path, U16BIT *ecm_pid, U16BIT *video_ecm_pid, U16BIT *audio_ecm_pid,
                          U16BIT *text_ecm_pid, U16BIT *data_ecm_pid, U16BIT *ad_ecm_pid)
{
   BOOLEAN retval;

   FUNCTION_START(DMX_GetEcmPIDs);
   retval = FALSE;

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
      *ecm_pid = demuxes[path].ecm_pid;
      *video_ecm_pid = demuxes[path].video_ecm_pid;
      *audio_ecm_pid = demuxes[path].audio_ecm_pid;
      *text_ecm_pid = demuxes[path].text_ecm_pid;
      *data_ecm_pid = demuxes[path].data_ecm_pid;
      *ad_ecm_pid = demuxes[path].ad_ecm_pid;
      retval = TRUE;
   }

   FUNCTION_FINISH(DMX_GetEcmPIDs);
   return retval;
}

U16BIT DMX_GetLiveAudioPID(void)
{
   U16BIT retval = INVALID_PID;

   FUNCTION_START(DMX_GetLiveAudioPID);

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else
   {
      retval = demuxes[LIVE_PATH].audio_pid;
   }

   FUNCTION_FINISH(DMX_GetLiveAudioPID);
   return retval;
}

/*!**************************************************************************
 * @fn      DMX_GetRecordPath
 * @brief
 * @path	the record path
 * @return  the demux path for ts recording
 ****************************************************************************/
U8BIT DMX_GetRecordPath(U8BIT path)
{
   return path;
}

/*!**************************************************************************
 * @fn      DemuxTask
 * @brief   DemuxTask to monitor the section filters
 * @param   param - demux path
 ****************************************************************************/
static void DemuxTask(void *param)
{
   U16BIT timeout;
   E_DEMUX_CMD cmd;

   FUNCTION_START(DemuxTask);
   USE_UNWANTED_PARAM(param);

   STB_OSMutexLock(dmx_task_termination_mutex);

   timeout = TIMEOUT_NEVER;

   while (TRUE)
   {
      if (STB_OSReadQueue(dmx_queue, &cmd, sizeof(E_DEMUX_CMD), timeout))
      {
         if (cmd == DEMUX_CMD_TASK_TERMINATE)
            break;

         if (active_pid_filters_num == 0)
         {
            timeout = TIMEOUT_NEVER;
         }
         else
         {
            if (cmd == DEMUX_CMD_DATA_READY) {
               for (U8BIT path = 0; path < dmxs_number; path++) {
                  U8BIT * pData = NULL;

                  STB_OSMutexLock(dmx_mutex);
                  if (! sectionBufferLists[path].empty()) {
                     pData = (U8BIT *)sectionBufferLists[path].front();
                     sectionBufferLists[path].pop_front();
                  }
                  STB_OSMutexUnlock(dmx_mutex);

                  if (NULL != pData) {
                     U16BIT dmx_channel_index;
                     U16BIT dmx_filter_index;
                     S_PID_FILTER* dmx_channel_ptr;
                     S_SECT_FILTER* dmx_filter_ptr;
                     
                     U16BIT filt_id =  ((U16BIT)pData[0] << 8) + pData[1];
                     U32BIT length = ((U32BIT)pData[2] << 24) + ((U32BIT)pData[3] << 16) + ((U32BIT)pData[4] << 8) + pData[5];
                     U8BIT * section = pData + 6;

                     /* Section filter id includes the PID filter id */
                     dmx_channel_index = pData[0]; //filt_id >> 8;
                     dmx_filter_index = pData[1];  //filt_id & 0x0f;

                     dmx_channel_ptr = (S_PID_FILTER *)&pid_filters[dmx_channel_index];
                     dmx_filter_ptr = (S_SECT_FILTER *)&dmx_channel_ptr->section_filters[dmx_filter_index];
 
                     if (NULL != dmx_filter_ptr->callback) {
                         dmx_filter_ptr->length = length;
                         dmx_filter_ptr->buffer = section;
                         dmx_filter_ptr->callback (path, (U16BIT)length, filt_id );
                     }
                     STB_MEMFreeSysRAM(pData);  // consumer        
                  }
               }
            } // if (cmd == DEMUX_CMD_DATA_READY)
         }
      }  // if (STB_OSReadQueue ...) 
   }  // while (TRUE)

   STB_OSMutexUnlock(dmx_task_termination_mutex);
   FUNCTION_FINISH(DemuxTask);
}

static void DMX_PESCallback( U8BIT path, U16BIT bytes, U16BIT pfilt_id )
{
   USE_UNWANTED_PARAM(path);
   USE_UNWANTED_PARAM(bytes);
   USE_UNWANTED_PARAM(pfilt_id);
}

static U16BIT DMX_GrabChannel(U8BIT path, U8BIT chan_type, U16BIT pid, FILTER_CALLBACK func_ptr)
{
   U16BIT dmx_channel_index;
   U16BIT dmx_filter_index;
   U16BIT pfilt_id = DEMUX_FILTER_NOT_ALLOCATED;
   int i;
   S_PID_FILTER* dmx_channel_ptr;

   FUNCTION_START(DMX_GrabChannel);

   if (!is_dmx_initialised)
   {
      DMX_ERROR("DMX module is not initialised");
   }
   else if (path >= dmxs_number)
   {
      DMX_ERROR("Wrong demux path");
   }
   else
   {
      for (dmx_channel_index = 0; dmx_channel_index < max_pid_filters_num; dmx_channel_index++)
      {
		   if(pid_filters[dmx_channel_index].is_valid)
		   {
			   if((pid_filters[dmx_channel_index].demux_path == path)&&(pid_filters[dmx_channel_index].pid == pid))
			   {
				   pfilt_id = dmx_channel_index << 8;
				   dmx_channel_ptr = &pid_filters[dmx_channel_index];

				   if(chan_type == CHANNEL_TYPE_SECTION)
				   {
					   dmx_filter_index = 0;
					   for (i=0; i<SECTION_FILTERS_A_CHANNEL; i++)
					   {
						   if (!dmx_channel_ptr->section_filters[i].is_valid)
						   {
							   dmx_filter_index = i;
							   pid_filters[dmx_channel_index].section_callback[dmx_filter_index] = func_ptr;
							   dmx_channel_ptr->demux_path = path;
							   dmx_channel_ptr->is_valid = TRUE;
							   dmx_channel_ptr->is_active = FALSE;
							   dmx_channel_ptr->chan_type = (E_CHANNEL_TYPE) chan_type;
							   pfilt_id |= dmx_filter_index;
							   break;
						   }
					   }
				   }
				   else if(chan_type == CHANNEL_TYPE_PES)
				   {
					   if ((dmx_channel_ptr->chan_type == (E_CHANNEL_TYPE) chan_type) && (TRUE == dmx_channel_ptr->is_valid))
					   {
						   pid_filters[dmx_channel_index].pes_callback = func_ptr;
						   dmx_channel_ptr->demux_path = path;
						   dmx_channel_ptr->is_valid = TRUE;
						   dmx_channel_ptr->is_active = FALSE;
						   dmx_channel_ptr->chan_type = (E_CHANNEL_TYPE) chan_type;
						   //pfilt_id |= dmx_filter_index;
						   break;
					   }
				   }
				   else
				   {
					   dmx_filter_index = 0;
					   break;
				   }
				   break;
            }
         }
      }  // for (dmx_channel_index ...)

      if(dmx_channel_index < max_pid_filters_num)
      {
         DMX_DEBUG("Found a using PID filter, dmx_channel_index=%d, dmx_filter_index=%d\n", dmx_channel_index, dmx_filter_index);
      }
      else
      {
         for (dmx_channel_index = 0; dmx_channel_index < max_pid_filters_num; dmx_channel_index++)
         {
            if (!pid_filters[dmx_channel_index].is_valid)
               break;
         }
		
         if (dmx_channel_index == max_pid_filters_num)
         {
            DMX_ERROR("Failed to allocate new PID filter: no free slots left");
         }
         else
         {
            /* TODO: Check if PID is valid */
            if (chan_type == CHANNEL_TYPE_SECTION) {
                  dmx_channel_ptr = &pid_filters[dmx_channel_index];
                  dmx_channel_ptr->demux_path = path;
                  //dmx_channel_ptr->callback = func_ptr;
                  dmx_channel_ptr->is_valid = TRUE;
                  dmx_channel_ptr->is_active = FALSE;
                  dmx_channel_ptr->chan_type = (E_CHANNEL_TYPE) chan_type;
                  dmx_channel_ptr->pid = pid;
                  pid_filters_num++;
				 
                  pfilt_id = dmx_channel_index;
                  pfilt_id <<= 8;
                  for (i = 0; i != SECTION_FILTERS_A_CHANNEL; i++)
                  {
                     memset(&pid_filters[dmx_channel_index].section_filters[i], 0, sizeof(S_SECT_FILTER));
                  }
                  dmx_filter_index = 0;
                  dmx_channel_ptr->section_callback[dmx_filter_index] = func_ptr;
                  pfilt_id |= dmx_filter_index;
            }
            else if(chan_type == CHANNEL_TYPE_PES)
            {
               dmx_channel_ptr = &pid_filters[dmx_channel_index];
               dmx_channel_ptr->pes_callback = func_ptr;
               dmx_channel_ptr->demux_path = path;
               dmx_channel_ptr->is_valid = TRUE;
               dmx_channel_ptr->is_active = FALSE;
               dmx_channel_ptr->chan_type = (E_CHANNEL_TYPE) chan_type;
               dmx_channel_ptr->pid = pid;
               pid_filters_num++;
               pfilt_id = dmx_channel_index;
               pfilt_id <<= 8;
            }
         }
      }  //  if(dmx_channel_index < max_pid_filters_num)
   }

   FUNCTION_FINISH(DMX_GrabChannel);
   return pfilt_id;
}

/*!**************************************************************************
 * @fn      AcquireTextPESFilter
 * @brief   Create PES filter for subtitles by their PID
 * @param   path - demux path
 * @param   pid - subtitles PID
 * @return  TRUE if successfull, FALSE otherwise
 ****************************************************************************/
static BOOLEAN AcquireTextPESFilter(U8BIT path, U16BIT pid)
{
   BOOLEAN ret = FALSE;
   U16BIT retval;

   FUNCTION_START(AcquireTextPESFilter);
   ASSERT(is_dmx_initialised)
   ASSERT(path < dmxs_number)

   retval = DMX_GrabChannel(path, CHANNEL_TYPE_PES, pid, DMX_PESCallback);
   if (DEMUX_FILTER_NOT_ALLOCATED != retval) {
      demuxes[path].text_datalen = 0;
      if(NULL == demuxes[path].text_buffer) {
         demuxes[path].text_buffer = (U8BIT *)STB_MEMGetSysRAM(TEXT_BUFFER_SIZE);
      }
      demuxes[path].text_channel = retval;
      demuxes[path].text_pid = pid;
      STB_DMXStartPIDFilter(path, retval);
      ret = TRUE;
   }

   FUNCTION_FINISH(AcquireTextPESFilter);
   return ret;
}



/*!**************************************************************************
 * @fn      ReleaseTextPESFilter
 * @brief   Release text PID filter
 * @param   path - demux path
 ****************************************************************************/
static void ReleaseTextPESFilter(U8BIT path)
{
   FUNCTION_START(ReleaseTextPESFilter);

   ASSERT(is_dmx_initialised)
   ASSERT(path < dmxs_number)
   
   if ((demuxes[path].text_channel >> 8) < max_pid_filters_num)
   {
      STB_DMXStopPIDFilter(path, demuxes[path].text_channel);
      STB_DMXReleasePIDFilter(path, demuxes[path].text_channel);
      demuxes[path].text_channel = 0xFFFF;
      demuxes[path].text_datalen = 0;
      if(NULL != demuxes[path].text_buffer) {
         STB_MEMFreeSysRAM(demuxes[path].text_buffer);
         demuxes[path].text_buffer = NULL;
      }
   }

   demuxes[path].text_pid = INVALID_PID;

   FUNCTION_FINISH(ReleaseTextPESFilter);
}

static void rtk_section_callback (
    const RTK_SectionFilter handle,
    RTK_HAL_SECTION_CB_STATUS enStatus,
    const unsigned char *section,
    const unsigned int length,
    void * userParam )
{
	//UNUSED(handle); UNUSED(section); UNUSED(length); UNUSED(userParam);
	//printf("%s() handle=%p, enStatus=%d, table_id=0x%x, length=%u, userParam=%p\n",
    //    __FUNCTION__,handle, enStatus, section[0], length, userParam);

    if (HAL_SECTION_FILTER_HIT == enStatus) {
        S_SECT_FILTER * dmx_filter_ptr = (S_SECT_FILTER *)userParam;

        if (NULL != dmx_filter_ptr) { 
            U8BIT path = pid_filters[dmx_filter_ptr->pidFilterId].demux_path;
            U16BIT pfilt_id = (dmx_filter_ptr->pidFilterId << 8) + dmx_filter_ptr->sectionFilterId;
            U8BIT * pData = (U8BIT *)STB_MEMGetSysRAM(2 /*pfilt_id*/ + 4 /*length*/ + length /*data*/);  // producer
            if (NULL != pData) {
               // big endian / network alignment
               // type: pfilt_id
               pData[0] = dmx_filter_ptr->pidFilterId;  //(pfilt_id >> 8) & 0xFF;
               pData[1] = dmx_filter_ptr->sectionFilterId;  //(pfilt_id >> 0) & 0xFF;
               // length
               pData[2] = (length >> 24) & 0xFF;
               pData[3] = (length >> 16) & 0xFF;
               pData[4] = (length >> 8) & 0xFF;
               pData[5] = (length >> 0) & 0xFF;
               // data
               memcpy(pData + 6, section, length);
               STB_OSMutexLock(dmx_mutex);
               sectionBufferLists[path].push_back(pData);
               STB_OSMutexUnlock(dmx_mutex);
               // trigger
               E_DEMUX_CMD cmd = DEMUX_CMD_DATA_READY;
               STB_OSWriteQueue(dmx_queue, &cmd, sizeof(E_DEMUX_CMD), TIMEOUT_NOW);
            }
        }
    }
}

static void rtk_pes_callback(uint32_t filterID , void * pes, uint32_t length, void * userParam)
{
   S_PID_FILTER * dmx_channel_ptr = (S_PID_FILTER *)userParam;   
   if (NULL != dmx_channel_ptr) {
      U8BIT path = dmx_channel_ptr->demux_path;
      if (path < dmxs_number) {
         void * pData = STB_MEMGetSysRAM(length);  // producer
         if (NULL != pData) {
            std::pair<void *,uint32_t> p = std::make_pair(pData, length);
            memcpy(pData, pes, length);
            STB_OSMutexLock(dmx_mutex);
            textBufferList.push_back(p);
            STB_OSMutexUnlock(dmx_mutex);
         }
      }
   }
}

static void rtk_player_callback (RTK_PLAYER_CB_TYPE_e type, void * data, void * user_param)
{
   USE_UNWANTED_PARAM(data);
   RealtekPlayer * pRealtekPlayer = (RealtekPlayer *)user_param;
   if (NULL == pRealtekPlayer)
      return;
      
   U8BIT path = (U8BIT) pRealtekPlayer->getUserData();
   pRealtekPlayer->middlewarePlayerCallback(type, data);
   
   switch(type)
   {
      case RTK_PLAYER_TS_FIRSTRECEIVE:
         break;
      case RTK_PLAYER_FIRSTIFRAME_DISPLAYED:
         STB_OSSendEvent(FALSE, HW_EV_CLASS_DECODE, HW_EV_TYPE_VIDEO_STARTED, &path, sizeof(U8BIT));
         break;
      case RTK_PLAYER_EOS:
         break;
      default:
         break;
   }
}

