#include <string.h>
#include <math.h>
#include <iostream>
#include <sstream>
#include "stbhwtun.h"

/* STB header files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbhwmem.h"
#include "stbhwtun.h"

#include "internal_generic.h"
#include "internal.h"

#include "stbhwsdk.h"

extern "C" {
#include "stbdpc.h"
}

/*--- Preprocessor definitions ------------------------------------------------*/

#define TNR_TASK_STACK_SIZE            4096
#define TNR_TASK_PRIORITY              10
#define CONNECT_TIMEOUT_MS          1500

#define TUNER_LOCK_ATTEMPTS            7
#define TUNER_LOCK_INITIAL_TIMEOUT     10
#define TUNER_RELOCK_TIMEOUT           1000

#define MAX_COMBO_TUNER_NUM           4

#define STB_RETUNE_PERIOD           200
#define STB_MONITOR_PERIOD          1000
#define STB_MONITOR_RETRY_TIMES     5
#define STB_MONITOR_RETRY_PERIOD    20

#ifndef INVALID_RES_ID
#define INVALID_RES_ID     ((U8BIT)0xFF)     /* ID used to represent an invalid resource */
#endif

#define TUNER_DEBUGS
#define TUNER_WARNINGS
#define TUNER_ERRORS

#ifdef TUNER_DEBUGS
#define TUN_DEBUG(format, args...) STB_DEBUG_LOG("TUN", format, ##args)
#else
#define TUN_DEBUG(format, args...)
#endif

#ifdef TUNER_WARNINGS
#define TUN_WARNING(format, args...) STB_WARNING_LOG("TUN", format, ##args)
#else
#define TUN_WARNING(format, args...)
#endif

#ifdef TUNER_ERRORS
#define TUN_ERROR(format, args...) STB_ERROR_LOG("TUN", format, ##args)
#else
#define TUN_ERROR(format, args...)
#endif


/*--- Local types definitions -------------------------------------------------*/
typedef enum
{
   TUNER_TASK_NOT_RUNNING,
   TUNER_TASK_RUNNING,
   TUNER_TASK_STOPPING
} E_TUNER_TASK_STATUS;

typedef enum
{
    STBTUNE_COMMAND_CONNECT,
    STBTUNE_COMMAND_DISCONNECT,
    STBTUNE_COMMAND_RECONNECT,
    STBTUNE_COMMAND_TIMEOUT
}STBTUNE_COMMAND;

typedef enum tuner_cmd
{
   TUNER_CMD_LOCK,
   TUNER_CMD_TASK_TERMINATE
} E_TUNER_CMD;

typedef struct tuner_msg
{
   E_TUNER_CMD cmd;
   U8BIT tuner_path;
} S_TUNER_MSG;

typedef enum
{
    TUNER_IDLE,
    TUNER_TUNNING,
    TUNER_CONNECTED
}STBTUNE_STATE;


typedef struct tuner
{
   /* General status*/
   U16BIT lo_freq;
   STBTUNE_STATE tune_state;
   U32BIT freq;
   U32BIT srate;
   E_STB_TUNE_FEC fec;
   E_STB_TUNE_TMODE tmode;
   E_STB_TUNE_TBWIDTH tbwidth;
   E_STB_TUNE_CMODE cmode;
   E_STB_TUNE_SIGNAL_TYPE sig_type;
   E_STB_TUNE_MODULATION modulation;
   E_STB_TUNE_LNB_VOLTAGE lnb_power;
   BOOLEAN islock;
   U8BIT plp_pending;
   U8BIT plp;
   
   U8BIT path;
   BOOLEAN auto_relock;
   volatile BOOLEAN start_tuning;
   volatile BOOLEAN in_use;
   BOOLEAN sys_type_2_pending;	//Used to pass pseudo tune parameter for dvbt2
   E_STB_TUNE_SYSTEM_TYPE sys_type; //DVB-T or DVB-T2

   /* Tuner setup parameters */
   U32BIT quality;
   U32BIT strength;
   U32BIT snr;/*support to get snr. Jesse Hou 2019/06/12*/
   U32BIT ber[3];/*support to get ber. Jesse Hou 2019/06/12*/
} S_TUNER;

typedef struct 
{
	BOOLEAN 				is_combo_tuner;
	U16BIT 					tuner_type_caps;	/* it may support more than one type */
	E_STB_TUNE_SIGNAL_TYPE	tuner_type_used;
	U8BIT tuner_types;
	U8BIT tuner_id[4];
} COMBO_TUN_STATUS;

typedef struct{
    void* task_hdl; /* task handle */
    volatile E_TUNER_TASK_STATUS task_status;
    U8BIT new_tun_id;
    void* tune_queue;
    U8BIT cur_tun_id;
    int task_id; /* task number */
} TUN_TASK;

static U8BIT m_committed_command[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static U8BIT m_uncommitted_command[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static U8BIT m_22k[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static U8BIT m_last_committed_command[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static U8BIT m_last_uncommitted_command[8] = {0, 0, 0, 0, 0, 0, 0, 0};
static U8BIT m_last_22k[8] = {0, 0, 0, 0, 0, 0, 0, 0};

static COMBO_TUN_STATUS combo_tuner_status[MAX_COMBO_TUNER_NUM];
static int combo_tuner_counter = 0;
static U8BIT num_tuner_task = 1;
static TUN_TASK *tuner_task_status = NULL;
// FIXME: 'static' variable results in incorrect behavior
int thread_running_flag=1;


/*--- Local (static) variable declarations ------------------------------------*/

static void *tuner_task = NULL;
static U8BIT tuners_num = 0;
static S_TUNER *tuner_status = NULL;
static BOOLEAN is_tuner_initialized = FALSE;


/*--- Local function prototypes -----------------------------------------------*/

static BOOLEAN SetupTuner(U8BIT tuner_id, S_TUNER *tuner);
static void TunerTask(void *param);
static void combo_tuner_init(void);
static U8BIT TuneGetActualPath(U8BIT path);
static U8BIT TuneGetRealPath(U8BIT path);
static void tune_nim(U8BIT cmd,int tun_id);
static void tune(U8BIT cmd, U8BIT path);
static void untune(U8BIT path);


/*--- Global function definitions ---------------------------------------------*/

/**
 * @brief   Initialises the tuner component
 * @param   paths number of tuning paths to initialise
 */
void STB_TuneInitialise(U8BIT paths)
{
   S32BIT ret;
   U32BIT tuner_id;
   int i = 0;

   FUNCTION_START(STB_TuneInitialise);
   
   if (is_tuner_initialized)
   {
      TUN_WARNING("Already initialized. Exit");
   }
   else
   {
      RTK_Tuner_Init();

      /* Make sure the following variables are initialized with NULLs, because
       * it will indicate which resources need to be released in case of failure */
      tuner_task = NULL;
      tuner_status = NULL;
      tuners_num = 0;
      memset(combo_tuner_status, 0, sizeof(COMBO_TUN_STATUS)*MAX_COMBO_TUNER_NUM);
      combo_tuner_counter = 0;

      ret = 0;
      {
         tuner_status = (S_TUNER*) STB_MEMGetSysRAM(sizeof(S_TUNER) * paths);
         if (tuner_status == NULL)
         {
            TUN_ERROR("Failed to allocate memory for tuners");
         }
         else
         {
            memset(tuner_status, 0, sizeof(S_TUNER) * paths);
			tuners_num = 0;
            /* Open tuners and set their arguments */
            for (tuner_id = 0; tuner_id < paths; tuner_id++)
            {
               /* TODO: Posibly move it to STB_TuneStartTuner, because starting tuner takes
                * quite some time, and I guess no need to start all tuners at the boot up time */
               if (!SetupTuner(tuner_id, &tuner_status[tuner_id]))
               {
                  break;
               }
               tuner_status[tuner_id].sys_type_2_pending = FALSE;
               tuner_status[tuner_id].auto_relock = TRUE;
               tuner_status[tuner_id].in_use = FALSE;
               tuner_status[tuner_id].tune_state = TUNER_IDLE;
               tuner_status[tuner_id].islock = FALSE;
               tuner_status[tuner_id].path = tuner_id;
               m_committed_command[tuner_id] = 0;
               m_uncommitted_command[tuner_id] = 0;
               m_22k[tuner_id] = 0;
               m_last_committed_command[tuner_id] = 0;
               m_last_uncommitted_command[tuner_id] = 0;
               m_last_22k[tuner_id] = 0;
			   tuners_num++;
            }

            if (tuners_num > 0 && tuners_num < paths)
            {
               TUN_WARNING("Failed to open %d tuners, only %d tuners openned", paths, tuners_num);
               tuner_status = (S_TUNER*) STB_MEMResizeSysRAM(tuner_status, sizeof(S_TUNER) * tuners_num);
            }
         }

		 combo_tuner_init();
		 if(combo_tuner_counter > 0)
		 	num_tuner_task = combo_tuner_counter;
		 else
		 	num_tuner_task = tuners_num;
		 
		 tuner_task_status = (TUN_TASK*)STB_MEMGetSysRAM((U32BIT)(sizeof(TUN_TASK) * num_tuner_task));
		 if (tuner_task_status != NULL) {
			 for(i=0; i<num_tuner_task; ++i) {
				 tuner_task_status[i].task_status = TUNER_TASK_NOT_RUNNING;
				 tuner_task_status[i].tune_queue = STB_OSCreateQueue(sizeof(U8BIT), 32);
				 tuner_task_status[i].task_id = i;
				 tuner_task_status[i].new_tun_id = INVALID_RES_ID;
				 tuner_task_status[i].cur_tun_id = INVALID_RES_ID;
				 tuner_task_status[i].task_hdl = STB_OSCreateTask(TunerTask,(void*)&tuner_task_status[i],
									 TNR_TASK_STACK_SIZE, TNR_TASK_PRIORITY, (U8BIT*)"TunerTask");
			 }
			 is_tuner_initialized = TRUE;
		 }

         /* Release resources in case of failure */
         if (!is_tuner_initialized)
         {
            TUN_Terminate(paths);
            TUN_ERROR("Failed to initialize tuner");
         }
      }
   }

   FUNCTION_FINISH(STB_TuneInitialise);
}

/**
 * @brief   Set the demodulator's signal type. This function must be called
 *          before each call to STB_TuneStartTuner in a dvb-t2 system and
 *          never in a dvb-t system.
 * @param   U8BIT path - the tuner path to set up
 * @param   E_STB_TUNE_TERR_TYPE type: TUNE_TERR_TYPE_DVBT,
 *          TUNE_TERR_TYPE_DVBT2 or TUNE_TERR_TYPE_UNKNOWN. When the signal
 *          type has been set to TUNE_TERR_TYPE_UNKNOWN, a call to
 *          STB_TuneStartTuner will force the driver to try with DVB-T first,
 *          and if no signal is found, with DVB-T2. When a signal has been
 *          found, STB_TuneGetTerrType will return the actual signal type.
 */
void STB_TuneSetSystemType(U8BIT path, E_STB_TUNE_SYSTEM_TYPE sys_type)
{
   FUNCTION_START(STB_TuneSetSystemType);

   if (!is_tuner_initialized)
   {
      TUN_WARNING("Tuner is not initialized");
   }
   else if (path >= tuners_num)
   {
      TUN_ERROR("Wrong tuner path: %d", path);
   }
   else
   {
      if (sys_type == TUNE_SYSTEM_TYPE_UNKNOWN)
      {
         TUN_ERROR("Wrong system type");
      }
      else
      {
		 path = TuneGetActualPath(path);
		 if (tuner_status != NULL) {
			switch(sys_type) {
			case TUNE_SYSTEM_TYPE_DVBT2:
			case TUNE_SYSTEM_TYPE_DVBS2:
				tuner_status[path].sys_type_2_pending = TRUE;
				break;
			default:
				tuner_status[path].sys_type_2_pending = FALSE;
				break;
			}
		 }
      }
   }
   FUNCTION_FINISH(STB_TuneSetSystemType);
}

/**
 * @brief   Returns the signal type as set by STB_TuneSetTerrType or as
 *          re-written by the driver.
 * @param   path the tuner path to query
 * @return  Signal type.
 */
E_STB_TUNE_SYSTEM_TYPE STB_TuneGetSystemType(U8BIT path)
{
    return TUNE_SYSTEM_TYPE_DVBC;
}

/**
 * @brief   Enables or disabled auto tuner relocking
 * @param   path the tuner path to configure
 * @param   state TRUE enables relocking, FALSE disables it
 */
void STB_TuneAutoRelock(U8BIT path, BOOLEAN state)
{
   FUNCTION_START(STB_TuneAutoRelock);

   if (!is_tuner_initialized)
   {
      TUN_ERROR("Tuner is not initialised");
   }
   else if (path >= tuners_num)
   {
      TUN_ERROR("Wrong tuner path: %d", path);
   }
   else
   {
	  path = TuneGetActualPath(path);
	  if (tuner_status != NULL) {
		 tuner_status[path].auto_relock = state;
	  }
   }

   FUNCTION_FINISH(STB_TuneAutoRelock);
}

/**
 * @brief   Gets the signal types of the given tuner path.
 *          This will be a bitmask of supported types defined by E_STB_TUNE_SIGNAL_TYPE
 * @param   path tuner path
 * @return  the signal types supported by the given tuner
 */
U16BIT STB_TuneGetSignalType(U8BIT path)
{
   E_STB_TUNE_SIGNAL_TYPE retval;

   FUNCTION_START(STB_TuneGetSignalType);

   retval = TUNE_SIGNAL_NONE;
   if (!is_tuner_initialized)
   {
      TUN_ERROR("Tuner is not initialised");
   }
   else if (path >= tuners_num)
   {
      TUN_ERROR("Wrong tuner path: %d", path);
   }
   else
   {
	  if(combo_tuner_status[path].is_combo_tuner)
	  {
		 if (combo_tuner_status[path].tuner_type_used == TUNE_SIGNAL_NONE)
		 {
			retval = (E_STB_TUNE_SIGNAL_TYPE) combo_tuner_status[path].tuner_type_caps;
		 }
		 else
		 {
			retval = combo_tuner_status[path].tuner_type_used;
		 }
	  }
	  else
	  {
		  retval = tuner_status[path].sig_type;
	  }
   }

   FUNCTION_FINISH(STB_TuneGetSignalType);
   
   // FIXME
   retval = TUNE_SIGNAL_QAM;
   return retval;
}

/**
 * @brief   This function is only relevant for tuners that support more than one signal type;
 *          for tuners that don't support more than one signal type it can be a blank function.
 *          It will be called to inform the platform which of the supported signal types is being
 *          used.
 * @param   path tuner path
 * @param   type signal type that is being used for this tuner
 */
void STB_TuneSetSignalType(U8BIT path, E_STB_TUNE_SIGNAL_TYPE type)
{
	FUNCTION_START(STB_TuneSetSignalType);

	ASSERT(path < tuners_num);

	if(combo_tuner_status[path].is_combo_tuner)	/* This function is only for combo tuner. */
	{
      if (type == TUNE_SIGNAL_NONE)
      {
         /* Tuner has been released so set it back to be able to support any signal type */
         combo_tuner_status[path].tuner_type_used = (E_STB_TUNE_SIGNAL_TYPE) combo_tuner_status[path].tuner_type_caps;
      }
      else
      {
         combo_tuner_status[path].tuner_type_used = type;
      }
	}

	FUNCTION_FINISH(STB_TuneSetSignalType);
}

/**
 * @brief   Returns the minimum tuner frequency in KHz
 * @param   path the tuner path to query
 * @return  minimum frequency in Khz
 */
U32BIT STB_TuneGetMinTunerFreqKHz(U8BIT path)
{
    return 48000;
}

/**
 * @brief   Returns the maximum tuner frequency in KHz
 * @param   path the tuner path to query
 * @return  maximum frequency in Khz
 */
U32BIT STB_TuneGetMaxTunerFreqKHz(U8BIT path)
{
    return 999000;
}

/**
 * @brief   Starts the tuner, it will then attempt to lock specified signal.
 *          Unrequired parameters can be passed as 0 (zero)
 * @param   path the tuner path to start
 * @param   freq the frequency to tune to
 * @param   srate the symbol rate to lock
 * @param   fec The forward error correction rate
 * @param   freq_off The frequency offset to use
 * @param   tmode The COFDM mode
 * @param   tbwidth The signal bandwidth
 * @param   cmode The QAM mode
 * @param   anlg_vtype The type of video for analogue tuner
 */
void STB_TuneStartTuner(U8BIT path, U32BIT freq, U32BIT srate, E_STB_TUNE_FEC fec,
   S8BIT freq_off, E_STB_TUNE_TMODE tmode, E_STB_TUNE_TBWIDTH tbwidth,
   E_STB_TUNE_CMODE cmode, E_STB_TUNE_ANALOG_VIDEO_TYPE anlg_vtype)
{
    volatile S_TUNER* p_ts = NULL;
    volatile TUN_TASK* task_p;
    U8BIT cmd;
    U8BIT tuner_task_id = path;
    U8BIT rpath = path;

    FUNCTION_START(STB_TuneStartTuner);
    
    ASSERT(path < tuners_num);

	path = TuneGetActualPath(path);
    USE_UNWANTED_PARAM(anlg_vtype);
    USE_UNWANTED_PARAM(freq_off);
    if ((tuner_status != NULL) && (path < tuners_num)) {
        p_ts = &tuner_status[path];
        /* Is the current demod already connected to this tuner path */
        if (tuner_task_status[tuner_task_id].cur_tun_id == path) {
            if((p_ts->islock && (TUNER_TASK_RUNNING == tuner_task_status[tuner_task_id].task_status))) {
              switch(p_ts->sig_type){
                  case TUNE_SIGNAL_QAM:
                      if((TUNER_IDLE != p_ts->tune_state)&&(p_ts->freq == freq)&&(p_ts->srate == srate)&&(p_ts->cmode == cmode)) {
                          TUN_DEBUG("%s(%u): Same QAM MUX\n", __FUNCTION__, path);
                          STB_OSSendEvent(FALSE, HW_EV_CLASS_TUNER, HW_EV_TYPE_LOCKED, &rpath, sizeof(U8BIT));
                          p_ts->tune_state = TUNER_CONNECTED;
                          return;
                      }
                      break;
                  case TUNE_SIGNAL_COFDM:
                      if (
                           (((p_ts->sys_type==TUNE_SYSTEM_TYPE_DVBT)&&(!p_ts->sys_type_2_pending)) || // dvbt
                           (((p_ts->sys_type==TUNE_SYSTEM_TYPE_DVBT2)&&p_ts->sys_type_2_pending)&&(p_ts->plp == p_ts->plp_pending))  // for dvbt2

                            )
                            && (p_ts->freq == freq)
                            &&(p_ts->tbwidth == tbwidth)) {

                          TUN_DEBUG("%s(%u): Same COFDM MUX\n", __FUNCTION__, path);
                          STB_OSSendEvent(FALSE, HW_EV_CLASS_TUNER, HW_EV_TYPE_LOCKED, &rpath, sizeof(U8BIT));
                          p_ts->tune_state = TUNER_CONNECTED;
                          return;
                      }
                      break;
                  case TUNE_SIGNAL_QPSK:
                      if(
                        (((p_ts->sys_type==TUNE_SYSTEM_TYPE_DVBS)&&(!p_ts->sys_type_2_pending)) || ((p_ts->sys_type==TUNE_SYSTEM_TYPE_DVBS2)&&p_ts->sys_type_2_pending)) &&
                        (p_ts->freq == freq)&&(p_ts->srate == srate)) {
                          TUN_DEBUG("%s(%u): Same QPSK MUX\n", __FUNCTION__, path);
                          STB_OSSendEvent(FALSE, HW_EV_CLASS_TUNER, HW_EV_TYPE_LOCKED, &rpath, sizeof(U8BIT));
                          p_ts->tune_state = TUNER_CONNECTED;
                          return;
                      }
                      break;
                  default:
                      break;
              }
              p_ts->islock = FALSE;
            }
        }

        switch(p_ts->sig_type) {
            case TUNE_SIGNAL_QAM:
                if(p_ts->sys_type == TUNE_SYSTEM_TYPE_DVBC) {
                    /* cable signal */
                    p_ts->freq = freq;
                    p_ts->srate = srate;
                    p_ts->cmode = cmode;
                    p_ts->tbwidth = tbwidth;
                } else {
                    /* Send the not locked event */
                    TUN_DEBUG("%s(%u): NOT_LOCKED on %u Hz\n", __FUNCTION__, path, freq);
                    STB_OSSendEvent(FALSE, HW_EV_CLASS_TUNER, HW_EV_TYPE_NOTLOCKED, &rpath, sizeof(U8BIT));
                    return;
                }
                break;
            case TUNE_SIGNAL_COFDM:
                /* terrestrial signal */
                if (p_ts->sys_type_2_pending) {
                    p_ts->sys_type_2_pending = FALSE;
                    p_ts->sys_type = TUNE_SYSTEM_TYPE_DVBT2;
                    p_ts->plp = p_ts->plp_pending;
                } else {
                    p_ts->sys_type = TUNE_SYSTEM_TYPE_DVBT;
                }
                p_ts->freq = freq;
                p_ts->tbwidth = tbwidth;
                break;
            case TUNE_SIGNAL_QPSK:
                /* satellite signal */
                if (p_ts->sys_type_2_pending) {
                    p_ts->sys_type_2_pending = FALSE;
                    p_ts->sys_type = TUNE_SYSTEM_TYPE_DVBS2;
                } else {
                    p_ts->sys_type = TUNE_SYSTEM_TYPE_DVBS;
                }
                if((p_ts->sys_type == TUNE_SYSTEM_TYPE_DVBS)||(p_ts->sys_type == TUNE_SYSTEM_TYPE_DVBS2)) {
                    p_ts->freq = freq;
                    p_ts->srate = srate;
                    p_ts->fec = fec;
                    m_committed_command[path] = m_last_committed_command[path];
                    m_uncommitted_command[path] = m_last_uncommitted_command[path];
                    m_22k[path] = m_last_22k[path];
                    m_last_committed_command[path] = 0;
                    m_last_uncommitted_command[path] = 0;
                } else {
                    /* Send the not locked event */
                    TUN_DEBUG("%s(%u): NOT_LOCKED on %u Hz\n", __FUNCTION__, path, freq);
                    STB_OSSendEvent(FALSE, HW_EV_CLASS_TUNER, HW_EV_TYPE_NOTLOCKED, &rpath, sizeof(U8BIT));
                    return;
                }
                break;
            default:
                /* Send the not locked event */
                TUN_DEBUG("%s(%u): NOT_LOCKED on %u Hz\n", __FUNCTION__, path, freq);
                STB_OSSendEvent(FALSE, HW_EV_CLASS_TUNER, HW_EV_TYPE_NOTLOCKED, &rpath, sizeof(U8BIT));
                return;
        }
    }
    /* Record the parameters in case we need to re-tune */
    if (p_ts) {
        p_ts->start_tuning = FALSE;
        p_ts->in_use = FALSE;
    }
    /* send message to tune task. */

    task_p = &tuner_task_status[tuner_task_id];  //kim
    task_p->new_tun_id = path;
    task_p->cur_tun_id = path;
    cmd = STBTUNE_COMMAND_CONNECT;
    STB_OSWriteQueue(task_p->tune_queue, (void *)&cmd, sizeof(U8BIT), TIMEOUT_NEVER);

    FUNCTION_FINISH(STB_TuneStartTuner);
}

/**
 * @brief   Restarts tuner and attempts to lock to signal in StartTuner call
 * @param   path the tuner path to restart
 */
void STB_TuneRestartTuner(U8BIT path)
{
    return;
}

/**
 * @brief   Stops any locking attempt, or unlocks if locked
 * @param   path the tuner path to stop
 */
void STB_TuneStopTuner(U8BIT path)
{
   volatile S_TUNER* p_ts = NULL;
   volatile TUN_TASK* task_p = NULL;
   U8BIT tuner_task_id;
   U8BIT cmd;

   FUNCTION_START(STB_TuneStopTuner);

   if (!is_tuner_initialized)
   {
      TUN_ERROR("Tuner is not initialised");
   }
   else if ((path & 0x7F) >= tuners_num)
   {
      TUN_ERROR("Wrong tuner path: %d", path);
   }
   else
   {
      untune(path);

      path &= 0x7F;
      tuner_task_id = path;
      path = TuneGetActualPath(path);
      task_p = &tuner_task_status[tuner_task_id];
      task_p->new_tun_id = path;
      p_ts = &tuner_status[path];
      p_ts->islock = FALSE;
      p_ts->tune_state = TUNER_IDLE;
      cmd = STBTUNE_COMMAND_DISCONNECT;
      STB_OSWriteQueue(task_p->tune_queue, (void *)&cmd, sizeof(U8BIT), TIMEOUT_NEVER);
   }

   FUNCTION_FINISH(STB_TuneStopTuner);
}

/**
 * @brief   Returns the current signal strength
 * @param   path the tuner path to query
 * @return  the signal strength as percentage of maximum (0-100)
 */
U8BIT STB_TuneGetSignalStrength(U8BIT path)
{
   U8BIT retval;

   FUNCTION_START(STB_TuneGetSignalStrength);

   retval = 0;
   if (!is_tuner_initialized)
   {
      TUN_ERROR("Tuner is not initialised");
   }
   else if (path >= tuners_num)
   {
      TUN_ERROR("Wrong tuner path: %d", path);
   }
   else
   {
	  path = TuneGetActualPath(path);
      if (tuner_status[path].islock)
      {
         retval = (U8BIT)tuner_status[path].strength;
      }
	  else
	  	 retval = 0;
   }


   FUNCTION_FINISH(STB_TuneGetSignalStrength);

   return retval;
}

/**
 * @brief   Returns the Signal Noise Ratio
 * @param   path - the tuner path to query
 * @return  the unit of snr is 0.1dB
 */
U32BIT STB_TuneGetSnr(U8BIT path)
{
   U32BIT retval;

   FUNCTION_START(STB_TuneGetSnr);

   retval = 0;
   if (!is_tuner_initialized)
   {
      TUN_ERROR("Tuner is not initialised");
   }
   else if (path >= tuners_num)
   {
      TUN_ERROR("Wrong tuner path: %d", path);
   }
   else
   {
	  path = TuneGetActualPath(path);
      if (tuner_status[path].islock)
      {
         retval = tuner_status[path].snr;
      }
	  else
	  	 retval = 0;
   }

   FUNCTION_FINISH(STB_TuneGetSnr);
   
   return retval;
}

/**
 * @brief   Returns the Bit Error Rate
 * @param   path - the tuner path to query
 * @return  bit_error_ratio (BER) in 1.0E-7(i.e. 10^-7) unit
 */
U32BIT STB_TuneGetBer(U8BIT path)
{
   U32BIT bit_error_ratio;
   U32BIT exp_diff;
   
   FUNCTION_START(STB_TuneGetBer);

   bit_error_ratio = 1;

   if (!is_tuner_initialized)
   {
      TUN_ERROR("Tuner is not initialised");
   }
   else if (path >= tuners_num)
   {
      TUN_ERROR("Wrong tuner path: %d", path);
   }
   else
   {
	  path = TuneGetActualPath(path);
      if (tuner_status[path].islock)
      {
#if 1
         bit_error_ratio = tuner_status[path].ber[0];
#else
         bit_error_ratio = tuner_status[path].ber[0] * 1000 + tuner_status[path].ber[1];
         exp_diff = 7 - tuner_status[path].ber[2];

         if(exp_diff > 3)
         {
            while(exp_diff > 3)
            {
               bit_error_ratio *= 10;
               exp_diff--;
            }
         }
         else if (exp_diff < 3)
         {
            while(exp_diff < 3)
            {
               bit_error_ratio /= 10;
               exp_diff++;
            }
         }
#endif
      }
   }

   FUNCTION_FINISH(STB_TuneGetBer);

   return bit_error_ratio;
}

/**
 * @brief   Returns the current data integrity
 * @param   path the tuner path to query
 * @return  the data integrity as percentage of maximum possible (0-100)
 */
U8BIT STB_TuneGetDataIntegrity(U8BIT path)
{
   U8BIT retval;

   FUNCTION_START(STB_TuneGetDataIntegrity);

   retval = 0;
   if (!is_tuner_initialized)
   {
      TUN_ERROR("Tuner is not initialised");
   }
   else if (path >= tuners_num)
   {
      TUN_ERROR("Wrong tuner path: %d", path);
   }
   else
   {
	  path = TuneGetActualPath(path);
      if (tuner_status[path].islock)
      {
         retval = (U8BIT)tuner_status[path].quality;
      }
	  else
	  	 retval = 0;
   }

   FUNCTION_FINISH(STB_TuneGetDataIntegrity);
   retval =100;  //FIXME
   return retval;
}

/**
 * @brief   Returns the actual frequency of the current terrestrial signal
 * @param   path the tuner path to query
 * @return  the frequency in Hz
 */
U32BIT STB_TuneGetActualTerrFrequency(U8BIT path)
{
    return 0;
}

/**
 * @brief   Returns the actual freq offset of the current terrestrial signal
 * @param   path the tuner path to query
 * @return  the frequency offset in Hz
 */
S8BIT STB_TuneGetActualTerrFreqOffset(U8BIT path)
{
    return 0;
}

/**
 * @brief   Returns the actual mode of the current terrestrial signal
 * @param   path the tuner path to query
 * @return  the tuning mode
 */
E_STB_TUNE_TMODE STB_TuneGetActualTerrMode(U8BIT path)
{
    return TUNE_MODE_COFDM_UNDEFINED;
}

/**
 * @brief   Returns the actual bandwidth of the current terrestrial signal
 * @param   path the tuner path to query
 * @return  the signal bandwidth
 */
E_STB_TUNE_TBWIDTH STB_TuneGetActualTerrBwidth(U8BIT path)
{
    return TUNE_TBWIDTH_6MHZ;
}

/**
 * @brief   Returns the constellation of the current terrestrial signal
 * @param   path the tuner path to query
 * @return  the constellation
 */
E_STB_TUNE_TCONST STB_TuneGetActualTerrConstellation(U8BIT path)
{
    return TUNE_TCONST_UNDEFINED;
}

/**
 * @brief   Returns the heirarchy of the current terrestrial signal.
 * @param   path the tuner path to query
 * @return  the heirarchy, i.e. the muximum PLP id possibly present at the current frequency.
 */
E_STB_TUNE_THIERARCHY STB_TuneGetActualTerrHierarchy(U8BIT path)
{
    return TUNE_THIERARCHY_UNDEFINED;
}

/**
 * @brief   Returns the LP code rate of the current terrestrial signal
 * @param   path the tuner path to query
 * @return  The LP code rate
 */
E_STB_TUNE_TCODERATE STB_TuneGetActualTerrLpCodeRate(U8BIT path)
{
    return TUNE_TCODERATE_UNDEFINED;
}

/**
 * @brief   Returns the HP code rate of the current terrestrial signal
 * @param   path the tuner path to query
 * @return  The HP code rate
 */
E_STB_TUNE_TCODERATE STB_TuneGetActualTerrHpCodeRate(U8BIT path)
{
    return TUNE_TCODERATE_UNDEFINED;
}

/**
 * @brief   Returns the guard interval of the current terrestrial signal
 * @param   path the tuner path to query
 * @return  the guard interval
 */
E_STB_TUNE_TGUARDINT STB_TuneGetActualTerrGuardInt(U8BIT path)
{
    return TUNE_TGUARDINT_UNDEFINED;
}

/**
 * @brief   Returns the cell id the current terrestrial signal
 * @param   path the tuner path to query
 * @return  the cell id
 */
U16BIT STB_TuneGetActualTerrCellId(U8BIT path)
{
    return 0;
}

/**
 * @brief   Sets the Physical Layer Pipe to be acquired
 * @param   path the tuner path to set up
 * @param   plp Physical Layer Pipe to be acquired
 */
void STB_TuneSetPLP(U8BIT path, U8BIT plp)
{
    return;
}

/**
 * @brief   Gets the Physical Layer Pipe to be acquired
 * @param   path  the tuner path to query
 * @return  Physical Layer Pipe to be acquired
 */
U8BIT STB_TuneGetPLP(U8BIT path)
{
    return 0;
}

/**
 * @brief   Enables/disables aerial power for DVB-T
 * @param   path tuner path
 * @param   enabled TRUE to enable
 */
void STB_TuneActiveAerialPower(U8BIT path, BOOLEAN enabled)
{
    return;
}

/**
 * @brief   Sets the local oscillator frequency used by the LNB
 * @param   path the tuner path to query
 */
void STB_TuneSetLOFrequency(U8BIT tuner, U16BIT lo_freq)
{
    return;
}

/**
 * @brief   Sets the type of modulation for the specified tuner
 * @param   path tuner path
 * @param   modulation type of modulation
 */
void STB_TuneSetModulation(U8BIT path, E_STB_TUNE_MODULATION modulation)
{
    return;
}

/**
 * @brief   Sets the LNB voltage for the given tuner
 * @param   path tuner path
 * @param   voltage voltage setting
 */
void STB_TuneSetLNBVoltage(U8BIT path, E_STB_TUNE_LNB_VOLTAGE voltage)
{
    return;
}

/**
 * @brief   Turns the 22 kHz tone on or off
 * @param   path tuner path
 * @param   state TRUE to turn the tone on, FALSE to turn it off
 */
void STB_TuneSet22kState(U8BIT path, BOOLEAN state)
{
    return;
}

/**
 * @brief   Sets the 12V switch for the given tuner
 * @param   path tuner path
 * @param   state TRUE for on
 */
void STB_TuneSet12VSwitch(U8BIT path, BOOLEAN state)
{
    return;
}

/**
 * @brief   Receives a DisEqc reply
 * @param   path tuner path
 * @param   data pointer to the received data
 * @param   timeout maximum number of milliseconds to wait for a reply
 * @return  The number of bytes received
 */
U8BIT STB_TuneGetDISEQCReply(U8BIT path, U8BIT *data, U32BIT timeout)
{
    return 0;
}

/**
 * @brief   Sends the DisEqc message
 * @param   path - tuner path
 * @param   data - message data
 * @param   size - number of bytes in message data
 */
void STB_TuneSendDISEQCMessage(U8BIT path, U8BIT *data, U8BIT size)
{
    return;
}

/**
 * @brief   Sets the pulse limit for the east
 * @param   path tuner path
 * @param   count east limit count
 */
void STB_TuneSetPulseLimitEast(U8BIT path, U16BIT count)
{
    return;
}

/**
 * @brief   Sets the pulse limit for the west
 * @param   path tuner path
 * @param   count west limit count
 */
void STB_TuneSetPulseLimitWest(U8BIT path, U16BIT count)
{
    return;
}

void STB_TuneChangePulsePosition(U8BIT path, U16BIT count)
{
    return;
}

/**
 * @brief   Returns the current pulse position
 * @param   path tuner path
 * @return  Current puls position
 */
U16BIT STB_TuneGetPulsePosition(U8BIT path)
{
    return 0;
}

void STB_TuneAtPulsePosition(U8BIT path, U16BIT position)
{
    return;
}

/**
 * @brief Changes the value of skew position count
 * @param path tuner path
 * @param count skew position count
 */
void STB_TuneChangeSkewPosition(U8BIT path, U16BIT count)
{
    return;
}

/**
 * @brief   Returns the carrier signal strength as a percentage
 * @param   path tuner path
 * @param   freq carrier frequency
 * @return  Strength as a percentage
 */
U8BIT STB_TuneSatGetCarrierStrength(U8BIT path, U32BIT freq)
{
    return 0;
}

/**
 * @brief   Returns the actual symbol rate when a tuner has locked
 * @param   path tuner path
 * @return  Symbol rate in symbols per second
 */
U32BIT STB_TuneGetActualSymbolRate(U8BIT path)
{
    U32BIT ret = 0;
 
    if (NULL != tuner_status)
        ret = tuner_status[path].srate;
        
    return ret;
}

/**
 * @brief   Returns the cable mode when the tuner has locked
 * @param   path tuner path
 * @return  QAM mode
 */
E_STB_TUNE_CMODE STB_TuneGetActualCableMode(U8BIT path)
{
    E_STB_TUNE_CMODE ret = TUNE_MODE_QAM_UNDEFINED;
 
    if (NULL != tuner_status) {
        ret = tuner_status[path].cmode;
    }
        
    return ret;
}

/**
 * @brief   Returns the system type supported by the path. This function
 *          differs from STB_TuneGetSystemType which only returns T2 or S2 if
 *          the tuner is currently performing T2 or S2 operations.
 * @param   path  the tuner path to query
 * @return  (E_STB_TUNE_SYSTEM_TYPE) the system type supported by this path.
 *          TUNE_SYSTEM_TYPE_DVBT2 means both DVBT and DVBT2 are supported,
 *          TUNE_SYSTEM_TYPE_DVBS2 means both DVBS and DVBS2 are supported
 */
E_STB_TUNE_SYSTEM_TYPE STB_TuneGetSupportedSystemType(U8BIT path)
{
    return TUNE_SYSTEM_TYPE_DVBC;
}

/*!**************************************************************************
 * @fn      TUN_Terminate
 * @param   paths
 * @note    Declared in internal.h, Used in stbhwini.c
 ****************************************************************************/
void TUN_Terminate(int paths)
{
   U32BIT tuner_id;
   int i;

   FUNCTION_START(TUN_Terminate);

   USE_UNWANTED_PARAM(paths);

   thread_running_flag=0;
   for (i = 0; i < num_tuner_task; i++){
	   STB_OSDestroyTask(tuner_task_status[i].task_hdl);
   }

   if (tuner_status != NULL)
   {
      for (tuner_id = 0; tuner_id < tuners_num; tuner_id++)
      {

      }
      STB_MEMFreeSysRAM(tuner_status);
      tuner_status = NULL;
      tuners_num = 0;
   }

   is_tuner_initialized = FALSE;

   FUNCTION_FINISH(TUN_Terminate);
}


/*!**************************************************************************
 * @fn      SetupTuner
 * @brief   Temporary function to set hardcoded tuner parameters
 * @param
 * @return
 ****************************************************************************/
static BOOLEAN SetupTuner(U8BIT tuner_id, S_TUNER *tuner)
{
    FUNCTION_START(SetupTuner);

    USE_UNWANTED_PARAM(tuner_id);

    tuner->auto_relock = FALSE;
    tuner->sys_type = TUNE_SYSTEM_TYPE_DVBC;
    tuner->sig_type = TUNE_SIGNAL_QAM;

   FUNCTION_FINISH(SetupTuner);

   return TRUE;
}

int check_for_lock(U8BIT cmd,int tun_id,int timeout,int send_unlock_event)
{
    //int loop_times = CONNECT_TIMEOUT_MS / 100;
    U16BIT event_type = HW_EV_TYPE_NOTLOCKED;
    U8BIT path = (U8BIT) tun_id;
	int i;

    printf("DTVKit Tuner %u is tuning.....\n", tun_id);
	for(i=6;  i> 0;	i--)
	{
         usleep(10000);
		 if (RTK_OK == RTK_Tuner_IsLocked(tun_id)) {
             tuner_status[tun_id].tune_state = TUNER_CONNECTED;
             tuner_status[tun_id].islock = TRUE;
             event_type=HW_EV_TYPE_LOCKED;
			 printf("RTK tuner %d is locked %u (Hz) !!\n", tun_id, tuner_status[tun_id].freq);
             break;
         }
		printf("Tuner is tuning..... %d ms  tuner_id %d\n", i*100, tun_id);
  		usleep(40000);
	}

    TUN_DEBUG("%s: %d) Channel is %s\n",
        (cmd==STBTUNE_COMMAND_CONNECT)?"CONNECT":((cmd==STBTUNE_COMMAND_TIMEOUT)?"TIMEOUT":"RECONNECT"), tun_id,
        (event_type == HW_EV_TYPE_NOTLOCKED) ? "unlocked" : "locked"
    );

    if (event_type != HW_EV_TYPE_NOTLOCKED || send_unlock_event) {
		U8BIT rpath = TuneGetRealPath(tuner_status[tun_id].path);
        STB_OSSendEvent(FALSE, HW_EV_CLASS_TUNER,event_type,&rpath, sizeof(U8BIT));
    }

    return timeout;
}

static void TunerTask(void* param)
{
    TUN_TASK* ts;
    U8BIT cmd;
    BOOLEAN read_cmd_from_queue;
    int timeout = STB_RETUNE_PERIOD;
    int retry_times = 0;
    RTK_Error ret;
    U8BIT path;

    FUNCTION_START(TunerTask);

    ts = (TUN_TASK*)param;

    int task_id = ts->task_id;
    int tun_id = 0;

    tuner_task_status[task_id].task_status = TUNER_TASK_RUNNING;

    TUN_DEBUG("***TunerTask  TUNER_TASK_RUNNING***, tuner_id=%d,task_id=%d\n", tun_id, task_id);

    while((tuner_task_status[task_id].task_status==TUNER_TASK_RUNNING) && (0 != thread_running_flag)) {
       if(tuner_task_status[task_id].tune_queue == 0) {
            tuner_task_status[task_id].tune_queue = STB_OSCreateQueue(sizeof(U8BIT), 32);
        }
        read_cmd_from_queue = STB_OSReadQueue(tuner_task_status[task_id].tune_queue, (void *)&cmd, sizeof(U8BIT), timeout);
        if (read_cmd_from_queue != TRUE) {
            cmd = STBTUNE_COMMAND_TIMEOUT;
        }
        tun_id = tuner_task_status[task_id].new_tun_id;
        if ((tun_id < 0) || (INVALID_RES_ID == tun_id)) {
           usleep(100000);
           continue;
        }

        switch(cmd) {
            case STBTUNE_COMMAND_CONNECT:
            [[fallthrough]];
            case STBTUNE_COMMAND_RECONNECT:
                tune(cmd, tun_id);
                timeout = check_for_lock(cmd,tun_id,timeout,TRUE);
                break;
            case STBTUNE_COMMAND_DISCONNECT:
                tuner_status[tun_id].start_tuning = FALSE;
                tuner_status[tun_id].in_use = FALSE;
                tuner_status[tun_id].tune_state = TUNER_IDLE;
                timeout = 300;
                break;
            case STBTUNE_COMMAND_TIMEOUT:
                switch(tuner_status[tun_id].tune_state) {
                    case TUNER_IDLE:
                        timeout = 1000;
                        break;
                    case TUNER_TUNNING:
                        if(tuner_status[tun_id].auto_relock) {
							tune(cmd, tun_id);

//                            STB_OSTaskDelay(100);

                            /* update timeout based on having a lock or not
                               send a LOCK event if we get one.
                            */
                            timeout = check_for_lock(cmd,tun_id,timeout,FALSE);
                        } else {
                            timeout = 1000;
                        }
                        break;
                    case TUNER_CONNECTED:
                        RTK_TunerInfo info;
                        for(retry_times=0; retry_times < STB_MONITOR_RETRY_TIMES; retry_times++) {
                           /* We check 5 times, to reduce black screen when signal is not good enough. */
                           if (RTK_OK == RTK_Tuner_IsLocked(tun_id))
                                   break;

                           STB_OSTaskDelay(STB_MONITOR_RETRY_PERIOD);
                        }
                        if (RTK_OK == RTK_Tuner_IsLocked(tun_id)) {
                           RTK_Tuner_GetInfo(tun_id, &info);
                           tuner_status[tun_id].quality = info.quality;
                           tuner_status[tun_id].strength = info.powerLevel / 10;  // dBuV

                           tuner_status[tun_id].ber[0] = (U32BIT)(info.ber * 10000000);  // E-7
                           tuner_status[tun_id].ber[1] = 0;
                           tuner_status[tun_id].ber[2] = 0;

                           tuner_status[tun_id].snr = info.snr;
                           timeout = STB_MONITOR_PERIOD;
                        } else {
                           U8BIT rpath = TuneGetRealPath(tuner_status[tun_id].path);
                            //~ STB_TUN_PRINT(("Lock lost"));
                            if(tuner_status[tun_id].auto_relock) {
                                tuner_status[tun_id].tune_state = TUNER_TUNNING;
                                timeout = STB_RETUNE_PERIOD;
                            } else {
                                tuner_status[tun_id].tune_state = TUNER_IDLE;
                                timeout = TIMEOUT_NEVER;
                            }
                            tuner_status[tun_id].islock = FALSE;
                            STB_OSSendEvent(FALSE, HW_EV_CLASS_TUNER, HW_EV_TYPE_NOTLOCKED,&rpath, sizeof(U8BIT));
                        }
                        break;
                }  // switch(tuner_status[tun_id].tune_state)
                break;
        }  // switch(cmd)
    }
    tuner_task_status[task_id].task_status = TUNER_TASK_NOT_RUNNING;
//    pthread_exit(0);
    FUNCTION_FINISH(TunerTask);
}

U8BIT TuneGetActualPath(U8BIT rpath)
{
	U8BIT path = 0;
	int i = 0;

	if(combo_tuner_counter > 0)
	{
		for(i=0; i<combo_tuner_status[rpath].tuner_types; i++)
		{
			path = combo_tuner_status[rpath].tuner_id[i];
			if (tuner_status[path].sig_type == combo_tuner_status[rpath].tuner_type_used)
			{
				return path;
			}
		}
	}
	else
		path = rpath;

	return path;
}

static U8BIT TuneGetRealPath(U8BIT path)
{
	U8BIT rpath = 0;
	int i = 0;
	int j = 0;

	ASSERT(path < tuners_num);

	if(combo_tuner_counter > 0)
	{
		for(i=0; i<combo_tuner_counter; i++)
		{
			for(j=0; j<combo_tuner_status[i].tuner_types; j++)
			{
				if(combo_tuner_status[i].tuner_id[j] == path)
				{
					return i;
				}
			}
		}
	}
	else
		rpath = path;

	return rpath;
}

static void combo_tuner_init(void)
{
	int i = 0; 
	int j = 0;
	int counter = 0;
	BOOLEAN bFound = FALSE;
	
	memset(combo_tuner_status, 0, sizeof(COMBO_TUN_STATUS) * MAX_COMBO_TUNER_NUM);
	combo_tuner_counter = 0;
	for(i=0; i<tuners_num; i++)
	{
		counter = 0;
		bFound = FALSE;
		
		if(bFound)
			combo_tuner_counter++;
	}
}

U8BIT TuneGetNumTuners(void)
{
	FUNCTION_START(TuneGetNumTuners);

	FUNCTION_FINISH(TuneGetNumTuners);
	return 4;
}

static void tune(U8BIT cmd, U8BIT path /*tuner ID*/)
{
   RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(path);
   if (NULL != pRealtekPlayer) {
      /*
      TUNE_MODE_QAM_4:
      TUNE_MODE_QAM_8 = 1,
      TUNE_MODE_QAM_16 = 2,
      TUNE_MODE_QAM_32 = 3,
      TUNE_MODE_QAM_64 = 4,
      TUNE_MODE_QAM_128 = 5,
      TUNE_MODE_QAM_256 = 6,
      TUNE_MODE_QAM_UNDEFINED = 255
      */
      uint32_t qam = pow(2,(tuner_status[path].cmode + 2));
      if (qam > 256)
         qam = 256;
     
      std::stringstream ss;
      ss << "tuner://DVBC:" << (tuner_status[path].freq / 1000000) << ":" << qam << ":" << tuner_status[path].srate;
      if(pRealtekPlayer->RT_PlayerCreate(ss.str().c_str(), 0) != RTK_OK) {
         // RSTB_PLAYER_PARAM_ERR;
      }
   }
}

static void untune(U8BIT path /*tuner ID*/)
{
   bool preload = (path & (1 << 7)) ? true : false;
   path &= 0x7F;
   RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(path);
   if (NULL != pRealtekPlayer) {
      if (true == preload) {
         pRealtekPlayer->RT_PlayerStop(RTK_FALSE);
         pRealtekPlayer->removeAllSectionFilters();
      }
      else {
         pRealtekPlayer->removeAllSectionFilters();
         pRealtekPlayer->RT_PlayerStop(RTK_TRUE);
      }
      pRealtekPlayer->RT_PlayerDestory();
   }
}

/****************************************************************************
** End of file
*****************************************************************************/
