/*--- Includes ----------------------------------------------------------------*/

/* System header files */
#include <stdio.h>
#include <string.h>

#include <sstream>
#include <iostream>

/* Third party header files */


/* STB header files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbhwmem.h"
#include "stbhwdsk.h"
#include "stbpvrpr.h"
#include "stbhwav.h"
#include "stbhwdmx.h"
#include "stbhwsdk.h"

#include "internal_generic.h"
#include "internal.h"
// #include "benchfuncs.h"


extern "C" {
#include "stbsitab.h"
#include "ap_dbdef.h"
}

#include "rtk_psi.h"

/*--- Preprocessor definitions ------------------------------------------------*/

#define INVALID_RECORDER            0xFF
#define INVALID_PLAYER              0xFF
#define FILENAME_EXT_TS             "ts"
#define MIN_SPACE_REQUIRED_KB       (80 * 1024)                // 80 MiB
#define MAX_PID_NUM                 32
#define MAX_FILENAME_LEN            128
#define MIN_REC_DURATION_TIMESHIFT  1500
#define TS_BUFFER_SIZE              (4 * 1024 * 1024)          // 4 MiB
#define RECORDING_TASK_STACK_SIZE   (32 * 1024)
#define RECORDING_TASK_PRIORITY     11
#define CRYPTION_KEY_MAX_LEN        16
#define PVR_MAX_FILENAME_LEN        256

#define PVR_DEBUGS
#define PVR_WARNINGS
#define PVR_ERRORS

#ifdef PVR_DEBUGS
#define PVR_DEBUG(format, args...) STB_DEBUG_LOG("PVR", format, ##args)
#else
#define PVR_DEBUG(format, args...)
#endif

#ifdef PVR_WARNINGS
#define PVR_WARNING(format, args...) STB_WARNING_LOG("PVR", format, ##args)
#else
#define PVR_WARNING(format, args...)
#endif

#ifdef PVR_ERRORS
#define PVR_ERROR(format, args...) STB_ERROR_LOG("PVR", format, ##args)
#else
#define PVR_ERROR(format, args...)
#endif




/*--- Local types definitions -------------------------------------------------*/

typedef enum recording_cmd
{
   RECORDING_CMD_START_RECORDING,
   RECORDING_CMD_STOP_RECORDING,
   RECORDING_CMD_RELEASE_RECORDER,
   RECORDING_CMD_TASK_TERMINATE
} E_RECORDING_CMD;

typedef enum recorder_status
{
   RECORDER_STATUS_INVALID,
   RECORDER_STATUS_IDLE,
   RECORDER_STATUS_STARTING,
   RECORDER_STATUS_RUNNING,
   RECORDER_STATUS_PAUSED,
   RECORDER_STATUS_STOPPING
} E_RECORDER_STATUS;

typedef enum player_status
{
   PLAYER_STATUS_IDLE,
   PLAYER_STATUS_RUNNING,
   PLAYER_STATUS_PAUSED,
   PLAYER_STATUS_TRICKMODE
} E_PLAYER_STATUS;

typedef struct recording_msg
{
   U8BIT recorder_id;
   E_RECORDING_CMD cmd;
} S_RECORDING_MSG;

typedef struct
{
    U8BIT                          key[CRYPTION_KEY_MAX_LEN];
    U8BIT                          iv[CRYPTION_KEY_MAX_LEN];
    U32BIT                         key_len;
} S_CRYPTION_KEY;

typedef struct recorder
{
   /* Resources */
   U8BIT demux_id;
   U8BIT tuner_id;

   /* Status */
   E_RECORDER_STATUS status;
   BOOLEAN is_video_recording;
   BOOLEAN is_audio_recording;
   //HI_U32 channel_id;

   /* Start recording attributes*/
   E_STB_PVR_START_MODE start_mode;
   U32BIT start_mode_param;
   U16BIT rec_disk_id;
   U8BIT rec_basename[MAX_FILENAME_LEN];
   U16BIT rec_pids_num;
   S_PVR_PID_INFO rec_pids_array[MAX_PID_NUM];
   BOOLEAN						  encrypt;
   S_CRYPTION_KEY				  cryption_key;
} S_RECORDER;

typedef struct player
{
   E_PLAYER_STATUS status;
   E_STB_PVR_START_MODE start_mode;
   BOOLEAN should_have_video;
   BOOLEAN						  encrypt;
   S_CRYPTION_KEY				  cryption_key;
} S_PLAYER;

/*--- Local (static) variable declarations ------------------------------------*/

static BOOLEAN is_pvr_recording_initialised = FALSE;
static BOOLEAN is_pvr_playback_initialised = FALSE;
static U32BIT recorders_num = 0;
static S_RECORDER *recorders = NULL;
static U32BIT players_num = 0;
static S_PLAYER *players = NULL;
static void *recording_task = NULL;
static void *recording_mutex = NULL;
static void *recording_task_termination_mutex = NULL;
static void *recording_queue = NULL;
static void *playback_mutex = NULL;
void* pSemaphore_stopRecord = NULL;

/*--- Local function prototypes -----------------------------------------------*/

static BOOLEAN GetFilenameFromBasename(U16BIT disk_id, U8BIT *basename,
                                       U8BIT* filename, U32BIT max_filename_len);
static uint16_t VideoCodecToDvbStreamType(E_STB_AV_VIDEO_CODEC codec);
static uint16_t AudioCodecToDvbStreamType(E_STB_AV_AUDIO_CODEC codec);
static void RecordingTask(void *param);
static void PVR_DeinitPlayback(void);

/*--- Global function definitions ---------------------------------------------*/
/**
 * @brief   Initialisation for playback
 * @param   num_audio_decoders number of audio decoders available
 * @param   num_video_decoders number of video decoders available
 * @return  Number of players, 0 if unsuccessful or unsupported
 */
U8BIT STB_PVRInitPlayback(U8BIT num_audio_decoders, U8BIT num_video_decoders)
{
   U32BIT i;

   FUNCTION_START(STB_PVRInitPlayback);

   ASSERT(num_audio_decoders <= STB_HWGetAudioDecodePaths());
   ASSERT(num_video_decoders <= STB_HWGetVideoDecodePaths());

   if (is_pvr_playback_initialised)
   {
      PVR_DEBUG("PVR playback is already initialised");
   }
   else
   {
      {
         players = (S_PLAYER*) STB_MEMGetSysRAM(sizeof(S_PLAYER) * num_video_decoders);
         if (players == NULL)
         {
            PVR_ERROR("Failed to allocate memory for players");
         }
         else
         {
            players_num = num_video_decoders;
            for (i = 0; i < players_num; i++)
            {
               players[i].status = PLAYER_STATUS_IDLE;
               players[i].start_mode = START_RUNNING;
               players[i].should_have_video = FALSE;
            }

            playback_mutex = STB_OSCreateMutex();
            if (playback_mutex == NULL)
            {
               PVR_ERROR("Failed to create Playback Mutex");
            }
            else
            {
               is_pvr_playback_initialised = TRUE;
            }
         }
      }
   }

   if (!is_pvr_playback_initialised)
   {
      PVR_DeinitPlayback();
      PVR_ERROR("Failed to initialise PVR playback");
   }

   FUNCTION_FINISH(STB_PVRInitPlayback);
   return 1;
}

/**
 * @brief   Initialisation for recording
 * @param   num_tuners number of tuners available for recording
 * @return  Number of recorders, 0 if unsuccessful or unsupported
 */
U8BIT STB_PVRInitRecording(U8BIT num_tuners)
{
   U32BIT i;

   FUNCTION_START(STB_PVRInitRecording);

   ASSERT(num_tuners <= STB_HWGetNumRecorders());

   if (is_pvr_recording_initialised)
   {
      PVR_DEBUG("PVR recording is already initialised");
   }
   else
   {
      recorders_num = 0;
      {
         recorders = (S_RECORDER*) STB_MEMGetSysRAM(sizeof(S_RECORDER) * num_tuners);
         if (recorders == NULL)
         {
            PVR_ERROR("Failed to allocate memory for recorders");
         }
         else
         {
            recorders_num = num_tuners;
            for (i = 0; i < recorders_num; i++)
            {
               recorders[i].status = RECORDER_STATUS_INVALID;
               recorders[i].is_video_recording = FALSE;
               recorders[i].is_audio_recording = FALSE;
            }

            recording_mutex = STB_OSCreateMutex();
            recording_queue = STB_OSCreateQueue(sizeof(S_RECORDING_MSG), 32);
            recording_task_termination_mutex = STB_OSCreateMutex();
            if (recording_mutex == NULL || recording_queue == NULL ||
                recording_task_termination_mutex == NULL)
            {
               PVR_ERROR("Failed to prepare resources for RecordingTask");
            }
            else
            {
               recording_task = STB_OSCreateTask(RecordingTask, NULL, RECORDING_TASK_STACK_SIZE,
                                                 RECORDING_TASK_PRIORITY, (U8BIT*)"RecordingTask");
               if (recording_task == NULL)
               {
                  PVR_ERROR("Failed to create RecordingTask");
               }
               else
               {
                  pSemaphore_stopRecord = STB_OSCreateCountSemaphore(0);
                  if(pSemaphore_stopRecord == NULL)
                  {
                  	PVR_ERROR("Failed to create pSemaphore_stopRecord");
                  }
                  else
                  {
                  	is_pvr_recording_initialised = TRUE;
                  }
               }
            }
         }
      }
   }

   if (!is_pvr_recording_initialised)
   {
      PVR_DeinitRecording();
      PVR_ERROR("Failed to initialise PVR recording");
   }

   FUNCTION_FINISH(STB_PVRInitRecording);

   return recorders_num;
}

/**
 * @brief   Set startup mode for playback
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 * @param   mode playback startup mode
 */
void STB_PVRSetPlayStartMode(U8BIT audio_decoder, U8BIT video_decoder, E_STB_PVR_START_MODE mode)
{
   U8BIT id;

   FUNCTION_START(STB_PVRSetPlayStartMode);

   id = audio_decoder;
   if (!is_pvr_playback_initialised)
   {
      PVR_ERROR("PVR playback is not initialised");
   }
   else if (id >= players_num)
   {
      PVR_ERROR("Wrong decoder");
   }
   else 
   {
      STB_OSMutexLock(playback_mutex);
      if (players[id].status != PLAYER_STATUS_IDLE)
      {
         PVR_ERROR("Cannot change start mode during active playback");
      }
      else
      {
         players[id].start_mode = mode;
         if(START_PAUSED == mode)
		 {
            // An work around for DVBCore, unblank video if it is a timeshift.
            STB_AVBlankVideo(0, FALSE);  
		 }
      }  
      STB_OSMutexUnlock(playback_mutex);
   }

   FUNCTION_FINISH(STB_PVRSetPlayStartMode);
}

/**
 * @brief   Informs the platform whether there's video in the file to be played.
 *          Should be called before playback is started.
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 * @param   has_video TRUE if the recording contains video, FALSE otherwise
 */
void STB_PVRPlayHasVideo(U8BIT audio_decoder, U8BIT video_decoder, BOOLEAN has_video)
{
   U8BIT id;

   FUNCTION_START(STB_PVRPlayHasVideo);

   id = audio_decoder;
   if (!is_pvr_playback_initialised)
   {
      PVR_ERROR("PVR playback is not initialised");
   }
   else if (id >= players_num)
   {
      PVR_ERROR("Wrong decoder");
   }
   else
   {
      STB_OSMutexLock(playback_mutex);
      if (players[id].status != PLAYER_STATUS_IDLE)
      {
         PVR_ERROR("Cannot toggle video decoding during active playback");
      }
      else
      {
         players[id].should_have_video = has_video;
      }
      STB_OSMutexUnlock(playback_mutex);
   } 

   FUNCTION_FINISH(STB_PVRPlayHasVideo);
}

/**
 * @brief   Sets the time the next notification event should be sent during playback.
 *          This is required for CI+, but may also be used for other purposes.
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 * @param   notify_time time in seconds the next notification event is to be sent
 */
void STB_PVRSetPlaybackNotifyTime(U8BIT audio_decoder, U8BIT video_decoder, U32BIT notify_time)
{
   FUNCTION_START(STB_PVRSetPlaybackNotifyTime);

   FUNCTION_FINISH(STB_PVRSetPlaybackNotifyTime);
}

/**
 * @brief   Starts playback
 * @param   disk_id disk containing the recording to be played
 * @param   audio_decoder audio decoder to be used for playback
 * @param   video_decoder video decoder to be used for playback
 * @param   demux demux to be used for playback
 * @param   basename basename of the recording to be played
 * @return  TRUE if playback is started, FALSE otherwise
 */
BOOLEAN STB_PVRPlayStart(U16BIT disk_id, U8BIT audio_decoder, U8BIT video_decoder, U8BIT demux,
   U8BIT *basename)
{
   S_RECORDING_USER_DATA rec_info;
   U8BIT id;

   BOOLEAN avplay_created;
   BOOLEAN buffer_created;
   BOOLEAN channel_created;
   BOOLEAN playback_started;

   BOOLEAN has_video;
   BOOLEAN has_audio;
   BOOLEAN video_failed;
   BOOLEAN audio_failed;

   FUNCTION_START(STB_PVRPlayStart);

   playback_started = FALSE;
   avplay_created = FALSE;
   buffer_created = FALSE;
   channel_created = FALSE;

   id = audio_decoder;
   if (!is_pvr_playback_initialised)
   {
      PVR_ERROR("PVR playback is not initialised");
   }
   else if (id >= players_num)
   {
      PVR_ERROR("Wrong decoder");
   }
   else 
   {

      STB_OSMutexLock(playback_mutex);
      if (players[id].status != PLAYER_STATUS_IDLE)
      {
         PVR_ERROR("PVR playback is already started");
      }
      else if (AV_IsAudioDecoding(audio_decoder) || AV_IsVideoDecoding(video_decoder))
      {
         PVR_ERROR("Decoder is occupied");
      }
      else
      {
         /* Create player */
         U8BIT file_name[PVR_MAX_FILENAME_LEN];
         if (!GetFilenameFromBasename(disk_id, basename, (U8BIT*) file_name, PVR_MAX_FILENAME_LEN))
         {
            PVR_ERROR("Failed to get recording filename from basename \"%s\"", (char*) basename);
         }
         else
         {
            RealtekPlayer * pRealtekPlayer = NULL;
            U8BIT path = STB_DPGetPlaybackPath();

            if (INVALID_RES_ID != path)
               pRealtekPlayer = STB_SdkGetRealtekPlayer(path);

            if (NULL != pRealtekPlayer) {
               std::stringstream ss;

               ss << "pvr://" << file_name;
               if(pRealtekPlayer->RT_PlayerCreate(ss.str().c_str(), 0) != RTK_OK) {

               }

               pRealtekPlayer->RT_Player_SetAVSyncMode(RTK_AVSYNC_AUDIO_MASTER_AUTO);
               if((pRealtekPlayer->RT_PlayerStart()) != RTK_OK) {

               }
               else {
                  playback_started = TRUE;
                  players[id].status = PLAYER_STATUS_RUNNING;
               }

               if (pRealtekPlayer->RT_SetAudioMute(RTK_FALSE) != RTK_OK) {

               }
            }
         }
         /* Create demux buffer */
         /* Create channel */
      }
      STB_OSMutexUnlock(playback_mutex);
   }

   if (playback_started)
   {
      STB_OSSendEvent(FALSE, HW_EV_CLASS_PVR, HW_EV_TYPE_PVR_PLAY_START, NULL, 0);
   }
   else
   {
      if (AV_IsVideoDecoding(video_decoder))
      {
         STB_AVStopVideoDecoding(video_decoder);
      }
      if (AV_IsAudioDecoding(audio_decoder))
      {
         STB_AVStopAudioDecoding(audio_decoder);
      }
   }

   FUNCTION_FINISH(STB_PVRPlayStart);

   return playback_started;
}

/**
 * @brief   Changes the main audio PID being decoded during playback. This can
 *          be used to switch between main audio and broadcaster mix AD.
 * @param   audio_decoder - audio decoder for playback
 * @param   video_decoder - video decoder for playback
 * @param   pid - new audio PID to decode
 * @param   codec - new audio codec
 * @return  TRUE if the PID is changed successfully, FALSE otherwise
 */
BOOLEAN STB_PVRPlayChangeAudio(U8BIT audio_decoder, U8BIT video_decoder, U16BIT pid, U8BIT codec)
{
   BOOLEAN retval;
   U8BIT id;

   FUNCTION_START(STB_PVRPlayChangeAudio);

   retval = FALSE;
   id = audio_decoder;
   if (!is_pvr_playback_initialised)
   {
      PVR_ERROR("PVR playback is not initialised");
   }
   else if (id >= players_num)
   {
      PVR_ERROR("Wrong decoder");
   }
   else
   {
      STB_OSMutexLock(playback_mutex);
      if (players[id].status == PLAYER_STATUS_IDLE)
      {
         PVR_ERROR("There is no active playback");
      }
      else if (codec > AV_AUDIO_CODEC_AAC_ADTS)
      {
         PVR_ERROR("Invalid codec");
      }
      else
      {
         PVR_ERROR("Not implemented");
      }
      STB_OSMutexUnlock(playback_mutex);
   }

   FUNCTION_FINISH(STB_PVRPlayChangeAudio);
   return retval;
}

/**
 * @brief   Returns status of playback with the given decoders
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 * @return  TRUE if playback is in progress with the given decoders
 */
BOOLEAN STB_PVRIsPlayStarted(U8BIT audio_decoder, U8BIT video_decoder)
{
   BOOLEAN retval;
   U8BIT id;
   FUNCTION_START(STB_PVRIsPlayStarted);

   retval = FALSE;
   id = audio_decoder;
   if (!is_pvr_playback_initialised)
   {
      PVR_ERROR("PVR playback is not initialised");
   }
   else if (id >= players_num)
   {
      PVR_ERROR("Wrong decoder");
   }
   else
   {
      STB_OSMutexLock(playback_mutex);
      if (players[id].status != PLAYER_STATUS_IDLE)
      {
         retval = TRUE;
      }
      STB_OSMutexUnlock(playback_mutex);
   }

   FUNCTION_FINISH(STB_PVRIsPlayStarted);
   return retval;
}

/**
 * @brief   Sets the playback position after playback has started (i.e. jump to bookmark)
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 * @param   position_in_seconds position to jump to in the recording in seconds from the beginning
 * @return  TRUE if position is set successfully, FALSE otherwise
 */
BOOLEAN STB_PVRPlaySetPosition(U8BIT audio_decoder, U8BIT video_decoder, U32BIT position_in_seconds)
{
   U8BIT id;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRPlaySetPosition);

   retval = FALSE;
   id = audio_decoder;
   if (!is_pvr_playback_initialised)
   {
      PVR_ERROR("PVR playback is not initialised");
   }
   else if (id >= players_num)
   {
      PVR_ERROR("Wrong decoder");
   }
   else
   {
      STB_OSMutexLock(playback_mutex);
      if (players[id].status == PLAYER_STATUS_IDLE)
      {
         PVR_ERROR("There is no active playback");
      }
      else
      {
         {
            // ??????
            retval = TRUE;
         }
      }
      STB_OSMutexUnlock(playback_mutex);
   }

   FUNCTION_FINISH(STB_PVRPlaySetPosition);
   return retval;
}

/**
 * @brief   Stops playback
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 */
void STB_PVRPlayStop(U8BIT audio_decoder, U8BIT video_decoder)
{
   U8BIT id;
   BOOLEAN playback_stopped;

   FUNCTION_START(STB_PVRPlayStop);

   id = audio_decoder;
   playback_stopped = FALSE;
   if (!is_pvr_playback_initialised)
   {
      PVR_ERROR("PVR playback is not initialised");
   }
   else if (id >= players_num)
   {
      PVR_ERROR("Wrong decoder");
   }
   else
   {
      STB_OSMutexLock(playback_mutex);
      if (players[id].status == PLAYER_STATUS_IDLE)
      {
         PVR_WARNING("There is no active playback");
      }
      else
      {
         {
            {
               {
                  //????AV_PreventStop(audio_decoder, FALSE);
                  if (AV_IsVideoDecoding(video_decoder))
                  {
                     STB_AVStopVideoDecoding(video_decoder);
                  }
                  if (AV_IsAudioDecoding(audio_decoder))
                  {
                     STB_AVStopAudioDecoding(audio_decoder);
                  }

                  players[id].status = PLAYER_STATUS_IDLE;

                  playback_stopped = TRUE;
               }
            }
         }
      }
      STB_OSMutexUnlock(playback_mutex);
   }

   if (playback_stopped)
   {
      STB_OSSendEvent(FALSE, HW_EV_CLASS_DECODE, HW_EV_TYPE_AUDIO_STOPPED, &audio_decoder, sizeof(U8BIT));
      STB_OSSendEvent(FALSE, HW_EV_CLASS_DECODE, HW_EV_TYPE_VIDEO_STOPPED, &video_decoder, sizeof(U8BIT));
      STB_OSSendEvent(FALSE, HW_EV_CLASS_PVR, HW_EV_TYPE_PVR_PLAY_STOP, NULL, 0);
   }

   FUNCTION_FINISH(STB_PVRPlayStop);
}

/**
 * @brief   Returns whether audio and video playback has been started
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 * @param   video returned as TRUE if video is being decoded
 * @param   audio returned as TRUE if audio is being decoded
 */
void STB_PVRPlayEnabled(U8BIT audio_decoder, U8BIT video_decoder, BOOLEAN *video, BOOLEAN *audio)
{
   U8BIT id;
   FUNCTION_START(STB_PVRPlayEnabled);

   id = audio_decoder;
   if (!is_pvr_playback_initialised)
   {
      PVR_ERROR("PVR playback is not initialised");
   }
   else if (id >= players_num)
   {
      PVR_ERROR("Wrong decoder");
   }
   else
   {
      *video = AV_IsVideoDecoding(video_decoder);
      *audio = AV_IsAudioDecoding(audio_decoder);
   }

   FUNCTION_FINISH(STB_PVRPlayEnabled);
}

/**
 * @brief   Set the retention limit for the playback. This function is used for CI+
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 * @param   retention_limit Retention limit in minutes
 * @param   rec_data data when the recording was taken
 * @param   rec_hour hour when the recording was taken
 * @param   rec_min minute when the recording was taken
 */
void STB_PVRPlaySetRetentionLimit(U8BIT audio_decoder, U8BIT video_decoder, U32BIT retention_limit,
   U16BIT rec_date, U8BIT rec_hour, U8BIT rec_min)
{
   FUNCTION_START(STB_PVRPlaySetRetentionLimit);

   FUNCTION_FINISH(STB_PVRPlaySetRetentionLimit);
}

/**
 * @brief   Sets the startup mode for a recording. This function should be called
 *          before the recording is started and is used to when pausing live TV
 *          in which case the additional param defines the length of the pause
 *          buffer to be used, in seconds.
 * @param   rec_index recording index to be used for the recording
 * @param   mode startup mode
 * @param   param additional parameter linked to the mode. When pausing live TV,
 *          this is the length of the pause buffer, in seconds.
 */
void STB_PVRSetRecordStartMode(U8BIT rec_index, E_STB_PVR_START_MODE mode, U32BIT param)
{
   FUNCTION_START(STB_PVRSetRecordStartMode);

   if (!is_pvr_recording_initialised)
   {
      PVR_ERROR("PVR recording is not initialised");
   }
   else
   {
      STB_OSMutexLock(recording_mutex);
      if (rec_index >= recorders_num || recorders[rec_index].status == RECORDER_STATUS_INVALID)
      {
         PVR_ERROR("Invalid recorder[%d]", rec_index);
      }
      else
      {
         recorders[rec_index].start_mode = mode;
         recorders[rec_index].start_mode_param = param;
      }
      STB_OSMutexUnlock(recording_mutex);
   }

   FUNCTION_FINISH(STB_PVRSetRecordStartMode);
}

/**
 * @brief   Starts recording
 * @param   disk_id disk on which the recording is to be saved
 * @param   rec_index recording index to be used for the recording
 * @param   basename base filename to be used for the recording
 * @param   num_pids number of PIDs to be recorded
 * @param   pid_array PIDs to be recorded
 * @return  TRUE if recording is started, FALSE otherwise
 */
BOOLEAN STB_PVRRecordStart(U16BIT disk_id, U8BIT rec_index, U8BIT *basename,
   U16BIT num_pids, S_PVR_PID_INFO *pid_array)
{
   U32BIT i;
   BOOLEAN retval;
   S_RECORDING_MSG msg;

   FUNCTION_START(STB_PVRRecordStart);

   retval = FALSE;
   if (!is_pvr_recording_initialised)
   {
      PVR_ERROR("PVR recording is not initialised");
   }
   /* Check arguments */
   else if (rec_index >= recorders_num)
   {
      PVR_ERROR("Invalid recorder index: %d", rec_index);
   }
   else if (!STB_DSKIsMounted(disk_id))
   {
      PVR_ERROR("Disk does not exist or is not mounted");
   }
   else if (strlen((char*) basename) >= MAX_FILENAME_LEN)
   {
      PVR_ERROR("Basename is too long. Max supported: %d", MAX_FILENAME_LEN);
   }
   else if (num_pids > MAX_PID_NUM)
   {
      PVR_ERROR("Too many PIDs: %d. Max supported: %d", num_pids, MAX_PID_NUM);
   }
   else
   {
      STB_OSMutexLock(recording_mutex);
      if (recorders[rec_index].status != RECORDER_STATUS_IDLE)
      {
         PVR_ERROR("Recorder [%d] is busy or invalid: %d", rec_index, recorders[rec_index].status);
      }
      else
      {
         recorders[rec_index].rec_disk_id = disk_id;
         recorders[rec_index].rec_pids_num = num_pids;
         strncpy((char*) recorders[rec_index].rec_basename, (char*) basename, MAX_FILENAME_LEN);
         for (i = 0; i < num_pids; i++)
         {
            recorders[rec_index].rec_pids_array[i] = pid_array[i];
         }

         msg.cmd = RECORDING_CMD_START_RECORDING;
         msg.recorder_id = rec_index;
         if (!STB_OSWriteQueue(recording_queue, &msg, sizeof(S_RECORDING_MSG), TIMEOUT_NEVER))
         {
            PVR_ERROR("Failed to write message queue");
         }
         else
         {
            recorders[rec_index].status = RECORDER_STATUS_STARTING;
            retval = TRUE;
         }
      }
      STB_OSMutexUnlock(recording_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordStart);
   return retval;
}

/**
 * @brief   Pauses a recording currently taking place
 * @param   rec_index recording index
 * @return  TRUE if the recording is successfully paused, FALSE otherwise
 */
BOOLEAN STB_PVRRecordPause(U8BIT rec_index)
{
   BOOLEAN retval;

   FUNCTION_START(STB_PVRRecordPause);

   retval = FALSE;
   if (!is_pvr_recording_initialised)
   {
      PVR_ERROR("PVR recording is not initialised");
   }
   else if (rec_index >= recorders_num)
   {
      PVR_ERROR("Invalid recorder index: %d", rec_index);
   }
   else
   {
      STB_OSMutexLock(recording_mutex);
      if (recorders[rec_index].status != RECORDER_STATUS_RUNNING)
      {
         PVR_ERROR("Recorder is not active");
      }
      else
      {
         {
            recorders[rec_index].status = RECORDER_STATUS_PAUSED;
            retval = TRUE;
         }
      }
      STB_OSMutexUnlock(recording_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordPause);
   return retval;
}

/**
 * @brief   Resumes a paused recording
 * @param   rec_index recording index
 * @return  TRUE if the recording is successfully resumed, FALSE otherwise
 */
BOOLEAN STB_PVRRecordResume(U8BIT rec_index)
{
   BOOLEAN retval;

   FUNCTION_START(STB_PVRRecordResume);

   retval = FALSE;
   if (!is_pvr_recording_initialised)
   {
      PVR_ERROR("PVR recording is not initialised");
   }
   else if (rec_index >= recorders_num)
   {
      PVR_ERROR("Invalid recorder index: %d", rec_index);
   }
   else
   {
      STB_OSMutexLock(recording_mutex);
      if (recorders[rec_index].status != RECORDER_STATUS_PAUSED)
      {
         PVR_ERROR("Recorder is not paused");
      }
      else
      {
         {
            recorders[rec_index].status = RECORDER_STATUS_RUNNING;
            retval = TRUE;
         }
      }
      STB_OSMutexUnlock(recording_mutex);
   }
   FUNCTION_FINISH(STB_PVRRecordResume);
   return retval;
}

/**
 * @brief   Changes the PIDs while recording
 * @param   rec_index current recording index  to be updated
 * @param   num_pids number of PIDs in PID array
 * @param   pid_array new PID list to be recorded
 * @return  TRUE if the PIDs have been successfully changed, FALSE otherwise
 */
BOOLEAN STB_PVRRecordChangePids(U8BIT rec_index, U16BIT num_pids, S_PVR_PID_INFO *pids_array)
{
    return TRUE;
}

/**
 * @brief   Stops a recording
 * @param   rec_index recording index
 */
void STB_PVRRecordStop(U8BIT rec_index)
{
   S_RECORDING_MSG msg;

   FUNCTION_START(STB_PVRRecordStop);

   if (!is_pvr_recording_initialised)
   {
      PVR_ERROR("PVR recording is not initialised");
   }
   else if (rec_index >= recorders_num)
   {
      PVR_ERROR("Invalid recorder index: %d", rec_index);
   }
   else
   {
      STB_OSMutexLock(recording_mutex);
      if (recorders[rec_index].status != RECORDER_STATUS_RUNNING &&
          recorders[rec_index].status != RECORDER_STATUS_STARTING &&
          recorders[rec_index].status != RECORDER_STATUS_PAUSED)
      {
         PVR_WARNING("Recorder is not active");
      }
      else
      {
         msg.cmd = RECORDING_CMD_STOP_RECORDING;
         msg.recorder_id = rec_index;
         if (!STB_OSWriteQueue(recording_queue, &msg, sizeof(S_RECORDING_MSG), TIMEOUT_NEVER))
         {
            PVR_ERROR("Failed to write message queue");
         }
         else
         {
            recorders[rec_index].status = RECORDER_STATUS_STOPPING;
         }
      }
      STB_OSMutexUnlock(recording_mutex);
   }

   U32BIT time_start = STB_OSGetClockMilliseconds();
   STB_OSSemaphoreWaitTimeout(pSemaphore_stopRecord,5000);
   PVR_DEBUG("stop timeout = %d.", STB_OSGetClockMilliseconds()-time_start);

   FUNCTION_FINISH(STB_PVRRecordStop);
}

/**
 * @brief   Returns whether recording has been started
 * @param   rec_index recording index being queried
 * @return  TRUE if recording has been started
 */
BOOLEAN STB_PVRIsRecordStarted(U8BIT rec_index)
{
   BOOLEAN retval;
   FUNCTION_START(STB_PVRIsRecordStarted);

   retval = FALSE;
   if (!is_pvr_recording_initialised)
   {
      PVR_ERROR("PVR recording is not initialised");
   }
   else if (rec_index >= recorders_num)
   {
      PVR_ERROR("Invalid recorder index: %d", rec_index);
   }
   else
   {
      STB_OSMutexLock(recording_mutex);
      if (recorders[rec_index].status == RECORDER_STATUS_RUNNING  ||
          recorders[rec_index].status == RECORDER_STATUS_STOPPING ||
          recorders[rec_index].status == RECORDER_STATUS_PAUSED)
      {
         retval = TRUE;
      }
      STB_OSMutexUnlock(recording_mutex);
   }

   FUNCTION_FINISH(STB_PVRIsRecordStarted);
   return retval;
}

/**
 * @brief   Returns status of audio/video recording
 * @param   rec_index recording index being used for recording
 * @param   video pointer to a boolean value that indicates whether the video data is being recorded
 * @param   audio pointer to a boolean value that indicates whether the audio data is being recorded
 */
void STB_PVRRecordEnabled(U8BIT rec_index, BOOLEAN *video, BOOLEAN *audio)
{
   FUNCTION_START(STB_PVRRecordEnabled);

   *video = FALSE;
   *audio = FALSE;
   if (!is_pvr_recording_initialised)
   {
      PVR_ERROR("PVR recording is not initialised");
   }
   else if (rec_index >= recorders_num)
   {
      PVR_ERROR("Invalid recorder index: %d", rec_index);
   }
   else
   {
      STB_OSMutexLock(recording_mutex);
      if (recorders[rec_index].status == RECORDER_STATUS_RUNNING  ||
          recorders[rec_index].status == RECORDER_STATUS_STOPPING ||
          recorders[rec_index].status == RECORDER_STATUS_PAUSED)
      {
         *video = recorders[rec_index].is_video_recording;
         *audio = recorders[rec_index].is_audio_recording;
      }
      STB_OSMutexUnlock(recording_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordEnabled);
}

/**
 * @brief   Enables or disables encryption and sets the encryption key to be used
 * @param   rec_index recording index to be set
 * @param   state whether encryption is enabled of disabled
 * @param   key encryption key, ignored if state is FALSE
 * @param   iv initialisation vector, ignored if state is FALSE
 * @param   key_len length of encryption key, ignored if state is FALSE
 */
void STB_PVRSetRecordEncryptionKey(U8BIT rec_index, BOOLEAN state, U8BIT *key, U8BIT *iv, U32BIT key_len)
{
   FUNCTION_START(STB_PVRSetRecordEncryptionKey);
   recorders[rec_index].encrypt = state;
   if(state)
   {
	   if(key_len > CRYPTION_KEY_MAX_LEN)
		   key_len = CRYPTION_KEY_MAX_LEN;
	   memcpy(recorders[rec_index].cryption_key.key, key, key_len);
	   memcpy(recorders[rec_index].cryption_key.iv, iv, key_len);
	   recorders[rec_index].cryption_key.key_len = key_len;
   }
   FUNCTION_FINISH(STB_PVRSetRecordEncryptionKey);
}

/**
 * @brief   Enables and sets the key that will be used to decrypt an encrypted
 *          recording during playback
 * @param   audio_decoder audio decoder used for playback
 * @param   video_decoder video decoder used for playback
 * @param   state whether decryption is enabled of disabled
 * @param   key decryption key, ignored if state is FALSE
 * @param   iv initialisation vector, ignored if state is FALSE
 * @param   key_len length of decryption key, ignored if state is FALSE
 */
void STB_PVRSetPlaybackDecryptionKey(U8BIT audio_decoder, U8BIT video_decoder, BOOLEAN state,
   U8BIT *key, U8BIT *iv, U32BIT key_len)
{
   FUNCTION_START(STB_PVRSetPlaybackDecryptionKey);
   players[0].encrypt = state;
   if(state)
   {
	   if(key_len > CRYPTION_KEY_MAX_LEN)
		   key_len = CRYPTION_KEY_MAX_LEN;
	   memcpy(players[0].cryption_key.key, key, key_len);
	   memcpy(players[0].cryption_key.iv, iv, key_len);
	   players[0].cryption_key.key_len = key_len;
   }
   FUNCTION_FINISH(STB_PVRSetPlaybackDecryptionKey);
}

/**
 * @brief   Sets trick mode during playback
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 * @param   mode trick mode to be used
 * @param   speed playback speed to be used as a percentage (100% = normal playback)
 */
void STB_PVRPlayTrickMode(U8BIT audio_decoder, U8BIT video_decoder, E_STB_PVR_PLAY_MODE mode, S16BIT speed)
{
    return;
}

/**
 * @brief   Save current frame of playback
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 */
void STB_PVRSaveFrame(U8BIT audio_decoder, U8BIT video_decoder)
{
    return;
}

/**
 * @brief   Returns the current playback speed
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 * @return  current playback speed as a percentage
 */
S16BIT STB_PVRGetPlaySpeed(U8BIT audio_decoder, U8BIT video_decoder)
{
   S16BIT speed = 0;
   char ret;
   RealtekPlayer * pRealtekPlayer;
   U8BIT path = STB_DPGetPlaybackPath();

   if (INVALID_RES_ID == path)
       goto end;

   pRealtekPlayer = STB_SdkGetRealtekPlayer(path);
   if (NULL == pRealtekPlayer)
      goto end;

   float playSpeed;
   ret = pRealtekPlayer->RT_PlayerGetPlaySpeed(&playSpeed);
   if (RTK_OK != ret)
      goto end;

   speed = (S16BIT)(playSpeed * 100);

end:
   return speed;
}

/**
 * @brief   Set the play speed for the specified decoder
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 * @param   speed Play speed as a percentage (i.e 100% = normal playback)
 * @return  TRUE if successful
 */
BOOLEAN STB_PVRSetPlaySpeed(U8BIT audio_decoder, U8BIT video_decoder, S16BIT speed)
{
   BOOLEAN ret = FALSE;
   RealtekPlayer * pRealtekPlayer = NULL;
   U8BIT path = STB_DPGetPlaybackPath();

   if (INVALID_RES_ID == path)
       goto end;

   pRealtekPlayer = STB_SdkGetRealtekPlayer(path);
   if (NULL == pRealtekPlayer)
      goto end;

   char rr;
   rr = pRealtekPlayer->RT_PlayerSetPlaySpeed(((float)speed) / 100);
   if (RTK_OK != rr)
      goto end;


   int32_t mode;
	if((100 == speed) || (0 == speed))
      mode = RTK_AVSYNC_AUDIO_MASTER_AUTO;
   else
      mode = RTK_AVSYNC_SYSTEM_MASTER;
      
	pRealtekPlayer->RT_Player_SetAVSyncMode(mode);

   ret = TRUE;

end:
    return ret;
}

/**
 * @brief   Checks whether any of the files already exist that would be created
 *          by a recording with the given base filename.
 * @param   disk_id disk to be checked
 * @param   basename base filename to be used for a recording
 * @return  TRUE if none of the files exist, FALSE otherwise
 */
BOOLEAN STB_PVRIsValidRecording(U16BIT disk_id, U8BIT *basename)
{
   BOOLEAN retval;
   U8BIT filename[MAX_FILENAME_LEN];

   FUNCTION_START(STB_PVRIsValidRecording);

   retval = FALSE;
   if (STB_DSKFileExists(disk_id, basename))
   {
      if (GetFilenameFromBasename(disk_id, basename, (U8BIT*) filename, MAX_FILENAME_LEN))
      {
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_PVRIsValidRecording);
   return retval;

}

/**
 * @brief   Checks whether any of the files already exist that would be created
 *          by a recording with the given base filename.
 * @param   disk_id disk to be checked
 * @param   basename base filename to be used for a recording
 * @return  TRUE if none of the files exist, FALSE otherwise
 */
BOOLEAN STB_PVRCanBeUsedForRecording(U16BIT disk_id, U8BIT *basename)
{
   BOOLEAN retval;
   FUNCTION_START(STB_PVRCanBeUsedForRecording);

   retval = FALSE;
   if (!is_pvr_recording_initialised)
   {
      PVR_ERROR("PVR recording is not initialised");
   }
   else
   {
      retval = STB_DSKIsMounted(disk_id) && !STB_DSKFileExists(disk_id, basename);
   }

   FUNCTION_FINISH(STB_PVRCanBeUsedForRecording);
   return retval;
}

/**
 * @brief   Deletes any files associated with the given base filename that were created
 *          as a result of the recording being performed.
 * @param   disk_id disk containing the recording to be deleted
 * @param   basename base filename used for the recording
 * @return  TRUE if successful, FALSE otherwise
 */
BOOLEAN STB_PVRDeleteRecording(U16BIT disk_id, U8BIT *basename)
{
   BOOLEAN retval;

   FUNCTION_START(STB_PVRDeleteRecording);

   retval = FALSE;
   /* This call may take few seconds, so it might be a good idea to move it out to RecordingTask.
    * For now I kept it here, because I'm not sure if it should be blocking */
   retval = STB_DSKDeleteDirectory(disk_id, basename);

   FUNCTION_FINISH(STB_PVRDeleteRecording);
   return retval;
}

/**
 * @brief   Returns the size in kilobytes of the recording defined by the given base filename.
 * @param   disk_id disk containing the recording to be queried
 * @param   basename base filename of recording to get info about
 * @param   rec_size_kb returned size of recording in kilobytes
 * @return  TRUE if the information is successfully gathered
 */
BOOLEAN STB_PVRGetRecordingSize(U16BIT disk_id, U8BIT *basename, U32BIT *rec_size_kb)
{
   U8BIT file_name[PVR_MAX_FILENAME_LEN];
   U32BIT filesize;
   BOOLEAN ret = FALSE;

   snprintf((char*) file_name, MAX_FILENAME_LEN, "%s/0.%s", (char*) basename, FILENAME_EXT_TS);
   // FIXME: calculate the folder size
   if (TRUE == STB_DSKFileSize(disk_id, file_name, &filesize)) {
      *rec_size_kb = filesize;
      ret = TRUE;
   }

   return ret;
}

/**
 * @brief   Returns the elapsed playback time in hours, mins & secs
 * @param   audio_decoder audio decoder being used for playback
 * @param   video_decoder video decoder being used for playback
 * @param   elapsed_hours current number of hours into the playback
 * @param   elapsed_mins current number of minutes into the playback
 * @param   elapsed_secs current number of seconds into the playback
 * @return  TRUE if the info has been successfully gathered
 */
BOOLEAN STB_PVRGetElapsedTime(U8BIT audio_decoder, U8BIT video_decoder, U8BIT *elapsed_hours,
   U8BIT *elapsed_mins, U8BIT *elapsed_secs)
{
   long long first;
   long long current;
   long long elapsed;
   RealtekPlayer * pRealtekPlayer;
   U8BIT path;
   BOOLEAN ret = FALSE;
   
   path = STB_DPGetPlaybackPath();
   if (INVALID_RES_ID == path)
       goto end;

   pRealtekPlayer = STB_SdkGetRealtekPlayer(path);
   if (NULL == pRealtekPlayer)
      goto end;
      
   pRealtekPlayer->RT_PlayerGetFirstDisplayingPTS(&first);
   pRealtekPlayer->RT_PlayerGetCurrentDisplayingPTS(&current);
   if ((first >= 0) && (current >= 0)) {
      if (current >= first)
         elapsed = current - first;
      else
         elapsed = 0x200000000 - first + current;  // PTS 33bits@90kHz

      elapsed /= 90000;  // seconds

      *elapsed_hours = (U8BIT)(elapsed / 3600);
      elapsed %= 3600;  // in an hour
      *elapsed_mins = (U8BIT)(elapsed / 60);
      *elapsed_secs = (U8BIT)(elapsed % 60);
      ret = TRUE;
   }
   
end:
   return ret;
}

/**
 * @brief   Acquires an index to be used to reference a recording
 * @param   tuner tuner to be used for the recording
 * @param   demux demux to be used for the recording
 * @return  recording index, 255 if none available
 */
U8BIT STB_PVRAcquireRecorderIndex(U8BIT tuner, U8BIT demux)
{
   U32BIT i;
   U8BIT retval;

   FUNCTION_START(STB_PVRAcquireRecorderIndex);

   retval = INVALID_RECORDER;
   if (!is_pvr_recording_initialised)
   {
      PVR_ERROR("PVR recording is not initialised");
   }
   else
   {
      STB_OSMutexLock(recording_mutex);
      for (i = 0; i < recorders_num; i++)
      {
         if (recorders[i].status == RECORDER_STATUS_INVALID)
         {
            recorders[i].status = RECORDER_STATUS_IDLE;
            recorders[i].is_video_recording = FALSE;
            recorders[i].is_audio_recording = FALSE;
            recorders[i].tuner_id = tuner;
			recorders[i].demux_id = DMX_GetRecordPath(demux);
            retval = i;
            PVR_DEBUG("Recorder [%d] aquired", retval);
            break;
         }
      }
      STB_OSMutexUnlock(recording_mutex);
   }

   FUNCTION_FINISH(STB_PVRAcquireRecorderIndex);
   return retval;
}

/**
 * @brief   Releases a recording index when no longer needed
 * @param   rec_index recoding index
 */
void STB_PVRReleaseRecorderIndex(U8BIT rec_index)
{
   S_RECORDING_MSG msg;

   FUNCTION_START(STB_PVRReleaseRecorderIndex);

   if (!is_pvr_recording_initialised)
   {
      PVR_ERROR("PVR recording is not initialised");
   }
   else if (rec_index >= recorders_num)
   {
      PVR_ERROR("Invalid recorder index: %d", rec_index);
   }
   else
   {
      STB_OSMutexLock(recording_mutex);
      if (recorders[rec_index].status != RECORDER_STATUS_INVALID)
      {
         if (recorders[rec_index].status == RECORDER_STATUS_RUNNING  ||
             recorders[rec_index].status == RECORDER_STATUS_STARTING ||
             recorders[rec_index].status == RECORDER_STATUS_PAUSED)
         {
            PVR_ERROR("Failed to release recorder: stop recording first");
         }
         else
         {
            /* Wait for recording to stop */
            msg.cmd = RECORDING_CMD_RELEASE_RECORDER;
            msg.recorder_id = rec_index;
            if (!STB_OSWriteQueue(recording_queue, &msg, sizeof(S_RECORDING_MSG), TIMEOUT_NEVER))
            {
               PVR_ERROR("Failed to write message queue");
            }
         }
      }
      STB_OSMutexUnlock(recording_mutex);
   }
   FUNCTION_FINISH(STB_PVRReleaseRecorderIndex);
}

/**
 * @brief   Called to apply the given descrambler key to the PID data being recorded.
 *          This function may be called before the recording has actually started.
 * @param   rec_index recording index
 * @param   desc_type descrambler type
 * @param   parity key parity
 * @param   key key data
 * @param   iv provides an initialisation vector data, if required for the descrambler type
 * @return  TRUE if successful, FALSE otherwise
 */
BOOLEAN STB_PVRApplyDescramblerKey(U8BIT rec_index, E_STB_DMX_DESC_TYPE desc_type,
   E_STB_DMX_DESC_KEY_PARITY parity, U8BIT *key, U8BIT *iv)
{
   FUNCTION_START(STB_PVRApplyDescramblerKey);
   USE_UNWANTED_PARAM(rec_index);
   USE_UNWANTED_PARAM(desc_type);
   USE_UNWANTED_PARAM(parity);
   USE_UNWANTED_PARAM(key);
   USE_UNWANTED_PARAM(iv);
   FUNCTION_FINISH(STB_PVRApplyDescramblerKey);
   return FALSE;
}

/*!**************************************************************************
 * @fn      PVR_DeinitPlayback
 * @brief
 * @param
 * @return
 ****************************************************************************/
void PVR_DeinitPlayback(void)
{
   FUNCTION_START(PVR_DeinitPlayback);

   if (players != NULL)
   {
      STB_MEMFreeSysRAM(players);
      players_num = 0;
      players = NULL;
   }
   if (playback_mutex != NULL)
   {
      STB_OSDeleteMutex(playback_mutex);
      playback_mutex = NULL;
   }

   is_pvr_playback_initialised = FALSE;

   FUNCTION_FINISH(PVR_DeinitPlayback);
}

/*!**************************************************************************
 * @fn      GetFilenameFromBasename
 * @brief
 * @param
 * @return  TRUE if successfull, FALSE otherwise
 ****************************************************************************/
static BOOLEAN GetFilenameFromBasename(U16BIT disk_id, U8BIT *basename,
                                       U8BIT* filename, U32BIT max_filename_len)
{
   BOOLEAN retval;
   U8BIT dirname[MAX_FILENAME_LEN];
   U32BIT size;

   FUNCTION_START(GetFilenameFromBasename);

   retval = FALSE;
   if (STB_DSKFullPathname(disk_id, basename, dirname, MAX_FILENAME_LEN))
   {
      size = snprintf((char*) filename, max_filename_len, "%s/0.%s", (char*) dirname, FILENAME_EXT_TS);
      if (size < max_filename_len)
      {
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(GetFilenameFromBasename);
   return retval;
}

/*!**************************************************************************
 * @fn      RecordingTask
 * @brief   Recording task
 * @param   param - not used
 ****************************************************************************/
static void RecordingTask(void *param)
{
   S_RECORDING_MSG msg;
   BOOLEAN recording_started;
   BOOLEAN dubplicated_pid;
   U32BIT i;
   U32BIT j;

   BOOLEAN is_video_recording;
   BOOLEAN is_audio_recording;
   U8BIT rec_demux_id;
   E_STB_PVR_START_MODE rec_start_mode;
   U32BIT rec_start_mode_param;
   U16BIT rec_disk_id;
   U8BIT rec_basename[MAX_FILENAME_LEN];
   U16BIT rec_pids_num;
   S_PVR_PID_INFO rec_pids_array[MAX_PID_NUM];
   S_RECORDING_USER_DATA rec_user_data;

   FUNCTION_START(RecordingTask);
   STB_OSMutexLock(recording_task_termination_mutex);

   while (TRUE)
   {
      if (STB_OSReadQueue(recording_queue, &msg, sizeof(S_RECORDING_MSG), TIMEOUT_NEVER))
      {
         if (msg.cmd == RECORDING_CMD_TASK_TERMINATE)
         {
            break;
         }
         else if (msg.recorder_id >= recorders_num)
         {
            PVR_ERROR("Invalid recorder ID received: %d, max: %d", msg.recorder_id, recorders_num);
         }

         else if (msg.cmd == RECORDING_CMD_START_RECORDING)
         {
            ADB_SERVICE_REC * pService = (ADB_SERVICE_REC * )DBDEF_GetTunedService(rec_demux_id);
            if (NULL == pService)
               continue;
   
            STB_OSMutexLock(recording_mutex);
            /* Copying this data would allow to not lock the rest of the recording
             * starting code by mutex. */
            rec_demux_id = recorders[msg.recorder_id].demux_id;
            rec_start_mode = recorders[msg.recorder_id].start_mode;
            rec_start_mode_param = recorders[msg.recorder_id].start_mode_param;
            rec_disk_id = recorders[msg.recorder_id].rec_disk_id;
            rec_pids_num = recorders[msg.recorder_id].rec_pids_num;
            strncpy((char*) rec_basename, (char*) recorders[msg.recorder_id].rec_basename,
                    MAX_FILENAME_LEN);
            for (i = 0; i < rec_pids_num; i++)
            {
               rec_pids_array[i] = recorders[msg.recorder_id].rec_pids_array[i];
            }
            STB_OSMutexUnlock(recording_mutex);

            recording_started = FALSE;
            /* STB_DSKGetUsed() is the most expensive call in STB_PVRRecordStart(), so remove
             * this check. If this check is really needed, I propose to get and cache used
             * size in CheckMountedDisks() (which is called regularly from DiskMonitorTask(),
             * stbhwdsk.c) and then reimplement STB_DSKGetUsed() to read the cached value */
            // if (STB_DSKGetSize(disk_id) - STB_DSKGetUsed(disk_id) < MIN_SPACE_REQUIRED_KB)
            // {
            //    PVR_ERROR("Disk is almost full. Cannot start recording");
            // }
            if (!STB_DSKCreateDirectory(rec_disk_id, rec_basename))
            {
               PVR_ERROR("Failed to create directory for recording");
            }
            else
            {
               /* Set recording attributes */

               /*** Set the filename */
               U8BIT fileName[PVR_MAX_FILENAME_LEN];            
               if (!GetFilenameFromBasename(rec_disk_id, rec_basename,
                                            fileName, PVR_MAX_FILENAME_LEN))
               {
                  STB_DSKDeleteDirectory(rec_disk_id, rec_basename);
                  PVR_ERROR("Failed to obtain filename");
               }
               else
               {
                  TS_PSI_Program program;
                  memset (&program, 0, sizeof(TS_PSI_Program));
                  //program.program_number = 0; // invalid
                  program.program_number = pService->serv_id;
                  program.map_pid = pService->pmt_pid;
                        
                  /*** Find index PID */
                  is_audio_recording = FALSE;
                  is_video_recording = FALSE;
                  rec_user_data.video_pid = INVALID_PID;
                  rec_user_data.pcr_pid = INVALID_PID;
                  rec_user_data.audio.audio_index_default = 0;
                  rec_user_data.audio.audio_num = 0;
                  for (i = 0; i < rec_pids_num; i++)
                  {
                     if (rec_pids_array[i].type == PVR_PID_TYPE_AUDIO)
                     {
                        // SDK setting
                        int idx = rec_user_data.audio.audio_num;
                        program.audio_pids[idx].pid = rec_pids_array[i].pid;
                        program.audio_pids[idx].codec = AudioCodecToDvbStreamType((E_STB_AV_AUDIO_CODEC)rec_pids_array[i].u.audio_codec);
                        program.num_audio_pids ++;
                        
                        is_audio_recording = TRUE;
                        rec_user_data.audio.audio_pid[rec_user_data.audio.audio_num] = rec_pids_array[i].pid;
                        rec_user_data.audio.audio_codec[rec_user_data.audio.audio_num] = rec_pids_array[i].u.audio_codec;
						      rec_user_data.audio.audio_num++;

                        //rec_user_data.pcr_pid = rec_pids_array[i].pid;
                     }

                     if (rec_pids_array[i].type == PVR_PID_TYPE_VIDEO && !is_video_recording)
                     {
                        // SDK
                        program.video_pids[0].pid = rec_pids_array[i].pid;
                        program.video_pids[0].codec = VideoCodecToDvbStreamType((E_STB_AV_VIDEO_CODEC)rec_pids_array[i].u.video_codec);
                        program.num_video_pids = 1;
                              
                        is_video_recording = TRUE;
                        rec_user_data.video_pid = rec_pids_array[i].pid;
                        rec_user_data.video_codec = rec_pids_array[i].u.video_codec;
                     }

                     if (rec_pids_array[i].type == PVR_PID_TYPE_PCR)
                     {
                        rec_user_data.pcr_pid = rec_pids_array[i].pid;
                     }
                  }
                  if (rec_user_data.pcr_pid == INVALID_PID)
                  {
                     rec_user_data.pcr_pid = rec_user_data.video_pid;
                  }
                  if(rec_user_data.audio.audio_num > 1)
                  {
                     for(i=0; i<rec_user_data.audio.audio_num; i++)
                     {
                        if(rec_user_data.audio.audio_pid[i] == DMX_GetLiveAudioPID())
                        {
                           /* found the default audio track */
                           rec_user_data.audio.audio_index_default = i;
                           break;
                        }
                     }
                  }
                  /*** Decide between timeshift and normal recording */

                  /* Start recording channel */
                  {
                     /* Add recording PIDs */
                     for (i = 0; i < rec_pids_num; i++)
                     {
                        /* Handle the case if PID is already added. */
                        dubplicated_pid = FALSE;
                        for (j = 0; j < i; j++)
                        {
                           if (rec_pids_array[i].pid == rec_pids_array[j].pid)
                           {
                              dubplicated_pid = TRUE;
                           }
                        }
                        if (dubplicated_pid)
                        {
                           continue;
                        }
                     }

                     RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(rec_demux_id);
                     if (NULL != pRealtekPlayer) {
                        if((pRealtekPlayer->RT_PlayerOpenPVR((const char *)fileName)) != RTK_OK) {

                        }
                        if(pRealtekPlayer->RT_PlayerStartPVR() != RTK_OK) {

                        }
                        pRealtekPlayer->RT_PlayerUpTsTSFilter(&program);

                        PVR_DEBUG("Recorder [%d] is started", msg.recorder_id);
                        {
                           STB_OSMutexLock(recording_mutex);
                           recorders[msg.recorder_id].status = RECORDER_STATUS_RUNNING;
                           recorders[msg.recorder_id].is_audio_recording = is_audio_recording;
                           recorders[msg.recorder_id].is_video_recording = is_video_recording;
                           //recorders[msg.recorder_id].channel_id = channel_id;
                           STB_OSMutexUnlock(recording_mutex);

                           recording_started = TRUE;

                           /* Workaround: Upper layers may start timeshift playback immediately
                            * after recording is started, but HiSDK playback API would fail if
                            * recording is less than 1500ms long */
                            /*
                           if (rec_start_mode == START_PAUSED)
                           {
                              STB_OSTaskDelay(MIN_REC_DURATION_TIMESHIFT);
                           }
                           */

                           STB_OSSendEvent(FALSE, HW_EV_CLASS_PVR, HW_EV_TYPE_PVR_REC_START,
                                           &msg.recorder_id, sizeof(U8BIT));
                        }
                     }
                  }
               }
            }

            if (!recording_started)
            {
               STB_OSMutexLock(recording_mutex);
               recorders[msg.recorder_id].status = RECORDER_STATUS_IDLE;
               recorders[msg.recorder_id].is_audio_recording = FALSE;
               recorders[msg.recorder_id].is_video_recording = FALSE;
               STB_OSMutexUnlock(recording_mutex);
            }
         }

         else if (msg.cmd == RECORDING_CMD_STOP_RECORDING)
         {
            STB_OSMutexLock(recording_mutex);
            rec_demux_id = recorders[msg.recorder_id].demux_id;
            STB_OSMutexUnlock(recording_mutex);

            RealtekPlayer * pRealtekPlayer = STB_SdkGetRealtekPlayer(rec_demux_id);
            if (NULL != pRealtekPlayer) {
               if(pRealtekPlayer->RT_PlayerStopPVR() != RTK_OK) {

               }
               if((pRealtekPlayer->RT_PlayerClosePVR()) != RTK_OK) {

               }
            }

            PVR_DEBUG("Recorder [%d] is stopped", msg.recorder_id);

            STB_OSMutexLock(recording_mutex);
            recorders[msg.recorder_id].status = RECORDER_STATUS_IDLE;
            recorders[msg.recorder_id].is_audio_recording = FALSE;
            recorders[msg.recorder_id].is_video_recording = FALSE;
            STB_OSMutexUnlock(recording_mutex);

            STB_OSSendEvent(FALSE, HW_EV_CLASS_PVR, HW_EV_TYPE_PVR_REC_STOP,
                            &msg.recorder_id, sizeof(U8BIT));

            STB_OSSemaphoreSignal(pSemaphore_stopRecord);
         }

         else if (msg.cmd == RECORDING_CMD_RELEASE_RECORDER)
         {
            STB_OSMutexLock(recording_mutex);
            if (recorders[msg.recorder_id].status != RECORDER_STATUS_IDLE)
            {
               PVR_ERROR("Failed to release recorder [%d]: it was not stopped properly", msg.recorder_id);
            }
            else
            {
               recorders[msg.recorder_id].status = RECORDER_STATUS_INVALID;
               PVR_DEBUG("Recorder [%d] released", msg.recorder_id);
            }
            STB_OSMutexUnlock(recording_mutex);
         }
      }
   }

   STB_OSMutexUnlock(recording_task_termination_mutex);
   FUNCTION_FINISH(RecordingTask);
}

/*!**************************************************************************
 * @fn      PVR_DeinitRecording
 * @brief
 * @param
 * @return
 ****************************************************************************/
void PVR_DeinitRecording(void)
{
   U32BIT i;
   S_RECORDING_MSG msg;

   FUNCTION_START(PVR_DeinitRecording);

   for (i = 0; i < recorders_num; i++)
   {
      if (recorders[i].status != RECORDER_STATUS_INVALID)
      {
         STB_PVRReleaseRecorderIndex(i);
      }
   }

   if (recording_task != NULL)
   {
      msg.cmd = RECORDING_CMD_TASK_TERMINATE;
      msg.recorder_id = INVALID_RECORDER;
      STB_OSWriteQueue(recording_queue, &msg, sizeof(S_RECORDING_MSG), TIMEOUT_NEVER);
      STB_OSMutexLock(recording_task_termination_mutex);
      STB_OSDestroyTask(recording_task);
      STB_OSMutexUnlock(recording_task_termination_mutex);
      recording_task = NULL;
   }
   if (recording_task_termination_mutex != NULL)
   {
      STB_OSDeleteMutex(recording_task_termination_mutex);
      recording_task_termination_mutex = NULL;
   }
   if (recording_queue != NULL)
   {
      STB_OSDestroyQueue(recording_queue);
      recording_queue = NULL;
   }
   if (recorders != NULL)
   {
      STB_MEMFreeSysRAM(recorders);
      recorders_num = 0;
      recorders = NULL;
   }
   if (recording_mutex != NULL)
   {
      STB_OSDeleteMutex(recording_mutex);
      recording_mutex = NULL;
   }

   is_pvr_recording_initialised = FALSE;
   PVR_DEBUG("Terminated");

   FUNCTION_FINISH(PVR_DeinitRecording);
}

static uint16_t VideoCodecToDvbStreamType(E_STB_AV_VIDEO_CODEC codec)
{
/*
   typedef enum e_stb_av_video_codec
   {
      AV_VIDEO_CODEC_AUTO = 0,
      AV_VIDEO_CODEC_MPEG1 = 1,
      AV_VIDEO_CODEC_MPEG2 = 2,
      AV_VIDEO_CODEC_H264 = 3,
      AV_VIDEO_CODEC_H265 = 4,
   } E_STB_AV_VIDEO_CODEC;
*/
   static uint16_t streamType[] = {
      TS_PSI_ST_14496_10_Video /*FIXME: default to H.264*/,
      TS_PSI_ST_11172_2_Video,
      TS_PSI_ST_13818_2_Video,
      TS_PSI_ST_14496_10_Video,
      TS_PSI_ST_23008_2_Video
   };

   uint16_t type = 0;
   if (codec < sizeof(streamType))
      type = streamType[codec];
   
   printf("\n===> %s  %d  %s %d -> 0x%X\n", __BASE_FILE__, __LINE__, __func__, codec, type);
   return type;
}

static uint16_t AudioCodecToDvbStreamType(E_STB_AV_AUDIO_CODEC codec)
{
/*
   typedef enum e_stb_av_audio_codec
   {
      AV_AUDIO_CODEC_AUTO = 0,
      AV_AUDIO_CODEC_MP2 = 1,
      AV_AUDIO_CODEC_MP3 = 2,
      AV_AUDIO_CODEC_AC3 = 3,
      AV_AUDIO_CODEC_EAC3 = 4,
      AV_AUDIO_CODEC_AAC = 5,
      AV_AUDIO_CODEC_HEAAC = 6,
      AV_AUDIO_CODEC_AAC_ADTS = 7
   } E_STB_AV_AUDIO_CODEC;
*/
   static uint16_t streamType[] = {
      TS_PSI_ST_13818_3_Audio /*FIXME: default to mpeg2 audio*/,
      TS_PSI_ST_13818_3_Audio,
      TS_PSI_ST_13818_3_Audio,
      TS_PSI_ST_ATSC_AC3,
      TS_PSI_ST_ATSC_EAC3,
      TS_PSI_ST_13818_7_AAC,
      TS_PSI_ST_13818_7_AAC,
      TS_PSI_ST_13818_7_AAC
   };

   uint16_t type = 0;
   if (codec < sizeof(streamType))
      type = streamType[codec];
  
   printf("\n===> %s  %d  %s %d -> 0x%X\n", __BASE_FILE__, __LINE__, __func__, codec, type); 
   return type;
}
