/*******************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Memory-related function
 * @file    heap.h
 * @date    08 November 2006
 * @author  Omri Barel
 */

#ifndef _HEAP_H
#define _HEAP_H


/*--- Includes ----------------------------------------------------------------*/

/*--- Preprocessor definitions ------------------------------------------------*/

/*--- Global types definitions ------------------------------------------------*/

/*--- Global function declarations --------------------------------------------*/

/*!**************************************************************************
 * @fn      HEAPOpen
 * @brief   Initialise memory manager module. Allocate component memory
 *          region. Allocates memory region from the default memory pool.
 *          This function must be called to initialise the MHEG5 memory
 *          manager. To close the memory manager a call to HEAPClose must be
 *          made. This function must not be called 'out of turn', i.e. after
 *          calling HEAPOpen a call to HEAPClose MUST be made before another
 *          call to HEAPOpen.
 * @param   memory_region - Pointer to memory region, or NULL if the region
 *          is to be allocated.
 *          memory_region_size - Size of region, or if the region is NULL,
 *          the size to be used for the allocation.
 * @return  TRUE if completed successfully, FALSE if unable to allocate
 *          memory region.
 ****************************************************************************/
BOOLEAN HEAPOpen(void *memory_region, U32BIT memory_region_size);



/*!**************************************************************************
 * @fn      HEAPClose
 * @brief   Release component memory region.
 *          A call to this function MUST only be made after a successful call
 *          to HEAPOpen function.
 ****************************************************************************/
void HEAPClose(void);



/*!**************************************************************************
 * @fn      HEAPAlloc
 * @brief   Allocate memory from the previously allocated component memory
 *          region.
 *          Calls to this function may only be made after a successful call
 *          to HEAPOpen.
 * @param   mem_size - Required size of new memory block.
 * @return  Pointer to memory location.
 ****************************************************************************/
void* HEAPAlloc(U32BIT mem_size);



/*!**************************************************************************
 * @fn      HEAPCalloc
 * @brief   Allocate memory elements.
 *          Allocate memory from the previously allocated component memory
 *          region.
 *          Calls to this function may only be made after a successful call
 *          to HEAPOpen.
 * @param   mem_size - Size of each memory element.
 *          num_elems - Number of memory elements.
 * @return  Pointer to memory location.
 ****************************************************************************/
void* HEAPCalloc(U32BIT mem_size, S32BIT num_elems);



/*!**************************************************************************
 * @fn      HEAPRealloc
 * @brief   Reallocate memory from the previously allocated component memory
 *          region.
 *          Calls to this function may only be made after a successful call
 *          to HEAPOpen.
 * @param   old_addr - Address of currently allocated block
 *          new_size - Required size of new memory block.
 * @return  Pointer to memory location.
 *
 ****************************************************************************/
void* HEAPRealloc(void *old_addr, U32BIT new_size);



/*!**************************************************************************
 * @fn      HEAPFree
 * @brief   Release previously allocated memory.
 *          Calls to this function may only be made after a successful call
 *          to HEAPOpen.
 *          This function may only de-allocate memory previously allocated
 *          using HEAPAlloc or HEAPCalloc.
 * @param   data - Pointer to memory block.
 ****************************************************************************/
void HEAPFree(void *data);


#endif /* _HEAP_H */

/****************************************************************************
** End of file
*****************************************************************************/
