/*******************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Memory-related functions
 * @file    heap.c
 * @date    08 November 2006
 * @author  Omri Barel
 */


/*--- Includes ----------------------------------------------------------------*/

/* System header files */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

/* STB header files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwc.h"
#include "stbhwos.h"


/*--- Preprocessor definitions ------------------------------------------------*/

#ifndef USE_NATIVE_MALLOC
#define USE_NATIVE_MALLOC  0
#endif

#define ALIGNMENT          (8)            /* Must be power of 2 */
#define ROUND_UP(x)        (((x) + ALIGNMENT - 1) & (~(ALIGNMENT - 1)))
#define NULL_OFFSET        (0)
#define BLOCK_HEADER_SIZE  (sizeof(U32BIT))
#define MIN_BLOCK_SIZE     ROUND_UP(4 * (sizeof(U32BIT)))
#define SPLIT_THRESHOD     (2 * MIN_BLOCK_SIZE)

#define FLAT_THRESHOLD     (256)          /* Power of 2 */
#define FLAT_BLOCK_INC     (ALIGNMENT)    /* Power of 2 */
#define EXP_FACTOR         (4)            /* Power of 2 */
#define MAX_BLOCK_SIZE     (33554432)

#ifdef ASSERT
#undef ASSERT
#endif
#define ASSERT(x)          assert(x)



/*--- Local types definitions -------------------------------------------------*/

/* Memory manager status type definition */
typedef enum
{
   MEM_STATUS_CLOSED,  /* Memory manager closed */
   MEM_STATUS_OPEN     /* Memory manager opened */
} E_MEM_STATUS;


/*--- Local (static) variable declarations ------------------------------------*/

/* Memory region */
static void *s_region = NULL;
static U32BIT s_region_size = 0;

/* Memory manager information */
static U32BIT s_bin_count = 0;
static U32BIT s_header_size = 0;

/* Memory manager status */
static E_MEM_STATUS s_mem_status = MEM_STATUS_CLOSED;

/* Store of who created memory region */
static BOOLEAN s_mem_region_preallocated = FALSE;

static void *heap_mutex = NULL;


/*--- Local function prototypes -----------------------------------------------*/

static void *HeapAlloc(U32BIT size);
static void HeapFree(void *buffer);
static BOOLEAN HeapCreateTables(void);
static BOOLEAN HeapCreateHeader(void);
static void HeapSetBlockSize(U32BIT offset, U32BIT size);
static void HeapSetBlockSizeEnd(U32BIT offset, U32BIT size);
static U32BIT HeapGetBlockSize(U32BIT offset);
static U32BIT HeapGetBlockPrevSize(U32BIT offset);
static void HeapSetBlockNext(U32BIT offset, U32BIT next_offset);
static U32BIT HeapGetBlockNext(U32BIT offset);
static void HeapSetBlockPrev(U32BIT offset, U32BIT prev_offset);
static U32BIT HeapGetBlockPrev(U32BIT offset);
static void HeapSetBlockUsed(U32BIT offset, BOOLEAN used);
static BOOLEAN HeapGetBlockUsed(U32BIT offset);
static void HeapHeapSetBlockPrevUsed(U32BIT offset, BOOLEAN used);
static BOOLEAN HeapGetBlockPrevUsed(U32BIT offset);
static void *HeapGetBlockMem(U32BIT offset);
static U32BIT HeapGetBlockOffset(void *p);
static void HeapSetBinHead(U32BIT bin_index, U32BIT offset);
static U32BIT HeapGetBinHead(U32BIT bin_index);
static U32BIT HeapGetBinOffset(U32BIT bin_index);
static void HeapSetBinSize(U32BIT bin_index, U32BIT size);
static U32BIT HeapGetBinSize(U32BIT bin_index);
static U32BIT HeapGetBinIndex(U32BIT size);
static void HeapInsertBlock(U32BIT bin_index, U32BIT offset);
static void HeapRemoveBlock(U32BIT offset);


/*--- Global function definitions ---------------------------------------------*/


/*!**************************************************************************
 * @fn      HEAPOpen
 * @brief   Initialise memory manager module. Allocate component memory
 *          region. Allocates memory region from the default memory pool.
 *          This function must be called to initialise the MHEG5 memory
 *          manager. To close the memory manager a call to HEAPClose must be
 *          made. This function must not be called 'out of turn', i.e. after
 *          calling HEAPOpen a call to HEAPClose MUST be made before another
 *          call to HEAPOpen.
 * @param   memory_region - Pointer to memory region, or NULL if the region
 *          is to be allocated.
 *          memory_region_size - Size of region, or if the region is NULL,
 *          the size to be used for the allocation.
 * @return  TRUE if completed successfully, FALSE if unable to allocate
 *          memory region.
 ****************************************************************************/
BOOLEAN HEAPOpen(void *memory_region, U32BIT memory_region_size)
{
   BOOLEAN result;

   FUNCTION_START(HEAPOpen);

   if (s_mem_status == MEM_STATUS_OPEN)
   {
      /* Already open */
      result = TRUE;
   }
   else
   {

#if USE_NATIVE_MALLOC

      if (memory_region)
      {
         /* Cannot use native malloc, region already allocated */
         s_mem_region_preallocated = TRUE;

         /* Store memory region */
         s_region = memory_region;

         /* Round memory size down */
         s_region_size = memory_region_size & ~(ALIGNMENT - 1);

         /* Create memory manager's tables */
         if (HeapCreateTables())
         {
            /* Create mutex to protect heap accesses */
            heap_mutex = STB_OSCreateMutex();

            /* Success */
            s_mem_status = MEM_STATUS_OPEN;
            result = TRUE;
         }
         else
         {
            /* Not enough memory for tables */
            if (!s_mem_region_preallocated)
            {
               free(memory_region);
            }
            s_region = NULL;
            result = FALSE;
         }
      }
      else
      {
         /* Using native malloc - no need to allocate region */
         s_mem_region_preallocated = TRUE;
         s_mem_status = MEM_STATUS_OPEN;
         result = TRUE;
      }

#else  /* USE_NATIVE_MALLOC */

      if (memory_region)
      {
         /* Memory region pre-allocated */
         s_mem_region_preallocated = TRUE;
         result = TRUE;
      }
      else
      {
         /* Memory region not pre-allocated */
         memory_region = malloc(memory_region_size);
         if (memory_region)
         {
            /* Allocation succeeded */
            s_mem_region_preallocated = FALSE;
            result = TRUE;
         }
         else
         {
            /* Allocation failed */
            result = FALSE;
         }
      }  /* if (memory_region) */

      if (result)
      {
         /* Store memory region */
         s_region = memory_region;

         /* Round memory size down */
         s_region_size = memory_region_size & ~(ALIGNMENT - 1);

         /* Create memory manager's tables */
         if (HeapCreateTables())
         {
            /* Create mutex to protect heap accesses */
            heap_mutex = STB_OSCreateMutex();

            /* Success */
            s_mem_status = MEM_STATUS_OPEN;
         }
         else
         {
            /* Not enough memory for tables */
            if (!s_mem_region_preallocated)
            {
               free(memory_region);
            }
            s_region = NULL;
            result = FALSE;
         }

      }  /* if (err == MEM_OK) */

#endif  /* USE_NATIVE_MALLOC */

   }  /* if (s_mem_status == MEM_STATUS_OPEN) */

   FUNCTION_FINISH(HEAPOpen);

   return result;
}



/*!**************************************************************************
 * @fn      HEAPClose
 * @brief   Release component memory region.
 *          A call to this function MUST only be made after a successful call
 *          to HEAPOpen function.
 ****************************************************************************/
void HEAPClose(void)
{
   FUNCTION_START(HEAPClose);

   if (s_mem_status != MEM_STATUS_CLOSED)
   {
      if (!s_mem_region_preallocated)
      {
         /* Not pre-allocated, so we have to free */
         free(s_region);
      }

      if (heap_mutex != NULL)
      {
         STB_OSDeleteMutex(heap_mutex);
      }

      /* Close memory manager */
      s_region = 0;
      s_mem_status = MEM_STATUS_CLOSED;
   }


   FUNCTION_FINISH(HEAPClose);
}



/*!**************************************************************************
 * @fn      HEAPAlloc
 * @brief   Allocate memory from the previously allocated component memory
 *          region.
 *          Calls to this function may only be made after a successful call
 *          to HEAPOpen.
 * @param   mem_size - Required size of new memory block.
 * @return  Pointer to memory location.
 ****************************************************************************/
void* HEAPAlloc(U32BIT mem_size)
{
   void *data;

   FUNCTION_START(HEAPAlloc);

   if (s_mem_status == MEM_STATUS_OPEN)
   {
#if USE_NATIVE_MALLOC
      if (s_mem_region_preallocated)
      {
         data = malloc(mem_size);
      }
      else
      {
         data = HeapAlloc(mem_size);
      }
#else
      data = HeapAlloc(mem_size);
#endif
   }
   else
   {
      data = 0;
   }

   FUNCTION_FINISH(HEAPAlloc);

   return data;
}



/*!**************************************************************************
 * @fn      HEAPCalloc
 * @brief   Allocate memory elements.
 *          Allocate memory from the previously allocated component memory
 *          region.
 *          Calls to this function may only be made after a successful call
 *          to HEAPOpen.
 * @param   mem_size - Size of each memory element.
 *          num_elems - Number of memory elements.
 * @return  Pointer to memory location.
 ****************************************************************************/
void* HEAPCalloc(U32BIT mem_size, S32BIT num_elems)
{
   void *data;

   FUNCTION_START(HEAPCalloc);

   if (s_mem_status == MEM_STATUS_OPEN)
   {
#if USE_NATIVE_MALLOC
      if (s_mem_region_preallocated)
      {
         data = malloc(mem_size);
      }
      else
      {
         data = HeapAlloc(mem_size);
      }
#else
      data = HeapAlloc(mem_size * num_elems);
#endif
      if (data)
      {
         memset(data, 0, mem_size * num_elems);
      }
   }
   else
   {
      data = 0;
   }

   FUNCTION_FINISH(HEAPCalloc);

   return data;
}



/*!**************************************************************************
 * @fn      HEAPRealloc
 * @brief   Reallocate memory from the previously allocated component memory
 *          region.
 *          Calls to this function may only be made after a successful call
 *          to HEAPOpen.
 * @param   old_addr - Address of currently allocated block
 *          new_size - Required size of new memory block.
 * @return  Pointer to memory location.
 *
 ****************************************************************************/
void* HEAPRealloc(void *old_addr, U32BIT new_size)
{
   void *data;
   U32BIT offset;
   U32BIT block_size;

   FUNCTION_START(HEAPRealloc);

   if (s_mem_status == MEM_STATUS_OPEN)
   {
#if USE_NATIVE_MALLOC
      if (s_mem_region_preallocated)
      {
         data = realloc(old_addr, new_size);
      }
      else
      {
         data = HeapAlloc(new_size);
         if (data != NULL)
         {
            STB_OSMutexLock(heap_mutex);

            /* Copy data from old block to new block */
            offset = HeapGetBlockOffset(old_addr);
            block_size = HeapGetBlockSize(offset);

            STB_OSMutexUnlock(heap_mutex);

            if (new_size > block_size)
            {
               memcpy(data, old_addr, block_size);
            }
            else
            {
               memcpy(data, old_addr, new_size);
            }
         }

         /* Free the original block of memory */
         HeapFree(old_addr);
      }
#else
      data = HeapAlloc(new_size);
      if (data != NULL)
      {
         STB_OSMutexLock(heap_mutex);

         /* Copy data from old block to new block */
         offset = HeapGetBlockOffset(old_addr);
         block_size = HeapGetBlockSize(offset);

         STB_OSMutexUnlock(heap_mutex);

         if (new_size > block_size)
         {
            memcpy(data, old_addr, block_size);
         }
         else
         {
            memcpy(data, old_addr, new_size);
         }
      }

      /* Free the original block of memory */
      HeapFree(old_addr);
#endif
   }
   else
   {
      data = 0;
   }

   FUNCTION_FINISH(HEAPRealloc);

   return data;
}



/*!**************************************************************************
 * @fn      HEAPFree
 * @brief   Release previously allocated memory.
 *          Calls to this function may only be made after a successful call
 *          to HEAPOpen.
 *          This function may only de-allocate memory previously allocated
 *          using HEAPAlloc or HEAPCalloc.
 * @param   data - Pointer to memory block.
 ****************************************************************************/
void HEAPFree(void *data)
{
   FUNCTION_START(HEAPFree);

   if (s_mem_status == MEM_STATUS_OPEN)
   {
#if USE_NATIVE_MALLOC
      if (s_mem_region_preallocated)
      {
         free(data);
      }
      else
      {
         HeapFree(data);
      }
#else
      HeapFree(data);
#endif
   }

   FUNCTION_FINISH(HEAPFree);
}



/*--- Local function definitions ----------------------------------------------*/

/*!**************************************************************************
 * @fn      HeapAlloc
 * @brief   Allocates the required number of bytes from the region given in
 *          HEAPOpen. If there is not enough memory to perform this
 *          operation, or no region has been set, NULL is returned.
 * @param   size - Size for allocated buffer.
 * @return  Pointer newly allocated buffer, or NULL if buffer could not
 *          be allocated.
 ****************************************************************************/
static void *HeapAlloc(U32BIT size)
{
   BOOLEAN found;
   U32BIT reqd_size;
   U32BIT block_size;
   U32BIT offset;
   U32BIT bin_index, orig_bin_index;
   void *allocated;
   U32BIT new_block;
   U32BIT remainder;

   FUNCTION_START(HeapAlloc);

   STB_OSMutexLock(heap_mutex);

   if (s_region)
   {
      /* Calculate real size */
      size += BLOCK_HEADER_SIZE;
      reqd_size = (size + ALIGNMENT - 1) & ~(ALIGNMENT - 1);

      if (reqd_size < MIN_BLOCK_SIZE)
      {
         /* We need at least MIN_BLOCK_SIZE bytes for bookkeeping */
         reqd_size = MIN_BLOCK_SIZE;
      }

      /* Start from the minimal possible bin */
      bin_index = HeapGetBinIndex(reqd_size);
      orig_bin_index = bin_index;

      /* perform first-fit search for a large enough block */
      found = FALSE;
      while (!found && bin_index < s_bin_count)
      {
         offset = HeapGetBinHead(bin_index);
         while (offset && !found)
         {
            block_size = HeapGetBlockSize(offset);
            if (block_size >= reqd_size)
            {
               /* Found a good block */
               found = TRUE;
            }
            else
            {
               offset = HeapGetBlockNext(offset);
            }
         }
         if (!found)
         {
            ++bin_index;
         }
      }
   }
   else
   {
      /* Cannot allocate if the region is NULL */
      found = FALSE;
   }

   if (found)
   {
      /* Take block out of its bin */
      HeapRemoveBlock(offset);

      remainder = block_size - reqd_size;
      if (remainder < SPLIT_THRESHOD || bin_index == orig_bin_index)
      {
         /* Keep it as one block */
         HeapSetBlockUsed(offset, TRUE);
         HeapHeapSetBlockPrevUsed(offset + block_size, TRUE);

         /* Success */
         allocated = HeapGetBlockMem(offset);
      }
      else
      {
         /* Split into two */
         new_block = offset + reqd_size;

         HeapSetBlockSize(offset, reqd_size);
         HeapSetBlockSize(new_block, remainder);

         /* Mark allocated block as used */
         HeapSetBlockUsed(offset, TRUE);
         HeapHeapSetBlockPrevUsed(new_block, TRUE);

         /* Insert new block to its linked list */
         bin_index = HeapGetBinIndex(remainder);
         HeapInsertBlock(bin_index, new_block);

         /* Success */
         allocated = HeapGetBlockMem(offset);
      }
   }
   else
   {
      /* No free block big enough to satisfy the request */
      allocated = 0;
   }

   STB_OSMutexUnlock(heap_mutex);

   FUNCTION_FINISH(HeapAlloc);

   return allocated;
}



/*!**************************************************************************
 * @fn      HeapFree
 * @brief   Frees a previously-allocated memory buffer.
 * @param   buffer - The buffer to free.
 * @warning Passing a pointer that was not previously returned by HeapAlloc
 *          is a recipe for disaster.
 ****************************************************************************/
static void HeapFree(void *buffer)
{
   U32BIT offset;
   U32BIT block_size;
   BOOLEAN prev_used;
   BOOLEAN next_used;
   U32BIT prev_block_size;
   U32BIT prev_offset;
   U32BIT next_offset;
   U32BIT next_block_size;

   FUNCTION_START(HeapFree);

   if (s_region && buffer)
   {
      offset = HeapGetBlockOffset(buffer);
      block_size = HeapGetBlockSize(offset);
      prev_used = HeapGetBlockPrevUsed(offset);
      next_used = HeapGetBlockUsed(offset + block_size);
      U32BIT bin_index;

      STB_OSMutexLock(heap_mutex);

      if (!prev_used)
      {
         prev_block_size = HeapGetBlockPrevSize(offset);
         prev_offset = offset - prev_block_size;

         /* Remove previous block from list */
         HeapRemoveBlock(prev_offset);

         /* Merge blocks */
         offset = prev_offset;
         block_size += prev_block_size;
         HeapSetBlockSize(offset, block_size);
      }

      if (!next_used)
      {
         next_offset = offset + block_size;
         next_block_size = HeapGetBlockSize(next_offset);

         /* Remove previous next block from list */
         HeapRemoveBlock(next_offset);

         /* Merge blocks */
         block_size += next_block_size;
         HeapSetBlockSize(offset, block_size);
      }

      /* Insert merged block to its linked list */
      bin_index = HeapGetBinIndex(block_size);
      HeapInsertBlock(bin_index, offset);

      STB_OSMutexUnlock(heap_mutex);
   }

   FUNCTION_FINISH(HeapFree);
}



/*!**************************************************************************
 * @fn      HeapCreateTables
 * @brief   Create memory manager's tables
 * @return  TRUE if operation ended successfully, FALSE if operation failed
 *          (not enough memory in region for tables).
 ****************************************************************************/
static BOOLEAN HeapCreateTables(void)
{
   BOOLEAN success;
   U32BIT offset;
   U32BIT available;

   FUNCTION_START(HeapCreateTables);

   if (HeapCreateHeader())
   {
      offset = s_header_size;
      available = s_region_size - s_header_size;

      if (available >= sizeof(U32BIT))
      {
         U32BIT bin_size;
         BOOLEAN found;
         S32BIT bin_index;

         /* Last word is never used */
         available -= sizeof(U32BIT);
         available &= ~(ALIGNMENT - 1);

         /* Put region into smallest possible bin */
         found = FALSE;
         for (bin_index = s_bin_count - 1; !found && bin_index >= 0; --bin_index)
         {
            bin_size = HeapGetBinSize(bin_index);
            if (bin_size > 0 && available >= bin_size)
            {
               /* Start linked list */
               HeapSetBinHead(bin_index, offset);

               /* Set block parameters */
               HeapSetBlockSize(offset, available);
               HeapSetBlockSizeEnd(offset, available);
               HeapSetBlockPrev(offset, HeapGetBinOffset(bin_index));
               HeapSetBlockNext(offset, NULL_OFFSET);

               /* Block is unused */
               HeapSetBlockUsed(offset, FALSE);
               HeapHeapSetBlockPrevUsed(offset + available, FALSE);

               /* Previous (non-existent) block is *always* "used" */
               HeapHeapSetBlockPrevUsed(offset, TRUE);

               /* Last (non-existent) block is *always* "used" */
               HeapSetBlockUsed(offset + available, TRUE);
               found = TRUE;
            }
         }

         if (found)
         {
            /* At least one block was setup */
            success = TRUE;
         }
         else
         {
            /* Not enough space in region */
            success = FALSE;
         }
      }
      else  /* if (available >= sizeof(U32BIT)) */
      {
         /* Not enough space in region */
         success = FALSE;
      }
   }
   else  /* if (HeapCreateHeader()) */
   {
      /* Not enough space in region */
      success = FALSE;
   }

   FUNCTION_FINISH(HeapCreateTables);

   return success;
}



/*!**************************************************************************
 * @fn      HeapCreateHeader
 * @brief   Create header for memory manager's tables
 * @return  TRUE if operation ended successfully, FALSE if operation failed
 *          (not enough memory in region for tables).
 ****************************************************************************/
static BOOLEAN HeapCreateHeader(void)
{
   U32BIT bin_index;
   U32BIT i, j;
   BOOLEAN success;

   FUNCTION_START(HeapCreateHeader);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);

   /* Count number of bins required */
   s_bin_count = 0;

   /* Flat increment */
   for (i = MIN_BLOCK_SIZE; i < FLAT_THRESHOLD; i += FLAT_BLOCK_INC)
   {
      ++s_bin_count;
   }

   /* Exponential increment */
   for (i = FLAT_THRESHOLD; i < MAX_BLOCK_SIZE; i *= 2)
   {
      s_bin_count += EXP_FACTOR;
   }

   /* Last bin */
   ++s_bin_count;

   if (s_region_size >= 2 * s_bin_count * sizeof(U32BIT))
   {
      /* We have enough space for the header */
      bin_index = 0;

      /* Flat increment */
      for (i = MIN_BLOCK_SIZE; i < FLAT_THRESHOLD; i += FLAT_BLOCK_INC)
      {
         HeapSetBinSize(bin_index, i);
         HeapSetBinHead(bin_index, NULL_OFFSET);
         ++bin_index;
      }

      /* Exponential increment */
      for (i = FLAT_THRESHOLD; i < MAX_BLOCK_SIZE; i *= 2)
      {
         for (j = 0; j < EXP_FACTOR; ++j)
         {
            HeapSetBinSize(bin_index, i + j * (i / EXP_FACTOR));
            HeapSetBinHead(bin_index, NULL_OFFSET);
            ++bin_index;
         }
      }

      /* Last bin */
      HeapSetBinSize(bin_index, MAX_BLOCK_SIZE);
      HeapSetBinHead(bin_index, NULL_OFFSET);

      s_header_size = 2 * s_bin_count * sizeof(U32BIT);
      success = TRUE;
   }
   else
   {
      /* Not enough space for header */
      success = FALSE;
   }

   FUNCTION_FINISH(HeapCreateHeader);

   return success;
}



/*!**************************************************************************
 * @fn      HeapSetBlockSize
 * @brief   Set block's 'size' field
 * @param   offset - Offset of block in region.
 *          size - Size of block in bytes.
 ****************************************************************************/
static void HeapSetBlockSize(U32BIT offset, U32BIT size)
{
   FUNCTION_START(HeapSetBlockSize);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);
   *((U32BIT*)((char*) s_region + offset)) &= 0x3;
   *((U32BIT*)((char*) s_region + offset)) |= size;

   FUNCTION_FINISH(HeapSetBlockSize);
}



/*!**************************************************************************
 * @fn      HeapSetBlockSizeEnd
 * @brief   Set block's 'size' field at the end of the block.
 * @param   offset - Offset of block in region.
 *          size - Size of block in bytes.
 ****************************************************************************/
static void HeapSetBlockSizeEnd(U32BIT offset, U32BIT size)
{
   FUNCTION_START(HeapSetBlockSizeEnd);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT-1)) == 0);
   ASSERT(offset + size < s_region_size);
   *((U32BIT*)((char*) s_region + offset + size - sizeof(U32BIT))) = size;

   FUNCTION_FINISH(HeapSetBlockSizeEnd);
}



/*!**************************************************************************
 * @fn      HeapGetBlockSize
 * @brief   Return block's 'size' field.
 * @param   offset - Offset of block in region.
 * @return  Size of block in bytes.
 ****************************************************************************/
static U32BIT HeapGetBlockSize(U32BIT offset)
{
   FUNCTION_START(HeapGetBlockSize);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);

   FUNCTION_FINISH(HeapGetBlockSize);

   return *((U32BIT*)((char*) s_region + offset)) & ~0x3;
}



/*!**************************************************************************
 * @fn      HeapGetBlockPrevSize
 * @brief   Return previous block's 'size' field (from end of previous block)
 * @param   offset - Offset of block in region.
 * @return  Size of previous block in bytes.
 ****************************************************************************/
static U32BIT HeapGetBlockPrevSize(U32BIT offset)
{
   FUNCTION_START(HeapGetBlockPrevSize);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT-1)) == 0);
   ASSERT(!HeapGetBlockPrevUsed(offset));

   FUNCTION_FINISH(HeapGetBlockPrevSize);

   return *((U32BIT*)((char*) s_region + offset - sizeof(U32BIT)));
}



/*!**************************************************************************
 * @fn      HeapSetBlockNext
 * @brief   Set block's 'next' field.
 * @param   offset - Offset of block in region.
 *          next_offset - Offset of block to be assigned to 'next' field.
 ****************************************************************************/
static void HeapSetBlockNext(U32BIT offset, U32BIT next_offset)
{
   FUNCTION_START(HeapSetBlockNext);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);
   *((U32BIT*)((char*) s_region + offset + sizeof(U32BIT))) = next_offset;

   FUNCTION_FINISH(HeapSetBlockNext);
}



/*!**************************************************************************
 * @fn      HeapGetBlockNext
 * @brief   Return block's 'next' field.
 * @param   offset -Offset of block in region.
 * @return  Value stored in 'next' field.
 ****************************************************************************/
static U32BIT HeapGetBlockNext(U32BIT offset)
{
   FUNCTION_START(HeapGetBlockNext);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);

   FUNCTION_FINISH(HeapGetBlockNext);

   return *((U32BIT*)((char*) s_region + offset + sizeof(U32BIT)));
}



/*!**************************************************************************
 * @fn      HeapSetBlockPrev
 * @brief   Set block's 'prev' field.
 * @param   offset - Offset of block in region.
 *          prev_offset - Offset of block to be assigned to 'prev' field.
 ****************************************************************************/
static void HeapSetBlockPrev(U32BIT offset, U32BIT prev_offset)
{
   FUNCTION_START(HeapSetBlockPrev);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);
   *((U32BIT*)((char*) s_region + offset + 2 * sizeof(U32BIT))) = prev_offset;

   FUNCTION_FINISH(HeapSetBlockPrev);
}



/*!**************************************************************************
 * @fn      HeapGetBlockPrev
 * @brief   Return block's 'prev' field.
 * @param   offset - Offset of block in region.
 * @return  Value stored in 'prev' field.
 ****************************************************************************/
static U32BIT HeapGetBlockPrev(U32BIT offset)
{
   FUNCTION_START(HeapGetBlockPrev);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);

   FUNCTION_FINISH(HeapGetBlockPrev);

   return *((U32BIT*)((char*) s_region + offset + 2 * sizeof(U32BIT)));
}



/*!**************************************************************************
 * @fn      HeapSetBlockUsed
 * @brief   Set a flag of whether the block is in use
 * @param   offset - Offset of block in region.
 *          used - TRUE if the block is to be marked as 'used', FALSE otherwise.
 ****************************************************************************/
static void HeapSetBlockUsed(U32BIT offset, BOOLEAN used)
{
   FUNCTION_START(HeapSetBlockUsed);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);
   if (used)
   {
      *((U32BIT*)((char*) s_region + offset)) &= 0xfffffffd;
   }
   else
   {
      *((U32BIT*)((char*) s_region + offset)) |= 0x02;
   }

   FUNCTION_FINISH(HeapSetBlockUsed);
}



/*!**************************************************************************
 * @fn      HeapGetBlockUsed
 * @brief   Tell whether a block is in use.
 * @param   offset - Offset of block in region.
 * @return  TRUE if the block is in use, FALSE if the block is not in use.
 ****************************************************************************/
static BOOLEAN HeapGetBlockUsed(U32BIT offset)
{
   BOOLEAN retval;

   FUNCTION_START(HeapGetBlockUsed);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);

   retval = TRUE;
   if (*((U32BIT*)((char*) s_region + offset)) & 0x2)
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(HeapGetBlockUsed);
   return retval;
}



/*!**************************************************************************
 * @fn      HeapHeapSetBlockPrevUsed
 * @brief   Set a flag of whether the previous block is in use.
 * @param   offset - Offset of block in region.
 *          used - TRUE if the previous block is in use, FALSE otherwise.
 ****************************************************************************/
static void HeapHeapSetBlockPrevUsed(U32BIT offset, BOOLEAN used)
{
   FUNCTION_START(HeapHeapSetBlockPrevUsed);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);
   if (used)
   {
      *((U32BIT*)((char*) s_region + offset)) &= 0xfffffffe;
   }
   else
   {
      *((U32BIT*)((char*) s_region + offset)) |= 0x01;
   }

   FUNCTION_FINISH(HeapHeapSetBlockPrevUsed);
}



/*!**************************************************************************
 * @fn      HeapGetBlockPrevUsed
 * @brief   Tell whether the previous block is in use.
 * @param   offset - Offset of block in region.
 * @return  TRUE if the previous block is in use, FALSE otherwise.
 ****************************************************************************/
static BOOLEAN HeapGetBlockPrevUsed(U32BIT offset)
{
   BOOLEAN retval;

   FUNCTION_START(HeapGetBlockPrevUsed);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);

   retval = TRUE;
   if (*((U32BIT*)((char*) s_region + offset)) & 0x1)
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(HeapGetBlockPrevUsed);
   return retval;
}



/*!**************************************************************************
 * @fn      HeapGetBlockMem
 * @brief   Return a pointer to the allocated memory of a block.
 * @param   offset - Offset of block in region.
 * @return  A pointer to the allocated memory of the block.
 ****************************************************************************/
static void *HeapGetBlockMem(U32BIT offset)
{
   FUNCTION_START(HeapGetBlockMem);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);

   FUNCTION_FINISH(HeapGetBlockMem);

   return (char*) s_region + offset + sizeof(U32BIT);
}



/*!**************************************************************************
 * @fn      HeapGetBlockOffset
 * @brief   Return the block's offset given a pointer to an allocated buffer.
 * @param   buffer - A pointer to a previously-allocated buffer.
 * @return  The offset of the block that holds this buffer.
 * @warning Passing a pointer to a buffer that was not allocated by
 *          HeapAlloc is a recipe for disaster.
 ****************************************************************************/
static U32BIT HeapGetBlockOffset(void *buffer)
{
   FUNCTION_START(HeapGetBlockOffset);

   ASSERT(buffer);
   ASSERT((char*) buffer - (char*) s_region >= s_header_size);
   ASSERT((char*) buffer - (char*) s_region < s_region_size);

   FUNCTION_FINISH(HeapGetBlockOffset);

   return (char*) buffer - (char*) s_region - sizeof(U32BIT);
}



/*!**************************************************************************
 * @fn      HeapSetBinSize
 * @brief   Set the minimum size associated with a bin.
 * @param   bin_index - The index of the bin.
 *          size - The size associated with the bin (in bytes).
 ****************************************************************************/
static void HeapSetBinSize(U32BIT bin_index, U32BIT size)
{
   FUNCTION_START(HeapSetBinSize);

   ASSERT(s_region);
   ASSERT(s_region_size >= 2 * s_bin_count * sizeof(U32BIT));
   ASSERT(bin_index >= 0);
   ASSERT(bin_index < s_bin_count);
   ((U32BIT*) s_region)[bin_index * 2] = size;

   FUNCTION_FINISH(HeapSetBinSize);
}



/*!**************************************************************************
 * @fn      HeapGetBinSize
 * @brief   Return the minimum size associated with a given bin.
 * @param   bin_index - The index of the bin.
 * @return  The size associated with the bin (in bytes).
 ****************************************************************************/
static U32BIT HeapGetBinSize(U32BIT bin_index)
{
   FUNCTION_START(HeapGetBinSize);

   ASSERT(s_region);
   ASSERT(s_region_size >= 2 * s_bin_count * sizeof(U32BIT));
   ASSERT(bin_index >= 0);
   ASSERT(bin_index < s_bin_count);

   FUNCTION_FINISH(HeapGetBinSize);

   return ((U32BIT*) s_region)[bin_index * 2];
}



/*!**************************************************************************
 * @fn      HeapSetBinHead
 * @brief   Set the head of the linked list for a given bin.
 * @param   bin_index - The index of the bin.
 *          offset - The offset of the block that will be the head of the
 *          list, of NULL_OFFSET if the list is to become empty.
 ****************************************************************************/
static void HeapSetBinHead(U32BIT bin_index, U32BIT offset)
{
   FUNCTION_START(HeapSetBinHead);

   ASSERT(bin_index >= 0 && bin_index < s_bin_count);
   ((U32BIT*) s_region)[bin_index * 2 + 1] = offset;

   FUNCTION_FINISH(HeapSetBinHead);
}



/*!**************************************************************************
 * @fn      GetbinHead
 * @brief   Return the head of the linked list for a given bin.
 * @param   bin_index - The index of the bin.
 * @return  The offset of the block that is the head of the list, or
 *          NULL_OFFSET if the list is empty.
 ****************************************************************************/
static U32BIT HeapGetBinHead(U32BIT bin_index)
{
   FUNCTION_START(HeapGetBinHead);

   ASSERT(bin_index >= 0 && bin_index < s_bin_count);

   FUNCTION_FINISH(HeapGetBinHead);

   return ((U32BIT*) s_region)[bin_index * 2 + 1];
}



/*!**************************************************************************
 * @fn       HeapGetBinOffset
 * @brief    Return the offset of the bin, to be used as a dummy block.
 * @param    bin_index - The index of the bin.
 * @return   An offset that can be used for setting the 'next' field ONLY(!)
 * @warning  Do not use the return value for purpose other than setting
 *           the 'next' field (in fact the head of the linked list), using
 *           HeapSetBlockNext. Any other use will cause problems, as the offset
 *           is not a real block offset.
 ****************************************************************************/
static U32BIT HeapGetBinOffset(U32BIT bin_index)
{
   FUNCTION_START(HeapGetBinOffset);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(bin_index >= 0 && bin_index < s_bin_count);

   FUNCTION_FINISH(HeapGetBinOffset);

   return 2 * bin_index * sizeof(U32BIT);
}



/*!**************************************************************************
 * @fn       HeapGetBinIndex
 * @brief    Return the index of the bin that can hold a block of the given
 *           size.
 * @param    size -The size of the block to be stored in one of the bins.
 * @return   The index of the bin that can hold a block of the given size.
 * @warning  Calling this function with a value greater than MAX_BLOCK_SIZE
 *           will return an incorrect index.
 ****************************************************************************/
static U32BIT HeapGetBinIndex(U32BIT size)
{
   U32BIT bin_index;
   U32BIT bottom;
   U32BIT top;
   U32BIT bin_size;

   FUNCTION_START(HeapGetBinIndex);

   ASSERT(size < MAX_BLOCK_SIZE);

   bin_index = s_bin_count / 2;
   bottom = 0;
   top = s_bin_count - 1;
   bin_size = HeapGetBinSize(bin_index);

   /* Binary search */
   while (top - bottom > 1)
   {
      if (size < bin_size)
      {
         top = bin_index;
      }
      else
      {
         bottom = bin_index;
      }
      bin_index = (top + bottom) / 2;
      bin_size = HeapGetBinSize(bin_index);
   }

   ASSERT(size >= HeapGetBinSize(bin_index) && size < HeapGetBinSize(bin_index + 1));

   FUNCTION_FINISH(HeapGetBinIndex);

   return bin_index;
}



/*!**************************************************************************
 * @fn       HeapInsertBlock
 * @brief    Insert block into the linked list of the given bin.
 * @param    bin_index - The index of the bin that will hold the block.
 *           offset - Offset of the block to be stored.
 ****************************************************************************/
static void HeapInsertBlock(U32BIT bin_index, U32BIT offset)
{
   U32BIT old_offset;
   U32BIT size;

   FUNCTION_START(HeapInsertBlock);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);
   ASSERT((offset & (ALIGNMENT - 1)) == 0);
   ASSERT(bin_index >= 0 && bin_index < s_bin_count);

   /* Prepend to linked list */
   old_offset = HeapGetBinHead(bin_index);
   HeapSetBlockPrev(offset, HeapGetBinOffset(bin_index));
   HeapSetBlockNext(offset, old_offset);
   if (old_offset)
   {
      HeapSetBlockPrev(old_offset, offset);
   }
   HeapSetBinHead(bin_index, offset);

   /* Mark block as unused */
   size = HeapGetBlockSize(offset);
   HeapSetBlockUsed(offset, FALSE);
   HeapHeapSetBlockPrevUsed(offset + size, FALSE);

   /* Set 'size' field at the end */
   HeapSetBlockSizeEnd(offset, size);

   FUNCTION_FINISH(HeapInsertBlock);
}



/*!**************************************************************************
 * @fn       HeapRemoveBlock
 * @brief    Remove a given block from the linked list it is stored in.
 * @param    offset -Offset of the block to be removed from the list.
 * @warning  The block must be in the list, otherwise bad things happen.
 ****************************************************************************/
static void HeapRemoveBlock(U32BIT offset)
{
   U32BIT prev;
   U32BIT next;

   FUNCTION_START(HeapRemoveBlock);

   ASSERT(s_region);
   ASSERT(s_region_size > 0);
   ASSERT(offset >= s_header_size);
   ASSERT(offset < s_region_size);

   prev = HeapGetBlockPrev(offset);
   next = HeapGetBlockNext(offset);

   /* Connect previous block to next block */
   HeapSetBlockNext(prev, next);

   /* Connect next block to previous block */
   if (next)
   {
      HeapSetBlockPrev(next, prev);
   }

   FUNCTION_FINISH(HeapRemoveBlock);
}


#if 0
/*!**************************************************************************
 * @fn      CheckHeap
 * @note    Never used
 ****************************************************************************/
static void CheckHeap(void)
{
   U32BIT bin_index;
   U32BIT offset;

   for (bin_index = 0; bin_index < s_bin_count; bin_index++)
   {
      offset = HeapGetBinHead(bin_index);
      while (offset != 0)
      {
         if ((offset < s_header_size) || (offset >= s_region_size))
         {
            printf("  ***** HEAP corrupted, offset=%x\n", offset);
         }

         offset = HeapGetBlockNext(offset);
      }

      offset = HeapGetBinHead(bin_index);
      while (offset != 0)
      {
         if ((offset < s_header_size) || (offset >= s_region_size))
         {
            printf("  ***** HEAP corrupted, offset=%x\n", offset);
         }

         offset = HeapGetBlockPrev(offset);
      }
   }
}
#endif


/****************************************************************************
** End of file
*****************************************************************************/
