/*******************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Disk Functions
 * @file    stbhwdsk.c
 * @date    June 2008
 * @author  Steve Ford
 */


/*--- Includes ----------------------------------------------------------------*/

/* System header files */
#include <stdio.h>
#include <stdlib.h>
#include <stdio_ext.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/mount.h>
#include <sys/ioctl.h>
#include <sys/statfs.h>
#include <dirent.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>
#include <linux/hdreg.h>

/* STB header Files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwc.h"
#include "stbhwos.h"
#include "stbhwmem.h"
#include "stbhwdsk.h"

extern "C" {
#include "cldsmcc.h"
}


#ifdef system
#undef system
#endif
#define system(cmd) execl("/vendor/bin/sh", "sh", "-c", cmd, NULL)

#ifdef ioctl
#undef ioctl
#endif
#define ioctl(...) (-1)

/*--- Preprocessor definitions ------------------------------------------------*/

// #define  DISK_DEBUG

#define MOUNT_BASE_NAME    "/mnt/"
#define DISK_NAME_SIZE     64
#define DISK_PATH_SIZE     256

#undef BUILD_FOR_OBS_DISK_MOUNT

#ifndef  DISK_DEBUG
#define  DISK_DBG(X)
#else
#define  DISK_DBG(X)    STB_SPDebugWrite X
#endif



/*--- Local types definitions -------------------------------------------------*/

typedef struct s_disk_info
{
   struct s_disk_info *next;
   char dev_path[DISK_PATH_SIZE];      /* Device pathname */
   BOOLEAN is_mounted;     /* TRUE if the disk is mounted */
   BOOLEAN is_removeable;  /* TRUE if the disk isn't an internal disk */
   char mount_path[DISK_PATH_SIZE];    /* Base pathname for the mounted disk */
   U16BIT disk_id;         /* Disk id */
   char disk_name[DISK_NAME_SIZE];
   U32BIT disk_size;       /* Disk size in KB */
   BOOLEAN blocked;
   BOOLEAN found;
} S_DISK_INFO;



/*--- Local (static) variable declarations ------------------------------------*/

static S_DISK_INFO *disks;
static void *disk_sem;
static void *dsk_task_ptr = NULL;
static BOOLEAN dsk_task_enable;


/*--- Local function prototypes -----------------------------------------------*/

static void AddDisk(S_DISK_INFO *disk_info);
static void DeleteDisk(S_DISK_INFO *del_disk);
static void CheckMountedDisks(BOOLEAN send_events);
static void DiskMonitorTask(void *param);
static S_DISK_INFO* FindDisk(U16BIT disk_id);
static BOOLEAN AssignDiskName(S_DISK_INFO *disk);



/*--- Global function definitions ---------------------------------------------*/

/*!**************************************************************************
 * @fn      STB_DSKInitialise
 * @brief   Initialise the hard disk component
 ****************************************************************************/
void STB_DSKInitialise(void)
{
   FUNCTION_START(STB_DSKInitialise);

   dsk_task_enable = TRUE;
   /* Create a semaphore to control access to disk info */
   disk_sem = STB_OSCreateSemaphore();

   /* Create a task to monitor addition/removal of disks */
   dsk_task_ptr = STB_OSCreateTask(DiskMonitorTask, NULL, 4096, 7, (U8BIT*) "DiskMonitor");

   FUNCTION_FINISH(STB_DSKInitialise);
}



/*!**************************************************************************
 * @fn      STB_DSKGetNumDisks
 * @brief   Returns the number of disks currently detected
 * @return  Number of disks
 ****************************************************************************/
U16BIT STB_DSKGetNumDisks(void)
{
   U16BIT num_disks;
   S_DISK_INFO *disk;

   FUNCTION_START(STB_DSKGetNumDisks);

   num_disks = 0;
   STB_OSSemaphoreWait(disk_sem);

   for (disk = disks; disk != NULL; disk = disk->next)
   {
      num_disks++;
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKGetNumDisks);

   return num_disks;
}



/*!**************************************************************************
 * @fn      STB_DSKGetDiskIdByIndex
 * @brief   Returns the id of the disk at the given index
 * @param   index - zero based index
 * @return  Disk id, 0 if no disk found
 ****************************************************************************/
U16BIT STB_DSKGetDiskIdByIndex(U16BIT index)
{
   U16BIT disk_id;
   S_DISK_INFO *disk;

   FUNCTION_START(STB_DSKGetDiskIdByIndex);

   STB_OSSemaphoreWait(disk_sem);

   for (disk = disks; (index > 0) && (disk != NULL); index--)
   {
      disk = disk->next;
   }

   disk_id = INVALID_DISK_ID;
   if (disk != NULL)
   {
      disk_id = disk->disk_id;
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKGetDiskIdByIndex);

   return disk_id;
}



/*!**************************************************************************
 * @fn      STB_DSKIsRemoveable
 * @brief   Checks if the given disk is removeable
 * @param   disk_id - ID of the disk to be checked
 * @return  TRUE if the disk is removeable, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DSKIsRemoveable(U16BIT disk_id)
{
   BOOLEAN retval;
   S_DISK_INFO *disk;

   FUNCTION_START(STB_DSKIsRemoveable);

   STB_OSSemaphoreWait(disk_sem);

   retval = FALSE;
   disk = FindDisk(disk_id);
   if (disk != NULL)
   {
      retval = disk->is_removeable;
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKIsRemoveable);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_DSKIsMounted
 * @brief   Checks if the given disk is mounted
 * @param   disk_id - ID of the disk to be checked
 * @return  TRUE if the disk is mounted, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DSKIsMounted(U16BIT disk_id)
{
   BOOLEAN retval;
   S_DISK_INFO *disk;

   FUNCTION_START(STB_DSKIsMounted);

   STB_OSSemaphoreWait(disk_sem);

   retval = FALSE;
   disk = FindDisk(disk_id);

#ifdef BUILD_FOR_OBS_DISK_MOUNT
   if ((disk != NULL) && disk->is_mounted)
   {
      retval = TRUE;
   }
#else
   if (disk != NULL)
   {
      retval = TRUE;
   }
#endif

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKIsMounted);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_DSKMountDisk
 * @brief   Attempts to mount the given disk, if it isn't already mounted
 * @param   disk_id - ID of the disk to be mounted
 * @return  TRUE if the disk is mounted, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DSKMountDisk(U16BIT disk_id)
{
   BOOLEAN retval;

   FUNCTION_START(STB_DSKMountDisk);

   retval = FALSE;

#ifdef BUILD_FOR_OBS_DISK_MOUNT
   BOOLEAN disk_mounted;
   S_DISK_INFO *disk;
   int res;

   STB_OSSemaphoreWait(disk_sem);

   disk_mounted = FALSE;
   disk = FindDisk(disk_id);
   if (disk != NULL)
   {
      if (!disk->is_mounted)
      {
         /* Attempt to mount it as a FAT partition first */
         res = mount(disk->dev_path, disk->mount_path, "vfat", MS_SYNCHRONOUS, NULL);

         if (res == 0)
         {
            /* Mount has been successful */
            disk->is_mounted = TRUE;
            disk->blocked = FALSE;
            disk_mounted = TRUE;

            DISK_DBG(("Mounted %s", disk->mount_path));
         }
      }

      if (disk->is_mounted)
      {
         retval = TRUE;
      }
   }

   STB_OSSemaphoreSignal(disk_sem);

   if (disk_mounted)
   {
      /* Send an event to indicate the disk is available */
      STB_OSSendEvent(FALSE, HW_EV_CLASS_DISK, HW_EV_TYPE_DISK_CONNECTED, NULL, 0);
   }
#endif

   FUNCTION_FINISH(STB_DSKMountDisk);
   return retval;
}



/*!**************************************************************************
 * @fn      STB_DSKUnmountDisk
 * @brief   Attempts to unmount the given disk, if it isn't already unmounted
 * @param   disk_id - ID of the disk to be unmounted
 * @return  TRUE if the disk is unmounted, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DSKUnmountDisk(U16BIT disk_id)
{
   BOOLEAN retval;

   FUNCTION_START(STB_DSKUnmountDisk);

   retval = FALSE;

#ifdef BUILD_FOR_OBS_DISK_MOUNT
   BOOLEAN disk_unmounted;
   S_DISK_INFO *disk;
   int res;

   STB_OSSemaphoreWait(disk_sem);

   disk_unmounted = FALSE;
   disk = FindDisk(disk_id);
   if (disk != NULL)
   {
      if (disk->is_mounted)
      {
         res = umount(disk->mount_path);
         if (res == 0)
         {
            /* Unmount has been successful */
            disk->is_mounted = FALSE;

            /* To prevent the background task from automatically
             * remounting the disk, set it to blocked */
            disk->blocked = TRUE;

            disk_unmounted = TRUE;

            DISK_DBG(("Unmounted %s", disk->mount_path));
         }
      }

      if (!disk->is_mounted)
      {
         retval = TRUE;
      }
   }

   STB_OSSemaphoreSignal(disk_sem);

   if (disk_unmounted)
   {
      /* Send an event to indicate the disk is no longer available */
      STB_OSSendEvent(FALSE, HW_EV_CLASS_DISK, HW_EV_TYPE_DISK_REMOVED, NULL, 0);
   }
#endif

   FUNCTION_FINISH(STB_DSKUnmountDisk);
   return retval;
}



/*!**************************************************************************
 * @fn      STB_DSKGetDiskName
 * @brief   Gets the name of a disk and copies it into the array provided
 * @param   disk_id - ID of the disk
 * @param   name - array of U8BIT into which the name will be copied
 * @param   name_len - max number of characters in the name
 * @return  TRUE if the disk is found, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DSKGetDiskName(U16BIT disk_id, U8BIT *name, U16BIT name_len)
{
   BOOLEAN disk_found;
   S_DISK_INFO *disk;

   FUNCTION_START(STB_DSKGetDiskName);

   STB_OSSemaphoreWait(disk_sem);

   disk_found = FALSE;
   disk = FindDisk(disk_id);
   if (disk != NULL)
   {
      if (strlen(disk->disk_name) > name_len)
      {
         strncpy((char*) name, disk->disk_name, name_len - 1);
         name[name_len - 1] = '\0';
      }
      else
      {
         strcpy((char*) name, disk->disk_name);
      }

      disk_found = TRUE;
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKGetDiskName);

   return disk_found;
}



/*!**************************************************************************
 * @fn      STB_DSKSetStandby
 * @brief   Put all disks into or out of standby mode
 * @param   state - standby mode, TRUE=Standby FALSE=On
 ****************************************************************************/
void STB_DSKSetStandby(BOOLEAN state)
{
   FUNCTION_START(STB_DSKSetStandby);
   USE_UNWANTED_PARAM(state);
   FUNCTION_FINISH(STB_DSKSetStandby);
}



/*!**************************************************************************
 * @fn      STB_DSKOpenFile
 * @brief   Opens an existing file or creates a new one
 * @param   name - The filename (including path)
 * @param   mode - The access mode
 * @return  The file handle
 ****************************************************************************/
void* STB_DSKOpenFile(U16BIT disk_id, U8BIT *name, E_STB_DSK_FILE_MODE mode)
{
   FILE *file;
   S_DISK_INFO *disk;
   char *fullpath;

   FUNCTION_START(STB_DSKOpenFile);

   ASSERT(name != NULL);
   file = NULL;

   STB_OSSemaphoreWait(disk_sem);

   disk = FindDisk(disk_id);
   if ((disk != NULL) && disk->is_mounted)
   {
      fullpath = (char *)STB_MEMGetSysRAM(strlen(disk->mount_path) + strlen((char*) name) + 2);
      if (fullpath != NULL)
      {
         sprintf(fullpath, "%s/%s", disk->mount_path, name);
         DISK_DBG(("STB_DSKOpenFile: Opening %s", fullpath));

         switch (mode)
         {
            case FILE_MODE_READ :
               file = fopen((const char*) fullpath, "rb");
               break;

            case FILE_MODE_WRITE :
               file = fopen((const char*) fullpath, "rb+");
               break;

            case FILE_MODE_OVERWRITE :
               file = fopen((const char*) fullpath, "wb");
               break;

            default :
               DISK_DBG(("STB_DSKOpenFile(%s): Unknown file mode %u", name, mode));
               file = NULL;
               break;
         }

#ifdef DISK_DEBUG
         if (file != NULL)
         {
            DISK_DBG(("STB_DSKOpenFile: returning %p", file));
         }
#endif

         STB_MEMFreeSysRAM(fullpath);
      }
   }
#ifdef DISK_DEBUG
   else
   {
      DISK_DBG(("STB_DSKOpenFile: Failed to find disk with ID 0x%x", disk_id));
   }
#endif

#ifdef DISK_DEBUG
   if (file == NULL)
   {
      DISK_DBG(("Failed to open \"%s\", error %d", name, errno));
   }
#endif

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKOpenFile);

   return (void*) file;
}



/*!**************************************************************************
 * @fn      STB_DSKReadFile
 * @brief   Reads data from an open file
 * @param   file - The file handle
 * @param   data - The caller's buffer
 * @param   size - Number of bytes to be read
 * @return  Number of bytes successfully read
 ****************************************************************************/
U32BIT STB_DSKReadFile(void *file, U8BIT *data, U32BIT size)
{
   U32BIT ret_val;

   FUNCTION_START(STB_DSKReadFile);

   ASSERT(file != NULL);
   ASSERT(data != NULL);

   if (feof((FILE*) file))
   {
      /* Attempting to read from the file again, so clear the eof condition as the file may
       * now have had more data written into it */
      clearerr((FILE*) file);
   }

   ret_val = (U32BIT) fread((void*) data, 1, (size_t) size, (FILE*) file);

   FUNCTION_FINISH(STB_DSKReadFile);

   return ret_val;
}



/*!**************************************************************************
 * @fn      STB_DSKWriteFile
 * @brief   Writes data to an open file
 * @param   file - The file handle
 * @param   data - Pointer to the data to be written
 * @param   size - Number of bytes to  write
 * @return  Number of bytes successfully written
 ****************************************************************************/
U32BIT STB_DSKWriteFile(void *file, U8BIT *data, U32BIT size)
{
   U32BIT ret_val;

   FUNCTION_START(STB_DSKWriteFile);

   ASSERT(file != NULL);
   ASSERT(data != NULL);

   ret_val = (U32BIT) fwrite((void*) data, 1, (size_t) size, (FILE*) file);

   FUNCTION_FINISH(STB_DSKWriteFile);

   return ret_val;
}



/*!**************************************************************************
 * @fn      STB_DSKSeekFile
 * @brief   Sets the read/write position of an open file
 * @param   file - The file handle
 * @param   position - Position to move relative to, i.e. start, end, current
 * @param   offset - Where to move to relative to position
 * @retval  TRUE success
 * @retval  FALSE error
 ****************************************************************************/
BOOLEAN STB_DSKSeekFile(void *file, E_STB_DSK_FILE_POSITION position, S32BIT offset)
{
   BOOLEAN ret_val;

   FUNCTION_START(STB_DSKSeekFile);

   ASSERT(file != NULL);

   ret_val = FALSE;
   switch (position)
   {
      case FILE_POSITION_START:
      {
         if (fseek((FILE*) file, offset, SEEK_SET) == 0)
         {
            ret_val = TRUE;
         }
         break;
      }

      case FILE_POSITION_END:
      {
         if (fseek((FILE*) file, offset, SEEK_END) == 0)
         {
            ret_val = TRUE;
         }
         break;
      }

      case FILE_POSITION_CURRENT:
      {
         if (offset == 0L || (fseek((FILE*) file, offset, SEEK_CUR) == 0))
         {
            ret_val = TRUE;
         }
         break;
      }

      default:
      {
         /* If we have gone off the end of the file, clear the
          * error condition and reposition to exact EOF */
         if (feof((FILE*) file))
         {
            clearerr((FILE*) file);
            fseek((FILE*) file, 0, SEEK_END);
         }
         break;
      }
   }

   FUNCTION_FINISH(STB_DSKSeekFile);

   return ret_val;
}



/*!**************************************************************************
 * @fn      STB_DSKTellFile
 * @brief   Gets the current position in an open file
 * @param   file - The file handle
 * @param   offset - Variable to contain result (byte position in file)
 * @retval  TRUE on success
 ****************************************************************************/
BOOLEAN STB_DSKTellFile(void *file, U32BIT *offset)
{
   FUNCTION_START(STB_DSKTellFile);
   ASSERT(file != NULL);
   *offset = (U32BIT) ftell((FILE *)file);
   FUNCTION_FINISH(STB_DSKTellFile);
   return TRUE;
}



/*!**************************************************************************
 * @fn      STB_DSKCloseFile
 * @brief   Flushes and closes an open file
 * @param   file - The file handle
 ****************************************************************************/
void STB_DSKCloseFile(void *file)
{
   FILE *fp;

   FUNCTION_START(STB_DSKCloseFile);

   ASSERT(file != NULL);

   DISK_DBG(("STB_DSKCloseFile(%p)", file));

   fp = (FILE*) file;

   if (__fwritable(fp))
   {
      fflush(fp);
      fsync(fileno(fp));
   }

   fclose(fp);

   FUNCTION_FINISH(STB_DSKCloseFile);
}



/*!**************************************************************************
 * @fn      STB_DSKDeleteFile
 * @brief   Deletes the file from the given disk
 * @param   disk_id - disk ID
 * @param   filename - pathname on the disk of the file to be deleted
 * @retval  TRUE if successful, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DSKDeleteFile(U16BIT disk_id, U8BIT *filename)
{
   BOOLEAN retval;
   S_DISK_INFO *disk;
   char *fullpath;

   FUNCTION_START(STB_DSKDeleteFile);

   STB_OSSemaphoreWait(disk_sem);

   retval = FALSE;
   disk = FindDisk(disk_id);
   if ((disk != NULL) && disk->is_mounted)
   {
      fullpath = (char *)STB_MEMGetSysRAM(strlen(disk->mount_path) + strlen((char*) filename) + 2);
      if (fullpath != NULL)
      {
         sprintf(fullpath, "%s/%s", disk->mount_path, filename);
         if (unlink(fullpath) == 0)
         {
            retval = TRUE;
         }
#ifdef DISK_DEBUG
         else
         {
            DISK_DBG(("STB_DSKDeleteFile: failed to delete \"%s\", errno=0x%x", fullpath, errno));
         }
#endif

         STB_MEMFreeSysRAM(fullpath);
      }
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKDeleteFile);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_DSKFileExists
 * @brief   Checks whether a file/directory will the given name exists
 * @param   disk_id - disk ID
 * @param   filename - pathname on the disk of the file
 * @retval  TRUE if exists, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DSKFileExists(U16BIT disk_id, U8BIT *filename)
{
   BOOLEAN retval;
   S_DISK_INFO *disk;
   char *fullpath;
   struct stat statbuf;

   FUNCTION_START(STB_DSKFileExists);

   retval = FALSE;
   STB_OSSemaphoreWait(disk_sem);

   disk = FindDisk(disk_id);
   if ((disk != NULL) && disk->is_mounted)
   {
      fullpath = (char *)STB_MEMGetSysRAM(strlen(disk->mount_path) + strlen((char*) filename) + 2);
      if (fullpath != NULL)
      {
         sprintf(fullpath, "%s/%s", disk->mount_path, filename);

         if (stat(fullpath, &statbuf) == 0)
         {
            retval = TRUE;
         }

         STB_MEMFreeSysRAM(fullpath);
      }
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKFileExists);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_DSKFileSize
 * @brief   Returns the size in KB of the given file
 * @param   disk_id - disk on which the file exists
 * @param   filename - name of the file on disk
 * @param   filesize - returned value giving the file size in KB
 * @return  TRUE if the file exists, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DSKFileSize(U16BIT disk_id, U8BIT *filename, U32BIT *filesize)
{
   BOOLEAN result;
   S_DISK_INFO *disk;
   char *fullpath;
   struct stat statbuf;

   FUNCTION_START(STB_DSKFileSize);

   STB_OSSemaphoreWait(disk_sem);

   result = FALSE;
   disk = FindDisk(disk_id);
   if ((disk != NULL) && disk->is_mounted)
   {
      fullpath = (char *)STB_MEMGetSysRAM(strlen(disk->mount_path) + strlen((char*) filename) + 2);
      if (fullpath != NULL)
      {
         sprintf(fullpath, "%s/%s", disk->mount_path, filename);

         if (stat(fullpath, &statbuf) == 0)
         {
            *filesize = statbuf.st_size / 1024;
            result = TRUE;
         }
#ifdef DISK_DEBUG
         else
         {
            DISK_DBG(("STB_DSKFileSize: Failed to stat \"%s\", errno=%d", fullpath, errno));
         }
#endif

         STB_MEMFreeSysRAM(fullpath);
      }
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKFileSize);

   return result;
}



/*!**************************************************************************
 * @fn      STB_DSKOpenDirectory
 * @brief   Opens a directory in order to read the entries
 * @param   disk_id - disk containing to the directory to be read
 * @param   dir_name - name of the directory to open
 * @return  Handle to be used in all other operations, NULL if the open fails
 ****************************************************************************/
void* STB_DSKOpenDirectory(U16BIT disk_id, U8BIT *dir_name)
{
   U8BIT *fullpath;
   U16BIT path_len;
   S_DISK_INFO *disk;
   DIR *dir;

   FUNCTION_START(STB_DSKOpenDirectory);

   STB_OSSemaphoreWait(disk_sem);

   dir = NULL;
   /* Get the full pathname of the directory to be opened */
   disk = FindDisk(disk_id);
   if (disk != NULL)
   {
      path_len = strlen((char*) dir_name) + strlen(disk->mount_path) + 3;
      fullpath = (U8BIT*) STB_MEMGetSysRAM(path_len);

      STB_OSSemaphoreSignal(disk_sem);

      if (fullpath != NULL)
      {
         if (STB_DSKFullPathname(disk_id, dir_name, fullpath, path_len))
         {
            dir = opendir((char*) fullpath);
#ifdef DISK_DEBUG
            if (dir == NULL)
            {
               DISK_DBG(("STB_DSKOpenDirectory(0x%x): Failed to open directory \"%s\", errno %d",
                         disk_id, fullpath, errno));
            }
#endif
         }

         STB_MEMFreeSysRAM(fullpath);
      }
   }
   else
   {
      STB_OSSemaphoreSignal(disk_sem);
   }

   FUNCTION_FINISH(STB_DSKOpenDirectory);

   return (void*) dir;
}



/*!**************************************************************************
 * @fn      STB_DSKReadDirectory
 * @brief   Reads the next entry from the directory, returning the name of the
 *          entry and the type of the entry
 * @param   dir - handle returned when the directory was opened
 * @param   filename - array in which the name is returned
 * @param   filename_len - size of the filename array
 * @param   entry_type - type of entry
 * @return  TRUE if and entry is read, FALSE otherwise which could indicate end of the directory
 ****************************************************************************/
BOOLEAN STB_DSKReadDirectory(void *dir, U8BIT *filename, U16BIT filename_len,
                             E_STB_DIR_ENTRY_TYPE *entry_type)
{
   BOOLEAN result;
   struct dirent *entry;

   FUNCTION_START(STB_DSKReadDirectory);

   result = FALSE;
   if (dir != NULL)
   {
      entry = readdir((DIR*) dir);
      if (entry != NULL)
      {
         strncpy((char*) filename, entry->d_name, filename_len - 1);
         filename[filename_len - 1] = '\0';

         switch (entry->d_type)
         {
            case 0:
            case 8:
               *entry_type = DIR_ENTRY_FILE;
               break;

            case 1:
               *entry_type = DIR_ENTRY_DIRECTORY;
               break;

            default:
               *entry_type = DIR_ENTRY_OTHER;
               break;
         }

         result = TRUE;
      }
   }

   FUNCTION_FINISH(STB_DSKReadDirectory);

   return result;
}



/*!**************************************************************************
 * @fn      STB_DSKCloseDirectory
 * @brief   Closes the directory for reading
 * @param   dir - directory handle
 ****************************************************************************/
void STB_DSKCloseDirectory(void *dir)
{
   FUNCTION_START(STB_DSKCloseDirectory);

   if (dir != NULL)
   {
      closedir((DIR*) dir);
   }

   FUNCTION_FINISH(STB_DSKCloseDirectory);
}



/*!**************************************************************************
 * @fn      STB_DSKCreateDirectory
 * @brief   Creates a directory with the given name
 * @param   disk_id - disk on which the directory is to be created
 * @param   dir_name - name of the directory to be created
 * @return  TRUE is the directory is successfully created
 ****************************************************************************/
BOOLEAN STB_DSKCreateDirectory(U16BIT disk_id, U8BIT *dir_path)
{
   S_DISK_INFO *disk;
   char *pathname;
   char *dirname;
   struct stat statbuf;
   BOOLEAN result;

   FUNCTION_START(STB_DSKCreateDirectory);

   STB_OSSemaphoreWait(disk_sem);
   result = FALSE;

   disk = FindDisk(disk_id);
   if ((disk != NULL) && disk->is_mounted)
   {
      pathname = (char*) STB_MEMGetSysRAM(strlen(disk->mount_path) + strlen((char*) dir_path) + 2);
      if (pathname != NULL)
      {
         result = TRUE;
         dirname = (char*) dir_path;

         /* Create all directories in the path upto the final name */
         while ((dirname = strchr(dirname, '/')) != NULL)
         {
            /* Terminate the directory path at the next slash char */
            *dirname = '\0';

            sprintf(pathname, "%s/%s", disk->mount_path, dir_path);

            /* Check whether the directory already exists */
            if (stat((char*) pathname, &statbuf) != 0)
            {
               /* Create the directory up to this point */
               if (mkdir((char*) pathname, (S_IRWXU | S_IRWXG | S_IRWXO)) != 0)
               {
                  /* Restore the slash char */
                  *dirname = '/';
                  result = FALSE;
                  break;
               }
            }

            /* Restore the slash char and move to the next char to start search for the next one */
            *dirname = '/';
            dirname++;
         }

         if (result)
         {
            sprintf(pathname, "%s/%s", disk->mount_path, dir_path);

            /* Check whether the directory already exists */
            if (stat((char*) pathname, &statbuf) != 0)
            {
               /* Now create the final directory in the path */
               if (mkdir((char*) pathname, (S_IRWXU | S_IRWXG | S_IRWXO)) != 0)
               {
                  result = FALSE;
               }
            }
         }

         STB_OSSemaphoreSignal(disk_sem);

         STB_MEMFreeSysRAM(pathname);
      }
      else
      {
         STB_OSSemaphoreSignal(disk_sem);
      }
   }
   else
   {
      STB_OSSemaphoreSignal(disk_sem);
   }

   FUNCTION_FINISH(STB_DSKCreateDirectory);

   return result;
}



/*!**************************************************************************
 * @fn      STB_DSKDeleteDirectory
 * @brief   Deletes a directory and all it contents, so use with care!
 * @param   disk_id - disk
 * @param   dir_name - name of the directory to be deleted
 ****************************************************************************/
BOOLEAN STB_DSKDeleteDirectory(U16BIT disk_id, U8BIT *dir_name)
{
   S_DISK_INFO *disk;
   char *dir_path;
   char *pathname;
   DIR *dp;
   struct dirent *entry;
   struct stat statbuf;
   BOOLEAN result;

   FUNCTION_START(STB_DSKDeleteDirectory);

   DISK_DBG(("STB_DSKDeleteDirectory(0x%x, \"%s\")", disk_id, dir_name));

   STB_OSSemaphoreWait(disk_sem);
   result = TRUE;

   disk = FindDisk(disk_id);
   if ((disk != NULL) && disk->is_mounted)
   {
      dir_path = (char*) STB_MEMGetSysRAM(strlen(disk->mount_path) + strlen((char*) dir_name) + 2);
      if (dir_path != NULL)
      {
         sprintf(dir_path, "%s/%s", disk->mount_path, dir_name);

         STB_OSSemaphoreSignal(disk_sem);

         dp = opendir((char*) dir_path);
         if (dp != NULL)
         {
            while ((entry = readdir(dp)) != NULL)
            {
               if ((strcmp(entry->d_name, ".") != 0) && (strcmp(entry->d_name, "..") != 0))
               {
                  pathname = (char*) STB_MEMGetSysRAM(strlen((char*) dir_path) +
                                                      strlen(entry->d_name) + 2);
                  if (pathname != NULL)
                  {
                     sprintf(pathname, "%s/%s", (char*) dir_path, entry->d_name);

                     if (stat(pathname, &statbuf) == 0)
                     {
                        if (S_ISDIR(statbuf.st_mode))
                        {
                           DISK_DBG(("Recursing into \"%s\"...", pathname));
                           /* Recurse down into this directory to delete its contents */
                           if (!STB_DSKDeleteDirectory(disk_id, (U8BIT*) pathname))
                           {
                              result = FALSE;
                           }
                        }
                        else
                        {
                           DISK_DBG(("Deleting \"%s\"", pathname));
                           if (unlink(pathname) != 0)
                           {
                              /* Failed to remove an item but continue
                               * trying to delete the rest */
                              result = FALSE;
                              DISK_DBG(("STB_DSKDeleteDirectory: Failed to delete "
                                        "\"%s\", errno %d", pathname, errno));
                           }
                        }
                     }
#ifdef DISK_DEBUG
                     else
                     {
                        DISK_DBG(("STB_DSKDeleteDirectory: Failed to stat \"%s\", "
                                  "so not deleted, errno %d", pathname, errno));
                     }
#endif

                     STB_MEMFreeSysRAM(pathname);
                  }
               }
            }

            closedir(dp);

            if (rmdir((char*) dir_path) != 0)
            {
               result = FALSE;
               DISK_DBG(("STB_DSKDeleteDirectory: Failed to delete directory \"%s\", errno %d",
                         dir_path, errno));
            }

            sync();
         }

         STB_MEMFreeSysRAM(dir_path);
      }
      else
      {
         STB_OSSemaphoreSignal(disk_sem);
      }
   }
   else
   {
      STB_OSSemaphoreSignal(disk_sem);
   }

   FUNCTION_FINISH(STB_DSKDeleteDirectory);

   return result;
}



/*!**************************************************************************
 * @fn      STB_DSKFullPathname
 * @brief   Copies the full pathname for the given filename, including the
 *          mount directory, to the given string array
 * @param   disk_id - disk
 * @param   filename - name of the file on the disk
 * @param   pathname - array into which the full pathname will be copied
 * @param   max_pathname_len - size of the pathname array
 * @return  TRUE if the disk is valid exists and is mounted and the destination
 *          string is long enough to take the full pathname
 ****************************************************************************/
BOOLEAN STB_DSKFullPathname(U16BIT disk_id, U8BIT *filename,
                            U8BIT *pathname, U16BIT max_pathname_len)
{
   BOOLEAN result;
   S_DISK_INFO *disk;

   FUNCTION_START(STB_DSKFullPathname);

   STB_OSSemaphoreWait(disk_sem);

   result = FALSE;
   disk = FindDisk(disk_id);
   if ((disk != NULL) && disk->is_mounted)
   {
      if (strlen((char*) filename) + strlen(disk->mount_path) + 2 <= max_pathname_len)
      {
         sprintf((char*) pathname, "%s/%s", disk->mount_path, filename);
         result = TRUE;
      }
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKFullPathname);

   return result;
}



/*!**************************************************************************
 * @fn      STB_DSKGetSize
 * @brief   Returns the size (capacity) of the disk
 * @return  The size of the disk in kilobytes
 ****************************************************************************/
U32BIT STB_DSKGetSize(U16BIT disk_id)
{
   U32BIT disk_size;
   S_DISK_INFO *disk;

   FUNCTION_START(STB_DSKGetSize);

   STB_OSSemaphoreWait(disk_sem);

   disk_size = 0;
   disk = FindDisk(disk_id);
   if ((disk != NULL) && disk->is_mounted)
   {
      disk_size = disk->disk_size;
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKGetSize);

   return disk_size;
}



/*!**************************************************************************
 * @fn      STB_DSKGetUsed
 * @brief   Returns the amount of space used on the disk
 * @param   disk_id - ID of the disk
 * @return  Space used in kilobytes
 ****************************************************************************/
U32BIT STB_DSKGetUsed(U16BIT disk_id)
{
   U32BIT disk_used;
   S_DISK_INFO *disk;
   struct statfs fs_info;
   U32BIT blocks_used;

   FUNCTION_START(STB_DSKGetUsed);

   STB_OSSemaphoreWait(disk_sem);

   disk_used = 0;
   disk = FindDisk(disk_id);
   if ((disk != NULL) && disk->is_mounted)
   {
      /* Found the disk */
      if (statfs(disk->mount_path, &fs_info) == 0)
      {
         /* Calculate the disk space used in KB */
         blocks_used = fs_info.f_blocks - fs_info.f_bavail;

         if ((fs_info.f_bsize > 1024) && ((fs_info.f_bsize % 1024) == 0))
         {
            disk_used = (fs_info.f_bsize / 1024) * blocks_used;
         }
         else
         {
            disk_used = (fs_info.f_bsize * blocks_used) / 1024;
         }
      }
#ifdef DISK_DEBUG
      else
      {
         DISK_DBG(("STB_DSKGetUsed(0x%x): Failed to get info on the disk, errno=%u",
                   disk_id, errno));
      }
#endif
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKGetUsed);

   return disk_used;
}



/*!**************************************************************************
 * @fn      STB_DSKFormat
 * @brief    Initiates formatting and partitioning of the hard disk
 * @warning  This will erase all data on the disk!
 ****************************************************************************/
void STB_DSKFormat(U16BIT disk_id)
{
   FUNCTION_START(STB_DSKFormat);
   USE_UNWANTED_PARAM(disk_id);
   FUNCTION_FINISH(STB_DSKFormat);
}



/*!**************************************************************************
 * @fn      STB_DSKGetFormatProgress
 * @brief    Gets the progress of the format operation
 * @return   The progress as percent complete
 ****************************************************************************/
U8BIT STB_DSKGetFormatProgress(U16BIT disk_id)
{
   FUNCTION_START(STB_DSKGetFormatProgress);
   USE_UNWANTED_PARAM(disk_id);
   FUNCTION_FINISH(STB_DSKGetFormatProgress);
   return 100;
}



/*!**************************************************************************
 * @fn      STB_DSKIsFormatted
 * @brief   Queries whether the disk is formatted
 * @retval  FALSE unformatted
 ****************************************************************************/
BOOLEAN STB_DSKIsFormatted(U16BIT disk_id)
{
   BOOLEAN is_formatted;
   S_DISK_INFO *disk;

   FUNCTION_START(STB_DSKIsFormatted);

   STB_OSSemaphoreWait(disk_sem);

   is_formatted = FALSE;
   disk = FindDisk(disk_id);
   if ((disk != NULL) && disk->is_mounted)
   {
      /* Disk is formatted if it's mounted */
      is_formatted = TRUE;
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKIsFormatted);

   return is_formatted;
}



/*!**************************************************************************
 * @fn      STB_DSKRepair
 * @brief    Initiates a data repair of the hard disk
 * @warning  This *may* cause the disk to become unreadable
 ****************************************************************************/
void STB_DSKRepair(U16BIT disk_id)
{
   FUNCTION_START(STB_DSKRepair);
   USE_UNWANTED_PARAM(disk_id);

   STB_OSSendEvent(FALSE, HW_EV_CLASS_DISK, HW_EV_TYPE_REPAIR_COMPLETE, NULL, 0);

   FUNCTION_FINISH(STB_DSKRepair);
}



/*!**************************************************************************
 * @fn      STB_DSKGetRepairProgress
 * @brief   Gets the progress of the repair operation
 * @return  The progress as percent complete
 ****************************************************************************/
U8BIT STB_DSKGetRepairProgress(U16BIT disk_id)
{
   FUNCTION_START(STB_DSKGetRepairProgress);
   USE_UNWANTED_PARAM(disk_id);
   FUNCTION_FINISH(STB_DSKGetRepairProgress);
   return 100;
}



/*!**************************************************************************
 * @fn      STB_DSKGetIntegrity
 * @brief    Returns a summary of the disk integrity
 * @retval   TRUE if disk integrity ok
 ****************************************************************************/
BOOLEAN STB_DSKGetIntegrity(U16BIT disk_id)
{
   FUNCTION_START(STB_DSKGetIntegrity);
   USE_UNWANTED_PARAM(disk_id);
   FUNCTION_FINISH(STB_DSKGetIntegrity);
   return TRUE;
}




/*--- Not declared in stbhwdsk.h ----------------------------------------------*/

/*!**************************************************************************
 * @fn      DSK_Terminate
 * @brief   Finalize the hard disk component
 * @note    Declared in internal.h
 ****************************************************************************/
void DSK_Terminate(void)
{
   FUNCTION_START(DSK_Terminate);

   dsk_task_enable = FALSE;
   /* Delete a semaphore to control access to disk info */
   STB_OSDeleteSemaphore(disk_sem);

   /* Destroy a task to monitor addition/removal of disks */
   STB_OSDestroyTask(dsk_task_ptr);

   FUNCTION_FINISH(DSK_Terminate);
}



/*!**************************************************************************
 * @fn      DSK_GetDiskIdByDevPath
 * @brief   Returns the id of the disk at the given index
 * @param   disk_name - disk name
 * @return  Disk id, 0 if no disk found
 * @note    Declared in internal.h, never called
 ****************************************************************************/
U16BIT DSK_GetDiskIdByDevPath(char *dev_path)
{
   U16BIT disk_id;
   S_DISK_INFO *disk;

   FUNCTION_START(STB_DSKGetDiskIdByIndex);

   STB_OSSemaphoreWait(disk_sem);

   disk_id = INVALID_DISK_ID;
   for (disk = disks; disk != NULL; disk = disk->next)
   {
      if (strcmp(disk->dev_path, dev_path) == 0)
      {
         disk_id = disk->disk_id;
         break;
      }
   }

   STB_OSSemaphoreSignal(disk_sem);

   FUNCTION_FINISH(STB_DSKGetDiskIdByIndex);

   return disk_id;
}



/*!**************************************************************************
 * @fn      DSK_GetFree
 * @brief   Returns the amount of space used on the disk
 * @param   disk_id - ID of the disk
 * @return  Space used in kilobytes
 * @note    Declared in internal.h
 ****************************************************************************/
U32BIT DSK_GetFree(U16BIT disk_id)
{
   U32BIT disk_free;
   S_DISK_INFO *disk;
   struct statfs fs_info;

   FUNCTION_START(STB_DSKGetUsed);

   STB_OSSemaphoreWait(disk_sem);

   disk_free = 0;
   disk = FindDisk(disk_id);
   if ((disk != NULL) && disk->is_mounted)
   {
      /* Found the disk */
      if (statfs(disk->mount_path, &fs_info) == 0)
      {
         if ((fs_info.f_bsize > 1024) && ((fs_info.f_bsize % 1024) == 0))
         {
            disk_free = (fs_info.f_bsize / 1024) * fs_info.f_bavail;
         }
         else
         {
            disk_free = (fs_info.f_bsize * fs_info.f_bavail) / 1024;
         }
      }
#ifdef DISK_DEBUG
      else
      {
         DISK_DBG(("STB_DSKGetUsed(0x%x): Failed to get info on the disk, errno=%u",
                   disk_id, errno));
      }
#endif
   }

   STB_OSSemaphoreSignal(disk_sem);

#ifdef DISK_DEBUG
   DISK_DBG(("[DSK_GetFree]disk_free = 0x%x(%ld)", disk_free, disk_free));
#endif

   FUNCTION_FINISH(STB_DSKGetUsed);
   return disk_free;
}





/*--- Local function definitions ----------------------------------------------*/

/*!**************************************************************************
 * @fn      AddDisk
 ****************************************************************************/
static void AddDisk(S_DISK_INFO *disk_info)
{
   S_DISK_INFO *ptr;

   FUNCTION_START(AddDisk);
   disk_info->next = NULL;

   if (disks == NULL)
   {
      disks = disk_info;
   }
   else
   {
      ptr = disks;

      while (ptr->next != NULL)
      {
         ptr = ptr->next;
      }

      ptr->next = disk_info;
   }
   FUNCTION_FINISH(AddDisk);
}



/*!**************************************************************************
 * @fn      DeleteDisk
 ****************************************************************************/
static void DeleteDisk(S_DISK_INFO *del_disk)
{
   S_DISK_INFO *prev;
   S_DISK_INFO *disk;

   FUNCTION_START(DeleteDisk);
   prev = NULL;

   for (disk = disks; disk != del_disk;)
   {
      prev = disk;
      disk = disk->next;
   }

   if (disk != NULL)
   {
      if (prev == NULL)
      {
         /* First disk in the list is being deleted */
         disks = disk->next;
      }
      else
      {
         prev->next = disk->next;
      }

      STB_MEMFreeSysRAM(disk);
   }
   FUNCTION_FINISH(DeleteDisk);
}



/*!**************************************************************************
 * @fn      FindDisk
 ****************************************************************************/
static S_DISK_INFO* FindDisk(U16BIT disk_id)
{
   S_DISK_INFO *disk;

   FUNCTION_START(FindDisk);
   for (disk = disks; (disk != NULL) && (disk->disk_id != disk_id);)
   {
      disk = disk->next;
   }
   FUNCTION_FINISH(FindDisk);

   return disk;
}



/*!**************************************************************************
 * @fn      DiskMonitorTask
 ****************************************************************************/
static void DiskMonitorTask(void *param)
{
   FUNCTION_START(DiskMonitorTask);

   USE_UNWANTED_PARAM(param);

   /* Mount any disks already available on startup, but don't send mount
    * events as there won't be anything to handle them during startup */
   CheckMountedDisks(FALSE);

   while (dsk_task_enable)
   {
      CheckMountedDisks(TRUE);

      /* Run the task every 3 seconds */
      STB_OSTaskDelay(3000);
   }
   pthread_exit(0);

   FUNCTION_FINISH(DiskMonitorTask);
}



/*!**************************************************************************
 * @fn      AssignDiskName
 ****************************************************************************/
static BOOLEAN AssignDiskName(S_DISK_INFO *disk)
{
   BOOLEAN result;
   char string[DISK_PATH_SIZE];
   char disk_name[DISK_NAME_SIZE];
   unsigned long num_blocks;
   unsigned long block_size;
   FILE *fp;

   FUNCTION_START(AssignDiskName);

   result = FALSE;

#if 0
   /* Execute the command and save the output for processing */
   sprintf(string, "/vendor/bin/sdparm --command=capacity %s > /data/vendor/dtvkit/diskinfo", disk->dev_path);
   system(string);

   fp = fopen("/data/vendor/dtvkit/diskinfo", "r");
   if (fp != NULL)
   {
      if (fgets(string, sizeof(string), fp) != NULL)
      {
         /* The full product name is normally split into manufacturer and device name,
          * but as the number of chars allocated to each seems to be fixed, they are
          * read separately */
         if ((sscanf(string, "%*[^:]: %s", disk_name) == 1) && (strlen(disk_name) > 0))
         {
            /* Now add the rest of the string to the manufacturer's name to give the
             * full product name */
            strcat(disk_name, " ");
            sscanf(string, "%*[^:]: %*s %[^\n]", &disk_name[strlen(disk_name)]);
            strcpy(disk->disk_name, disk_name);
            result = TRUE;
         }

         /* The disk capacity should also be in the output file */
         if (fgets(string, sizeof(string), fp) != NULL)
         {
            /* Number of blocks first */
            if (sscanf(string, "blocks: %lu", &num_blocks) == 1)
            {
               /* Block size next */
               if (fgets(string, sizeof(string), fp) != NULL)
               {
                  if (sscanf(string, "block_length: %lu", &block_size) == 1)
                  {
                     /* Calculate the disk size in KB. It's done in two ways to avoid overflow */
                     if ((block_size > 1024) && ((block_size % 1024) == 0))
                     {
                        disk->disk_size = (block_size / 1024) * num_blocks;
                     }
                     else
                     {
                        disk->disk_size = (num_blocks / 1024) * block_size;
                     }
                  }
               }
            }
         }
      }

      fclose(fp);
      unlink("/data/vendor/dtvkit/diskinfo");
   }
#endif

   FUNCTION_FINISH(AssignDiskName);

   return result;
}



/*!**************************************************************************
 * @fn      CheckMountedDisks
 ****************************************************************************/
#ifdef BUILD_FOR_OBS_DISK_MOUNT
static void CheckMountedDisks(BOOLEAN send_events)
{
   FILE *fp;
   char string[100];
   char disk_letter;
   char disk_id;
   unsigned int partition_num;
   int retval;
   char dev[32];
   char mnt[12];
   S_DISK_INFO *disk;
   S_DISK_INFO *next_disk;
   struct stat statbuf;
   struct statfs fs_info;
   BOOLEAN disk_change;
   int sscanfed;
   int partition_index;

   // int fd;
   // struct hd_driveid hdinfo;
   // int i;

   FUNCTION_START(CheckMountedDisks);

   /* Read the partition list to see if there are any new disks */
   fp = fopen("/proc/partitions", "r");
   if (fp != NULL)
   {
      STB_OSSemaphoreWait(disk_sem);

      /* Mark all disks as not found so that any that have been removed can be detected */
      for (disk = disks; disk != NULL; disk = disk->next)
      {
         disk->found = FALSE;
      }

      STB_OSSemaphoreSignal(disk_sem);

      disk_change = FALSE;

      while (fgets(string, sizeof(string), fp) != NULL)
      {
         sscanfed = sscanf(string, "%*d %*d %*d %cd%c%u", &disk_letter, &disk_id, &partition_num);
         if ((sscanfed == 2) || (sscanfed == 3))
         {
            if (sscanfed == 2)
            {
               /*To check if there is any sd[a-z][0-9]*/
               for (partition_index = 0; partition_index < 10; partition_index++)
               {
                  sprintf(dev, "/dev/%cd%c%u", disk_letter, disk_id, partition_index);
                  if (stat(dev, &statbuf) == 0)
                  {
                     break;
                  }
               }
               if (partition_index < 10)
               {
                  /* There is partition, let's ignore this mount point. */
                  continue;
               }
               else
               {
                  /* There is no partition, let's mount it without partition number. */
                  sprintf(dev, "/dev/%cd%c", disk_letter, disk_id);
               }
            }
            else
            {
               sprintf(dev, "/dev/%cd%c%u", disk_letter, disk_id, partition_num);
            }

            /* Check to see if this disk is already known */
            STB_OSSemaphoreWait(disk_sem);

            for (disk = disks; (disk != NULL) && (strcmp(disk->dev_path, dev) != 0);)
            {
               disk = disk->next;
            }

            /* Check that the dev path is valid before creating an entry in the disk list */
            if ((disk == NULL) && (stat(dev, &statbuf) == 0))
            {
               /* Create a directory on which to mount the device */
               // sprintf(mnt, "%s%cd%c%u", MOUNT_BASE_NAME, disk_letter, disk_id, partition_num);
               sprintf(mnt, "%shdd_1", MOUNT_BASE_NAME);

               retval = mkdir(mnt, (S_IFDIR | S_IRWXU | S_IRWXG | S_IRWXO));
               if ((retval == 0) || (errno == EEXIST))
               {
                  /* Directory has been created or already exists, so it can be mounted */
                  disk = (S_DISK_INFO*) STB_MEMGetSysRAM(sizeof(S_DISK_INFO));

                  memset(disk, 0, sizeof(S_DISK_INFO));

                  strcpy(disk->mount_path, mnt);
                  disk->disk_id = (U16BIT) ((disk_letter - 'a') << 9) +
                                            ((disk_id - 'a') << 4) + partition_num;
                  disk->is_mounted = FALSE;
                  disk->found = TRUE;
                  disk->blocked = FALSE;
                  strcpy(disk->dev_path, dev);

                  /* All disks are removeable */
                  disk->is_removeable = TRUE;

                  /* Try getting the name of the device using a Linux utility */
                  if (!AssignDiskName(disk))
                  {
                     /* Assign a name based on the disk ID which should be unique */
                     sprintf(disk->disk_name, "USB%u", disk->disk_id);
                  }

                  /* Add this disk to the list */
                  AddDisk(disk);

#if 0
                  /* Get the name of the disk */
                  fd = open(disk->dev_path, 0);
                  if (fd >= 0)
                  {
                     retval = ioctl(fd, HDIO_GET_IDENTITY, &hdinfo);
                     if (retval == 0)
                     {
                        strncpy(disk->disk_name, (char*) hdinfo.model, DISK_NAME_SIZE - 1);
                        disk->disk_name[DISK_NAME_SIZE - 1] = '\0';

                        /* Disk names are padded with spaces, so strip trailing spaces from
                         * the stored name */
                        for (i = DISK_NAME_SIZE - 2; (i > 0) && (disk->disk_name[i] == ' '); i--)
                        {
                           disk->disk_name[i] = '\0';
                        }

                        DISK_DBG(("Disk \"%s\" found", disk->disk_name));
                     }
                     else
                     {
                        DISK_DBG(("Failed to ioctl the device \"%s\", errno=%d",
                                  disk->dev_path, errno));

                        /* Assign a name based on the disk ID which should be unique */
                        sprintf(disk->disk_name, "USB%u", disk->disk_id);
                     }

                     close(fd);
                  }
                  else
                  {
                     DISK_DBG(("Failed to open the device \"%s\", errno=%d",
                               disk->dev_path, errno));
                  }
#endif
               }
#ifdef DISK_DEBUG
               else
               {
                  DISK_DBG(("STB_DSKInitialise: Failed to create mount point \"%s\", errno=%ld",
                        mnt, errno));
               }
#endif
            }

            if (disk != NULL)
            {
               if (!disk->is_mounted && !disk->blocked)
               {
                  retval = mount(disk->dev_path, disk->mount_path, "ext2", 0, NULL);
                  if ((retval == 0) || (errno == EBUSY))
                  {
                     /* Mount has been successful */
                     disk->is_mounted = TRUE;
                     disk->found = TRUE;

                     disk_change = TRUE;

                     /* The size of the disk may have been set when getting its name,
                      * so only do the following if the size is still 0 */
                     if (disk->disk_size == 0)
                     {
                        /* statfs can take a long time the first time it's called for a disk,
                         * so release the semaphore to prevent a lockup while it's running */
                        STB_OSSemaphoreSignal(disk_sem);
                        retval = statfs(disk->mount_path, &fs_info);
                        STB_OSSemaphoreWait(disk_sem);

                        if (retval == 0)
                        {
                           /* Calculate the disk size in KB. It's done various ways to
                            * avoid overflow */
                           if ((fs_info.f_bsize > 1024) && ((fs_info.f_bsize % 1024) == 0))
                           {
                              disk->disk_size = (fs_info.f_bsize / 1024) * fs_info.f_blocks;
                           }
                           else
                           {
                              disk->disk_size = (fs_info.f_bsize * fs_info.f_blocks) / 1024;
                           }
                        }
                     }

                     DISK_DBG(("Mounted ext2 partition on %s, size=%lu KB",
                               disk->mount_path, disk->disk_size));
                  }
                  else
                  {
                     /* Attempt to mount the disk  by trying the 2 supported file formats.
                      * For vfat, all file accesses are made synchronous to attempt to prevent file
                      * corruption as the device could be removed at any time! */
                     // retval = mount(disk->dev_path, disk->mount_path, "vfat", MS_SYNCHRONOUS, NULL);
                     retval = mount(disk->dev_path, disk->mount_path, "vfat", 0, NULL);

                     /* Mount returns EBUSY if the mount path is already being used */
                     if ((retval == 0) || (errno == EBUSY))
                     {
                        /* Mount has been successful */
                        disk->is_mounted = TRUE;
                        disk->found = TRUE;

                        disk_change = TRUE;

                        /* The size of the disk may have been set when getting its name,
                         * so only do the following if the size is still 0 */
                        if (disk->disk_size == 0)
                        {
                           /* statfs can take a long time the first time it's called for a disk,
                            * so release the semaphore to prevent a lockup while it's running */
                           STB_OSSemaphoreSignal(disk_sem);
                           retval = statfs(disk->mount_path, &fs_info);
                           STB_OSSemaphoreWait(disk_sem);

                           if (retval == 0)
                           {
                              /* Calculate the disk size in KB. It's done various ways to
                               * avoid overflow */
                              if ((fs_info.f_bsize > 1024) && ((fs_info.f_bsize % 1024) == 0))
                              {
                                 disk->disk_size = (fs_info.f_bsize / 1024) * fs_info.f_blocks;
                              }
                              else
                              {
                                 disk->disk_size = (fs_info.f_bsize * fs_info.f_blocks) / 1024;
                              }
                           }
                        }

                        DISK_DBG(("Mounted vfat partition on %s, size=%lu KB",
                                  disk->mount_path, disk->disk_size));
                     }
                     else
                     {
                        /* As this mount failed, prevent the mount from being attempted again
                         * until the disk is first removed */
                        disk->blocked = TRUE;
                        DISK_DBG(("Failed to mount %s on %s, errno=%d",
                                  disk->dev_path, disk->mount_path, errno));
                     }
                  }
               }
               else
               {
                  disk->found = TRUE;
               }
            }

            STB_OSSemaphoreSignal(disk_sem);
         }
      }
      fclose(fp);

      if (disk_change && send_events)
      {
         /* Send an event to indicate a device has been attached */
         STB_OSSendEvent(FALSE, HW_EV_CLASS_DISK, HW_EV_TYPE_DISK_CONNECTED, NULL, 0);
      }

      STB_OSSemaphoreWait(disk_sem);

      /* Now check all disks and unmount any that weren't found in the partitions file */
      disk_change = FALSE;

      for (disk = disks; disk != NULL;)
      {
         if (!disk->found)
         {
            if (disk->is_mounted)
            {
               retval = umount(disk->mount_path);
               if (retval == 0)
               {
                  DISK_DBG(("Unmounted \"%s\"", disk->mount_path));
               }
               else
               {
                  DISK_DBG(("Unmounting \"%s\" failed, errno=%d", disk->mount_path, errno));
               }

               /* Whatever happened, mark it as unmounted because it isn't available anymore */
               disk->is_mounted = FALSE;
            }

            /* Now the disk has disappeared, delete it from the list of known disks */
            next_disk = disk->next;
            DeleteDisk(disk);
            disk = next_disk;

            disk_change = TRUE;
         }
         else
         {
            disk = disk->next;
         }
      }

      STB_OSSemaphoreSignal(disk_sem);

      if (disk_change && send_events)
      {
         /* Send an event to indicate a device has been removed */
         STB_OSSendEvent(FALSE, HW_EV_CLASS_DISK, HW_EV_TYPE_DISK_REMOVED, NULL, 0);
      }
   }
   FUNCTION_FINISH(CheckMountedDisks);
}

#else /* BUILD_FOR_OBS_DISK_MOUNT */

static void CheckMountedDisks(BOOLEAN send_events)
{
   FILE *fp;
   char string[DISK_PATH_SIZE];
   char dev_name[32];
   char dev_path[32];
   char dev_mp[64];
   char dev_fs[32];
   int sscanfed;
   S_DISK_INFO *disk;
   S_DISK_INFO *next_disk;
   char disk_letter;
   char disk_id;
   unsigned int partition_num;
   BOOLEAN disk_change;


   FUNCTION_START(CheckMountedDisks);

   partition_num = 0;
   disk_change = FALSE;

   fp = fopen("/proc/mounts", "r");
   if (fp != NULL)
   {
      STB_OSSemaphoreWait(disk_sem);

      /* Mark all disks as not found so that any that have been removed can be detected */
      for (disk = disks; disk != NULL; disk = disk->next)
      {
         disk->found = FALSE;
      }

      STB_OSSemaphoreSignal(disk_sem);

      disk_change = FALSE;

      while (fgets(string, sizeof(string), fp) != NULL)
      {
         sscanfed = sscanf(string, "%[^ ] %[^ ] %[^ ] ", dev_name, dev_mp, dev_fs);
         if (strncmp(dev_mp, "/storage/", 9) == 0 && strncmp(dev_name, "/mnt/media_rw/", 14) == 0 )
         {
            STB_OSSemaphoreWait(disk_sem);
            
            for (disk = disks; (disk != NULL) && (strcmp(disk->dev_path, dev_path) != 0);)
            {
               disk = disk->next;
            }
            if (disk == NULL)
            {
               disk = (S_DISK_INFO*) STB_MEMGetSysRAM(sizeof(S_DISK_INFO));
               strcpy(disk->dev_path, dev_path);
               strcpy(disk->mount_path, dev_mp);
               disk->is_mounted = TRUE;
               disk->found = TRUE;
               disk->blocked = FALSE;
               disk->next = NULL;
               disk->is_removeable = TRUE;

               /* e.g. /mnt/media_rw/4675-7103 /storage/4675-7103 sdcardfs rw,nosuid, ... */
               string[DISK_PATH_SIZE - 1] = 0;  // FIXME: force the terminatation
               U32BIT crc32 = CDSM_UtilCalculateCRC((U8BIT *)string, (U32BIT)strlen(string));
               disk->disk_id = (U16BIT)(crc32 & 0xFFFF);
               if (INVALID_DISK_ID == disk->disk_id)
                   disk->disk_id = (U16BIT)((crc32 >> 16) & 0xFFFF);

               /* Try getting the name of the device using a Linux utility */
               if (!AssignDiskName(disk))
               {
                  /* Assign a name based on the disk ID which should be unique */
                  sprintf(disk->disk_name, "USB%u", disk->disk_id);
               }
               /* Add this disk to the list */
               AddDisk(disk);
               disk_change = TRUE;
            }
            else
            {
               disk->found = TRUE;
            }
            STB_OSSemaphoreSignal(disk_sem);
         }
      }   
      
      fclose(fp);
   }
   if (disk_change && send_events)
   {
      /* Send an event to indicate a device has been attached */
      STB_OSSendEvent(FALSE, HW_EV_CLASS_DISK, HW_EV_TYPE_DISK_CONNECTED, NULL, 0);
   }

   STB_OSSemaphoreWait(disk_sem);

   /* Now check all disks and unmount any that weren't found in the partitions file */
   disk_change = FALSE;

   for (disk = disks; disk != NULL;)
   {
      if (!disk->found)
      {
         /* Whatever happened, mark it as unmounted because it isn't available anymore */
         disk->is_mounted = FALSE;

         /* Now the disk has disappeared, delete it from the list of known disks */
         next_disk = disk->next;
         DeleteDisk(disk);
         disk = next_disk;

         disk_change = TRUE;
      }
      else
      {
         disk = disk->next;
      }
   }

   STB_OSSemaphoreSignal(disk_sem);

   if (disk_change && send_events)
   {
      /* Send an event to indicate a device has been removed */
      STB_OSSendEvent(FALSE, HW_EV_CLASS_DISK, HW_EV_TYPE_DISK_REMOVED, NULL, 0);
   }

   FUNCTION_FINISH(CheckMountedDisks);
}

#endif /* BUILD_FOR_OBS_DISK_MOUNT */



/****************************************************************************
** End of file
*****************************************************************************/
