/*******************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *****************************************************************************/
/**
 * @brief   Set Top Box - STB810 Hardware Layer, Memory Management
 * @file    stbhwmem.c
 * @date    September 2007
 * @author  Dave Carter
 */


/*--- Includes ----------------------------------------------------------------*/

/* System header files */
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <pthread.h>          // Temporary
#include <errno.h>
#include <sys/stat.h>
#include <libgen.h>
#include <dirent.h>

/* Third party header files */
#ifndef USE_SECURE_FILE
// #include "systemAPILib.h"
#endif

/* STB header files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbhwmem.h"
#include "stbhwdsk.h"
#include "stbhwdef.h"
#include "stbhwc.h"           // To get STB_MEMConfigMloaderForUpgrade to compile
#include "stbhwcrypt.h"
#include "heap.h"
#include "internal.h"
#include "internal_generic.h"


#ifdef system
#undef system
#endif
#define system(cmd) execl("/vendor/bin/sh", "sh", "-c", cmd, NULL)

/*--- Preprocessor definitions ------------------------------------------------*/

#define  HEAP_DEBUG
//#define NVM_DEBUG
//#define SECURE_DEBUG
//#define USE_SECURE_FILE

/* Only one of these flags can be enabled, and if MONITOR_MEM_USAGE is set then CHECK_MEMORY
 * won't have any effect */
// #define USE_PRIVATE_HEAP     // Enable this to use functions defined in heap.c
// #define MONITOR_MEM_USAGE    // Enable this to store info on amount of memory allocated, etc
// #define CHECK_MEMORY         // Enable this to check for writing outside of allocated memory


#define HEAP_SIZE             (7 * 1024 * 1024)
#define MAX_SECURE_FILENAME   16
#define PLATFORM_MAXPATH      256
#define KB                    1024

#define PVR_ENC_KEY_SIZE	  16

#ifndef DTVKIT_NVM_PATH
#define DTVKIT_NVM_PATH       "/data/vendor/dtvkit"
#endif

#ifndef  HEAP_DEBUG
#define  HEAP_DBG(X)
#else
#define  HEAP_DBG(X)    STB_SPDebugWrite X
#endif

#ifndef  NVM_DEBUG
#define  NVM_DBG(X)
#else
#define  NVM_DBG(X)     STB_SPDebugWrite X
#endif

#ifndef  SECURE_DEBUG
#define  SEC_DBG(X)
#else
#define  SEC_DBG(X)     STB_SPDebugWrite X
#endif

#define AES_PAD(x)      (((x) + 0xf) & ~0xf)


/*--- Local types definitions -------------------------------------------------*/

typedef enum stb_ci_key_type
{
   STB_CI_KEY_ROOT_CERT,
   STB_CI_KEY_BRAND_CERT,
   STB_CI_KEY_DEVICE_CERT,
   STB_CI_KEY_PRNG_SEED,
   STB_CI_KEY_PRNG_KEY_K,
   STB_CI_KEY_DH_P,
   STB_CI_KEY_DH_G,
   STB_CI_KEY_DH_Q,
   STB_CI_KEY_HDQ,
   STB_CI_KEY_SIV,
   STB_CI_KEY_SLK,
   STB_CI_KEY_CLK
} E_STB_CI_KEY_TYPE;

typedef struct nvm_status
{
   U32BIT size;
   U32BIT offset;
} S_NVM_STATUS;

typedef struct s_secure_data
{
   struct s_secure_data *next;
   U8BIT key;
   U32BIT size;
   void *value;
} S_SECURE_DATA;



/*--- Local (static) variable declarations ------------------------------------*/
static U8BIT pvr_enc_aes_key[PVR_ENC_KEY_SIZE] = {0};
static U8BIT pvr_enc_iv_key[PVR_ENC_KEY_SIZE] = {0};

#ifndef SQLITE_DATABASE
static S_NVM_STATUS nvm_status;
static char *nvm_file_string = DTVKIT_NVM_PATH "/nvm.dat";
#endif

static S_SECURE_DATA *secure_const_list;
static S_SECURE_DATA *secure_data_list;
static void *nvm_mutex;

#ifdef MONITOR_MEM_USAGE
static U32BIT mem_used;
static U32BIT peak_mem_used;
static U32BIT total_alloc_time;
static U32BIT total_free_time;
static U32BIT num_allocs;
static U32BIT num_frees;
static U32BIT total_num_allocs;
static U32BIT total_num_frees;
static void *heap_mutex;
#endif



/*--- Local function prototypes -----------------------------------------------*/

static void* AllocMem(U32BIT bytes);
static void* ReallocMem(void *addr, U32BIT new_size);
static void FreeMem(void *addr);
static void ReadSecureData(void);
static void PrefixPath(char *name, char *path);



/*--- Global function definitions ---------------------------------------------*/

/*!**************************************************************************
 * @fn      STB_MEMInitialiseRAM
 * @brief   Initialises the heap
 * @note    Does nothing on this platform, already taken care of by Linux
 ****************************************************************************/
void STB_MEMInitialiseRAM(void)
{
   FUNCTION_START(STB_MEMInitialiseRAM);

#ifdef USE_PRIVATE_HEAP
   if (!HEAPOpen(NULL, HEAP_SIZE))
   {
      printf("***** STB_MEMInitialiseRAM: Failed to create heap of %lu bytes\n", HEAP_SIZE);
   }
#endif

#ifdef MONITOR_MEM_USAGE
   mem_used = 0;
   peak_mem_used = 0;
   heap_mutex = STB_OSCreateMutex();
#endif
   memset(pvr_enc_aes_key, 0, PVR_ENC_KEY_SIZE);
   memset(pvr_enc_iv_key, 0, PVR_ENC_KEY_SIZE);
#if 0
   STB_CRYPTGetDefaultKey(pvr_enc_aes_key, PVR_ENC_KEY_SIZE);
#endif

   FUNCTION_FINISH(STB_MEMInitialiseRAM);
}



/*!**************************************************************************
 * @fn      STB_MEMInitialiseNVM
 * @brief   Initialises Non-Volatile memory access for a new NVM device, this
 *          function formats it ready for use, otherwise it prepares existing
 *          data for access
 ****************************************************************************/
void STB_MEMInitialiseNVM(void)
{
#ifndef SQLITE_DATABASE
   U8BIT *temp_buffer;
   U32BIT file_size;
   BOOLEAN create_nvm;
   FILE *fhan;
   U16BIT eload_size;

   FUNCTION_START(STB_MEMInitialiseNVM);

   create_nvm = FALSE;

   fhan = fopen(nvm_file_string, "r");
   if (fhan == NULL)
   {
      create_nvm = TRUE;
      NVM_DBG(("STB_MEMInitialiseNVM(): NVM file does not exist. Attempting to create %s.\n",
               nvm_file_string));
   }
   else
   {
      NVM_DBG(("STB_MEMInitialiseNVM(): NVM file %s exists.\n", nvm_file_string));
      fread((U8BIT*) &eload_size, sizeof(U16BIT), 1, fhan);
      if ((eload_size < 0x0002) || (eload_size > 0x1000))
      {
         eload_size = 2;
      }
      eload_size = 256;
      fseek(fhan, 0, SEEK_END);
      file_size = ftell(fhan);
      fclose(fhan);

      if (file_size < NV_DEVICE_SIZE)
      {
         remove((char*) nvm_file_string);
         create_nvm = TRUE;
      }
   }

   if (create_nvm == TRUE)
   {
      fhan = fopen(nvm_file_string, "w+");
      ASSERT(fhan !=NULL);
      if (fhan != NULL)
      {
         temp_buffer = (U8BIT*) STB_MEMGetSysRAM(NV_DEVICE_SIZE);
         ASSERT(temp_buffer !=NULL);
         if (temp_buffer != NULL)
         {
            eload_size = 256;
            fwrite((U8BIT*) &eload_size, sizeof(U16BIT), 1, fhan);
            memset(temp_buffer, 0, NV_DEVICE_SIZE);
            fwrite(temp_buffer, 1, NV_DEVICE_SIZE - sizeof(U16BIT), fhan);
            fclose(fhan);
            STB_MEMFreeSysRAM(temp_buffer);
            NVM_DBG(("\nSTB_MEMInitialiseNVM(): NVM Datafile %s created\n", nvm_file_string));
         }
         else
         {
            fclose(fhan);
            remove((char*) nvm_file_string);
            NVM_DBG(("\nSTB_MEMInitialiseNVM(): memory could not be allocated to temp_buffer\n"));
         }

      }
      else
      {
         NVM_DBG(("STB_MEMInitialiseNVM: Failed to create NVM file, errno %d", errno));
      }
   }

   nvm_status.size = NV_DEVICE_SIZE;
   nvm_status.offset = (U32BIT) eload_size;

   nvm_mutex = STB_OSCreateMutex();

   /* Read secure data from NVM */
   ReadSecureData();

   FUNCTION_FINISH(STB_MEMInitialiseNVM);
#else
   FUNCTION_START(STB_MEMInitialiseNVM);

   nvm_mutex = STB_OSCreateMutex();

   /* Read secure data from NVM */
   ReadSecureData();

   FUNCTION_FINISH(STB_MEMInitialiseNVM);
#endif
}



/*!**************************************************************************
 * @fn      STB_MEMReadNVM
 * @brief   Read data from the NVM
 * @param   addr - The NVM start address
 * @param   bytes - The number of bytes to read
 * @param   dst_addr - Callers buffer for the data
 * @return  TRUE if successful, FALSE if there was a problem
 ****************************************************************************/
BOOLEAN STB_MEMReadNVM(U32BIT addr, U32BIT bytes, U8BIT *dst_addr)
{
#ifndef SQLITE_DATABASE
   BOOLEAN ret_val;
   FILE *fhan;

   FUNCTION_START(STB_MEMReadNVM);

   NVM_DBG(("<< STB_MEMReadNVM (%ld, %ld, *D) sz=%ld\n", addr, bytes, nvm_status.size));

   ret_val = FALSE;
   if ((addr + bytes) <= nvm_status.size)
   {
      ASSERT((addr + bytes) <= NV_DEVICE_SIZE);
      ASSERT(dst_addr != NULL);

      fhan = fopen(nvm_file_string, "r");
      ASSERT(fhan != NULL);
      if (fhan != NULL)
      {
         NVM_DBG(("   STB_MEMReadNVM : seek addr %ld\n", addr));

         fseek(fhan, addr, SEEK_SET);
         if ((U32BIT) ftell(fhan) == addr)
         {
            if (fread(dst_addr, 1, bytes, fhan) == bytes)
            {
               ret_val = TRUE;
            }
         }
         else
         {
            NVM_DBG(("Failed to locate address in NVM datafile!\n"));
         }

         fclose(fhan);
      }
      else
      {
         NVM_DBG(("Failed to open NVM datafile for read!\n"));
      }
   }

   ASSERT(ret_val == TRUE);

   NVM_DBG((">> STB_MEMReadNVM OUT (ret %d) [%d %d %d %d]\n",
            ret_val, dst_addr[0], dst_addr[1], dst_addr[2], dst_addr[3]));

   FUNCTION_FINISH(STB_MEMReadNVM);

   return ret_val;
#else
   FUNCTION_START(STB_MEMReadNVM);
   FUNCTION_FINISH(STB_MEMReadNVM);
   return(FALSE);
#endif
}



/*!**************************************************************************
 * @fn      STB_MEMWriteNVM
 * @brief   Writes data to the NVM
 * @param   addr - The NVM start address to write to
 * @param   bytes - The number of bytes to be written
 * @param   src_addr - Pointer to the data to be written
 * @return  TRUE if successful, FALSE if there was a problem
 ****************************************************************************/
BOOLEAN STB_MEMWriteNVM(U32BIT addr, U32BIT bytes, U8BIT *src_addr)
{
#ifndef SQLITE_DATABASE
   BOOLEAN ret_val;
   size_t actual;
   FILE *fhan;

   FUNCTION_START(STB_MEMWriteNVM);

   NVM_DBG(("<< STB_MEMWriteNVM \n"));

   ret_val = FALSE;
   if ((addr + bytes) <= nvm_status.size)
   {
      NVM_DBG(("   STB_MEMWriteNVM :addr %ld  bytes : %ld\n", addr,  bytes));
      ASSERT((addr + bytes) <= NV_DEVICE_SIZE);
      ASSERT(src_addr != NULL);

      fhan = fopen(nvm_file_string, "r+");
      ASSERT(fhan != NULL);

      if (fhan != NULL)
      {
         fseek(fhan, addr, SEEK_SET);
         if ((U32BIT) ftell(fhan) == addr)
         {
            actual = fwrite(src_addr, 1, bytes, fhan);
            fflush(fhan);
            if (actual == bytes)
            {
               ret_val = TRUE;
            }
            else
            {
               NVM_DBG(("NVM.DAT WRITE ERROR!!! Tried %ld, wrote %ld\n",
                        bytes, (unsigned long) actual));
            }
         }
         else
         {
            NVM_DBG(("Failed to locate address in NVM datafile!\n"));
         }

         fclose(fhan);
      }
      else
      {
         NVM_DBG(("Failed to open NVM datafile for write!\n"));
      }
   }

   ASSERT(ret_val == TRUE);

   NVM_DBG((">> STB_MEMWriteNVM OUT (ret %d) [%d %d %d %d]\n",
            ret_val, src_addr[0], src_addr[1], src_addr[2], src_addr[3]));

   FUNCTION_FINISH(STB_MEMWriteNVM);

   return ret_val;
#else
   FUNCTION_START(STB_MEMWriteNVM);
   FUNCTION_FINISH(STB_MEMWriteNVM);
   return(FALSE);
#endif
}



/*!**************************************************************************
 * @fn      STB_MEMReadSecureVariable
 * @brief   Read variable defined by the given key from secure NVM
 * @param   key - data item to be read
 * @param   len - returned length of data read
 * @return  Pointer to data read, or NULL on failure.
 * @note    The data must be released using STB_MEMReleaseSecureVariable
 ****************************************************************************/
void* STB_MEMReadSecureVariable(U8BIT key, U32BIT *len)
{
   S_SECURE_DATA *item;
   void *value;

   FUNCTION_START(STB_MEMReadSecureVariable);

   value = NULL;
   STB_OSMutexLock(nvm_mutex);

   for (item = secure_data_list; item != NULL; item = item->next)
   {
      if (item->key == key)
      {
         value = item->value;
         *len = item->size;
         break;
      }
   }

   STB_OSMutexUnlock(nvm_mutex);

   FUNCTION_FINISH(STB_MEMReadSecureVariable);

   return value;
}



/*!**************************************************************************
 * @fn      STB_MEMReleaseSecureVariable
 * @brief   Release the data returned by STB_MEMReadSecureVariable
 * @param   value - pointer to data to be released
 ****************************************************************************/
void STB_MEMReleaseSecureVariable(void *value)
{
   FUNCTION_START(STB_MEMReleaseSecureVariable);
   free(value);
   FUNCTION_FINISH(STB_MEMReleaseSecureVariable);
}



/*!**************************************************************************
 * @fn      STB_MEMWriteSecureVariable
 * @brief   Writes data defined by the given key to secure NVM
 * @param   key - data item to be written
 * @param   value - pointer to data to be written
 * @param   len - data size
 * @return  TRUE if the data is written successfully, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_MEMWriteSecureVariable(U8BIT key, void *value, U32BIT len)
{
   S_SECURE_DATA *item;
   void *new_value;
   BOOLEAN result;

   FUNCTION_START(STB_MEMWriteSecureVariable);

   result = FALSE;

   STB_OSMutexLock(nvm_mutex);

   for (item = secure_data_list; item != NULL; item = item->next)
   {
      if (item->key == key)
      {
         if (AES_PAD(item->size) == AES_PAD(len))
         {
            /* Can just overwrite the existing data */
            memcpy(item->value, value, len);
            if (AES_PAD(len) > len)
            {
               memset((char*) item->value + len, 0, AES_PAD(len) - len);
            }
            item->size = len;
            result = TRUE;
            break;
         }
         else
         {
            /* Need to change the size of data being stored */
            new_value = (void*) STB_MEMGetSysRAM(AES_PAD(len));
            if (new_value != NULL)
            {
               if (item->value != NULL)
               {
                  STB_MEMFreeSysRAM(item->value);
               }

               memcpy(new_value, value, len);
               if (AES_PAD(len) > len)
               {
                  memset((char*) item->value + len, 0, AES_PAD(len) - len);
               }
               item->value = new_value;
               item->size = len;
               result = TRUE;
            }
            break;
         }
      }
   }

   if (!result && (item == NULL))
   {
      /* Key hasn't been found so add it */
      item = (S_SECURE_DATA*) STB_MEMGetSysRAM(sizeof(S_SECURE_DATA));
      if (item != NULL)
      {
         item->value = (void*) STB_MEMGetSysRAM(AES_PAD(len));
         if (item->value != NULL)
         {
            item->key = key;
            item->size = len;

            memcpy(item->value, value, len);
            if (AES_PAD(len) > len)
            {
               memset((char*) item->value + len, 0, AES_PAD(len) - len);
            }

            item->next = secure_data_list;
            secure_data_list = item;

            result = TRUE;
         }
      }
   }
   STB_OSMutexUnlock(nvm_mutex);

   FUNCTION_FINISH(STB_MEMWriteSecureVariable);

   return result;
}



/*!**************************************************************************
 * @fn      STB_MEMReadSecureConstant
 * @brief   Read constant defined by the given key from secure NVM
 * @param   key - data item to be read
 * @param   len - returned length of data read
 * @return  Pointer to data read, or NULL on failure.
 * @note    The data must not be released (it is managed by the platform layer)
 ****************************************************************************/
const void* STB_MEMReadSecureConstant(U8BIT key, U32BIT *len)
{
   S_SECURE_DATA *item;
   void *value;

   FUNCTION_START(STB_MEMReadSecureConstant);

   value = NULL;
   STB_OSMutexLock(nvm_mutex);

   for (item = secure_const_list; item != NULL; item = item->next)
   {
      if (item->key == key)
      {
         value = item->value;
         *len = item->size;
         break;
      }
   }

   if(value == NULL)
   {
	   if(SECURE_NVM_DEFAULT_ENCRYPTION_KEY == key)
	   {
		   *len = PVR_ENC_KEY_SIZE;
		   value = pvr_enc_aes_key; 
	   }
	   else if(SECURE_NVM_DEFAULT_ENC_INIT_VECTOR == key)
	   {
		   *len = PVR_ENC_KEY_SIZE;
		   value = pvr_enc_iv_key; 
	   }
   }

   STB_OSMutexUnlock(nvm_mutex);

   FUNCTION_FINISH(STB_MEMReadSecureConstant);

   return value;
}



/*!**************************************************************************
 * @fn      STB_MEMGetNVMSize
 * @brief   Returns the size (capacity) of the NVM
 * @return  Size of the NVM in bytes
 ****************************************************************************/
U32BIT STB_MEMGetNVMSize(void)
{
#ifndef SQLITE_DATABASE
   U32BIT ret_val;

   FUNCTION_START(STB_MEMGetNVMSize);
   ret_val = nvm_status.size;
   FUNCTION_FINISH(STB_MEMGetNVMSize);

   return ret_val;
#else
   FUNCTION_START(STB_MEMGetNVMSize);
   FUNCTION_FINISH(STB_MEMGetNVMSize);
   return 0;
#endif
}



/*!**************************************************************************
 * @fn      STB_MEMGetNVMOffset
 * @brief   Returns any offset required in NVM to avoid private data
 * @return  The offset in bytes
 ****************************************************************************/
U32BIT STB_MEMGetNVMOffset(void)
{
#ifndef SQLITE_DATABASE
   U32BIT ret_val;

   FUNCTION_START(STB_MEMGetNVMOffset);
   ret_val = nvm_status.offset;
   FUNCTION_FINISH(STB_MEMGetNVMOffset);

   return ret_val;
#else
   FUNCTION_START(STB_MEMGetNVMOffset);
   FUNCTION_FINISH(STB_MEMGetNVMOffset);
   return(0);
#endif
}



/*!**************************************************************************
 * @fn      STB_MEMGetNVMAlign
 * @brief   Returns the word alignment size of the NVM device
 * @return  Alignment in bytes
 ****************************************************************************/
U8BIT STB_MEMGetNVMAlign(void)
{
   FUNCTION_START(STB_MEMGetNVMAlign);
   FUNCTION_FINISH(STB_MEMGetNVMAlign);
   return 1;
}



/*!**************************************************************************
 * @fn      STB_MEMGetSysRAM
 * @brief   Allocates a new block of memory for system use
 * @param   bytes - Size of required block in bytes
 * @return  A pointer to the allocated block of memory, NULL if allocation
 *          failed
 ****************************************************************************/
void* STB_MEMGetSysRAM(U32BIT bytes)
{
   void *retval;

   FUNCTION_START(STB_MEMGetSysRAM);
   retval = AllocMem(bytes);
   FUNCTION_FINISH(STB_MEMGetSysRAM);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_MEMResizeSysRAM
 * @brief   Changes the size of the given block of memory
 * @param   ptr - pointer to memory already allocated from the system heap
 * @param   new_num_bytes - size of the memory block to be returned
 * @return  Address of new block of memory
 ****************************************************************************/
void* STB_MEMResizeSysRAM(void *ptr, U32BIT new_num_bytes)
{
   void *new_ptr;

   FUNCTION_START(STB_MEMResizeSysRAM);

   if (ptr == NULL)
   {
      new_ptr = AllocMem(new_num_bytes);
   }
   else if (new_num_bytes == 0)
   {
      FreeMem(ptr);
      new_ptr = NULL;
   }
   else
   {
      new_ptr = ReallocMem(ptr, new_num_bytes);
   }

   FUNCTION_FINISH(STB_MEMResizeSysRAM);

   return new_ptr;
}



/*!**************************************************************************
 * @fn      STB_MEMFreeSysRAM
 * @brief   Releases a previously allocated block of system memory
 * @param   block_ptr - address of block to be released
 ****************************************************************************/
void STB_MEMFreeSysRAM(void *block_ptr)
{
   FUNCTION_START(STB_MEMFreeSysRAM);
   FreeMem(block_ptr);
   FUNCTION_FINISH(STB_MEMFreeSysRAM);
}



/*!**************************************************************************
 * @fn      STB_MEMSysRAMUsed
 * @brief   Returns the amount of available system memory consumed
 * @return  Memory used as a percentage of available
 * @todo    Add STB810 code
 ****************************************************************************/
U8BIT STB_MEMSysRAMUsed(void)
{
   U8BIT used_result;

   FUNCTION_START(STB_MEMSysRAMUsed);
   used_result = 0;
   FUNCTION_FINISH(STB_MEMSysRAMUsed);

   return used_result;
}



/*!**************************************************************************
 * @fn      STB_MEMGetAppRAM
 * @brief   Allocates a new block of memory for application use
 * @param   bytes - Size of required block in bytes
 * @return  A pointer to the allocated block of memory NULL if allocation
 *          failed
 ****************************************************************************/
void* STB_MEMGetAppRAM(U32BIT bytes)
{
   void *retval;

   FUNCTION_START(STB_MEMGetAppRAM);
   retval = AllocMem(bytes);
   FUNCTION_FINISH(STB_MEMGetAppRAM);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_MEMResizeAppRAM
 * @brief   Changes the size of the given block of app memory
 * @param   ptr - pointer to memory already allocated from the app heap
 * @param   new_num_bytes - size of the memory block to be returned
 * @return  Address of new block of memory
 ****************************************************************************/
void* STB_MEMResizeAppRAM(void *ptr, U32BIT new_num_bytes)
{
   void *new_ptr;

   FUNCTION_START(STB_MEMResizeAppRAM);

   if (ptr == NULL)
   {
      new_ptr = AllocMem(new_num_bytes);
   }
   else if (new_num_bytes == 0)
   {
      FreeMem(ptr);
      new_ptr = NULL;
   }
   else
   {
      new_ptr = ReallocMem(ptr, new_num_bytes);
   }

   FUNCTION_FINISH(STB_MEMResizeAppRAM);

   return new_ptr;
}



/*!**************************************************************************
 * @fn      STB_MEMFreeAppRAM
 * @brief   Releases a previously allocated block of system memory
 * @param   block_ptr - address of block to be released
 ****************************************************************************/
void STB_MEMFreeAppRAM(void *block_ptr)
{
   FUNCTION_START(STB_MEMFreeAppRAM);
   FreeMem(block_ptr);
   FUNCTION_FINISH(STB_MEMFreeAppRAM);
}



/*!**************************************************************************
 * @fn      STB_MEMAppRAMUsed
 * @brief   Returns the amount of available application memory consumed
 * @return  Memory used as a percentage of available
 * @todo    Add STB810 code
 ****************************************************************************/
U8BIT STB_MEMAppRAMUsed(void)
{
   U8BIT used_result;

   FUNCTION_START(STB_MEMFreeAppRAM);
   used_result = 0;
   FUNCTION_FINISH(STB_MEMFreeAppRAM);

   return  used_result;
}



/*!**************************************************************************
 * @fn      STB_MEMConfigMloaderForUpgrade
 * @brief   Configures the multiloader data area in NVM with the required
 *          data to start an upgrade.
 * @param   del_sys_desc - Terrestrial Delivery system descriptor as found in
 *          the NIT. Expected length is 11 bytes.
 * @param   num_attempts
 * @return  TRUE if successful, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_MEMConfigMloaderForUpgrade(void *loader_data, U32BIT data_size)
{
   FUNCTION_START(STB_MEMConfigMloaderForUpgrade);
   USE_UNWANTED_PARAM(loader_data);
   USE_UNWANTED_PARAM(data_size);
   FUNCTION_FINISH(STB_MEMConfigMloaderForUpgrade);
   return FALSE;
}



/*!**************************************************************************
 * @fn      STB_MEMReadMloaderData
 * @brief   Reads from the memory area allocated for the OTA loader
 * @param   buffer - pointer to the destination buffer
 * @param   size - number of bytes to read
 * @return  TRUE if successful, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_MEMReadMloaderData(void *buffer, U32BIT size)
{
   FUNCTION_START(STB_MEMReadMloaderData);
   USE_UNWANTED_PARAM(buffer);
   USE_UNWANTED_PARAM(size);
   FUNCTION_FINISH(STB_MEMReadMloaderData);
   return FALSE;
}



/*!**************************************************************************
 * @fn      STB_MEMWriteMloaderData
 * @brief   Writes from the memory area allocated for the OTA loader
 * @param   buffer - pointer to the source buffer
 * @param   size - number of bytes to write
 * @return  TRUE if successful, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_MEMWriteMloaderData(void *buffer, U32BIT size)
{
   FUNCTION_START(STB_MEMWriteMloaderData);
   USE_UNWANTED_PARAM(buffer);
   USE_UNWANTED_PARAM(size);
   FUNCTION_FINISH(STB_MEMWriteMloaderData);
   return FALSE;
}



/*!**************************************************************************
 * @fn      STB_NVMFileSize
 * @brief   Returns the size in KB of the given file
 * @param   filename - name of the file in the nvm
 * @param   filesize - returned value giving the file size in KB
 * @return  TRUE if the file exists, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_NVMFileSize(U8BIT *filename, U32BIT *filesize)
{
   char path[PLATFORM_MAXPATH];
   struct stat buf;
   BOOLEAN ret;

   FUNCTION_START(STB_NVMFileSize);

   ret = FALSE;
   PrefixPath((char*) filename, path);
   if (stat(path, &buf) == 0)
   {
      ret = TRUE;
      *filesize = buf.st_size / KB;
   }

   FUNCTION_FINISH(STB_NVMFileSize);

   return ret;
}



/*!**************************************************************************
 * @fn      STB_NVMOpenFile
 * @brief   Opens an existing file or creates a new one.
 * @param   name - The filename (including path). When the mode is
 *          FILE_MODE_OVERWRITE, the directories in the path will be created
 *          when they don't exist.
 * @param   mode - The access mode
 * @return  The file handle
 ****************************************************************************/
void* STB_NVMOpenFile(U8BIT *name, E_STB_DSK_FILE_MODE mode)
{
   FILE *file;
   char path[PLATFORM_MAXPATH];
   char cmd[PLATFORM_MAXPATH];
   char dir[PLATFORM_MAXPATH];
   char *directory;

   FUNCTION_START(STB_NVMOpenFile);

   PrefixPath((char*) name, path);

   switch (mode)
   {
      case FILE_MODE_READ:
         file = fopen(path, "r");
         break;

      case FILE_MODE_WRITE:
         file = fopen(path, "w");
         break;

      case FILE_MODE_OVERWRITE:
         file = fopen(path, "w");
         if (file == NULL)
         {
            /* assume that we need to create the path */
            strncpy(dir, path, PLATFORM_MAXPATH);
            directory = dirname(dir);
            snprintf(cmd, PLATFORM_MAXPATH, "mkdir -p %s", directory);
            system(cmd);
            file = fopen(path, "w");
         }
         break;
   }

   FUNCTION_FINISH(STB_NVMOpenFile);

   return (void*) file;
}



/*!**************************************************************************
 * @fn      STB_NVMCloseFile
 * @brief   Flushes and closes an open file
 * @param   file - The file handle
 ****************************************************************************/
void STB_NVMCloseFile(void *file)
{
   FUNCTION_START(STB_NVMCloseFile);
   fclose((FILE*) file);
   FUNCTION_FINISH(STB_NVMCloseFile);
}



/*!**************************************************************************
 * @fn      STB_NVMReadFile
 * @brief   Reads data from an open file
 * @param   file - The file handle
 * @param   data - The caller's buffer
 * @param   size - Number of bytes to be read
 * @return  Number of bytes successfully read
 ****************************************************************************/
U32BIT STB_NVMReadFile(void *file, U8BIT *data, U32BIT size)
{
   FUNCTION_START(STB_NVMReadFile);
   FUNCTION_FINISH(STB_NVMReadFile);
   return fread(data, 1, size, (FILE*) file);
}



/*!**************************************************************************
 * @fn      STB_NVMWriteFile
 * @brief   Writes data to an open file
 * @param   file - The file handle
 * @param   data - Pointer to the data to be written
 * @param   size - Number of bytes to  write
 * @return  Number of bytes successfully written
 ****************************************************************************/
U32BIT STB_NVMWriteFile(void *file, U8BIT *data, U32BIT size)
{
   FUNCTION_START(STB_NVMWriteFile);
   FUNCTION_FINISH(STB_NVMWriteFile);
   return fwrite(data, 1, size, (FILE*) file);
}



/*!**************************************************************************
 * @fn      STB_NVMDeleteFile
 * @brief   Deletes the file.
 * @param   filename - pathname of the file to be deleted
 * @return  TRUE if successful, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_NVMDeleteFile(U8BIT *name)
{
   char path[PLATFORM_MAXPATH];

   FUNCTION_START(STB_NVMDeleteFile);
   PrefixPath((char*) name, path);
   FUNCTION_FINISH(STB_NVMDeleteFile);

   return unlink(path);
}



/*!**************************************************************************
 * @fn      STB_NVMOpenDirectory
 * @brief   Opens a directory in order to read the entries
 * @param   dir_name - name of the directory to open
 * @return  Handle to be used in all other operations, NULL if the open fails
 ****************************************************************************/
void* STB_NVMOpenDirectory(U8BIT *name)
{
   char path[PLATFORM_MAXPATH];
   DIR *handle;

   FUNCTION_START(STB_NVMOpenDirectory);
   PrefixPath((char*) name, path);
   handle = opendir(path);
   FUNCTION_FINISH(STB_NVMOpenDirectory);

   return (void*) handle;
}



/*!**************************************************************************
 * @fn      STB_NVMReadDirectory
 * @brief   Reads the next entry from the directory, returning the name of the
 *          entry and the type of the entry
 * @param   dir - handle returned when the directory was opened
 * @param   filename - array in which the name is returned
 * @param   filename_len - size of the filename array
 * @param   entry_type - type of entry
 * @return  TRUE if and entry is read, FALSE otherwise which could indicate end
 *          of the directory
 ****************************************************************************/
BOOLEAN STB_NVMReadDirectory(void *dir, U8BIT *filename, U16BIT filename_len,
                             E_STB_DIR_ENTRY_TYPE *entry_type)
{

   DIR *dirp;
   struct dirent *entry;
   BOOLEAN ret;

   FUNCTION_START(STB_NVMReadDirectory);

   dirp = (DIR*) dir;
   ret = FALSE;
   entry = readdir(dirp);

   if (entry != NULL)
   {
      ret = TRUE;
      strncpy((char*) filename, entry->d_name, filename_len);
      switch (entry->d_type)
      {
         case DT_REG:
            *entry_type = DIR_ENTRY_FILE;
            break;
         case DT_DIR:
            *entry_type = DIR_ENTRY_DIRECTORY;
            break;
         case DT_BLK:
         case DT_CHR:
         case DT_FIFO:
         case DT_LNK:
         case DT_SOCK:
         case DT_UNKNOWN:
         default:
            *entry_type = DIR_ENTRY_OTHER;
            break;
      }
   }

   FUNCTION_FINISH(STB_NVMReadDirectory);

   return ret;
}



/*!**************************************************************************
 * @fn      STB_NVMCloseDirectory
 * @brief   Closes the directory for reading
 * @param   dir - directory handle
 ****************************************************************************/
void STB_NVMCloseDirectory(void *dir)
{
   DIR *dirp;

   FUNCTION_START(STB_NVMCloseDirectory);
   dirp = (DIR*) dir;
   closedir(dirp);
   FUNCTION_FINISH(STB_NVMCloseDirectory);
}




/*--- Global functions not declared in stbhwmem.h -----------------------------*/

/*!**************************************************************************
 * @fn      STB_MEMGetUsed
 * @note    Never declared, never used
 ****************************************************************************/
void STB_MEMGetUsed(U32BIT *current_mem, U32BIT *peak_mem, U32BIT *alloc_time,
                    U32BIT *free_time, U32BIT *total_allocs, U32BIT *total_frees)
{
   FUNCTION_START(STB_MEMGetUsed);

#ifdef MONITOR_MEM_USAGE
   STB_OSMutexLock(heap_mutex);
   *current_mem = mem_used;
   *peak_mem = peak_mem_used;
   *alloc_time = total_alloc_time;
   *free_time = total_free_time;
   *total_allocs = num_allocs;
   *total_frees = num_frees;

   /* Reset counts */
   total_alloc_time = 0;
   total_free_time = 0;
   num_allocs = 0;
   num_frees = 0;
   STB_OSMutexUnlock(heap_mutex);
#else
   *current_mem = 0;
   *peak_mem = 0;
   *alloc_time = 0;
   *free_time = 0;
   *total_allocs = 0;
   *total_frees = 0;
#endif

   FUNCTION_FINISH(STB_MEMGetUsed);
}



/*!**************************************************************************
 * @fn      STB_CIGetHostKey
 * @brief   Return CI+ host key
 * @param   type - type of host key
 * @param   key - pointer to the key data
 * @param   length - number of bytes in key data
 * @param   slot_id - Zero-based CI slot identifier (0, 1, ...)
 * @note    The pointer must remain valid while the CI+ stack is running
 ****************************************************************************/
void STB_CIGetHostKey(E_STB_CI_KEY_TYPE type, U8BIT **key, U16BIT *length)
{
   U8BIT ciplus_data;
   U32BIT len;
   U32BIT secure_constant;

   FUNCTION_START(STB_MEMReadSecureConstant);
   len = 0;
   secure_constant = 0;
   switch (type)
   {
      case STB_CI_KEY_ROOT_CERT:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_ROOT_CERT\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_ROOT_CERT;
         break;

      case STB_CI_KEY_BRAND_CERT:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_BRAND_CERT\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_BRAND_CERT;
         break;

      case STB_CI_KEY_DEVICE_CERT:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_DEVICE_CERT\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_DEVICE_CERT;
         break;

      case STB_CI_KEY_PRNG_SEED:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_PRNG_SEED\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_PRNG_SEED;
         break;

      case STB_CI_KEY_PRNG_KEY_K:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_PRNG_KEY_K\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_PRNG_KEY_K;
         break;

      case STB_CI_KEY_DH_P:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_DH_P\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_DH_P;
         break;

      case STB_CI_KEY_DH_G:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_DH_G\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_DH_G;
         break;

      case STB_CI_KEY_DH_Q:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_DH_Q\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_DH_Q;
         break;

      case STB_CI_KEY_HDQ:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_HDQ\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_DEVICE_KEY;
         break;

      case STB_CI_KEY_SIV:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_SIV\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_SIV;
         break;

      case STB_CI_KEY_SLK:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_SLK\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_SLK;
         break;

      case STB_CI_KEY_CLK:
         SEC_DBG(("STB_CIGetHostKey    STB_CI_KEY_CLK\n"));
         ciplus_data = SECURE_NVM_CI_PLUS_CLK;
         break;

      default:
         SEC_DBG(("STB_CIGetHostKey    UNKNOWN\n"));
         break;
   }

   secure_constant = (U32BIT) STB_MEMReadSecureConstant(ciplus_data, &len);
   *key = (U8BIT *) secure_constant;
   *length = (U16BIT) len;

   FUNCTION_FINISH(STB_MEMReadSecureConstant);

   return;
}



/*!**************************************************************************
 * @fn      STB_CIReadSecureNVM
 * @brief   Read data from NVM
 * @param   bytes - number of bytes to read
 * @param   dest_addr - destination address for data
 * @return  TRUE if data was read successfully, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_CIReadSecureNVM(U8BIT *dest_addr, U32BIT bytes)
{
   BOOLEAN success;

   FUNCTION_START(STB_CIReadSecureNVM);
   success = FALSE;
   FUNCTION_FINISH(STB_CIReadSecureNVM);

   return success;
}



/*!**************************************************************************
 * @fn      STB_CIWriteSecureNVM
 * @brief   Write data to NVM
 * @param   bytes - number of bytes to write
 * @param   src_addr - source address for data
 * @return  TRUE if data was written successfully, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_CIWriteSecureNVM(U8BIT *src_addr, U32BIT bytes)
{
   BOOLEAN success;

   FUNCTION_START(STB_CIWriteSecureNVM);
   success = FALSE;
   FUNCTION_FINISH(STB_CIWriteSecureNVM);

   return success;
}



/*!**************************************************************************
 * @fn      STB_CIClearSecureNVM
 * @brief   Clear data in NVM
 * @return  TRUE if nvram is cleared successfully, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_CIClearSecureNVM(void)
{
   BOOLEAN success;

   FUNCTION_START(STB_CIClearSecureNVM);
   success = FALSE;
   FUNCTION_FINISH(STB_CIClearSecureNVM);

   return success;
}





/*--- Local function definitions ----------------------------------------------*/

/*!**************************************************************************
 * @fn      AllocMem
 ****************************************************************************/
static void* AllocMem(U32BIT bytes)
{
   U8BIT *addr;

#ifdef MONITOR_MEM_USAGE
   U32BIT start_time;
   U32BIT time_taken;
#endif

   FUNCTION_START(AllocMem);
   addr = NULL;
   if (bytes > 0)
   {
#ifdef MONITOR_MEM_USAGE

      start_time = STB_OSGetClockMilliseconds();

#ifdef USE_PRIVATE_HEAP
      addr = HEAPAlloc((size_t) bytes + 4);
#else
      addr = malloc(bytes + 4);
#endif

      time_taken = STB_OSGetClockMilliseconds() - start_time;
      if (addr != NULL)
      {
         *addr = (U8BIT)(bytes >> 24) & 0xff;
         *(addr + 1) = (U8BIT)(bytes >> 16) & 0xff;
         *(addr + 2) = (U8BIT)(bytes >> 8) & 0xff;
         *(addr + 3) = (U8BIT)(bytes & 0xff);
         addr += 4;

         STB_OSMutexLock(heap_mutex);
         mem_used += bytes;
         if (mem_used > peak_mem_used)
         {
            peak_mem_used = mem_used;
         }

         total_alloc_time += time_taken;
         num_allocs++;
         total_num_allocs++;
         STB_OSMutexUnlock(heap_mutex);
      }
#elif defined(CHECK_MEMORY)

#ifdef USE_PRIVATE_HEAP
      addr = HEAPAlloc((size_t) bytes + 12);
#else
      addr = malloc(bytes + 12);
#endif

      if (addr != NULL)
      {
         /* Put check values at the start and end of the allocated block */
         *(addr + 1) = (U8BIT) 'O';
         *(addr + 3) = (U8BIT) 'B';
         *(addr + 5) = (U8BIT) 'S';
         *(addr + 7) = (U8BIT) 'L';

         *(addr + 8 + bytes) = (U8BIT) 'L';
         *(addr + 9 + bytes) = (U8BIT) 'S';
         *(addr + 10 + bytes) = (U8BIT) 'B';
         *(addr + 11 + bytes) = (U8BIT) 'O';

         /* Also need to store the size of the block so the end can be checked when freed */
         *addr = (U8BIT)(bytes >> 24) & 0xff;
         *(addr + 2) = (U8BIT)(bytes >> 16) & 0xff;
         *(addr + 4) = (U8BIT)(bytes >> 8) & 0xff;
         *(addr + 6) = (U8BIT)(bytes & 0xff);

         addr += 8;
      }
#else

#ifdef USE_PRIVATE_HEAP
      addr = HEAPAlloc((size_t) bytes);
#else
      addr = (U8BIT *)malloc(bytes);
#endif

#endif
      if (addr == NULL)
      {
         HEAP_DBG(("*** OUT OF MEMORY ***: Can't allocate %lu bytes\n", bytes));
      }
   }

   FUNCTION_FINISH(AllocMem);
   return addr;
}


/*!**************************************************************************
 * @fn      ReallocMem
 ****************************************************************************/
static void* ReallocMem(void *addr, U32BIT new_size)
{
   U8BIT *new_ptr;

#ifdef MONITOR_MEM_USAGE
   U8BIT *ptr;
   U32BIT num_bytes;
#elif defined(CHECK_MEMORY)
   U8BIT *ptr;
   U32BIT num_bytes;
#endif

   FUNCTION_START(ReallocMem);

   new_ptr = NULL;
   if ((addr != NULL) && (new_size > 0))
   {
#ifdef MONITOR_MEM_USAGE
      ptr = (U8BIT*) addr;
      ptr -= 4;

      num_bytes = *ptr << 24;
      num_bytes += *(ptr + 1) << 16;
      num_bytes += *(ptr + 2) << 8;
      num_bytes += *(ptr + 3);

#ifdef USE_PRIVATE_HEAP
      new_ptr = HEAPRealloc(ptr, new_size + 4);
#else
      new_ptr = realloc(ptr, new_size + 4);
#endif

      if (new_ptr != NULL)
      {
         *new_ptr = (U8BIT)(new_size >> 24) & 0xff;
         *(new_ptr + 1) = (U8BIT)(new_size >> 16) & 0xff;
         *(new_ptr + 2) = (U8BIT)(new_size >> 8) & 0xff;
         *(new_ptr + 3) = (U8BIT)(new_size & 0xff);
         new_ptr += 4;
      }

      STB_OSMutexLock(heap_mutex);
      mem_used += new_size;
      mem_used -= num_bytes;
      STB_OSMutexUnlock(heap_mutex);

#elif defined(CHECK_MEMORY)

      /* Check the memory still has the set values at the start and end */
      ptr = (U8BIT*) addr;
      ptr -= 8;

      if ((*(ptr + 1) == (U8BIT) 'O') && (*(ptr + 3) == (U8BIT) 'B') &&
          (*(ptr + 5) == (U8BIT) 'S') && (*(ptr + 7) == (U8BIT) 'L'))
      {
         /* The start of memory seems to be correct, so get the size of the block */
         num_bytes = *ptr << 24;
         num_bytes += *(ptr + 2) << 16;
         num_bytes += *(ptr + 4) << 8;
         num_bytes += *(ptr + 6);

         /* Now check the end of the block */
         if ((*(ptr + 8 + num_bytes) != (U8BIT) 'L') || (*(ptr + 9 + num_bytes) != (U8BIT) 'S') ||
             (*(ptr + 10 + num_bytes) != (U8BIT) 'B') || (*(ptr + 11 + num_bytes) != (U8BIT) 'O'))
         {
            printf("*** HEAP CORRUPTED: At %p, addr %p, size %lu\n",
                   ptr + 8 + num_bytes, addr, num_bytes);
         }
      }
      else
      {
         printf("*** HEAP CORRUPTED: At start of %p\n", addr);
      }

#ifdef USE_PRIVATE_HEAP
      new_ptr = HEAPRealloc(ptr, new_size + 12);
#else
      new_ptr = realloc(ptr, new_size + 12);
#endif

      if (new_ptr != NULL)
      {
         /* Put check values at the start and end of the allocated block */
         *(new_ptr + 1) = (U8BIT) 'O';
         *(new_ptr + 3) = (U8BIT) 'B';
         *(new_ptr + 5) = (U8BIT) 'S';
         *(new_ptr + 7) = (U8BIT) 'L';

         *(new_ptr + 8 + new_size) = (U8BIT) 'L';
         *(new_ptr + 9 + new_size) = (U8BIT) 'S';
         *(new_ptr + 10 + new_size) = (U8BIT) 'B';
         *(new_ptr + 11 + new_size) = (U8BIT) 'O';

         /* Also need to store the size of the block so the end can be checked when freed */
         *new_ptr = (U8BIT)(new_size >> 24) & 0xff;
         *(new_ptr + 2) = (U8BIT)(new_size >> 16) & 0xff;
         *(new_ptr + 4) = (U8BIT)(new_size >> 8) & 0xff;
         *(new_ptr + 6) = (U8BIT)(new_size & 0xff);

         new_ptr += 8;
      }

#else

#ifdef USE_PRIVATE_HEAP
      new_ptr = HEAPRealloc(addr, new_size);
#else
      new_ptr = (U8BIT *)realloc(addr, new_size);
#endif

#endif

      if (new_ptr == NULL)
      {
         HEAP_DBG(("*** OUT OF MEMORY *** Failed to reallocate %lu bytes for addr %p\n",
                   new_size, addr));
      }
   }

   FUNCTION_FINISH(ReallocMem);
   return new_ptr;
}


/*!**************************************************************************
 * @fn      FreeMem
 ****************************************************************************/
static void FreeMem(void *addr)
{
#ifdef MONITOR_MEM_USAGE
   U8BIT *ptr;
   U32BIT num_bytes;
   U32BIT start_time;
   U32BIT time_taken;
#elif defined(CHECK_MEMORY)
   U8BIT *ptr;
   U32BIT num_bytes;
#endif

   FUNCTION_START(FreeMem);
   if (addr != NULL)
   {
#ifdef MONITOR_MEM_USAGE
      ptr = (U8BIT*) addr;
      ptr -= 4;

      num_bytes = *ptr << 24;
      num_bytes += *(ptr + 1) << 16;
      num_bytes += *(ptr + 2) << 8;
      num_bytes += *(ptr + 3);

      start_time = STB_OSGetClockMilliseconds();

#ifdef USE_PRIVATE_HEAP
      HEAPFree(ptr);
#else
      free(ptr);
#endif

      time_taken = STB_OSGetClockMilliseconds() - start_time;

      STB_OSMutexLock(heap_mutex);
      if (num_bytes > mem_used)
      {
         printf("*** HEAP ERROR: Freeing block of %lu bytes, but only %lu bytes on the heap!\n",
                num_bytes, mem_used);
      }
      else
      {
         mem_used -= num_bytes;
      }

      total_free_time += time_taken;
      num_frees++;
      total_num_frees++;
      STB_OSMutexUnlock(heap_mutex);

#elif defined(CHECK_MEMORY)

      /* Check the memory still has the set values at the start and end */
      ptr = (U8BIT*) addr;
      ptr -= 8;

      if ((*(ptr + 1) == (U8BIT) 'O') && (*(ptr + 3) == (U8BIT) 'B') &&
          (*(ptr + 5) == (U8BIT) 'S') && (*(ptr + 7) == (U8BIT) 'L'))
      {
         /* The start of memory seems to be correct, so get the size of the block */
         num_bytes = *ptr << 24;
         num_bytes += *(ptr + 2) << 16;
         num_bytes += *(ptr + 4) << 8;
         num_bytes += *(ptr + 6);

         /* Now check the end of the block */
         if ((*(ptr + 8 + num_bytes) != (U8BIT) 'L') || (*(ptr + 9 + num_bytes) != (U8BIT) 'S') ||
             (*(ptr + 10 + num_bytes) != (U8BIT) 'B') || (*(ptr + 11 + num_bytes) != (U8BIT) 'O'))
         {
            printf("*** HEAP CORRUPTED: At end of %p, size %lu\n", addr, num_bytes);
         }
      }
      else
      {
         printf("*** HEAP CORRUPTED: At start of %p\n", addr);
      }

#ifdef USE_PRIVATE_HEAP
      HEAPFree(ptr);
#else
      free(ptr);
#endif

#else

#ifdef USE_PRIVATE_HEAP
      HEAPFree(addr);
#else
      free(addr);
#endif

#endif
   }

   FUNCTION_FINISH(FreeMem);
}


/*!**************************************************************************
 * @fn      ReadSecureData
 * @brief   Read secure data from a file
 * @param   filename - name of secure data file
 * @warning None.
 * @bug     None.
 ****************************************************************************/
static void ReadSecureData(void)
{
   void *secure_buffer = NULL;
   U8BIT *buffer;
   U8BIT key;
   U32BIT data_size;
   S32BIT padded_size;
   S_SECURE_DATA *item;
   U8BIT raw_data_size[4];
   int secure_buffer_len;
   S32BIT offset;

   FUNCTION_START(ReadSecureData);

#if 0
   secure_buffer_len = CI_GetSecureCiData(&secure_buffer);
#else
   secure_buffer_len = 0;
#endif
   if (secure_buffer_len > 0)
   {
      buffer = (U8BIT *)secure_buffer;
      offset = 0;

      /* Read item key */
      while (offset < secure_buffer_len)
      {
         key = buffer[offset++];

         /* Read data size: this is the "raw" size. The file actually
          * contains encrypted data which is padded for AES.
          */
         if (offset + (int) sizeof(raw_data_size) > secure_buffer_len)
         {
            /* Invalid data */
            break;
         }

         memcpy(raw_data_size, &buffer[offset], sizeof(raw_data_size));
         data_size = (raw_data_size[0] << 24 | raw_data_size[1] << 16 |
                      raw_data_size[2] << 8  | raw_data_size[3]);
         padded_size = AES_PAD(data_size);
         offset += sizeof(raw_data_size);

         SEC_DBG(("ReadSecureData: Reading key %d, len %ld (%ld)",
                  key, data_size, padded_size));

         item = (S_SECURE_DATA *)STB_MEMGetSysRAM(sizeof(S_SECURE_DATA));
         if (item != NULL)
         {
            item->value = STB_MEMGetSysRAM(padded_size);
            if (item->value != NULL)
            {
               /* Read item data (encrypted) */
               if (offset + padded_size > secure_buffer_len)
               {
                  /* Invalid data */
                  SEC_DBG(("ReadSecureData: Invalid data (offset=%d, padded_size=%d, len=%d)",
                           offset, padded_size, secure_buffer_len));
                  STB_MEMFreeSysRAM(item->value);
                  STB_MEMFreeSysRAM(item);
                  break;
               }

               memcpy(item->value, &buffer[offset], padded_size);
               offset += padded_size;

               /* Add item */
               item->key = key;
               item->size = data_size;
               item->next = secure_const_list;
               secure_const_list = item;
            }
            else
            {
               /* Out of memory */
               SEC_DBG(("ReadSecureData: Out of memory!"));
               STB_MEMFreeSysRAM(item);
            }
         }
      }

#if 0
      CI_ReleaseSecureCiData(secure_buffer);
#endif
   }
#ifdef SECURE_DEBUG
   else
   {
      SEC_DBG(("ReadSecureData: Please check if secure data is available."));
      SEC_DBG(("ReadSecureData: as CI+ will not work without it."));
   }
#endif

   FUNCTION_FINISH(ReadSecureData);
}



/*!**************************************************************************
 * @fn      PrefixPath
 * @brief   copy the file path in name to path prepending a
 *          writeable directory path.
 ****************************************************************************/
static void PrefixPath(char *name, char *path)
{
   if (name[0] == '/') {
      snprintf(path,PLATFORM_MAXPATH,"%s%s",DTVKIT_NVM_PATH,name);
   } else {
      snprintf(path,PLATFORM_MAXPATH,"%s/%s",DTVKIT_NVM_PATH,name);
   }
}


/****************************************************************************
** End of file
*****************************************************************************/
