/*!****************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 ******************************************************************************/
/**
 * @brief   Network functions
 * @file    stbhwnet.c
 * @date    June 2009
 * @author  Sergio Panseri
 */

/*--- Includes ----------------------------------------------------------------*/

/* System header Files */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/ioctl.h>
#include <netinet/ip.h>
#include <net/if.h>
#include <linux/sockios.h>
#include <linux/ethtool.h>
#include <linux/rtnetlink.h>
#include <errno.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <pthread.h>

/* STB header Files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwmem.h"
#include "stbhwos.h"
#include "stbhwc.h"
#include "stbhwnet.h"
#include "stbhwip.h"

#ifdef system
#undef system
#endif
#define system(cmd) execl("/vendor/bin/sh", "sh", "-c", cmd, NULL)

#ifdef ioctl
#undef ioctl
#endif
#define ioctl(...) (-1)


/*--- Preprocessor definitions ------------------------------------------------*/

#define MAX_IFS               64
#define GATEWAY_BUFSIZE       8192
#define RESOLV_CONF           "/etc/resolv.conf"
#define NW_TASK_STACK_SIZE    1024
#define NW_TASK_PRIORITY      8

#define NETWORK_ERROR
// #define NETWORK_DEBUG

#ifdef NETWORK_ERROR
#define NET_ERR(x)            STB_SPDebugWrite x
#else
#define NET_ERR(x)
#endif

#ifdef NETWORK_DEBUG
#define NET_DBG(x)            STB_SPDebugWrite x
#else
#define NET_DBG(x)
#endif

#if 0
#ifdef FUNCTION_START
#undef FUNCTION_START
#endif
#define FUNCTION_START(x)     printf(">>> %s\n", #x);

#ifdef FUNCTION_FINISH
#undef FUNCTION_FINISH
#endif
#define FUNCTION_FINISH(x)    printf("<<< %s\n", #x);
#endif



/*--- Local types definitions -------------------------------------------------*/

typedef struct
{
   int socket;
} S_SOCKET_DESC;

typedef struct ethernet_monitor
{
   struct ethernet_monitor *next;
   NW_eth_callback function;
} S_ETHERNET_MONITOR;

typedef struct route_info
{
   struct in_addr dst_addr;
   struct in_addr src_addr;
   struct in_addr gateway;
   char if_name[IF_NAMESIZE];
} S_ROUTE_INFO;



/*--- Local (static) variable declarations ------------------------------------*/

static S_ETHERNET_MONITOR *nw_monitor_list = NULL;
static void *nw_eth_task_ptr = NULL;
static void *nw_mutex = NULL;
static int nw_eth_status_running = 0;
static E_NW_LINK_STATUS current_ethernet_status;



/*--- Local function prototypes -----------------------------------------------*/

static void EthernetMonitorTask(void *arg);
static int ReadSockLength(int sock_fd, char *buf_ptr, int seq_num, int p_id);
static int ParseRoutes(struct nlmsghdr *nl_hdr, S_ROUTE_INFO *rt_info);


/*--- Global function definitions ---------------------------------------------*/

/*!**************************************************************************
 * @fn      STB_NWInitialise
 * @brief   Initialises the socket API, must be called once before using API
 * @return  TRUE if socket API initialised ok, FALSE if failed
 ****************************************************************************/
BOOLEAN STB_NWInitialise(void)
{
   FUNCTION_START(STB_NWInitialise);

   nw_eth_status_running = 1;
   nw_mutex = STB_OSCreateMutex();
#if 0
   nw_eth_task_ptr = STB_OSCreateTask(EthernetMonitorTask, NULL, NW_TASK_STACK_SIZE,
                                      NW_TASK_PRIORITY, (U8BIT*) "ethtsk");
#endif

   FUNCTION_FINISH(STB_NWInitialise);
   return TRUE;
}



/*!**************************************************************************
 * @fn      STB_NWSelectInterface
 * @brief   Sets the network interface that will be used for all network or IP operations
 * @param   interface - network interface type to use
 * @return  TRUE if interface is available, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_NWSelectInterface(E_NW_INTERFACE interface)
{
   BOOLEAN retval;
   S_ETHERNET_MONITOR *p_nw_monitor;

   FUNCTION_START(STB_NWSelectInterface);

   if (interface == NW_WIRED)
   {
      if (current_ethernet_status != NW_LINK_DISABLED)
      {
         STB_OSMutexLock(nw_mutex);

         current_ethernet_status = NW_LINK_DISABLED;

         NET_DBG(("%s: current ethernet status changed to %s", __FUNCTION__,
                 ((current_ethernet_status == NW_LINK_ACTIVE) ? "active" :
                  (current_ethernet_status == NW_LINK_INACTIVE) ? "inactive" : "disabled")));

         /* Inform all monitors that the state has changed */
         p_nw_monitor = nw_monitor_list;
         while (p_nw_monitor)
         {
            (p_nw_monitor->function)(interface, current_ethernet_status);
            p_nw_monitor = p_nw_monitor->next;
         }
         STB_OSMutexUnlock(nw_mutex);
      }
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(STB_NWSelectInterface);
   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWGetSelectedInterface
 * @brief   Returns the selected interface
 * @return  Selected interface
 ****************************************************************************/
E_NW_INTERFACE STB_NWGetSelectedInterface(void)
{
   FUNCTION_START(STB_NWGetSelectedInterface);
   FUNCTION_FINISH(STB_NWGetSelectedInterface);

   return NW_WIRED;
}



/*!**************************************************************************
 * @fn      STB_NWGetLinkStatus
 * @brief   Get the selected interface link status
 * @return  NW_LINK_ACTIVE   cable connected,
 *          NW_LINK_INACTIVE cable dis-connected
 *          NW_LINK_DISABLED no ethernet device or other error
 ****************************************************************************/
E_NW_LINK_STATUS STB_NWGetLinkStatus(void)
{
   FUNCTION_START(STB_NWGetLinkStatus);
   NET_DBG(("STB_NWGetLinkStatus: %s", ((current_ethernet_status == NW_LINK_ACTIVE) ? "active" :
           (current_ethernet_status == NW_LINK_INACTIVE) ? "inactive" : "disabled")));
   FUNCTION_FINISH(STB_NWGetLinkStatus);
   return current_ethernet_status;
}



/*!**************************************************************************
 * @fn      STB_NWStartEthernetMonitor
 * @brief   Start monitoring the ethernet status
 * @param   func - callback function to notify status change to ethernet device
 * @return  handle
 ****************************************************************************/
NW_handle STB_NWStartEthernetMonitor(NW_eth_callback func)
{
   S_ETHERNET_MONITOR *monitor;

   FUNCTION_START(STB_NWStartEthernetMonitor);
   if (func == NULL)
   {
      monitor = NULL;
   }
   else
   {
      monitor = (S_ETHERNET_MONITOR*) STB_MEMGetSysRAM(sizeof(S_ETHERNET_MONITOR));

      if (monitor)
      {
         monitor->function = func;

         STB_OSMutexLock(nw_mutex);
         monitor->next = nw_monitor_list;
         nw_monitor_list = monitor;
         STB_OSMutexUnlock(nw_mutex);
      }
   }
   FUNCTION_FINISH(STB_NWStartEthernetMonitor);

   return monitor;
}



/*!**************************************************************************
 * @fn      STB_NWStopEthernetMonitor
 * @brief   Stop monitoring the ethernet status
 ****************************************************************************/
void STB_NWStopEthernetMonitor(NW_handle hdl)
{
   S_ETHERNET_MONITOR *monitor;
   S_ETHERNET_MONITOR *prev_monitor;

   FUNCTION_START(STB_NWStopEthernetMonitor);

   if (hdl && nw_monitor_list)
   {
      monitor = (S_ETHERNET_MONITOR*) hdl;
      STB_OSMutexLock(nw_mutex);
      if (monitor == nw_monitor_list)
      {
         nw_monitor_list = monitor->next;
      }
      else
      {
         prev_monitor = nw_monitor_list;
         while (prev_monitor->next)
         {
            if (prev_monitor->next == monitor)
            {
               prev_monitor->next = monitor->next;
               break;
            }
            prev_monitor = prev_monitor->next;
         }
      }
      STB_OSMutexUnlock(nw_mutex);

      STB_MEMFreeSysRAM(hdl);
   }

   FUNCTION_FINISH(STB_NWStopEthernetMonitor);
}



/*!**************************************************************************
 * @fn      STB_NWGetMACAddress
 * @brief   Gets the MAC address of the default ethernet connection
 * @param   interface - NW_WIRED or NW_WIRELESS
 * @param   mac_address - 6 byte array in which the address will be returned
 *          with the most significant byte in mac_addr[0]
 * @return  TRUE if address is returned, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_NWGetMACAddress(E_NW_INTERFACE interface, U8BIT *mac_addr)
{
   BOOLEAN addr_found;
   struct ifreq ifr;
   struct ifreq *pifr;
   struct ifreq *last_ifr;
   struct ifconf ifc;
   U32BIT buffer_size;
   char *buffer;
   BOOLEAN overflowed;
   int sock;

   FUNCTION_START(STB_NWGetMACAddress);
   USE_UNWANTED_PARAM(interface);

   addr_found = FALSE;

   if ((sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP)) >= 0)
   {
      buffer = NULL;
      buffer_size = 1024;
      overflowed = FALSE;

      do
      {
         if (buffer != NULL)
         {
            /* Free the current buffer and allocate a bigger one */
            STB_MEMFreeSysRAM(buffer);
            buffer_size *= 2;
         }

         if ((buffer = (char *)STB_MEMGetSysRAM(buffer_size)) != NULL)
         {
            ifc.ifc_len = buffer_size;
            ifc.ifc_buf = buffer;
            overflowed = FALSE;

            if (ioctl(sock, SIOCGIFCONF, &ifc) >= 0)
            {
               if (ifc.ifc_len == buffer_size)
               {
                  /* Buffer passed in has been filled, so a bigger buffer is needed */
                  overflowed = TRUE;
               }
            }
            else
            {
               NET_ERR(("%s: ioctl(SIOCGIFCONF) failed, errno %d", __FUNCTION__, errno));
               STB_MEMFreeSysRAM(buffer);
               buffer = NULL;
            }
         }
#ifdef NETWORK_ERROR
         else
         {
            NET_ERR(("%s: Failed to allocate buffer of %lu bytes", __FUNCTION__, buffer_size));
         }
#endif
      }
      while (overflowed && (buffer != NULL));

      if (buffer != NULL)
      {
         last_ifr = ifc.ifc_req + (ifc.ifc_len / sizeof(struct ifreq));

         for (pifr = ifc.ifc_req; (pifr != last_ifr) && !addr_found; pifr++)
         {
            strcpy(ifr.ifr_name, pifr->ifr_name);
            if (ioctl(sock, SIOCGIFFLAGS, &ifr) == 0)
            {
               /* Skip the loopback network */
               if ((ifr.ifr_flags & IFF_LOOPBACK) == 0)
               {
                  if (ioctl(sock, SIOCGIFHWADDR, &ifr) == 0)
                  {
                     /* Found the address, copy it to the output */
                     memcpy(mac_addr, ifr.ifr_hwaddr.sa_data, 6);
                     addr_found = TRUE;
                     NET_DBG(("%s: MAC=%02x:%02x:%02x:%02x:%02x:%02x", __FUNCTION__, mac_addr[0],
                              mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5]));
                  }
               }
            }
         }

         STB_MEMFreeSysRAM(buffer);
      }
   }
#ifdef NETWORK_ERROR
   else
   {
      NET_ERR(("%s: Failed to open socket", __FUNCTION__));
   }
#endif

   FUNCTION_FINISH(STB_NWGetMACAddress);

   return addr_found;
}



/*!**************************************************************************
 * @fn      STB_NWLookupAddress
 * @brief   Performs a lookup to find the IP address(es) of the given host name
 * @param   name - host name
 * @param   nw_addrs - array of structures containing the results of the lookup.
 *          This array must be deleted using STB_MEMFreeSysRAM.
 * @return  The number of addresses returned in the array
 ****************************************************************************/
U16BIT STB_NWLookupAddress(U8BIT *name, S_NW_ADDR_INFO **nw_addrs)
{
   U16BIT num_addrs;
   struct addrinfo *addrs;
   struct addrinfo *ra;
   U16BIT i;

   FUNCTION_START(STB_NWLookupAddress);

   if (getaddrinfo((char*) name, NULL, NULL, &addrs) == 0)
   {
      /* Count the number of returned addresses */
      for (ra = addrs, num_addrs = 0; ra != NULL; ra = ra->ai_next)
      {
         if (((ra->ai_family == AF_INET) || (ra->ai_family == AF_INET6)) &&
             ((ra->ai_socktype == SOCK_DGRAM) || (ra->ai_socktype == SOCK_STREAM)) &&
             ((ra->ai_protocol == IPPROTO_UDP) || (ra->ai_protocol == IPPROTO_TCP) ||
              (ra->ai_protocol == IPPROTO_IP)))
         {
            num_addrs++;
         }
      }

      if (num_addrs > 0)
      {
         /* Allocate array to be returned */
         *nw_addrs = (S_NW_ADDR_INFO*) STB_MEMGetSysRAM(num_addrs * sizeof(S_NW_ADDR_INFO));
         if (*nw_addrs != NULL)
         {
            for (ra = addrs, i = 0; ra != NULL; ra = ra->ai_next)
            {
               if (((ra->ai_family == AF_INET) || (ra->ai_family == AF_INET6)) &&
                   ((ra->ai_socktype == SOCK_DGRAM) || (ra->ai_socktype == SOCK_STREAM)) &&
                   ((ra->ai_protocol == IPPROTO_UDP) || (ra->ai_protocol == IPPROTO_TCP) ||
                    (ra->ai_protocol == IPPROTO_IP)))
               {
                  if (ra->ai_family == AF_INET6)
                  {
                     (*nw_addrs)[i].af = NW_AF_INET6;
                  }
                  else
                  {
                     (*nw_addrs)[i].af = NW_AF_INET;
                  }

                  if (ra->ai_socktype == SOCK_DGRAM)
                  {
                     (*nw_addrs)[i].type = NW_SOCK_DGRAM;
                  }
                  else
                  {
                     (*nw_addrs)[i].type = NW_SOCK_STREAM;
                  }

                  if (ra->ai_protocol == IPPROTO_UDP)
                  {
                     (*nw_addrs)[i].protocol = NW_PROTOCOL_UDP;
                  }
                  else
                  {
                     (*nw_addrs)[i].protocol = NW_PROTOCOL_TCP;
                  }

                  if (ra->ai_family == AF_INET)
                  {
                     if (inet_ntop(ra->ai_family,
                                   (void*) &((struct sockaddr_in*) ra->ai_addr)->sin_addr,
                                   (char*)(*nw_addrs)[i].addr,
                                   sizeof((*nw_addrs)[i].addr)) == NULL)
                     {
                        NET_ERR(("%s: inet_ntop failed, errno %d", __FUNCTION__, errno));
                     }
                  }
                  else
                  {
                     if (inet_ntop(ra->ai_family,
                                   (void*) &((struct sockaddr_in6*) ra->ai_addr)->sin6_addr,
                                   (char*)(*nw_addrs)[i].addr,
                                   sizeof((*nw_addrs)[i].addr)) == NULL)
                     {
                        NET_ERR(("%s: inet_ntop failed, errno %d", __FUNCTION__, errno));
                     }
                  }

                  i++;
               }
            }
         }
      }

      freeaddrinfo(addrs);
   }
   else
   {
      NET_ERR(("STB_NWLookupAddress: Failed to lookup address for \"%s\", errno %d", name, errno));
      num_addrs = 0;
   }

   FUNCTION_FINISH(STB_NWLookupAddress);

   return num_addrs;
}



/*!**************************************************************************
 * @fn      STB_NWOpenSocket
 * @brief   Opens (creates) a new socket for subsequent use
 * @param   af - the address family that the socket will be used with
 * @param   type - the stream type the socket will be used with
 * @param   protocol - the protocol the socket will be used with
 * @return  The socket handle, or NULL if failed
 ****************************************************************************/
void* STB_NWOpenSocket(E_NW_AF af, E_NW_TYPE type, E_NW_PROTOCOL protocol, BOOLEAN nonblock)
{
   void *retval;
   int fd;
   int socket_family;
   int socket_type;
   S_SOCKET_DESC *socket_desc;

   FUNCTION_START(STB_NWOpenSocket);

   USE_UNWANTED_PARAM(protocol); /* used in windows socket implementation */
   USE_UNWANTED_PARAM(nonblock);

   switch (af)
   {
      case NW_AF_INET:
         socket_family = AF_INET;
         break;

      case NW_AF_INET6:
         socket_family = AF_INET6;
         break;

      default:
         socket_family = -1;
         break;
   }

   switch (type)
   {
      case NW_SOCK_STREAM:
         socket_type = SOCK_STREAM;
         break;

      case NW_SOCK_DGRAM:
         socket_type = SOCK_DGRAM;
         break;

      default:
         socket_type = -1;
         break;
   }

   retval = NULL;
   if ((socket_type != -1) && (socket_family != -1))
   {
      fd = socket(socket_family, socket_type, 0);
      if (fd >= 0)
      {
         socket_desc = (S_SOCKET_DESC*) STB_MEMGetAppRAM(sizeof(S_SOCKET_DESC));
         if (socket_desc != NULL)
         {
            socket_desc->socket = fd;
            retval = (void*) socket_desc;
         }
         else
         {
            close(fd);
         }
      }
   }
   else
   {
      NET_ERR(("STB_NWOpenSocket: wrong parameter af=%d, type=%d, prot=%d", af, type, protocol));
   }

   FUNCTION_FINISH(STB_NWOpenSocket);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWCloseSocket
 * @brief   Closes (destroys) a socket instance
 * @param   socket - handle of the socket to be closed
 ****************************************************************************/
BOOLEAN STB_NWCloseSocket(void *socket)
{
   S_SOCKET_DESC *socket_desc;
   BOOLEAN retval;

   FUNCTION_START(STB_NWCloseSocket);

   retval = FALSE;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;

      shutdown(socket_desc->socket, SHUT_RDWR);
      if (close(socket_desc->socket) == 0)
      {
         STB_MEMFreeAppRAM((void*) socket_desc);
         retval = TRUE;
      }
#ifdef NETWORK_ERROR
      else
      {
         NET_ERR(("STB_NWCloseSocket: close failed, errno=%u", errno));
      }
#endif
   }

   FUNCTION_FINISH(STB_NWCloseSocket);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWBind
 * @brief   Binds (names) a socket in the local address space
 * @param   socket - the handle of the socket to be bound
 * @param   af - the address family to be bound to
 * @param   address - the string address (NULL to let OS choose)
 * @param   port - the port number to be bound to (zero to let OS choose)
 * @return  TRUE if successfully bound, FALSE if failed to bind
 ****************************************************************************/
BOOLEAN STB_NWBind(void *socket, U8BIT *address, U32BIT port)
{
   S_SOCKET_DESC *socket_desc;
   struct sockaddr_in addr;
   BOOLEAN retval;

   FUNCTION_START(STB_NWBind);

   retval = FALSE;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;
      memset(&addr, 0, sizeof(struct sockaddr_in));
      addr.sin_family = AF_INET;
      if (address == NULL)
      {
         NET_DBG(("Binding, address ANY"));
         addr.sin_addr.s_addr = INADDR_ANY;
      }
      else
      {
         NET_DBG(("Binding with address %s", address));
         addr.sin_addr.s_addr = inet_addr((char*) address);
      }
      NET_DBG(("Binding port %d", port));
      addr.sin_port = htons(port);
      if (bind(socket_desc->socket, (struct sockaddr*) &addr, sizeof(struct sockaddr)) >= 0)
      {
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_NWBind);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWSetReuseaddr
 * @brief   Sets the socket option SO_REUSEADDR
 * @param   socket - the handle of the socket to be bound
 * @param   state - TRUE or FALSE to activate/deactivate the
 *          option REUSEADDR
 * @return  TRUE if successfully set, FALSE if failed to set
 ****************************************************************************/
BOOLEAN STB_NWSetReuseaddr(void *socket, BOOLEAN state)
{
   S_SOCKET_DESC *socket_desc;
   BOOLEAN retval;
   int optval;

   FUNCTION_START(STB_NWSetReuseaddr);

   retval = FALSE;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;
      if (state)
      {
         optval = 1;
      }
      else
      {
         optval = 0;
      }

      if (setsockopt(socket_desc->socket, SOL_SOCKET, SO_REUSEADDR, (char*) &optval, 4) == 0)
      {
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_NWSetReuseaddr);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWGetReuseaddr
 * @brief   Gets the socket option SO_REUSEADDR
 * @param   socket - the handle of the socket
 * @param   state - TRUE or FALSE if the option is actie/inactivate
 * @return  TRUE if successfully got, FALSE if failed to get
 ****************************************************************************/
BOOLEAN STB_NWGetReuseaddr(void *socket, BOOLEAN *state)
{
   S_SOCKET_DESC *socket_desc;
   BOOLEAN retval;
   int optval;
   socklen_t option_len;

   FUNCTION_START(STB_NWGetReuseaddr);

   retval = FALSE;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;
      if (getsockopt(socket_desc->socket, SOL_SOCKET, SO_REUSEADDR,
                     (char*) &optval, &option_len) == 0)
      {
         if (optval > 0)
         {
            *state = TRUE;
         }
         else
         {
            *state = FALSE;
         }

         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_NWGetReuseaddr);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWAddMembership
 * @brief   Sets the ip option IP_ADD_MEMBERSHIP
 * @param   socket - the handle of the socket
 * @param   group_address - address to add
 * @return  TRUE if successfully set, FALSE if failed to set
 ****************************************************************************/
BOOLEAN STB_NWAddMembership(void *socket, U8BIT *group_address)
{
   S_SOCKET_DESC *socket_desc;
   struct ip_mreq imr;
   BOOLEAN retval;

   FUNCTION_START(STB_NWAddMembership);

   retval = FALSE;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;

      /* This block configures the socket for multicast */
      memset(&imr, 0, sizeof(struct ip_mreq));
      imr.imr_multiaddr.s_addr = inet_addr((char*) group_address);
      imr.imr_interface.s_addr = INADDR_ANY;
      if (setsockopt(socket_desc->socket, IPPROTO_IP, IP_ADD_MEMBERSHIP,
                     (char*) &imr, sizeof(struct ip_mreq)) == 0)
      {
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_NWAddMembership);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWDropMembership
 * @brief   Sets the ip option IP_DROP_MEMBERSHIP
 * @param   socket - the handle of the socket
 * @param   group_address - address to drop
 * @return  TRUE if successfully set, FALSE if failed to set
 ****************************************************************************/
BOOLEAN STB_NWDropMembership(void *socket, U8BIT *group_address)
{
   S_SOCKET_DESC *socket_desc;
   struct ip_mreq imr;
   BOOLEAN retval;

   FUNCTION_START(STB_NWDropMembership);

   retval = FALSE;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;

      imr.imr_multiaddr.s_addr = inet_addr((char*) group_address);
      imr.imr_interface.s_addr = INADDR_ANY;
      if (setsockopt(socket_desc->socket, IPPROTO_IP, IP_DROP_MEMBERSHIP,
                     (char*) &imr, sizeof(struct ip_mreq)) == 0)
      {
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_NWDropMembership);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWGetSocketName
 * @brief   Gets the socket's "name" (family, address and port)
 * @param   socket - the handle of the socket
 * @param   af - pointer to the returned family
 * @param   address - pointer to the returned address string
 * @param   port - pointer to the returned port
 * @return  TRUE successfully set, FALSE failed to set
 ****************************************************************************/
BOOLEAN STB_NWGetSocketName(void *socket, E_NW_AF *af, U8BIT *address, U32BIT *port)
{
   S_SOCKET_DESC *socket_desc;
   BOOLEAN retval;
   struct sockaddr_in addr;
   socklen_t sock_len;
   char *tmp_address;

   FUNCTION_START(STB_NWGetSocketName);

   retval = FALSE;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;

      if (getsockname(socket_desc->socket, (struct sockaddr*) &addr, &sock_len) == 0)
      {
         retval = TRUE;
         tmp_address = inet_ntoa(addr.sin_addr);
         if ((tmp_address != NULL) && (address != NULL))
         {
            strcpy((char*) address, tmp_address);
         }
         if (port != NULL)
         {
            *port = addr.sin_port;
         }
         if (af != NULL)
         {
            switch (addr.sin_family)
            {
               case AF_INET:
                  *af = NW_AF_INET;
                  break;

               case AF_INET6:
                  *af = NW_AF_INET6;
                  break;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_NWGetSocketName);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWConnect
 * @brief   Connects the socket to a remote host
 * @param   socket - the handle of the socket
 * @param   address - address string
 * @param   port - port
 * @return  TRUE if successfully connected, FALSE if failed to set
 ****************************************************************************/
E_NW_ERROR STB_NWConnect(void *socket, U8BIT *address, U32BIT port)
{
   S_SOCKET_DESC *socket_desc;
   struct sockaddr_in addr;
   E_NW_ERROR retval;

   FUNCTION_START(STB_NWConnect);

   retval = NW_ERROR;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;
      memset(&addr, 0, sizeof(struct sockaddr_in));
      addr.sin_family = AF_INET;
      addr.sin_addr.s_addr = inet_addr((char*) address);
      addr.sin_port = htons(port);
      if (connect(socket_desc->socket, (struct sockaddr*) &addr, sizeof(struct sockaddr)) >= 0)
      {
         retval = NW_OK;
      }
#ifdef NETWORK_ERROR
      else
      {
         NET_ERR(("STB_NWConnect: connect failed, errno = %d", errno));
      }
#endif
   }
#ifdef NETWORK_ERROR
   else
   {
      NET_ERR(("STB_NWConnect: socket id NULL"));
   }
#endif

   FUNCTION_FINISH(STB_NWConnect);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWListen
 * @brief   Put socket into state of waiting for incoming connection
 * @param   socket - the handle of the socket to begin listening
 * @param   backlog - the maximum length of the queue of pending connections
 * @return  TRUE if successfully connected, FALSE if failed to set
 ****************************************************************************/
BOOLEAN STB_NWListen(void *socket, S32BIT backlog)
{
   S_SOCKET_DESC *socket_desc;
   BOOLEAN retval;

   FUNCTION_START(STB_NWListen);

   retval = FALSE;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;
      if (listen(socket_desc->socket, backlog) == 0)
      {
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_NWListen);

   return retval;
}


/*!**************************************************************************
 * @fn      STB_NWAccept
 * @brief   Accepts an incoming connection attempt on a socket
 * @param   socket - the handle of the socket in the listening state
 * @param   address - pointer to the returned address (string) of
 *          connecting entity
 * @param   port - pointer to the returned port of connecting entity
 * @return  handle of new socket actually connected, NULL if failed
 ****************************************************************************/
void* STB_NWAccept(void *socket, U8BIT *address, U32BIT *port)
{
   S_SOCKET_DESC *socket_desc;
   int new_socket;
   S_SOCKET_DESC *new_desc;
   socklen_t sock_len;
   struct sockaddr_in addr;
   char *tmp_address;

   FUNCTION_START(STB_NWAccept);

   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;

      new_socket = accept(socket_desc->socket, (struct sockaddr*) &addr, &sock_len);
      if (new_socket >= 0)
      {
         new_desc = (S_SOCKET_DESC*) STB_MEMGetAppRAM(sizeof(S_SOCKET_DESC));
         if (new_desc != NULL)
         {
            new_desc->socket = new_socket;
            tmp_address = inet_ntoa(addr.sin_addr);
            if ((tmp_address != NULL) && (address != NULL))
            {
               strcpy((char*) address, tmp_address);
            }
            if (port != NULL)
            {
               *port = ntohs(addr.sin_port);
            }
         }
      }
   }

   FUNCTION_FINISH(STB_NWAccept);

   return (void*) new_desc;
}



/*!**************************************************************************
 * @fn      STB_NWSend
 * @brief   Sends data on a connected socket
 * @param   socket - the handle of the (connected) socket on which to send
 * @param   buf - the buffer holding the data to be sent
 * @param   num_bytes - the number of bytes in buf to be sent
 * @return  the number of bytes sent (may be less than num_bytes), -1 if failed
 ****************************************************************************/
S32BIT STB_NWSend(void *socket, U8BIT *buf, U32BIT num_bytes)
{
   S_SOCKET_DESC *socket_desc;
   S32BIT retval;

   FUNCTION_START(STB_NWSend);

   retval = -1;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;
      retval = send(socket_desc->socket, buf, (size_t) num_bytes, 0);
      if (retval <= 0)
      {
         NET_ERR(("STB_NWSend: Send failed errno = %d", errno));
      }
   }
   else
   {
      NET_ERR(("STB_NWSend: socket id NULL"));
   }

   FUNCTION_FINISH(STB_NWSend);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWReceive
 * @brief   Receives data from a connected socket
 * @param   socket - the handle of the socket from which to read
 * @param   buf - the buffer to hold the read data
 * @param   max_bytes - the maximum bytes the caller can accept in the buffer
 * @return  the number of bytes read, -1 if there was an error reading,
 *          0 if the connection was closed
 ****************************************************************************/
S32BIT STB_NWReceive(void *socket, U8BIT *buf, U32BIT max_bytes)
{
   S_SOCKET_DESC *socket_desc;
   S32BIT retval = -1;

   FUNCTION_START(STB_NWReceive);

   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;
      retval = recv(socket_desc->socket, buf, (size_t) max_bytes, MSG_DONTWAIT);
#ifdef NETWORK_ERROR
      if (retval < 0)
      {
         NET_ERR(("STB_NWReceive: Receive failed errno = %d", errno));
      }
#endif
   }
#ifdef NETWORK_ERROR
   else
   {
      NET_ERR(("STB_NWReceive: socket id NULL"));
   }
#endif

   FUNCTION_FINISH(STB_NWReceive);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWReceiveFrom
 * @brief   Receives a datagram and returns the data and its source address
 * @param   socket - the handle of the socket from which to read
 * @param   buf - the buffer to hold the read data
 * @param   max_bytes - the maximum bytes the caller can accept in the buffer
 * @param   address - the source address of the data in string form
 * @param   port - the source port of the data
 * @return  the number of bytes read, -1 if there was an error reading,
 *          0 if the connection was closed
 ****************************************************************************/
S32BIT STB_NWReceiveFrom(void *socket, U8BIT *buf, U32BIT max_bytes, U8BIT *address, U32BIT *port)
{
   S_SOCKET_DESC *socket_desc;
   S32BIT retval;
   struct sockaddr_in addr;
   struct sockaddr_in *addr_ptr;
   socklen_t addr_len;
   ssize_t ret;
   char *tmp_address;

   FUNCTION_START(STB_NWReceiveFrom);

   retval = -1;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;

      if ((address == NULL) || (port == NULL))
      {
         /* Receive for a connected socket */
         addr_ptr = NULL;
      }
      else
      {
         addr_ptr = &addr;
      }

      ret = recvfrom(socket_desc->socket, (void*) buf, (size_t) max_bytes,
                     0, (struct sockaddr*) addr_ptr, &addr_len);
      retval = (S32BIT) ret;
      if (ret >= 0)
      {
         if (addr_ptr != NULL)
         {
            tmp_address = inet_ntoa(addr.sin_addr);
            if (tmp_address != NULL)
            {
               strcpy((char*) address, tmp_address);
            }
            *port = ntohs(addr.sin_port);
         }
      }
      else
      {
         NET_ERR(("STB_NWReceiveFrom: recvfrom error, errno = %d", errno));
      }
   }
   else
   {
      NET_ERR(("STB_NWReceiveFrom: socket id NULL"));
   }

   FUNCTION_FINISH(STB_NWReceiveFrom);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWSendTo
 * @brief   Sends data to a specific destination
 * @param   socket - the handle of the socket on which to send
 * @param   buf - the buffer holding the data to be sent
 * @param   num_bytes - the number of bytes in buf to be sent
 * @param   address - the address (in string form) of the target socket
 * @param   port - the port number of the target socket
 * @return  the number of bytes sent (may be less than num_bytes), -1 if failed
 ****************************************************************************/
S32BIT STB_NWSendTo(void *socket, U8BIT *buf, U32BIT num_bytes, U8BIT *address, U32BIT port)
{
   S_SOCKET_DESC *socket_desc;
   S32BIT retval;
   struct sockaddr_in addr;

   FUNCTION_START(STB_NWSendTo);

   retval = -1;
   if (socket != NULL)
   {
      socket_desc = (S_SOCKET_DESC*) socket;

      memset(&addr, 0, sizeof(struct sockaddr_in));
      addr.sin_family = AF_INET;
      addr.sin_addr.s_addr = inet_addr((char*) address);
      addr.sin_port = htons(port);
      retval = sendto(socket_desc->socket, buf, num_bytes, 0,
                      (struct sockaddr*) &addr, sizeof(struct sockaddr_in));
   }

   FUNCTION_FINISH(STB_NWSendTo);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWSockIsSet
 * @brief   Returns whether a specified socket is a member of a specified set
 * @param   socket - the socket to check
 * @param   socks - the set of sockets to check
 * @return  TRUE if socket is a member of set, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_NWSockIsSet(void *socket, S_NW_SOCKSET *socks)
{
   BOOLEAN retval;
   U16BIT i;

   FUNCTION_START(STB_NWSockIsSet);

   retval = FALSE;
   if ((socket != NULL) && (socks != NULL))
   {
      i = 0;
      while (!retval && (i < socks->sock_count))
      {
         if (socks->sock_array[i] == socket)
         {
            retval = TRUE;
         }
         else
         {
            i++;
         }
      }
   }

   FUNCTION_FINISH(STB_NWSockIsSet);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWSockZero
 * @brief   Clears the socket set
 * @param   socks - the set of sockets to check
 ****************************************************************************/
void STB_NWSockZero(S_NW_SOCKSET *socks)
{
   FUNCTION_START(STB_NWSockIsSet);

   if (socks != NULL)
   {
      socks->sock_count = 0;
   }

   FUNCTION_FINISH(STB_NWSockIsSet);
}



/*!**************************************************************************
 * @fn      STB_NWSockClear
 * @brief   Clears the specified socket from the specified set
 * @param   socket - the socket to clear
 * @param   socks - the set of sockets
 ****************************************************************************/
void STB_NWSockClear(void *socket, S_NW_SOCKSET *socks)
{
   U16BIT i;

   FUNCTION_START(STB_NWSockIsSet);

   if ((socket != NULL) && (socks != NULL))
   {
      for (i = 0; i < socks->sock_count; ++i)
      {
         if (socks->sock_array[i] == socket)
         {
            while (i < socks->sock_count - 1)
            {
               socks->sock_array[i] = socks->sock_array[i + 1];
               ++i;
            }
            --socks->sock_count;
            break;
         }
      }
   }

   FUNCTION_FINISH(STB_NWSockIsSet);
}



/*!**************************************************************************
 * @fn      STB_NWSockSet
 * @brief   Sets a specificed socket in a specified set
 * @param   socket - the socket to set
 * @param   socks - the set of sockets
 ****************************************************************************/
void STB_NWSockSet(void *socket, S_NW_SOCKSET *socks)
{
   FUNCTION_START(STB_NWSockIsSet);

   if ((socket != NULL) && (socks != NULL))
   {
      if (socks->sock_count < SOCK_SETSIZE)
      {
         socks->sock_array[socks->sock_count] = socket;
         ++socks->sock_count;
      }
   }

   FUNCTION_FINISH(STB_NWSockIsSet);
}




/*!**************************************************************************
 * @fn      STB_NWSelect
 * @brief   Determines the status of one or more sockets, blocking if necessary
 * @param   read_sockets - set of sockets to be checked for readability
 * @param   write_sockets - set of sockets to be checked for writability
 * @param   exceptfds - set of sockets to be checked for errors
 * @param   timeout_ms - maximum number of milliseconds to wait
 *          (-1 to block, 0 to return immediately)
 * @return  the total number of sockets that are ready, 0 if time out exceeded,
 *          -1 if an error occured
 ****************************************************************************/
S32BIT  STB_NWSelect(S_NW_SOCKSET *read_sockets, S_NW_SOCKSET *write_sockets,
                     S_NW_SOCKSET *except_sockets, S32BIT timeout_ms)
{
   S_SOCKET_DESC *socket_desc;
   S32BIT retval;
   U16BIT i;
   struct timeval *timeout_ptr;
   struct timeval timeout;
   fd_set readfds;
   fd_set *readfds_ptr;
   fd_set writefds;
   fd_set *writefds_ptr;
   fd_set exceptfds;
   fd_set *exceptfds_ptr;
   S_SOCKET_DESC *temp_socket;
   int nfds;

   FUNCTION_START(STB_NWSelect);

   readfds_ptr = NULL;
   writefds_ptr = NULL;
   exceptfds_ptr = NULL;
   retval = -1;

   if (timeout_ms >= 0)
   {
      timeout.tv_sec = timeout_ms / 1000;
      timeout.tv_usec = (timeout_ms % 1000) * 1000;
      timeout_ptr = &timeout;
   }
   else
   {
      timeout_ptr = NULL;
   }

   nfds = 0;

   /* remap the set of sockets from the OBS API to sets for the socket API */
   FD_ZERO(&readfds);
   if (read_sockets != NULL)
   {
      for (i = 0; i < read_sockets->sock_count; i++)
      {
         socket_desc = (S_SOCKET_DESC*) read_sockets->sock_array[i];
         FD_SET(socket_desc->socket, &readfds);
         if (socket_desc->socket > nfds)
         {
            nfds = socket_desc->socket;
         }
      }
      readfds_ptr = &readfds;
   }

   FD_ZERO(&writefds);
   if (write_sockets != NULL)
   {
      for (i = 0; i < write_sockets->sock_count; i++)
      {
         socket_desc = (S_SOCKET_DESC*) write_sockets->sock_array[i];
         FD_SET(socket_desc->socket, &writefds);
         if (socket_desc->socket > nfds)
         {
            nfds = socket_desc->socket;
         }
      }
      writefds_ptr = &writefds;
   }

   FD_ZERO(&exceptfds);
   if (except_sockets != NULL)
   {
      for (i = 0; i < except_sockets->sock_count; i++)
      {
         socket_desc = (S_SOCKET_DESC*) except_sockets->sock_array[i];
         FD_SET(socket_desc->socket, &exceptfds);
         if (socket_desc->socket > nfds)
         {
            nfds = socket_desc->socket;
         }
      }
      exceptfds_ptr = &exceptfds;
   }

   retval = select(nfds + 1, readfds_ptr, writefds_ptr, exceptfds_ptr, timeout_ptr);

   if (retval != -1)
   {
      retval = 0;

      /* remove all the socket handles from the OBS API sets that select did not report as ready */
      if (read_sockets != NULL)
      {
         retval = read_sockets->sock_count;

         for (i = 0; i < read_sockets->sock_count; i++)
         {
            temp_socket = (S_SOCKET_DESC*) read_sockets->sock_array[i];
            if (!FD_ISSET(temp_socket->socket, &readfds))
            {
               SOCK_CLR(temp_socket, read_sockets);
               retval--;
            }
         }
      }

      if (write_sockets != NULL)
      {
         retval += write_sockets->sock_count;

         for (i = 0; i < write_sockets->sock_count; i++)
         {
            temp_socket = (S_SOCKET_DESC*) write_sockets->sock_array[i];
            if (!FD_ISSET(temp_socket->socket, &writefds))
            {
               SOCK_CLR(temp_socket, write_sockets);
               retval--;
            }
         }
      }

      if (except_sockets != NULL)
      {
         retval += except_sockets->sock_count;

         for (i = 0; i < except_sockets->sock_count; i++)
         {
            temp_socket = (S_SOCKET_DESC*) except_sockets->sock_array[i];
            if (!FD_ISSET(temp_socket->socket, &exceptfds))
            {
               SOCK_CLR(temp_socket, except_sockets);
               retval--;
            }
         }
      }
   }
#ifdef NETWORK_ERROR
   else
   {
      NET_ERR(("STB_NWSelect: select failed, errno %d", errno));
   }
#endif

   FUNCTION_FINISH(STB_NWSelect);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_NWGetWirelessAccessPoints
 * @brief   Scans and returns all access points visible on the wireless network
 * @param   access_points - pointer to an array that will be allocated containing
 *          info on each access point found
 * @return  number of access points found
 ****************************************************************************/
U16BIT STB_NWGetWirelessAccessPoints(S_NW_ACCESS_POINT **access_points)
{
   U16BIT num_aps;

   FUNCTION_START(STB_NWGetWirelessAccessPoints);

   num_aps = 0;
   *access_points = NULL;

   FUNCTION_FINISH(STB_NWGetWirelessAccessPoints);

   return num_aps;
}



/*!**************************************************************************
 * @fn      STB_NWFreeWirelessAccessPoints
 * @brief   Frees the array of access points returned by STB_NWGetWirelessAccessPoints
 * @param   access_points - array of access points to be freed
 * @param   num_aps - number of access points in the array
 ****************************************************************************/
void STB_NWFreeWirelessAccessPoints(S_NW_ACCESS_POINT *access_points, U16BIT num_aps)
{
   FUNCTION_START(STB_NWFreeWirelessAccessPoints);
   USE_UNWANTED_PARAM(access_points);
   USE_UNWANTED_PARAM(num_aps);
   FUNCTION_FINISH(STB_NWFreeWirelessAccessPoints);
}



/*!**************************************************************************
 * @fn      STB_NWConnectToAccessPoint
 * @brief   Attempts to connect to the wireless network with the given SSID.
 *          If the network is open then 'password' can be NULL.
 * @param   essid - network to connect to
 * @param   password - password to be used for an encrypted network,
 *          can be NULL for an open network
 * @return  TRUE if connected successfully to the network, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_NWConnectToAccessPoint(U8BIT *essid, U8BIT *password)
{
   BOOLEAN retval;

   FUNCTION_START(STB_NWConnectToAccessPoint);

   USE_UNWANTED_PARAM(essid);
   USE_UNWANTED_PARAM(password);
   retval = FALSE;

   FUNCTION_FINISH(STB_NWConnectToAccessPoint);

   return retval;
}





/*--- Global functions not declared in stbhwnet.h -----------------------------*/

/*!**************************************************************************
 * @fn      NW_Terminate
 * @brief   Terminate the socket API, must be called once after using API
 ****************************************************************************/
void NW_Terminate(void)
{
   FUNCTION_START(NW_Terminate);

   nw_eth_status_running = 0;
   STB_OSDeleteMutex(nw_mutex);
   STB_OSDestroyTask(nw_eth_task_ptr);

   FUNCTION_FINISH(NW_Terminate);
}



/*!**************************************************************************
 * @fn      NW_EnableEthernetDevice
 * @note    Defined in internal.h, never used
 ****************************************************************************/
void NW_EnableEthernetDevice(BOOLEAN enable)
{
   struct ifreq ifr;
   int sock_fd;

   FUNCTION_START(NW_EnableEthernetDevice);

   sock_fd = socket(AF_INET, SOCK_DGRAM, 0);
   if (sock_fd < 0)
   {
      NET_ERR(("EthernetMonitorTask socket FAILURE"));
   }
   else
   {
      strcpy(ifr.ifr_name, "eth0");

      if (ioctl(sock_fd, SIOCGIFFLAGS, &ifr) < 0)
      {
         NET_ERR(("ioctl SIOCGIFFLAGS failed"));
      }
      else
      {
         if (!enable)
         {
            ifr.ifr_flags &= ~IFF_UP;
         }
         else
         {
            ifr.ifr_flags |= IFF_UP;
         }
         if (ioctl(sock_fd, SIOCSIFFLAGS, &ifr) < 0)
         {
            NET_ERR(("ioctl SIOCSIFFLAGS failed"));
         }
      }
      close(sock_fd);
   }

   FUNCTION_FINISH(NW_EnableEthernetDevice);
}



/*!**************************************************************************
 * @fn      STB_IPGetIPAddress
 * @note    Defined in stbhwip.h
 ****************************************************************************/
BOOLEAN STB_IPGetIPAddress(U8BIT ip_addr[4])
{
   struct ifreq *ifr;
   struct ifreq *ifend;
   struct ifconf ifc;
   struct ifreq ifs[MAX_IFS];
   struct sockaddr address;
   struct sockaddr_in *address_in;
   int sock_fd;
   BOOLEAN result;

   FUNCTION_START(STB_IPGetIPAddress);

   result = FALSE;

   sock_fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
   ifc.ifc_len = sizeof(ifs);
   ifc.ifc_req = ifs;

   if (ioctl(sock_fd, SIOCGIFCONF, &ifc) < 0)
   {
      NET_DBG(("ioctl(SIOCGIFCONF): Failed\n"));
   }
   else
   {
      ifend = ifs + (ifc.ifc_len / sizeof(struct ifreq));
      for (ifr = ifc.ifc_req; ifr < ifend; ifr++)
      {
         if (ifr->ifr_addr.sa_family == AF_INET)
         {
            if (strncmp(ifr->ifr_name, "eth", 3) == 0)
            {
               address = ifr->ifr_addr;
               address_in = (struct sockaddr_in*) &address;

               ip_addr[3] = (address_in->sin_addr.s_addr >> 24) & 0xFF;
               ip_addr[2] = (address_in->sin_addr.s_addr >> 16) & 0xFF;
               ip_addr[1] = (address_in->sin_addr.s_addr >> 8) & 0xFF;
               ip_addr[0] = address_in->sin_addr.s_addr & 0xFF;

               NET_DBG(("IP address: %d.%d.%d.%d", ip_addr[0], ip_addr[1], ip_addr[2], ip_addr[3]));
               result = TRUE;
            }
         }
      }

      close(sock_fd);
   }

   FUNCTION_FINISH(STB_IPGetIPAddress);

   return result;
}



/*!**************************************************************************
 * @fn      STB_IPGetSubnetMask
 * @note    Defined in stbhwip.h
 ****************************************************************************/
BOOLEAN STB_IPGetSubnetMask(U8BIT subnet_mask[4])
{
   struct ifreq *ifr;
   struct ifreq *ifend;
   struct ifreq ifreq;
   struct ifconf ifc;
   struct ifreq ifs[MAX_IFS];
   struct sockaddr address;
   struct sockaddr_in *address_in;
   int sock_fd;
   BOOLEAN result;

   FUNCTION_START(STB_IPGetSubnetMask);

   result = FALSE;

   sock_fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
   ifc.ifc_len = sizeof(ifs);
   ifc.ifc_req = ifs;

   if (ioctl(sock_fd, SIOCGIFCONF, &ifc) < 0)
   {
      NET_DBG(("ioctl(SIOCGIFCONF): Failed"));
   }
   else
   {
      ifend = ifs + (ifc.ifc_len / sizeof(struct ifreq));
      for (ifr = ifc.ifc_req; ifr < ifend; ifr++)
      {
         if (ifr->ifr_addr.sa_family == AF_INET)
         {
            if (strncmp(ifr->ifr_name, "eth", 3) == 0)
            {
               strncpy(ifreq.ifr_name, ifr->ifr_name, sizeof(ifreq.ifr_name));
               if (ioctl (sock_fd, SIOCGIFNETMASK, &ifreq) < 0)
               {
                  NET_DBG(("SIOCSIFNETMASK(%s): %m", ifreq.ifr_name));
               }
               else
               {
                  address = ifreq.ifr_netmask;
                  address_in = (struct sockaddr_in *) &address;

                  subnet_mask[3] = (address_in->sin_addr.s_addr >> 24) & 0xFF;
                  subnet_mask[2] = (address_in->sin_addr.s_addr >> 16) & 0xFF;
                  subnet_mask[1] = (address_in->sin_addr.s_addr >> 8) & 0xFF;
                  subnet_mask[0] = address_in->sin_addr.s_addr & 0xFF;

                  NET_DBG(("Subnet mask: %u.%u.%u.%u", subnet_mask[0], subnet_mask[1],
                           subnet_mask[2], subnet_mask[3]));
                  result = TRUE;
                  break;
               }
            }
         }
      }
      close(sock_fd);
   }

   FUNCTION_FINISH(STB_IPGetSubnetMask);

   return result;
}



/*!**************************************************************************
 * @fn      STB_IPGetGatewayIPAddress
 * @note    Defined in stbhwip.h
 ****************************************************************************/
BOOLEAN STB_IPGetGatewayIPAddress(U8BIT gateway_addr[4])
{
   // struct rtmsg *rt_msg;
   struct nlmsghdr *nl_msg;
   S_ROUTE_INFO *rt_info;
   char *msg_buf;
   int sock;
   int len;
   int msg_seq;
   BOOLEAN found_gateway;

   FUNCTION_START(STB_IPGetGatewayIPAddress);

   found_gateway = FALSE;
   msg_seq = 0;

   /* Create Socket */
   if ((sock = socket(PF_NETLINK, SOCK_DGRAM, NETLINK_ROUTE)) < 0)
   {
      NET_DBG(("STB_IPGetGatewayIPAddress: Failed to open socket"));
   }
   else
   {
      msg_buf = (char *)STB_MEMGetSysRAM(GATEWAY_BUFSIZE);
      if (msg_buf != NULL)
      {
         /* Initialize the buffer */
         memset(msg_buf, 0, GATEWAY_BUFSIZE);

         /* point the header and the msg structure pointers into the buffer */
         nl_msg = (struct nlmsghdr*) msg_buf;
         // rt_msg = (struct rtmsg*) NLMSG_DATA(nl_msg);

         /* Fill in the nlmsg header*/
         nl_msg->nlmsg_len = NLMSG_LENGTH(sizeof(struct rtmsg));        // Length of message.
         nl_msg->nlmsg_type = RTM_GETROUTE;        // Get the routes from kernel routing table.

         nl_msg->nlmsg_flags = NLM_F_DUMP | NLM_F_REQUEST;  // The message is a request for dump.
         nl_msg->nlmsg_seq = msg_seq++;                     // Sequence of the message packet.
         nl_msg->nlmsg_pid = getpid();                      // PID of process sending the request.

         /* Send the request */
         if (send(sock, nl_msg, nl_msg->nlmsg_len, 0) < 0)
         {
            NET_DBG(("STB_IPGetGatewayIPAddress: Failed to write to socket"));
         }
         else
         {
            /* Read the response */
            if ((len = ReadSockLength(sock, msg_buf, msg_seq, getpid())) < 0)
            {
               NET_DBG(("STB_IPGetGatewayIPAddress: Read from socket failed"));
            }
            else
            {
               /* Parse and print the response */
               rt_info = (S_ROUTE_INFO*) STB_MEMGetSysRAM(sizeof(S_ROUTE_INFO));
               if (rt_info != NULL)
               {
                  for (; NLMSG_OK(nl_msg, len);nl_msg = NLMSG_NEXT(nl_msg, len))
                  {
                     memset(rt_info, 0, sizeof(S_ROUTE_INFO));
                     if(ParseRoutes(nl_msg, rt_info) == 0)
                     {
                        // Check if default gateway
                        if (strstr((char*) inet_ntoa(rt_info->dst_addr), "0.0.0.0"))
	                    {
	                       found_gateway = TRUE;
	                       gateway_addr[3] = (rt_info->gateway.s_addr >> 24) & 0xFF;
	                       gateway_addr[2] = (rt_info->gateway.s_addr >> 16) & 0xFF;
	                       gateway_addr[1] = (rt_info->gateway.s_addr >> 8) & 0xFF;
	                       gateway_addr[0] = rt_info->gateway.s_addr & 0xFF;

	                       NET_DBG(("Gateway: %u.%u.%u.%u", gateway_addr[0], gateway_addr[1],
	                                 gateway_addr[2], gateway_addr[3]));
	                       break;
	                    }
                     }
                  }
                  STB_MEMFreeSysRAM(rt_info);
               }
            }
         }
         STB_MEMFreeSysRAM(msg_buf);
      }
      close(sock);
   }

   FUNCTION_FINISH(STB_IPGetGatewayIPAddress);

   return found_gateway;
}



/*!**************************************************************************
 * @fn      STB_IPGetDnsServerIPAddress
 * @note    Defined in stbhwip.h
 ****************************************************************************/
BOOLEAN STB_IPGetDnsServerIPAddress(U8BIT *dns_addr)
{
   char *token_ptr;
   FILE *file;
   char buffer[255];
   struct in_addr addr;
   BOOLEAN addr_found;

   FUNCTION_START(STB_IPGetDnsServerIPAddress);

   addr_found = FALSE;

   file = fopen(RESOLV_CONF, "r");
   if (file != NULL)
   {
      while (!addr_found && (fgets(buffer, 255, file) != 0))
      {
         token_ptr = strtok(buffer, " \n");
         while (token_ptr != NULL)
         {
            if (strcmp(token_ptr, "nameserver") == 0)
            {
               addr_found = TRUE;
            }

            token_ptr = strtok(NULL, " \n");
            if (addr_found && (token_ptr != NULL))
            {
               if (inet_aton(token_ptr, &addr) != 0)
               {
                  dns_addr[3] = (addr.s_addr >> 24) & 0xFF;
                  dns_addr[2] = (addr.s_addr >> 16) & 0xFF;
                  dns_addr[1] = (addr.s_addr >> 8) & 0xFF;
                  dns_addr[0] = addr.s_addr & 0xFF;
                  addr_found = TRUE;
                  token_ptr = NULL;
                  NET_DBG(("DNS: %u.%u.%u.%u", dns_addr[0], dns_addr[1], dns_addr[2], dns_addr[3]));
               }
            }
         }
      }
      fclose(file);
   }

   FUNCTION_FINISH(STB_IPGetDnsServerIPAddress);

   return addr_found;
}



/*!**************************************************************************
 * @fn      STB_IPSetIPAddress
 * @note    Defined in stbhwip.h
 ****************************************************************************/
void STB_IPSetIPAddress(const U8BIT *ip_addr)
{
   char cmd[128];

   FUNCTION_START(STB_IPSetIPAddress);

   sprintf(cmd, "ifconfig eth0 %u.%u.%u.%u", ip_addr[0], ip_addr[1], ip_addr[2], ip_addr[3]);
   system(cmd);

   FUNCTION_FINISH(STB_IPSetIPAddress);
}



/*!**************************************************************************
 * @fn      STB_IPSetSubnetMask
 * @note    Defined in stbhwip.h
 ****************************************************************************/
void STB_IPSetSubnetMask(const U8BIT *subnet_mask)
{
   char cmd[128];

   FUNCTION_START(STB_IPSetSubnetMask);

   sprintf(cmd, "ifconfig eth0 netmask %u.%u.%u.%u", subnet_mask[0], subnet_mask[1],
      subnet_mask[2], subnet_mask[3]);
   system(cmd);

   FUNCTION_FINISH(STB_IPSetSubnetMask);
}



/*!**************************************************************************
 * @fn      STB_IPSetGatewayIPAddress
 * @note    Defined in stbhwip.h
 ****************************************************************************/
void STB_IPSetGatewayIPAddress(const U8BIT *gateway_addr)
{
   char cmd[128];

   FUNCTION_START(STB_IPSetGatewayIPAddress);

   sprintf(cmd, "route add default gw %u.%u.%u.%u", gateway_addr[0], gateway_addr[1],
      gateway_addr[2], gateway_addr[3]);
   system(cmd);

   FUNCTION_FINISH(STB_IPSetGatewayIPAddress);
}



/*!**************************************************************************
 * @fn      STB_IPSetDnsServerIPAddress
 * @note    Defined in stbhwip.h
 ****************************************************************************/
void STB_IPSetDnsServerIPAddress(const U8BIT *dns_addr)
{
   FILE *fp;

   FUNCTION_START(STB_IPSetDnsServerIPAddress);

   fp = fopen(RESOLV_CONF, "w");
   if (fp != NULL)
   {
      fprintf(fp, "nameserver %u.%u.%u.%u\n", dns_addr[0], dns_addr[1], dns_addr[2], dns_addr[3]);
      fclose(fp);
   }

   FUNCTION_FINISH(STB_IPSetDnsServerIPAddress);
}



/*!**************************************************************************
 * @fn      STB_IPGetIPByDhcp
 * @note    Defined in stbhwip.h
 ****************************************************************************/
void STB_IPGetIPByDhcp(BOOLEAN wait_for_completion)
{
   char cmd[128];

   FUNCTION_START(STB_IPGetIPByDhcp);

   sprintf(cmd, "/bin/busybox udhcpc -n -q -s /etc/udhcpc.script");
   if (!wait_for_completion)
   {
      strcat(cmd, "&");
   }

   system(cmd);

   FUNCTION_FINISH(STB_IPGetIPByDhcp);
}



/*--- Local function definitions ----------------------------------------------*/

/*!**************************************************************************
 * @fn      EthernetMonitorTask
 ****************************************************************************/
static void EthernetMonitorTask(void *arg)
{
   S_ETHERNET_MONITOR *p_nw_monitor;
   E_NW_LINK_STATUS status_now;
   struct ifreq ifr;
   int sock_fd;
   struct ethtool_value edata;

   FUNCTION_START(EthernetMonitorTask);
   USE_UNWANTED_PARAM(arg);

   sock_fd = socket(AF_INET, SOCK_DGRAM, 0);
   if (sock_fd < 0)
   {
      NET_ERR(("EthernetMonitorTask socket FAILURE"));
   }
   else
   {
      strcpy(ifr.ifr_name, "eth0");

      while (nw_eth_status_running)
      {
         if (ioctl(sock_fd, SIOCGIFFLAGS, &ifr) < 0)
         {
            NET_ERR(("ioctl SIOCGIFFLAGS failed"));
            status_now = NW_LINK_DISABLED;
         }
         else
         {
            if (ifr.ifr_flags & (IFF_UP | IFF_RUNNING))
            {
               edata.cmd = ETHTOOL_GLINK;
               ifr.ifr_data = (char*) &edata;
               if (ioctl(sock_fd, SIOCETHTOOL, &ifr) < 0)
               {
                  NET_ERR(("ioctl SIOCETHTOOL failed"));
                  status_now = NW_LINK_DISABLED;
               }
               else
               {
                  if (edata.data)
                  {
                     status_now = NW_LINK_ACTIVE;
                  }
                  else
                  {
                     status_now = NW_LINK_INACTIVE;
                  }
               }
            }
            else
            {
               status_now = NW_LINK_DISABLED;
            }
         }
         if (current_ethernet_status != status_now)
         {
            current_ethernet_status = status_now;

            NET_DBG(("EthernetMonitorTask: current ethernet status changed to %s",
                    ((current_ethernet_status == NW_LINK_ACTIVE) ? "active" :
                     (current_ethernet_status == NW_LINK_INACTIVE) ? "inactive" : "disabled")));

            STB_OSMutexLock(nw_mutex);
            p_nw_monitor = nw_monitor_list;
            while (p_nw_monitor)
            {
               (p_nw_monitor->function)(NW_WIRED, current_ethernet_status);
               p_nw_monitor = p_nw_monitor->next;
            }
            STB_OSMutexUnlock(nw_mutex);
         }

         STB_OSTaskDelay( 100 );
      }
      close(sock_fd);
   }

   FUNCTION_FINISH(EthernetMonitorTask);

   pthread_exit(0);
}



/*!**************************************************************************
 * @fn      ReadSockLength
 ****************************************************************************/
static int ReadSockLength(int sock_fd, char *buf_ptr, int seq_num, int pid)
{
   struct nlmsghdr *nl_hdr;
   int read_len;
   int msg_len;

   FUNCTION_START(ReadSockLength);

   read_len = 0;
   msg_len = 0;

   do
   {
      /* Recieve response from the kernel */
      read_len = recv(sock_fd, buf_ptr, GATEWAY_BUFSIZE - msg_len, 0);
      if (read_len < 0)
      {
         perror("SOCK READ: ");
         msg_len = -1;
         break;
      }
      else
      {
         nl_hdr = (struct nlmsghdr*) buf_ptr;

         /* Check if the header is valid */
         if ((NLMSG_OK(nl_hdr, read_len) == 0) || (nl_hdr->nlmsg_type == NLMSG_ERROR))
         {
            perror("Error in recieved packet");
            msg_len = -1;
            break;
         }
         else
         {
            /* Check if the its the last message */
            if (nl_hdr->nlmsg_type == NLMSG_DONE)
            {
               break;
            }
            else
            {
               /* Else move the pointer to buffer appropriately */
               buf_ptr += read_len;
               msg_len += read_len;
            }

            /* Check if its a multi part message */
            if ((nl_hdr->nlmsg_flags & NLM_F_MULTI) == 0)
            {
               /* return if its not */
               break;
            }
         }

      }
   }
   while ((nl_hdr->nlmsg_seq != seq_num) || (nl_hdr->nlmsg_pid != pid));

   FUNCTION_FINISH(ReadSockLength);
   return msg_len;
}



/*!**************************************************************************
 * @fn      ParseRoutes
 * @brief   Parse the route info returned
 ****************************************************************************/
static int ParseRoutes(struct nlmsghdr *nl_hdr, S_ROUTE_INFO *rt_info)
{
   struct rtmsg *rt_msg;
   struct rtattr *rt_attr;
   int rt_len;
   int retval;

   FUNCTION_START(ParseRoutes);
   rt_msg = (struct rtmsg*) NLMSG_DATA(nl_hdr);
   retval = -1;

   /* If the route is not for AF_INET or does not belong to main routing table then return. */
   if ((rt_msg->rtm_family == AF_INET) && (rt_msg->rtm_table == RT_TABLE_MAIN))
   {
      /* get the rtattr field */
      rt_attr = (struct rtattr*) RTM_RTA(rt_msg);
      rt_len = RTM_PAYLOAD(nl_hdr);

      for (; RTA_OK(rt_attr, rt_len); rt_attr = RTA_NEXT(rt_attr, rt_len))
      {
         switch (rt_attr->rta_type)
         {
            case RTA_OIF:
               if_indextoname(*(int*) RTA_DATA(rt_attr), rt_info->if_name);
               break;

            case RTA_GATEWAY:
               memcpy(&rt_info->gateway, RTA_DATA(rt_attr), sizeof(rt_info->gateway));
               break;

            case RTA_PREFSRC:
               memcpy(&rt_info->src_addr, RTA_DATA(rt_attr), sizeof(rt_info->src_addr));
               break;

            case RTA_DST:
               memcpy(&rt_info->dst_addr, RTA_DATA(rt_attr), sizeof(rt_info->dst_addr));
               break;
         }
      }
      retval = 0;
   }

   FUNCTION_FINISH(ParseRoutes);
   return retval;
}



/****************************************************************************
** End of file
****************************************************************************/
