/*******************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Functions for writing upgrade modules to non volatile memory
 * @file    stbhwupg.c
 * @date    October 2009
 * @author  Sergio Panseri
 */


/*--- Includes ----------------------------------------------------------------*/

/* System header files */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>

/* Third party header files */

/* STB header files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbhwmem.h"
#include "stbhwdsk.h"
#include "stbhwdef.h"
#include "stbhwc.h"
#include "stbhwupg.h"

#ifdef system
#undef system
#endif
#define system(cmd) execl("/vendor/bin/sh", "sh", "-c", cmd, NULL)

/*--- Preprocessor definitions ------------------------------------------------*/

#define FACTORY_APP_FILE   "/data/vendor/dtvkit/zapper"
#define APP_FILE           "/data/vendor/dtvkit/ssu"
#define TMP_APP_FILE       "/data/vendor/dtvkit/tmp"

#define UPG_MODULE_APPLICATION   1
#define UPG_NUMBER_OF_MODULES    1


// #define UPGRADE_ERROR
// #define UPGRADE_DEBUG

#ifdef UPGRADE_ERROR
#define UPG_ERR(X)      STB_SPDebugWrite X
#else
#define UPG_ERR(X)
#endif

#ifdef UPGRADE_DEBUG
#define UPG_DBG(X)      STB_SPDebugWrite X
#else
#define UPG_DBG(X)
#endif

#if 0
#ifdef FUNCTION_START
#undef FUNCTION_START
#define FUNCTION_START(name)  printf(">>> %s\n", #name);fflush(stdout)
#endif

#ifdef FUNCTION_FINISH
#undef FUNCTION_FINISH
#define FUNCTION_FINISH(name)  printf("<<< %s\n", #name);fflush(stdout)
#endif
#endif


/*--- Local types definitions -------------------------------------------------*/

/*--- Local (static) variable declarations ------------------------------------*/

static FILE *read_file;
static FILE *write_file;
static U8BIT *real_filename;
static U8BIT *tmp_filename;


/*--- Local function prototypes -----------------------------------------------*/

static FILE* OpenFile(U8BIT image_type);



/*--- Global function definitions ---------------------------------------------*/

/*!**************************************************************************
 * @fn      STB_UPGInitialise
 * @brief   Initialisation of the necessary structures
 * @warning
 * @bug
 ****************************************************************************/
void STB_UPGInitialise(void)
{
   FUNCTION_START(STB_UPGInitialise);

   UPG_DBG(("STB_UPGInitialise()"));

   read_file = NULL;
   write_file = NULL;
   real_filename = NULL;
   tmp_filename = NULL;

   FUNCTION_FINISH(STB_UPGInitialise);
}



/*!**************************************************************************
 * @fn      STB_UPGStart
 * @brief   Opens the area ready for writing
 * @param   image_type - type of image to be written, this is meaningful
 *          for the platform code only, as each platform may have different
 *          upgradable modules (kernel, application, kernel modules...). The
 *          midware passes this parameter as it finds it in the upgrade
 *          stream without interpretation.
 * @param   filename
 * @return  TRUE if successful, FALSE otherwise
 * @warning
 * @bug
 ****************************************************************************/
BOOLEAN STB_UPGStart(U8BIT image_type, U8BIT *filename)
{
   BOOLEAN retval;

   FUNCTION_START(STB_UPGStart);

   UPG_DBG(("STB_UPGStart(image_type=%u, filename=\"%s\")",
            image_type, ((filename == NULL) ? "NULL" : (char*) filename)));

   retval = FALSE;

   if (image_type == UPG_MODULE_APPLICATION)
   {
      if (write_file == NULL)
      {
         if (filename != NULL)
         {
            /* Save the filename to tidy up at the end */
            real_filename = (U8BIT*) STB_MEMGetSysRAM(strlen((char*) filename) + 1);
            if (real_filename != NULL)
            {
               strcpy((char*) real_filename, (char*) filename);

               /* Create a temp file to write the data into */
               tmp_filename = (U8BIT*) STB_MEMGetSysRAM(strlen((char*) real_filename) + 5);
               if (tmp_filename != NULL)
               {
                  strcpy((char*) tmp_filename, (char*) real_filename);
                  strcat((char*) tmp_filename, ".tmp");

                  write_file = fopen((char*) tmp_filename, "w");
                  if (write_file != NULL)
                  {
                     retval = TRUE;
                  }
                  else
                  {
                     UPG_DBG(("%s: Failed to create file \"%s\", errno %d", __FUNCTION__, tmp_filename, errno));
                     STB_MEMFreeSysRAM(tmp_filename);
                     STB_MEMFreeSysRAM(real_filename);
                     tmp_filename = NULL;
                     real_filename = NULL;
                  }
               }
               else
               {
                  STB_MEMFreeSysRAM(real_filename);
                  real_filename = NULL;
               }
            }
         }
         else
         {
            /* The file hasn't been opened yet */
            write_file = OpenFile(image_type);
            retval = TRUE;
         }
      }
   }

   FUNCTION_FINISH(STB_UPGStart);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_UPGWrite
 * @brief   Write size bytes to the upgrade storage area specified by
 *          image_type.
 * @param   image_type - type of image to be written, this is meaningful
 *          for the platform code only, as each platform may have different
 *          upgradable modules (kernel, application, kernel modules...). The
 *          midware passes this parameter as it finds it in the upgrade
 *          stream without interpretation.
 * @param   offset - offset inside the specified area where to write to
 * @param   size - number of bytes to write
 * @param   buffer - buffer containing the data to be written
 * @return  TRUE if successful, FALSE otherwise
 * @warning
 * @bug
 ****************************************************************************/
BOOLEAN STB_UPGWrite(U8BIT image_type, U32BIT offset, U32BIT size, U8BIT *buffer)
{
   BOOLEAN result;

   FUNCTION_START(STB_UPGWrite);

   UPG_DBG(("STB_UPGWrite(image_type=%u, offset=%lu, size=%lu, buffer=%p)",
            image_type, offset, size, buffer));

   result = TRUE;
   if (image_type == UPG_MODULE_APPLICATION)
   {
      if (write_file == NULL)
      {
         /* The file hasn't been opened yet */
         write_file = OpenFile(image_type);
      }

      if (write_file != NULL)
      {
         if (fseek(write_file, offset, SEEK_SET) == 0)
         {
            if (fwrite(buffer, 1, size, write_file) != size)
            {
               result = FALSE;
               UPG_ERR(("fwrite failed errno=%d (%s)", errno, strerror(errno)));
            }
         }
         else
         {
            result = FALSE;
            UPG_ERR(("fseek failed errno=%d (%s)", errno, strerror(errno)));
         }
      }
      else
      {
         result = FALSE;
         UPG_ERR(("No file for image type %d", image_type));
      }
   }
   else
   {
      result = FALSE;
      UPG_ERR(("Wrong image type %d", image_type));
   }

   FUNCTION_FINISH(STB_UPGWrite);

   return result;
}



/*!**************************************************************************
 * @fn      STB_UPGRead
 * @brief   Read size bytes from the upgrade storage area specified by
 *          image_type.
 * @param   image_type - type of image to be read, this
 *          is meaningful for the platform code only, as each platform may
 *          have different upgradable modules (kernel, application, kernel
 *          modules...). The midware passes this parameter as it finds it in
 *          the upgrade stream without interpretation.
 * @param   offset - offset inside the specified area where to read from
 * @param   size - number of bytes to read
 * @param   buffer - buffer where to return the read data.
 * @return  TRUE if successful, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_UPGRead(U8BIT image_type, U32BIT offset, U32BIT size, U8BIT *buffer)
{
   BOOLEAN result;
   FILE *file;
   size_t read;

   FUNCTION_START(STB_UPGRead);

   UPG_DBG(("STB_UPGRead(image_type=%u, offset=%lu, size=%lu, buffer=%p)",
            image_type, offset, size, buffer));

   result = TRUE;
   if (image_type == UPG_MODULE_APPLICATION)
   {
      if (write_file == NULL)
      {
         /* Not reading from a file that's been written */
         if (read_file == NULL)
         {
            /* First time the file has been read from so attempt to open the app SSU file */
            read_file = OpenFile(image_type);
         }

         file = read_file;
      }
      else
      {
         /* The read will be from the file that has been or is being written */
         if (read_file != NULL)
         {
            /* Need to close the original file that was being read from */
            fclose(read_file);
            read_file = NULL;
         }

         file = write_file;
      }

      if (file != NULL)
      {
         if (fseek(file, offset, SEEK_SET) == 0)
         {
            read = fread(buffer, 1, size, file);
            if (read != size)
            {
               // if the file is not big enough fill the rest with 0
               memset(&buffer[read], 0, size -read);
            }
         }
         else
         {
            /* If fseek fails, pretend the file is big enough and fill the buffer with 0 */
            memset(buffer, 0, size);
         }
      }
      else
      {
         // result = FALSE;
         memset(buffer, 0, size);
         UPG_ERR(("No file for image type %d", image_type));
      }
   }
   else
   {
      result = FALSE;
      UPG_ERR(("Wrong image type %d", image_type));
   }

   FUNCTION_FINISH(STB_UPGRead);

   return result;
}



/*!**************************************************************************
 * @fn      STB_UPGFinish
 * @brief   Completes the upgrade, closing and renaming files as appropriate
 * @param   image_type - type of image to be read, this is meaningful for
 *          the platform code only, as each platform may have different
 *          upgradable modules (kernel, application, kernel modules...). The
 *          midware passes this parameter as it finds it in the upgrade
 *          stream without interpretation
 * @param   upgrade_successful - BOOLEAN flag indicating whether the upgrade
 *          was successful or not
 * @return  TRUE if all operations are completed successfully, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_UPGFinish(U8BIT image_type, BOOLEAN upgrade_successful)
{
   char cmd_str[256];
   BOOLEAN retval;

   FUNCTION_START(STB_UPGFinish);

   UPG_DBG(("STB_UPGFinish(image_type=%u, success=%u)", image_type, upgrade_successful));

   if (write_file != NULL)
   {
      fclose(write_file);
      write_file = NULL;
   }

   if (upgrade_successful)
   {
      if ((real_filename != NULL) && (tmp_filename != NULL))
      {
         /* Delete the file being replaced */
         sprintf(cmd_str, "rm -f %s", real_filename);
         system(cmd_str);

         /* The new file replaces the old file */
         sprintf(cmd_str, "mv %s %s", tmp_filename, real_filename);
         system(cmd_str);
      }
      else
      {
         /* Change the name of the new app to allow it to be used */
         sprintf(cmd_str, "rm -f %s", APP_FILE);
         system(cmd_str);

         sprintf(cmd_str, "mv %s %s", TMP_APP_FILE, APP_FILE);
         system(cmd_str);
      }
   }
   else
   {
      /* Delete the temp file */
      sprintf(cmd_str, "rm -f %s", TMP_APP_FILE);
      system(cmd_str);
   }

   if (real_filename != NULL)
   {
      STB_MEMFreeSysRAM(real_filename);
      real_filename = NULL;
   }

   if (tmp_filename != NULL)
   {
      STB_MEMFreeSysRAM(tmp_filename);
      tmp_filename = NULL;
   }

   retval = TRUE;

   FUNCTION_FINISH(STB_UPGFinish);

   return retval;
}



/*!**************************************************************************
 * @fn      STB_UPGGetApplicationSize
 * @brief   Returns the size of the NVM area available for the application
 *          and all the upgradable modules. In the last part of this area
 *          Intellibyte loader places some version information.
 * @return  Size of the application area.
 ****************************************************************************/
U32BIT STB_UPGGetApplicationSize(void)
{
   FUNCTION_START(STB_UPGGetApplicationSize);
   FUNCTION_FINISH(STB_UPGGetApplicationSize);
   return 0x400000;  /* 4MB */
}



/*!**************************************************************************
 * @fn      STB_UPGGetApplicationOffset
 * @brief   Returns the application's offset inside its area.
 * @return  Application offset.
 ****************************************************************************/
U32BIT STB_UPGGetApplicationOffset(void)
{
   FUNCTION_START(STB_UPGGetApplicationSize);
   FUNCTION_FINISH(STB_UPGGetApplicationSize);
   return 0;
}



/*--- Local function definitions ----------------------------------------------*/


/*!**************************************************************************
 * @fn      OpenFile
 * @brief   Performs all the necessary copies and then opens the file
 *          associated with the specified image_type.
 * @param   image_type - image type to open the file for.
 * @return  File pointer
 ****************************************************************************/
static FILE* OpenFile(U8BIT image_type)
{
   FILE *file_ptr;
   char cmd_str[50];
   struct stat sbuf;

   FUNCTION_START(OpenFile);

   file_ptr = NULL;
   switch (image_type)
   {
      case UPG_MODULE_APPLICATION:
         /* Delete temp file to allow download of a new file */
         sprintf(cmd_str, "rm -f %s", TMP_APP_FILE);
         system(cmd_str);

         if (stat(APP_FILE, &sbuf) == 0)
         {
            UPG_DBG(("Copying %s to %s", APP_FILE, TMP_APP_FILE));

            /* If the app file exists then copy it to the tmp file so that it can be updated */
            sprintf(cmd_str, "cp -pf %s %s", APP_FILE, TMP_APP_FILE);
            system(cmd_str);
         }

         UPG_DBG(("Opening file %s", TMP_APP_FILE));

         file_ptr = fopen(TMP_APP_FILE, "r+");
         if (file_ptr == NULL)
         {
            file_ptr = fopen(TMP_APP_FILE, "w+");
         }

#ifdef UPG_ERR
         if (file_ptr == NULL)
         {
            UPG_ERR(("fopen errno=%d (%s)", errno, strerror(errno)));
         }
#endif
         break;

      default:
         break;
   }

   FUNCTION_FINISH(OpenFile);
   return file_ptr;
}



/****************************************************************************
** End of file
*****************************************************************************/

