/*******************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Set Top Box - Operating System Interface for Mutexes.
 * @file    stbos_mutex.c
 * @date    10 November 2006
 */

#ifndef NDEBUG
#define DEBUG_ASSERT_STANDARD
#endif

/*--- Includes ----------------------------------------------------------------*/

/* System header Files */
#include <stdio.h>
#include <pthread.h>
#include <time.h>

/* STB header Files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwmem.h"
#include "stbhwc.h"

#include "stbhwos.h"

/*--- Preprocessor definitions ------------------------------------------------*/

#ifdef STATIC_MEM_MTX
#define MAX_MUTEXES     330
#define INVALID_COUNT   0xffffffff
#endif

/* Select-Deselect Local Debug Text Output */
/*#define  MUTEX_DEBUG*/
#ifndef MUTEX_DEBUG
#define MUTEX_DBG(X)
#else
#define MUTEX_DBG(X)    STB_SPDebugWrite X
#endif


#ifdef OS_CHECK_NULL_PTRS
#define NULLCHECK(ptr)  if (ptr == NULL) { \
                           STB_SPDebugWrite("line:%d, %s is Null", __LINE__, #ptr); \
                        } else
#else
#define NULLCHECK(ptr)
#endif


/*--- Local types definitions -------------------------------------------------*/

typedef struct mutex
{
   pthread_mutex_t lock;
   pthread_t thread_id;
   U32BIT lock_count;
} S_MUTEX;


/*--- Local (static) variable declarations ------------------------------------*/

#ifdef STATIC_MEM_MTX
static S_MUTEX mutexes[MAX_MUTEXES];
static S_MUTEX *mtx_end_ptr;
#endif

static unsigned int mtx_count = 0;


/*--- Local function prototypes -----------------------------------------------*/


/*--- Global function definitions ---------------------------------------------*/

/*!**************************************************************************
 * @fn      STB_OSCreateMutex
 * @brief   Create a mutex.
 * @return  Newly created mutex, or NULL.
 ****************************************************************************/
void* STB_OSCreateMutex(void)
{
   S_MUTEX *mutex_handle;
   pthread_mutexattr_t mutex_attr_ptr;

#ifdef STATIC_MEM_MTX
   int it;
#endif

   FUNCTION_START(STB_OSCreateMutex);

#ifdef STATIC_MEM_MTX
   if (mtx_count == 0)
   {
      /* Only execute on first initialisation */
      for (it = 0; it != MAX_MUTEXES; it++)
      {
         mutexes[it].lock_count = INVALID_COUNT;
      }
      mutex_handle = mutexes;
      mtx_end_ptr = mutex_handle + MAX_MUTEXES;
   }
   else
   {
      mutex_handle = NULL;
      for (it = 0; it != MAX_MUTEXES; it++)
      {
         if (mutexes[it].lock_count == INVALID_COUNT)
         {
            mutex_handle = mutexes + it;
            break;
         }
      }
   }
#else
   mutex_handle = (S_MUTEX*) STB_MEMGetSysRAM(sizeof(S_MUTEX));
#endif

   if (mutex_handle != NULL)
   {
      mtx_count++;
      pthread_mutexattr_init(&mutex_attr_ptr);
      if (pthread_mutex_init(&mutex_handle->lock, &mutex_attr_ptr) == 0)
      {
         mutex_handle->thread_id = 0;
         mutex_handle->lock_count = 0;
      }
   }

   MUTEX_DBG(("STB_OSCreateMutex: Created mutex 0x%x cnt=%d it=%d", mutex_handle, mtx_count, it));

   FUNCTION_FINISH(STB_OSCreateMutex);

   return (void*) mutex_handle;
}



/*!**************************************************************************
 * @fn      STB_OSMutexLock
 * @brief   Lock a mutex (a.k.a. 'enter', 'wait' or 'get').
 * @param   mutex_var - The mutex to lock.
 ****************************************************************************/
void STB_OSMutexLock(void *mutex)
{
   S_MUTEX *mutex_handle;

   FUNCTION_START(STB_OSMutexLock);

   // MUTEX_DBG(("STB_OSMutexLock: Locking mutex 0x%x", mutex));
   NULLCHECK(mutex)
   {
      mutex_handle = (S_MUTEX*) mutex;

#ifndef STATIC_MEM_MTX
      ASSERT(mutex != (void*) 0x0);
#else
      ASSERT(mutex_handle >= mutexes);
      ASSERT(mutex_handle < mtx_end_ptr);
      ASSERT(mutex_handle->lock_count < 0x20);
#endif

      if (pthread_equal(mutex_handle->thread_id, pthread_self()) != 0)
      {
         /* This thread already owns the mutex so just increment the lock count */
         mutex_handle->lock_count++;
      }
      else
      {
         /* The mutex is either already owned by another thread or
            isn't owned at all so just attempt to acquire it. */
         pthread_mutex_lock((pthread_mutex_t*) &mutex_handle->lock);

         mutex_handle->thread_id = pthread_self();
         mutex_handle->lock_count = 1;
      }
   }

   FUNCTION_FINISH(STB_OSMutexLock);
}



/*!**************************************************************************
 * @fn      STB_OSMutexUnlock
 * @brief   Unlock a mutex (a.k.a. 'leave', 'signal' or 'release')
 * @param   mutex_var - The mutex to unlock.
 ****************************************************************************/
void STB_OSMutexUnlock(void *mutex_var)
{
   S_MUTEX *mutex_handle;

   FUNCTION_START(STB_OSMutexUnlock);

   // MUTEX_DBG(("STB_OSMutexUnlock: Unlocking mutex 0x%x", mutex_var));

   NULLCHECK(mutex_var)
   {
      mutex_handle = (S_MUTEX*) mutex_var;

#ifndef STATIC_MEM_MTX
      ASSERT(mutex_var != (void*) 0x0);
#else
      ASSERT(mutex_handle >= mutexes);
      ASSERT(mutex_handle < mtx_end_ptr);
      ASSERT(mutex_handle->lock_count < 0x20);
#endif

      /* if the same thread as original lock, decrement counter */
      if (pthread_equal(mutex_handle->thread_id, pthread_self()) != 0)
      {
         mutex_handle->lock_count--;

         if (mutex_handle->lock_count == 0)
         {
            /* Mutex has now been released by this thread */
            mutex_handle->thread_id = 0;
            pthread_mutex_unlock((pthread_mutex_t*) &mutex_handle->lock);
         }
      }
      else
      {
         // MUTEX_DBG(("STB_OSMutexUnlock: Thread doesn't own the mutex"));
      }
   }

   FUNCTION_FINISH(STB_OSMutexUnlock);
}



/*!**************************************************************************
 * @fn      STB_OSDeleteMutex
 * @brief   Delete a mutex.
 * @param   mutex_var - The mutex to delete.
 ****************************************************************************/
void STB_OSDeleteMutex(void *mutex_var)
{
   S_MUTEX *mutex_handle;

   FUNCTION_START(STB_OSDeleteMutex);

   MUTEX_DBG(("STB_OSDeleteMutex: Deleting mutex 0x%x", mutex_var));

   NULLCHECK(mutex_var)
   {
      mutex_handle = (S_MUTEX*) mutex_var;

#ifndef STATIC_MEM_MTX
      ASSERT(mutex_var != (void*) 0x0);
#else
      ASSERT(mutex_handle >= mutexes);
      ASSERT(mutex_handle < mtx_end_ptr);
#endif

      mtx_count--;

      pthread_mutex_destroy((pthread_mutex_t*) &mutex_handle->lock);

#ifdef STATIC_MEM_MTX
      mutex_handle->lock_count = INVALID_COUNT;
#else
      STB_MEMFreeSysRAM(mutex_var);
#endif
   }

   FUNCTION_FINISH(STB_OSDeleteMutex);
}


/*--- Local function definitions ----------------------------------------------*/


/****************************************************************************
** End of file
*****************************************************************************/
