/*******************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Set Top Box - System Interface for Real Time Clock facility.
 * @file    stbos_rtc.c
 * @date    July 2006
 */

/*--- Includes ----------------------------------------------------------------*/

/* System header Files */
#include <time.h>
#include <sys/time.h>

/* STB header Files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwc.h"

#include "stbhwos.h"

/*--- Preprocessor definitions ------------------------------------------------*/

#define RTC_TICKS_PER_SEC       1000

/* Select-Deselect Local Debug Text Output */
/*#define  RTC_DEBUG*/
#ifndef  RTC_DEBUG
#define  RTC_DBG(X)
#else
#define  RTC_DBG(X)    STB_SPDebugWrite X
#endif


/*--- Local types definitions -------------------------------------------------*/


/*--- Local (static) variable declarations ------------------------------------*/

/* UTC Time in Seconds */
static U32BIT utc_seconds = 0;

/* System Time from when the Time was Last Set */
static U32BIT sync_time = 0;


/*--- Local function prototypes -----------------------------------------------*/

static U32BIT SysBootTime(void);


/*--- Global function definitions ---------------------------------------------*/

/*!**************************************************************************
 * @fn      STB_OSInitialise
 * @brief   Allows setting of initial boot time.
 * @return  nothing.
 ****************************************************************************/
void STB_OSInitialise(void)
{
   FUNCTION_START(STB_OSInitialise);

   // STBOS_SignalInitialise();

   FUNCTION_FINISH(STB_OSInitialise);
}



/*!**************************************************************************
 * @fn      STB_OSSetClockRTC
 * @brief   Set the time in seconds since midnight 1-1-1970
 * @param   num_seconds - time in seconds
 ****************************************************************************/
void STB_OSSetClockRTC(U32BIT num_seconds)
{
   FUNCTION_START(STB_OSSetClockRTC);

   utc_seconds = num_seconds;

   /* Save the system time at the point the clock has been set */
   sync_time = SysBootTime();

   RTC_DBG(("STB_OSSetClockRTC: Time set to %u secs at %u msecs", num_seconds, sync_time));

   FUNCTION_FINISH(STB_OSSetClockRTC);
}



/*!**************************************************************************
 * @fn      STB_OSGetClockRTC
 * @brief   Returns the current time in seconds. This is calculated by using
 *          the set UTC time and adding the difference between the system boot
 *          time when it was set (sync_time) and the system boot time now.
 * @return  The current time in seconds since midnight 1-1-1970.
 ****************************************************************************/
U32BIT STB_OSGetClockRTC(void)
{
   U32BIT time_now;

   FUNCTION_START(STB_OSGetClockRTC);

   /* Actual time is given by the value saved in
    * utc_seconds with the difference between the time
    * when it was set (sync_time) and the time now.   */
   time_now = utc_seconds + ((SysBootTime() - sync_time) / RTC_TICKS_PER_SEC);

   FUNCTION_FINISH(STB_OSGetClockRTC);

   return time_now;
}



/*!**************************************************************************
 * @fn      STB_OSGetClockDiff
 * @brief   Get Difference between Given Time and Current Time.
 * @param   timestamp - Given Clock Value to Compare Against.
 * @return  Time Difference in MilliSeconds.
 ****************************************************************************/
U32BIT STB_OSGetClockDiff(U32BIT timestamp)
{
   U32BIT diff;

   FUNCTION_START(STB_OSGetClockDiff);

   /*!- Calculate Difference between current time & given value */
   diff = SysBootTime() - timestamp;

   RTC_DBG(("STB_OSGetClockDiff: Timestamp = %u msecs, diff = %u msecs", timestamp, diff));

   FUNCTION_FINISH(STB_OSGetClockDiff);

   return diff;
}



/*!**************************************************************************
 * @fn      STB_OSGetClockPerSec
 * @brief   Get Number of Clock Ticks per Second.
 * @return  Number of Ticks.
 ****************************************************************************/
U32BIT STB_OSGetClockPerSec(void)
{
   FUNCTION_START(STB_OSGetClockPerSec);
   FUNCTION_FINISH(STB_OSGetClockPerSec);
   return RTC_TICKS_PER_SEC;
}



/*!**************************************************************************
 * @fn      STB_OSGetClockMilliseconds
 * @brief   Get Current Computer Clock Time.
 * @return  Time in Milliseconds.
 ****************************************************************************/
U32BIT STB_OSGetClockMilliseconds(void)
{
   U32BIT millisecs;

   FUNCTION_START(STB_OSGetClockMilliseconds);

   millisecs = SysBootTime();
   RTC_DBG(("STB_OSGetClockMilliseconds: %u msecs", millisecs));

   FUNCTION_FINISH(STB_OSGetClockMilliseconds);

   return millisecs;
}



/*!**************************************************************************
 * @fn      STB_OSSetClockGMT
 * @brief   Set the time in seconds since midnight 1-1-1970 in GMT
 * @param   num_seconds - time in seconds
 ****************************************************************************/
void STB_OSSetClockGMT(U32BIT num_seconds)
{
   struct timespec sys_time;

   FUNCTION_START(STB_OSSetClockGMT);

   /* Set the system clock with the date/time in GMT */
   sys_time.tv_sec = num_seconds;
   sys_time.tv_nsec = 0;

   clock_settime(CLOCK_REALTIME, &sys_time);

   RTC_DBG(("STB_OSSetClockGMT: Time set to %u secs", num_seconds));

   FUNCTION_FINISH(STB_OSSetClockGMT);
}



/*!**************************************************************************
 * @fn      STB_OSGetClockGMT
 * @brief   Gets the time in seconds from the system clock
 * @return  time in seconds
 ****************************************************************************/
U32BIT STB_OSGetClockGMT(void)
{
   struct timespec sys_time;

   FUNCTION_START(STB_OSGetClockGMT);

   clock_gettime(CLOCK_REALTIME, &sys_time);

   RTC_DBG(("STB_OSGetClockGMT: Time=%lu secs", sys_time.tv_sec));

   FUNCTION_FINISH(STB_OSGetClockGMT);

   return sys_time.tv_sec;
}




/*--- Local function definitions ----------------------------------------------*/

/*!**************************************************************************
 * @fn      SysBootTime
 * @brief   Get the time in milliseconds since the system was booted.
 * @return  Time in milliseconds since system booted
 ****************************************************************************/
static U32BIT SysBootTime(void)
{
   static BOOLEAN first_call = TRUE;
   static struct timespec first_ts;

   struct timespec ts;
   U32BIT sec;
   U32BIT msec;
   U32BIT nsec;

   FUNCTION_START(SysBootTime);

   if (first_call)
   {
      clock_gettime(CLOCK_MONOTONIC, &first_ts);
      first_call = FALSE;
   }

   clock_gettime(CLOCK_MONOTONIC, &ts);

   sec = ts.tv_sec - first_ts.tv_sec;
   if (ts.tv_nsec < first_ts.tv_nsec)
   {
      nsec = ts.tv_nsec + 1000000000 - first_ts.tv_nsec;
      --sec;
   }
   else
   {
      nsec = ts.tv_nsec - first_ts.tv_nsec;
   }

   msec = sec * 1000 + nsec / 1000000;

   FUNCTION_FINISH(SysBootTime);
   return msec;
}


/****************************************************************************
** End of file
*****************************************************************************/

