/*******************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief
 * @file    stbos_signal.c
 * @date    2010
 */


/******************************************************************************
 * Andrii Zymohliad:
 *
 * I'm not sure if this file is used anywhere. The only global function it
 * defines (STB_OSSignalInitialise) is not declared in stbhwos.h header.
 *
 ******************************************************************************
 * Original header:
 *
 * This source file is used to print out a stack-trace when your program
 * segfaults. It is relatively reliable and spot-on accurate.
 *
 * This code is in the public domain. Use it as you see fit, some credit
 * would be appreciated, but is not a prerequisite for usage. Feedback
 * on it's use would encourage further development and maintenance.
 *
 * Due to a bug in gcc-4.x.x you currently have to compile as C++ if you want
 * demangling to work.
 *
 * Author: Jaco Kroon <jaco@kroon.co.za>
 *
 * Copyright (C) 2005 - 2010 Jaco Kroon
 *****************************************************************************/

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif


/*--- Includes ----------------------------------------------------------------*/

/* System header Files */
#include <memory.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <ucontext.h>
#include <dlfcn.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <limits.h>
#include <sys/types.h>

/* STB header Files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbheap.h"

/*--- Preprocessor definitions ------------------------------------------------*/

#if defined(REG_RIP)
#define SIGSEGV_STACK_IA64
#define REGFORMAT              "%016lx"
#elif defined(REG_EIP)
#define SIGSEGV_STACK_X86
#define REGFORMAT              "%08x"
#else
#define SIGSEGV_STACK_GENERIC
#define REGFORMAT              "%x"
#endif

#define ARR_SIZEOF(x)           (sizeof(x)/sizeof(x[0]))
#define DBGPRINT(x, ...)        fprintf(stderr, x "\n", ##__VA_ARGS__)


/*--- Local types definitions -------------------------------------------------*/


/*--- Local (static) variable declarations ------------------------------------*/


/*--- Local function prototypes -----------------------------------------------*/

static void SignalHandler(int signal);
static void SignalSegv(int signum, siginfo_t *info, void *ptr);


/*--- Global function definitions ---------------------------------------------*/


/*!**************************************************************************
 * @fn      STB_OSSignalInitialise
 ****************************************************************************/
void STB_OSSignalInitialise(void)
{
   struct sigaction action;

   FUNCTION_START(STB_OSSignalInitialise);

   memset(&action, 0, sizeof(action));
   action.sa_sigaction = SignalSegv;
   action.sa_flags = SA_SIGINFO;

   if (sigaction(SIGSEGV, &action, NULL) < 0)
   {
      perror("sigaction");
   }

   signal(SIGILL, SignalHandler);
   signal(SIGBUS, SignalHandler);

   FUNCTION_FINISH(STB_OSSignalInitialise);
}

/*--- Local function definitions ----------------------------------------------*/


/*!**************************************************************************
 * @fn      SignalSegv
 ****************************************************************************/
static void SignalSegv(int signum, siginfo_t *info, void *ptr)
{
   static const char *SI_CODES[3] = {"", "SEGV_MAPERR", "SEGV_ACCERR"};

   int i;
   int f;
   void **bp;
   void *ip;
   void *bt[20];
   char **strings;
   ucontext_t *ucontext;
   Dl_info dlinfo;
   size_t sz;

   FUNCTION_START(SignalSegv);

   DBGPRINT("Segmentation Fault!");
   DBGPRINT("info.si_signo = %d", signum);
   DBGPRINT("info.si_errno = %d", info->si_errno);
   DBGPRINT("info.si_code  = %d (%s)", info->si_code, SI_CODES[info->si_code]);
   DBGPRINT("info.si_addr  = %p", info->si_addr);
#if 0
   ucontext = (ucontext_t*) ptr;
#ifndef SIGSEGV_NOSTACK?
   for (i = 0; i != 19; i++)
   {
      DBGPRINT("reg[%02d]       = 0x" REGFORMAT, i, ucontext->uc_mcontext.gregs[i]);
   }

#if defined(SIGSEGV_STACK_IA64) || defined(SIGSEGV_STACK_X86)
#if defined(SIGSEGV_STACK_IA64)
   ip = (void*) ucontext->uc_mcontext.gregs[REG_RIP];
   bp = (void**) ucontext->uc_mcontext.gregs[REG_RBP];
#elif defined(SIGSEGV_STACK_X86)
   ip = (void*) ucontext->uc_mcontext.gregs[REG_EIP];
   bp = (void**) ucontext->uc_mcontext.gregs[REG_EBP];
#endif

   DBGPRINT("Stack trace:");
   while (bp && ip)
   {
      if (!dladdr(ip, &dlinfo))
      {
         break;
      }

      const char *SYMNAME = dlinfo.dli_sname;


      DBGPRINT("% 2d: %p <%s+%lu> (%s)", ++f, ip, SYMNAME,
            (unsigned long) ip - (unsigned long) dlinfo.dli_saddr,
            dlinfo.dli_fname);

      if (dlinfo.dli_sname && !strcmp(dlinfo.dli_sname, "main"))
      {
         break;
      }

      ip = bp[1];
      bp = (void**) bp[0];
   }
#else
   DBGPRINT("Stack trace (non-dedicated):");
   sz = backtrace(bt, 20);
   strings = backtrace_symbols(bt, sz);
   for (i = 0; i < sz; ++i)
   {
      DBGPRINT("%s", strings[i]);
   }
   free(strings);
#endif
   DBGPRINT("End of stack trace.");
#else
   DBGPRINT("Not printing stack strace.");
#endif
   _exit(-1);
#endif

   FUNCTION_FINISH(SignalSegv);
}


/*!**************************************************************************
 * @fn      SignalHandler
 * @brief   Handle signals to application (for now just cleanup)
 * @param   signal - signal number to handle
 ****************************************************************************/
static void SignalHandler(int signal)
{
   FUNCTION_START(SignalHandler);

   /* Call shutdown for all the HW areas to clean up resources */
   printf("Got Signal: %s\n", strsignal(signal));
#ifdef PGW_ENABLED
   STB_MemoryDumpPGW(TRUE);
#endif

   FUNCTION_FINISH(SignalHandler);
}


/****************************************************************************
** End of file
*****************************************************************************/
