/*******************************************************************************
 * Copyright © 2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Set Top Box - Linux OS Interface for Task Mechanism
 * @file    stbos_task.c
 * @date    September 2006
 */


/*--- Includes ----------------------------------------------------------------*/

/* System Header Files */
#include <pthread.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <sys/prctl.h>

/* STB Header Files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"

#include "stbhwdef.h"
#include "stbhwc.h"

/*--- Preprocessor definitions ------------------------------------------------*/

#define MAX_TASKS          64
#define MAX_DEPTH          64
#define MAX_COUNT          (1 << 5)
#define MSK_COUNT          (MAX_COUNT - 1)

/* Select-Deselect Local Debug Text Output */
#ifndef NDEBUG
#define TASK_DEBUG
#define CHECK_INIT_TASK_DATA()   if (task_count == 0) { \
                                    task_count++; \
                                    InitTaskData(); \
                                 }
#else
#undef TASK_DEBUG
#undef STACK_DEBUGGING
#endif


#ifndef TASK_DEBUG
#define TASK_DBG(X)
#else
#define TASK_DBG(X)              STB_SPDebugWrite X
#endif


/*--- Local types definitions -------------------------------------------------*/

#ifndef NDEBUG

#ifdef STACK_DEBUGGING
typedef struct func_name
{
   unsigned int enter;
   unsigned int depth;
   char name[256];
} S_FUNC_NAME;
#endif

typedef struct tasks
{
   pthread_t handle;
   char name[32];
#ifdef STACK_DEBUGGING
   unsigned int depth;
   unsigned int f_cnt;
   S_FUNC_NAME funcs[MAX_DEPTH];
   S_FUNC_NAME last_funcs[MAX_COUNT];
#endif
} S_TASKS;

#endif /* NDEBUG */


/*--- Local (static) variable declarations ------------------------------------*/

static pthread_mutex_t sleep_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t sleep_cond = PTHREAD_COND_INITIALIZER;

#ifndef NDEBUG
static const char UNKNOWN_NAME[] = "Main";
static U32BIT task_count;
static S_TASKS task_details[MAX_TASKS];

#ifdef STACK_DEBUGGING
static U32BIT task_last_enter;
#endif
#endif /* NDEBUG */


/*--- Local function prototypes -----------------------------------------------*/

#ifndef NDEBUG
static void InitTaskData(void);
#endif


/*--- Global function definitions ---------------------------------------------*/


/*!**************************************************************************
 * @fn      STB_OSCreateTask
 * @brief   Create a New Task to the calling process.
 *          Upon success, the created task runs on its own stack, as part of
 *          the calling process context.  Task termination must be achieved
 *          by calling STB_OSDestroyTask()
 * @param   function - task entry point
 * @param   param - user defined parameter passed when task is started
 * @param   stack - stack size
 * @param   priority - task priority, min 0, max 15
 * @param   name - task name
 * @return  handle of task
 ****************************************************************************/
void* STB_OSCreateTask(void (*function)(void*), void *param, U32BIT stack, U8BIT priority, U8BIT *name)
{
   struct sched_param parm;
   pthread_attr_t attr;
   pthread_t handle;
   int err;
#ifndef NDEBUG
   U32BIT tid;
#endif

   FUNCTION_START(STB_OSCreateTask);

#ifndef NDEBUG
   CHECK_INIT_TASK_DATA();
   ASSERT(task_count < MAX_TASKS);
#endif

   /* Create a set of default creation attributes */
   pthread_attr_init(&attr);

   if (stack > 0)
   {
      if (stack < 4096)
         stack = 4096;

      pthread_attr_setstacksize(&attr, stack);
   }

   /* Ensure thread is detached so resources are freed on exit */
   pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
   // pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED);
   err = pthread_attr_setschedpolicy(&attr, SCHED_FIFO);
   if (err == 0)
   {
      /* Although the policy returns min and max thread priorities of 1 and 99,
       * NXP have said that only values from 1-20 should be used as these are
       * the ones supposedly defined for POSIX, though I can't find out whether
       * this is true. As the range is very similar to the OBS range, we just
       * add 1 to the OBS value and use that. */
      parm.sched_priority = priority + 1;
      pthread_attr_setschedparam(&attr, &parm);
   }

   /* Create the task */
   err = pthread_create(&handle, &attr, (void* (*)(void*)) function, param);
   if (err != 0)
   {
      TASK_DBG(("pthread_create FAILED for %s err=%d", name, err));
      handle = (pthread_t) NULL;
   }
   else
   {
      /* Set the task name */
      pthread_setname_np(handle, (char*) name);
#ifndef NDEBUG
      if (err == 0)
      {
         task_count++;
         for (tid = 0; tid < MAX_TASKS; tid++)
         {
            if (task_details[tid].handle == 0)
            {
               strncpy(task_details[tid].name, (char*) name, 32);
               task_details[tid].handle = handle;
               break;
            }
         }
      }
#endif
   }

   /* Destroy the creation attributes */
   pthread_attr_destroy(&attr);
   TASK_DBG(("================== CreateTask %s: %x ==================", name, handle));
   FUNCTION_FINISH(STB_OSCreateTask);

   return (void*) handle;
}



/*!**************************************************************************
 * @fn      STB_OSTaskSuspend
 * @brief   Suspend Task
 ****************************************************************************/
void STB_OSTaskSuspend(void)
{
   FUNCTION_START(STB_OSTaskSuspend);
   sched_yield();
   FUNCTION_FINISH(STB_OSTaskSuspend);
}



/*!**************************************************************************
 * @fn      STB_OSTaskDelay
 * @brief   Delay Task for Specifed Time Period.
 * @param   timeout - delay in milliseconds.
 ****************************************************************************/
void STB_OSTaskDelay(U16BIT timeout)
{
   struct timespec t;

   FUNCTION_START(STB_OSTaskDelay);
   t.tv_sec = timeout / 1000;
   t.tv_nsec = ((U32BIT) timeout % 1000) * 1000000L;

   while ((nanosleep(&t, &t) < 0) && (errno == EINTR));
   FUNCTION_FINISH(STB_OSTaskDelay);
}



/*!**************************************************************************
 * @fn      STB_OSTaskSleep
 * @brief   Put Calling Task to Sleep. This is the equivalent of a Task
 *          Reschedule which is useful when a task wants to release the CPU
 *          on an application preemption point.
 ****************************************************************************/
void STB_OSTaskSleep(void)
{
   int status;

   FUNCTION_START(STB_OSTaskSleep);

   /* Lock the mutex */
   pthread_mutex_lock(&sleep_mutex);

   /* Wait on the cond variable */
   status = pthread_cond_wait(&sleep_cond, &sleep_mutex);
   STB_SPDebugWrite("%s: pthread_cond_wait=%d", __FUNCTION__, status);

   pthread_mutex_unlock(&sleep_mutex);

   FUNCTION_FINISH(STB_OSTaskSleep);
}



/*!**************************************************************************
 * @fn      STB_OSTaskWakeUp
 * @brief   Wake up a task that was put to sleep when it called STB_OSTaskSleep().
 ****************************************************************************/
void STB_OSTaskWakeUp(void)
{
   FUNCTION_START(STB_OSTaskWakeUp);
   pthread_mutex_lock(&sleep_mutex);
   pthread_cond_broadcast(&sleep_cond);
   pthread_mutex_unlock(&sleep_mutex);
   FUNCTION_FINISH(STB_OSTaskWakeUp);
}



/*!**************************************************************************
 * @fn      STB_OSDestroyTask
 * @brief   Delete Task must be called upon termination of each task as it
 *          frees all OS specific resources allocated for the specific task.
 * @bug     None.
 ****************************************************************************/
void STB_OSDestroyTask(void *DestroyTaskPtr)
{
#ifndef NDEBUG
   int tid;
#endif
   void *status;

   FUNCTION_START(STB_OSDestroyTask);
   if (DestroyTaskPtr != NULL)
   {
      pthread_join((pthread_t) DestroyTaskPtr, &status);
   }


#ifndef NDEBUG
   task_count--;
   for (tid = 0; tid < MAX_TASKS; tid++)
   {
      if (task_details[tid].handle == (pthread_t) DestroyTaskPtr)
      {
         task_details[tid].handle = 0;
         task_details[tid].name[0] = 0;
         break;
      }
   }
#endif

   FUNCTION_FINISH(STB_OSDestroyTask);
}



/*!**************************************************************************
 * @fn      STB_OSTaskPriority
 * @brief   Set a New Priority Level for Specified Task.
 * @param   task - task whose priority is to be changed
 * @param   priority - new priority
 * @return  Old task priority
 ****************************************************************************/
U8BIT STB_OSTaskPriority(void *task, U8BIT priority)
{
   struct sched_param parm;

   FUNCTION_START(STB_OSTaskPriority);

   ASSERT(task != NULL);
   memset(&parm, 0, sizeof(parm));
   parm.sched_priority =  priority;
   pthread_setschedparam(*((pthread_t*) task), SCHED_FIFO, &parm);

   FUNCTION_FINISH(STB_OSTaskPriority);

   return 0;
}



/*!**************************************************************************
 * @fn      STB_OSTaskLock
 * @brief   Lock Task
 *          Prevents task scheduler from preempting calling task and should
 *          always be used as a pair with STB_OSTaskUnlock() to create a
 *          critical region of code execution that cannot be interrupted by
 *          another task.
 ****************************************************************************/
void STB_OSTaskLock(void)
{
   FUNCTION_START(STB_OSTaskLock);

   FUNCTION_FINISH(STB_OSTaskLock);
}



/*!**************************************************************************
 * @fn      STB_OSTaskUnlock
 * @brief   Unlock Task
 *          Allows task scheduler to preempting calling task and should
 *          always be used as a pair with STB_OSTaskLock() to create a
 *          critical region of code execution that cannot be interrupted by
 *          another task.
 ****************************************************************************/
void STB_OSTaskUnlock(void)
{
   FUNCTION_START(STB_OSTaskUnlock);

   FUNCTION_FINISH(STB_OSTaskUnlock);
}


/*!**************************************************************************
 * @fn      STB_OSGetCurrentTask
 * @brief   Returns the handle of the current task
 * @return  Handle of current task
 ****************************************************************************/
void* STB_OSGetCurrentTask(void)
{
   FUNCTION_START(STB_OSGetCurrentTask);
   FUNCTION_FINISH(STB_OSGetCurrentTask);
   return (void*) pthread_self();
}




/*--- Local function definitions ----------------------------------------------*/

#ifndef NDEBUG

/*!**************************************************************************
 * @fn      InitTaskData
 * @brief   Init task data
 ****************************************************************************/
static void InitTaskData(void)
{
   int tid;

   FUNCTION_START(InitTaskData);

   /* memset can cause crash here, so just */
   for (tid = 0; tid != MAX_TASKS; tid++)
   {
#ifdef STACK_DEBUGGING
      int j;
      for (j = 0; j != MAX_DEPTH; j++)
      {
         task_details[tid].funcs[j].name[0] = 0;
      }
      for (j = 0; j != MAX_COUNT; j++)
      {
         task_details[tid].last_funcs[j].depth = 0;
         task_details[tid].last_funcs[j].enter = 0;
         task_details[tid].last_funcs[j].name[0] = 0;
      }
      task_details[tid].depth = 0;
      task_details[tid].f_cnt = 0;
#endif
      task_details[tid].handle = 0;
      task_details[tid].name[0] = 0;
   }
   task_details[0].handle = pthread_self();
   strcpy(task_details[0].name, (char*) UNKNOWN_NAME);

   FUNCTION_FINISH(InitTaskData);
}

#endif /* NDEBUG */

/****************************************************************************
** End of file
*****************************************************************************/

