/*****************************************************************************
*Copyright (c) Realtek, Inc. http://www.realtek.com.All rights reserved.
*FileName: rtk_hal_base_private.h
* Create Time: 2019. 5. 6.
* Description: A  HAL_HANDLE_S class is defined to support the customer APIs.
* @file  RHAL_Base.h
* @brief  Realtek HAL base interface.\n
* @author Realtek team
*Note:
*****************************************************************************
*/

#ifndef _RHAL_BASE_H_
#define _RHAL_BASE_H_

#include "rtk_hal_base.h" ///include the local header file rtk_hal_base.h
#ifdef ANDROID //for rvsd reset	
#include "rtk_porting_rvsdSocket.h"
#include "NativeAudioRedirect.h"
#endif

#if defined(USE_RTKHAL_IN_ANDROID9) 
#include <HDMILib.h>
#endif


//#ifdef CLIENT_DTVSOURCE
#include <rtk_ion.h>
//#endif
#ifdef ENABLE_GLINUX_RTK_PORTING
#include <vector>
#endif

#define RTK_MAX_DTVFLOW_NO (32)  /// the amount number of flow to support,now up to 32 channel flow data
#define RTK_DEMUX_CHANNEL_BEGIN	(16) 
#define RTK_MAX_DTVSOURCE_MUM	(16)
#define RTK_SOURCE_DMX_CHANNEL_SHIFT	(32)


//#ifdef CLIENT_DTVSOURCE
#define MTP_BUF_SIZE 188*512 // 94KB
#define MTP_BUF_COUNT 4
/**
** @brief
**  mtp buffer.
*/
typedef struct {
    ION_DATA ion;
    Source_Handle source; // NULL means no user
    uint32_t write_ofs;
    uint32_t cache_ofs;
    uint32_t bufsize;
    uint32_t freesize;
    struct timeval last_time;
    bool pre_lend;
    Rtk_Porting_Source_Type_e type;
}Rtk_MtpBuf_t;
//#endif // #ifdef CLIENT_DTVSOURCE

enum PlayingStatus 
{
	RHAL_VIDEO_PLAYING=(1<<0),
	RHAL_AUDIO_PLAYING=(1<<1),	
};


#ifndef RTKPLAYER_ARCH_VER //xiongz
#error "RTKPLAYER_ARCH_VER not defined!"
#endif
/**
  * @brief
  *	This structure define RHAL handle parameter
  *  reference for a RHAL handle thread
  */
typedef struct
{
	RTK_Bool				create_by_duplicate;
	RTK_Bool				used;
	HAL_TYPE				type; /// HAL_TYPE_PLAYER or HAL_TYPE_DEMUX or HAL_TYPE_EMMDEMUX
	union
	{
		RTK_PlayerConfig	player_config;///the player handler configure parameter struct
		RTK_DemuxConfig		demux_config;
	};
	UINT32_T				channel;///which channel data flow, either 0 or 1
	UINT32_T				av_handle;///the av handler order number return 
	DEMUX_HANDLE_S*			dmx_handle;///
	DEMUX_PES_HANDLE_S*		dmx_pes_handle;///

	
	pthread_t				cc_recvdata_thread;
	RTK_Bool				stop_cc_thread;
	pthread_t				sec_callback_thread;///the callback  thread when section data is parsed out and needs to deal with
	pthread_t				pes_callback_thread;///the callback	thread when pes data is parsed out and needs to deal with
	UINT8_T					stop_thread;
	pthread_t				rec_callback_thread;///the callback  thread when record data needs to deal with
	RTK_Bool				isRecording;
	UINT32_T				uiTsFilterCnt;
	ULONG_T*				pulRecShareMemFd;
	UINT32_T				playingStatus;

	Mtp_Mem_Pool_t*			DemuxMtpMemPool;	///demux MTP buffer pointer
	UINT32_T				DemuxMtpMemSize;///demux MTP buffer size
	UINT32_T				DemuxMtpMemMapfd;///parameter related to the memory device
	ULONG_T					DemuxMtpMemSharefd;///the share memory size of demux MTPMemPool
	INT32_T					DemuxMtpMemIonFd;///parameter related to the memory device
	INT32_T					DemuxMtpMemRtkIonFd; /// memory descriptor
	ion_user_handle_t		DemuxMtpMemIonHandle;///parameter related to the memory device

	RTK_RingMem_Pool_t*		PlayerMtpMemPool;	///player share buffer pointer
	UINT32_T				PlayerMtpMemSize;///player share buffer size
	UINT32_T				PlayerMtpMemMapfd;///parameter related to the player memory device
	ULONG_T					PlayerMtpMemSharefd;///the share memory size of player MTPMemPool
	INT32_T					PlayerMtpMemIonFd;///parameter related to the palyer memory device
	INT32_T					PlayerMtpMemRtkIonFd;/// memory descriptor
	ion_user_handle_t		PlayerMtpMemIonHandle;///parameter related to the player memory device
	
	UINT8_T					got_mtpbuf;  /// mark the mtpbuf whether used,0 not use,1 in use
	int64_t					last_got_time;///mark the time to be used

	RTK_NAV_AVSYNC_MODE		AVSyncMode;///hal_handler in which avSync Mode

	pthread_mutex_t			mutex;///the mutex to protect the hal_handler
	float					speed;///the  play speed of hal_handler
#ifdef CLIENT_DTVSOURCE
	Source_Handle			internal_source;
	Source_Handle			connect_source;
	ION_DATA*				mtp_ion;
#endif
	void*					handle_ionMapped_esbuf;  ///the handle of element stream input buffer witch mapped from ion share memory.
	RTK_RingMem_Pool_Header_t*	pEsRingbuf_header; ///element stream input ring buffer header. 
	uint8_t*				pEsbufVirt; ///element stream input ring buffer data area begin address.
	Hal_CCData_t			m_CCData;///store cc data
	bool					m_bGetCCdata;
	RTK_Bool				rec_inSecurebuf;
	int						rec_secbuf_fd;

	//for secure play share memory.
	bool					isSecurePlayShmUsed;
	void*					pSecurePlayShmHandle;
	int						securePlayShm_fd;
	unsigned char*			pSecurePlayShm_virt;
	unsigned int			securePlayShm_size;
	Rtk_Subtitle_Info_t		m_subtitle;
} HAL_HANDLE_S;


/**
   * @brief
   *	This structure define RHAL handle thread parameter
   *  reference for a RHAL handle thread
   */

typedef struct
{
	void* instance;///the object of RTKHAL_base
	void* halHandle;///the pointer of HAL_HANDLE_S hanldler
	bool bUseSourceFilter;
}RHAL_ThreadParam_t;

/**
* @brief
*	This enum define the status of the DVB channel.
*/
typedef enum 
{
	RHAL_CHANNEL_free=0,
	RHAL_CHANNEL_allocated,
	RHAL_CHANNEL_using,
}RHAL_CHANNEL_USE_STATE_e;

typedef struct 
{
	RHAL_CHANNEL_USE_STATE_e channelState; 
	HAL_TYPE channelHalType;
}RHAL_ChannelStatus_t;

/**
* @brief
*	This enum define the source demux handle info.
*/
typedef struct 
{
	bool isUsing;
	Source_Handle source_handle;
	DEMUX_HANDLE_S* dmx_handle;
	bool sectionThread_stop;
	pthread_t sectionThread_id;
}RHAL_SourceDmxHandleInfo_t;



/**
   * @brief 
   *   RHAL_Base class definition body
   */

class RHAL_Base {
	
public:
	static HAL_HANDLE_S* m_pCurHal_Handle[RTK_MAX_DTVFLOW_NO];///an array of player handler pointer
	static Source_Handle m_pCurSource_Handle[RTK_MAX_DTVSOURCE_MUM];///an array of emmdemux handler pointer

	static pthread_mutex_t m_av_create_mutex;
	
	UINT32_T m_uiTunerChannel;
	
#if defined(USE_RTKHAL_IN_ANDROID9) 
	realtek::HDMILib  *mhal_hdmiLib;
#endif
	
	RHAL_Base(int dtvflow_num);///the class constructor function
	virtual ~RHAL_Base();///the class destructor function 

/**
   * @brief
   *This function creates player and does initial works for this player.
   *
   *@param[out]		player		 return the Player handle pointer created by this function
   *@param[in]		config		 the configs used when creating the player
   *@param[in]		type			 the tv flow handle type,HAL_TYPE_PLAYER or HAL_TYPE_DEMUX or HAL_TYPE_EMMDEMUX
   *
   * @retval		RTK_OK			 RTK_Player_Creat has successfully processed.
   * @retval		RTK_Fail 			 parameters error occurred or Create player return failed.
   *
   */
	virtual RTK_Error Player_Create(RTK_Player * player, RTK_PlayerConfig * config, HAL_TYPE type);


/**
   * @brief
   *This function update vo plane after player start.
   *
   * @param[in]	player		 the player
   * @param[in]	plane		 the plane
   *
   * @retval		RTK_OK	        success.
   * @retval		RTK_Fail 	        failed.
   *
   */
	virtual RTK_Error Player_Update_VO_Plane(RTK_Player player, RTK_WinPlane plane);

/**
   * @brief
   * This function starts player.
   *
   * @param[in]	player		 	the player handle
   *
   * @retval		RTK_OK			the player starts playing successfully.
   * @retval		RTK_Fail 			parameters error occurred or the player doesn't start to play.
   *
   */
	virtual RTK_Error Player_Start(RTK_Player player);
/**
   * @brief
   * This function stops player.
   *
   * @param[in]	player		 	the player handle
   * @param[in]	keepLastFrame	if true the last frame will be show when stop,if false nothing will be shown when stop
   *
   * @retval		RTK_OK			the player stops playing successfully.
   * @retval		RTK_Fail 			parameters error occurred or the player doesn't stop to play.
   *
   */
	virtual RTK_Error Player_Stop(RTK_Player player, RTK_Bool keepLastFrame);
/**
   * @brief
   * This function destories the player and frees related memories.
   *
   * @param[in]	player		 	the player handle
   *
   * @retval		RTK_OK			the player is destoried  successfully.
   * @retval		RTK_Fail 			parameters error occurred or the player doesn't stop to play.
   *
   */
	virtual RTK_Error Player_Destroy(RTK_Player player);
/**
   * @brief
   * This function makes the player paused. It is very similar to the RTK_Player_Stop,
   * but this function will not flush the memories.
   *
   * @param[in]	player		 	the player handle
   *
   * @retval		RTK_OK			the player is destoried  successfully.
   * @retval		RTK_Fail 			parameters error occurred or the player doesn't stop to play.
   *
   */
	virtual RTK_Error Player_Pause(RTK_Player player);
/**
   * @brief
   * This function resumes the playing. After calling RTK_Player_Pause or RTK_Player_Stop or RTK_Player_StopEx to stop playing, 
   * RTK_Player_Resume could resume the playing.
   *
   * @param[in]	player		 	the player handle
   *
   * @retval		RTK_OK			the player is replays again successfully.
   * @retval		RTK_Fail 			 parameters error occurred or the player doesn't replay.
   *
   */
	virtual RTK_Error Player_Resume(RTK_Player player);
/**
   * @brief
   * This function drops all previously fed TS data from internal buffer which is not yet processed
   * (not decoded, or waiting for display)
   * This function is used for "Seek" operation to clear si ring buffer,
   * AV buffer and demux share memory by calling rtk_porting_av_flush. 
   * If the input source is tuner, it should also flush tuner buffer.
   *
   * @param[in]	player		 	the player handle
   *
   * @retval		RTK_OK			RTK_Player_DropData has successfully processed.
   * @retval		RTK_Fail 			 parameters error occurred or data is dropped failed.
   *
   */
	virtual RTK_Error Player_DropData(RTK_Player player);
/**
   * @brief
   * This function opens and sets sectionFilter by specific pid, add the SectionFilter into SI engine with the player handle
   *
   *
   * @param[in]	player		 	handle of player to add section filter
   * @param[out]      filter_handle            handle of sectionFilter to specify
   * @param[in]      pattern       defines filter pid, mask, mode, compose type, monitor mode, 
   *                                           user param and a function pointer to SI section filter callback
   *
   * @retval		RTK_OK			AddSectionFilter has successfully processed
   * @retval		RTK_Fail 			parameters error occurred or filter open failed
   *
   */	
	virtual RTK_Error Player_AddSectionFilter(RTK_Player player, RTK_SectionFilter * filter_handle, RTK_SectionFilterMask * pattern);
/**
   * @brief
   * This function stops the sectionFilter, then it should not parse the specific sections.
   * This function asks for stopping filter the sections with specific filter_handle. 
   * When sectionFilter stops, it will not receive related sections.
   *
   *
   * @param[in]	player		 	handle of player to add section filter
   * @param[in]      filter_handle            handle of sectionFilter to stop
   *
   * @retval		RTK_OK			AddSectionFilter has successfully processed.
   * @retval		RTK_Fail 			parameters error occurred or filter open failed.
   *
  */
	virtual RTK_Error Player_RemoveSectionFilter(RTK_Player player, RTK_SectionFilter filter_handle);

/**
	* @brief
	* enable the sectionFilter status
	*
	* @param[in]	player			handle of player 
	* @param[in]	filter_handle  handle of filter to enable
	* @param[in]	enable			enable or disable section fitler
	*
	* @retval		RTK_OK			section filter has been enable or disable success
	* @retval		RTK_Fail		parameters error occurred or SI engine PID 
	*								filter enable or disable failed
	*/
	virtual RTK_Error Player_ControlSectionFilter(RTK_Player player, RTK_SectionFilter filter_handle,RTK_Bool enable);

/**
   * @brief
   * This function stops all sectionFilters that player added, then sectionFilters will not work .
   * This function will stop all opened filters with the player handle.
   *
   * @param[in]	player		 	handle of player to deal with
   *
   * @retval		RTK_OK			all section filters have been stopped parsing successfully.
   * @retval		RTK_Fail 			parameters error occurred or SI engine PID filter remove failed.
   *
   */
	virtual RTK_Error Player_RemoveAllSectionFilters(RTK_Player player);
/**
   * @brief
   * This function add TS filter and parse the TS data through callback function.
   *
   * @param[in]	player		 	handle of player to add TS filter
   * @param[in]	pattern			defines filter pid, a flag means the captured packets encrypted or not, A/V codec types,
   *                                                        isAnchor means use this pid to get PTS and Frame type or not
   * @param[out]	filter_handle		return TS handle of filter to specify
   *
   * @retval		RTK_OK			AddTSFilter has successfully processed
   * @retval		RTK_Fail 			parameters error occurred or filter add failed
   *
   */
	virtual RTK_Error Player_AddTSFilter(RTK_Player player, RTK_TSFilterPattern * pattern, RTK_TSFilter * filter_handle);
/**
   * @brief
   * removes the TSFilter, then it should not parse the specific TS data
   *
   * @param[in]	player		 	handle of player to deal with
   * @param[in]      filter_handle            handle of TsFilter to stop
   *
   * @retval		RTK_OK			TS filter has been stopped parsing with specific filter_handle successfully
   * @retval		RTK_Fail 			parameters error occurred or ts filter remove failed
   *
   */
	virtual RTK_Error Player_RemoveTSFilter(RTK_Player player, RTK_TSFilter filter_handle);
/**
   * @brief
   * removes all TSFilter added, then TSFilters will don't work .
   *
   * @param[in]	player		 	handle of player to deal with
   *
   * @retval		RTK_OK			all TS filters have been stopped parsing successfully
   * @retval		RTK_Fail 			parameters error occurred or TSFliter remove failed
   *
   */
	virtual RTK_Error Player_RemoveAllTSFilters(RTK_Player player);
	virtual RTK_Error Player_ResetTSFilterParser(RTK_Player player);
/**
   * @brief
   * This function will modifies pid of PCR. During playing, if audio and video are out of sync, 
   * Player_ChangePCR is used to  modify PCR PID for the player with specific handle, 
   * which is good for correcting audio/video 	out-sync problems.
   *
   * @param[in]	player		 	handle of player to deal with
   * @param[in]	pcr_pid                   pcr pid which the player will change to
   *
   * @retval		RTK_OK			the PCR has changed successfully
   * @retval		RTK_Fail 			parameters error occurred or PCR change failed
   *
   */
	virtual RTK_Error Player_ChangePCR(RTK_Player player, uint16_t pcr_pid);
/**
   * @brief
   * This function will change audio pid, audio codec type, then play back the audio.
   *
   * @param[in]	player		     handle of player to change audio configuration
   * @param[in]	audio                   pointer to a prepared RTK_AudioConfig object
   *
   * @retval		RTK_OK		     change audio and play successfully
   * @retval		RTK_Fail 		     parameter input error such as player or audio equals NULL
   *
   */
	virtual RTK_Error Player_ChangeAudio(RTK_Player player, RTK_AudioConfig * audio);
/**
   * @brief
   * changes each track audio PID, codec type, then plays back the multi-audio.
   *
   * @param[in]	player		     handle of player to change audio configuration
   * @param[in]	audio                   pointer to an Array of RTK_AudioConfig object
   * @param[in]      count                    the amount number of audio track
   *
   * @retval		RTK_OK		     change audio and play successfully
   * @retval		RTK_Fail 		     parameter input error such as player or audio equals NULL
   *
   */
	virtual RTK_Error Player_ChangeAudioEx(RTK_Player player, RTK_AudioConfig **audio, uint8_t count) ;
/**
   * @brief
   * changes video PID, video codec type, then play back the audio.
   *
   * @param[in]	player		     handle of player to change audio configuration
   * @param[in]	video                   defines video PID, codec type, a flag indicate PID stream is encrypted or not, Video window type
   *
   * @retval		RTK_OK		     change video and play successfully
   * @retval		RTK_Fail 		     parameter input error such as player or audio equals NULL
   *
   */
	virtual RTK_Error Player_ChangeVideo(RTK_Player player, RTK_VideoConfig * video);
/**
   * @brief
   * This function will apply a buffer from share memory to reserve TS data. 
   * If successfully, these information be will returned in the parameter buffer, this parameter is clear to recognize buffer features.
   *
   * @param[in]	player		     handle of player to get buffer
   * @param[out]	buffer                  get the viraddr, size, filled_size, islast when return input buffer
   *
   * @retval		RTK_OK		    got the buffer information successfully
   * @retval		RTK_Fail 		    parameters error occurred or share buff has full
   *
   */
	virtual RTK_Error Player_GetInputBuffer(RTK_Player player, RTK_TSInputBuffer * buffer);
	virtual RTK_Error Player_GetInputInfoBuffer(RTK_Player player, unsigned char **ppInfoBuf, unsigned int *pInfoBufSize);
	virtual RTK_Error Player_DecryptResetOffset(RTK_Player player, uint64_t ofs);	
	virtual RTK_Error Player_WaitDataGone(RTK_Player player, int32_t timeOut_MS);
	virtual RTK_Error Player_InputRedirect(RTK_Player player, bool isEnable, RTK_Demux redirect_from);
/**
   * @brief
   * This function will feed some TS data used for playing or parsed in input_buffer.
   *
   * @param[in]	player		     handle of player to feed buffer
   * @param[out]	buffer                  define input buffer viraddr, size, filled_size, islast
   *
   * @retval		RTK_OK		    FeedInputBuffer has successfully processed
   * @retval		RTK_Fail 		    parameters error occurred or file size not packet align
   *
   */
	virtual RTK_Error Player_FeedInputBuffer(RTK_Player player, RTK_TSInputBuffer * buffer);
/**
   * @brief
   * This function will discard input data buffer.
   *
   * @param[in]	player		     handle of player to discard input buffer
   * @param[in]	buffer                  define input buffer viraddr, size, filled_size, islast
   *
   * @retval		RTK_OK		    DiscardInputBuffer has successfully processed
   * @retval		RTK_Fail 		    parameters error occurred or doesn't get buffer first
   *
   */
	virtual RTK_Error Player_DiscardInputBuffer(RTK_Player player, RTK_TSInputBuffer * buffer);

   virtual RTK_Error Player_SetInputBufferAlign(RTK_Player player, uint32_t align);
   /**
   * @brief
   * Feed element stream into es buffer.
   *
   * @param[in] 	   player		   handle of player 
   * @param[in] 	   buffer		   element stream input buffer.
   * @param[in] 	   timeout		   feed timeout.(millisecond)
   *
   * @retval	   RTK_OK		   successfully
   * @retval	   RTK_Fail 	   fail or timeout.
   *
   */
	virtual RTK_Error Player_FeedEsInputBuffer(RTK_Player player, RTK_EsInputBuf_t *buffer, int32_t timeout);


/**
   * @brief
   * This function will set AV_sync_mode of a specific player.
   *
   * @param[in]	player		    handle of player to deal with
   * @param[in]	AvSyncMode       avsync type
   *
   * @retval		RTK_OK		    set the avsync mode successfully
   * @retval		RTK_Fail 		    parameters error occurred or rtk_porting_av_set_AV_sync_mode return error
   *
   */
	virtual RTK_Error Player_SetAVSyncMode(RTK_Player player, RTK_AVsyncMode AvSyncMode);
#ifdef ANDROID
/**
   * @brief
   * This function will create the side band handle about dvdplayer.
   *
   * @param[in]	player		    handle of player to deal with
   *
   * @retval		handle		    handle of  the side band
   *
   */
	virtual void* Player_GetSidebandHandle(RTK_Player player);
#endif
/**
   * @brief
   *This function can change the play speed of the player with specific handle.
   * speed is a multiplier factor, for example, speed is equal to 0.8, means the play speed slow down to 0.8 times, 1.5 means faster to 1.5 times, and so on.
   *
   * @param[in]	player		    handle of player to deal with
   * @param[in]	speed                 multiplier factor of change, should among  seven values of 0.5,0.6,0.8,1,1.2,1.5,2
   *
   * @retval		RTK_OK		     the play speed has changed successfully
   * @retval		RTK_Fail 		    parameters error occurred or the play speed change failed
   *
   */
	virtual RTK_Error Player_SetPlaySpeed(RTK_Player player, float speed);
/**
   * @brief
   *This function can get the play speed of the player with specific handle.
   * speed is a multiplier factor, for example, speed is equal to 0.8, means the play speed slow down to 0.8 times, 1.5 means faster to 1.5 times, and so on.
   *
   * @param[in]	player		    handle of player to deal with
   * @param[out]	speed                 get the multiplier factor of current paly speed
   *
   * @retval		RTK_OK		    the play speed has got successfully
   * @retval		RTK_Fail 		    parameters error occurred or play speed get failed
   *
   */
	virtual RTK_Error Player_GetPlaySpeed(RTK_Player player, float * speed);
/**
   * @brief
   *This function will get the current displaying frame's pts(presentation time stamp)
   *
   * @param[in]	player		    handle of player to deal with
   * @param[out]	_pts                    get the current displaying frame's pts
   *
   * @retval		RTK_OK		    the current displaying PTS has got successfully
   * @retval		RTK_Fail 		    parameters error occurred or PTS get failed
   *
   */
	virtual RTK_Error Player_GetCurrentDisplayingPTS(RTK_Player player, uint64_t * _pts);

/**
   * @brief
   *This function will get the current displaying frame's pts(presentation time stamp)
   *
   * @param[in] player			handle of player to deal with
   * @param[out]	pts_info	get the current displaying frame's video and audio pts
   *
   * @retval		RTK_OK			the current displaying video and audio PTS has got successfully
   * @retval		RTK_Fail			parameters error occurred or PTS get failed
   *
   */
	virtual RTK_Error Player_GetCurrentDisplayingPTSInfo(RTK_Player player, RTK_PTSInfo * pts_info);

/**
   * @brief
   * This function will set the player with specific handle video plane visible or not
   *
   * @param[in]	player		    handle of player to deal with
   * @param[in]	mute                   defines set video mute or not,RTK_FALSE means set video plane visible, RTK_TRUE means not
   *
   * @retval		RTK_OK		    set the specific player video plane visible mute successfully
   * @retval		RTK_Fail 		    parameters error occurred or set mute failed
   *
   */
	virtual RTK_Error Player_SetVideoMute(RTK_Player player, RTK_Bool mute);
/**
   * @brief
   * This function will get the player with specific handle visible or not status
   *
   * @param[in]	player		    handle of player to deal with
   * @param[out]	mute                   the value of video visible status currently,RTK_FALSE means video plane visible, RTK_TRUE means not
   *
   * @retval		RTK_OK		    get the specific player video visible status successfully
   * @retval		RTK_Fail 		  parameters error occurred or got video visible status failed
   *
   */
	virtual RTK_Error Player_GetVideoMute(RTK_Player player, RTK_Bool * mute);
/**
   * @brief
   * This function will set the player with specific handle audio mute or not
   *
   * @param[in]	player		    handle of player to deal with
   * @param[in]	mute                   defines set audio mute or not, RTK_FALSE means set audio not mute, RTK_TRUE means audio mute
   *
   * @retval		RTK_OK		    set the specific player audio mute successfully.
   * @retval		RTK_Fail 		    parameters error occurred or set mute failed
   *
   */
	virtual RTK_Error Player_SetAudioMute(RTK_Player player, RTK_Bool mute);
/**
   * @brief
   * This function will get the player with specific handle audio mute status
   *
   * @param[in]	player		  handle of player to deal with
   * @param[out]	mute                 the value of audio mute status currently,RTK_FALSE means audio not mute, RTK_TRUE means audio mute
   *
   * @retval		RTK_OK		  get the specific player video visible status successfully
   * @retval		RTK_Fail 		  parameters error occurred or got video visible status failed
   *
   */
	virtual RTK_Error Player_GetAudioMute(RTK_Player player, RTK_Bool * mute);
/**
   * @brief
   * This function will adjust the audio sound volume
   *
   * @param[in]	i32volume	 the volume value to set, range from 0 to 31
   * 
   * @retval		void		         void is returned after the volume is set.
   * 
   */
	virtual void Player_SetAudioVolume(uint32_t i32volume);
/**
   * @brief
   * This function will get the audio sound volume
   *
   * @param[out]	i32volume	 the volume value to set, range from 0 to 31
   * 
   * @retval		 void		        void is returned after the volume is got.
   * 
   */
	virtual void Player_GetAudioVolume(uint32_t* i32volume);
/**
   * @brief
   * This function is used to send I frame information to dvdplayer when it's in I-frame-only mode.
   *
   * @param[in]	player	         handle of player to add send I frame information
   * @param[in]	info    	         object composed of frame info
   *
   * @retval		RTK_OK		  I frame information is sent successfully.
   * @retval		RTK_Fail 		  parameters error occurred or sending I frame information failed.
   *
   */
	virtual RTK_Error Player_SendIFrameInfo(RTK_Player player, RTK_IframeOffset_t info);
	virtual RTK_Error Player_GetPacketDemuxInfo(RTK_Player player, Hal_Demux_Info_t* demuxInfo);
	virtual RTK_Error Player_EnableSecurePlayShareMem(RTK_Player player, RTK_Bool en);
	virtual RTK_Error Player_SetTrickMode(RTK_Player player, RTK_HAL_avTrickMode_e trickMode);

/**
   * @brief
   * Get Ts data input bitrate which calculated with PCR info.
   *
   * @param[in]	player		 	handle of player to deal with
   * @param[out]	pullBitrate 	pointer of the variable witch will store the result.
   *
   * @retval		RTK_OK			success.
   * @retval		RTK_ERR 		failed. 
   *
   */
	virtual RTK_Error Player_GetTsInputBitrate(RTK_Player player, UINT64_T* pullBitrate);

/**
   * @brief
   * Record start.
   *
   * @param[in]		player		 	handle of player to deal with
   *
   * @retval		RTK_OK			success.
   * @retval		RTK_ERR 		failed. 
   *
   */
	virtual RTK_Error Record_Start(RTK_Player player);

/**
   * @brief
   * Record stop.
   *
   * @param[in]		player		 	handle of player to deal with
   *
   * @retval		RTK_OK			success.
   * @retval		RTK_ERR 		failed. 
   *
   */
	virtual RTK_Error Record_Stop(RTK_Player player);

//RTKHAL 2.1 add TS demux.
#if 1
/**
   * @brief
   * The function will create demux and do initial works for the demux.
   *
   * @param[out]	demux	         return the Demux handle created by this function
   * @param[in]	demux_config   the configs used when creating the Demux
   * @retval		RTK_OK		  Demux created successfully
   * @retval		RTK_Fail 		  parameters error occurred or Demux created fail. 
   *
   */
	virtual RTK_Error Demux_Create(RTK_Demux * demux, RTK_DemuxConfig * demux_config);
/**
   * @brief
   * 	The function will destroy the Demux and destroy the recording flow.
   *
   * @param[in]	demux	         he Demux handle, this handle must be same with the created one
   *
   * @retval		RTK_OK		  Demux destoryed successfully
   * @retval		RTK_Fail 		  parameters error occurred or Demux destoryed fail. 
   *
   */
	virtual RTK_Error Demux_Destroy(RTK_Demux demux);
/**
   * @brief
   * 	The function will start the Demux flow.
   *
   * @param[in]	demux	         he Demux handle, this handle must be same with the created one
   *
   * @retval		RTK_OK		  Demux destoryed successfully
   * @retval		RTK_Fail 		  parameters error occurred or Demux destoryed fail. 
   *
   */
	virtual RTK_Error Demux_Start(RTK_Demux demux);
/**
   * @brief
   * 	The function will stop the Demux and do the opposite jobs of the RTK_Demux_Start. 
   *    You can keep the last frame to display.
   *
   * @param[in]	demux	         handle of Demux, this handle must be same with the created one
   *
   * @retval		RTK_OK		  Demux stopped successfully
   * @retval		RTK_Fail 		  parameters error occurred or stopped the Demux failed. 
   *
   */
	virtual RTK_Error Demux_Stop(RTK_Demux demux);

	virtual RTK_Error Demux_Pause(RTK_Demux demux);
	virtual RTK_Error Demux_Resume(RTK_Demux demux);
/**
   * @brief
   * This function will open and set sectionFilter by specific pid, then add the SectionFilter into SI engine with the demux handle.
   * This function is the same usage as Player_AddSectionFilter except handle difference.
   *
   * @param[in]      demux		 	handle of demux to add section filter
   * @param[out]      filter_handle            handle of sectionFilter to specify
   * @param[in]      pattern                    defines filter pid, mask, mode, compose type, monitor mode, 
   *                                                        user param and a function pointer to SI section filter callback.
   *
   * @retval		RTK_OK			AddSectionFilter has successfully processed
   * @retval		RTK_Fail 			parameters error occurred or filter open failed
   *
   */
	virtual RTK_Error Demux_AddSectionFilter(RTK_Demux demux, RTK_SectionFilter * filter_handle, RTK_SectionFilterMask * pattern);
/**
   * @brief
   * This function will stop the sectionFilter, then it should not parse the specific sections.
   * This function asks for stopping filter the sections with specific filter_handle. 
   * When sectionFilter stops, it will not receive related sections.
   * This function is the same usage as Player_RemoveSectionFilter except handle difference.
   *
   *
   * @param[in]	demux		 	handle of demux to add section filter
   * @param[in]       filter_handle            handle of sectionFilter to stop
   *
   * @retval		RTK_OK			RemoveSectionFilter has successfully processed.
   * @retval		RTK_Fail 			parameters error occurred or filter open failed.
   *
   */
	virtual RTK_Error Demux_RemoveSectionFilter(RTK_Demux demux, RTK_SectionFilter filter_handle);
/**
   * @brief
   * This function will stops all sectionFilters that demux added, then sectionFilters will not work .
   * This function is the same usage as Player_RemoveAllSectionFilter except handle difference.
   *
   * @param[in]	demux		 	handle of demux to deal with
   *
   * @retval		RTK_OK			all section filters have been stopped parsing successfully.
   * @retval		RTK_Fail 			parameters error occurred or SI engine PID filter remove failed.
   *
   */
	virtual RTK_Error Demux_RemoveAllSectionFilters(RTK_Demux demux);
/**
   * @brief
   * This function will add TF filter.
   * This function is the same usage as Player_AddTSFilter except handle difference.
   *
   * @param[in]	demux		 	handle of demux to add TS filter
   * @param[in]	pattern			defines filter pid, a flag means the captured packets encrypted or not, A/V codec types,
   *                                                        isAnchor means use this pid to get PTS and Frame type or not
   * @param[out]	filter_handle  		 return handle of filter to specify
   *
   * @retval		RTK_OK			AddTSFilter has successfully processed
   * @retval		RTK_Fail 			parameters error occurred or filter add failed
   *
   */
	virtual RTK_Error Demux_AddTSFilter(RTK_Demux demux, RTK_TSFilterPattern * pattern, RTK_TSFilter * filter_handle);
/**
   * @brief
   * removes the TSFilter, then it should not parse the specific TS data
   * This function is the same usage as Player_RemoveTSFilter except handle difference.
   *
   * @param[in]	demux		       handle of demux to deal with
   * @param[in]       filter_handle          handle of TsFilter to stop
   *
   * @retval		RTK_OK			TS filter has been stopped parsing with specific filter_handle successfully
   * @retval		RTK_Fail 			parameters error occurred or ts filter remove failed
   *
   */
	virtual RTK_Error Demux_RemoveTSFilter(RTK_Demux demux, RTK_TSFilter filter_handle);
/**
   * @brief
   * removes all TSFilter added, then TSFilters will don't work.
   * This function is the same usage as Player_RemoveAllTSFilter except handle difference.
   *
   * @param[in]	demux		 	handle of demux to deal with
   *
   * @retval		RTK_OK			all section filters have been stopped parsing successfully
   * @retval		RTK_Fail 			parameters error occurred or TSFliter remove failed
   *
   */
	virtual RTK_Error Demux_RemoveAllTSFilters(RTK_Demux demux);

	virtual RTK_Error Demux_EnableTSFilterSecurebuf(RTK_Demux demux, RTK_Bool isSecure);
	virtual RTK_Error Demux_ResetTSFilterParser(RTK_Demux demux);
/**
   * @brief
   * This function will apply a buffer from share memory to reserve TS data. 
   * If successfully, these information be will returned in the parameter buffer, this parameter is clear to recognize buffer features.
   *
   * @param[in]	demux		     handle of demux to get buffer
   * @param[out]	buffer                  get the viraddr, size, filled_size, islast when return input buffer
   *
   * @retval		RTK_OK		    got the buffer information successfully
   * @retval		RTK_Fail 		    parameters error occurred or share buff has full
   *
   */
	virtual RTK_Error Demux_GetInputBuffer(RTK_Demux demux, RTK_TSInputBuffer * buffer);
/**
   * @brief
   * This function will feed some TS data used for playing or parsed in input_buffer.
   * This function is the same usage as Player_FeedInputBuffer except handle difference.
   *
   * @param[in]	demux		     handle of demux to feed buffer
   * @param[in]	buffer                  define input buffer viraddr, size, filled_size, islast
   *
   * @retval		RTK_OK		    FeedInputBuffer has successfully processed
   * @retval		RTK_Fail 		    parameters error occurred or file size not packet align
   *
   */
	virtual RTK_Error Demux_FeedInputBuffer(RTK_Demux demux, RTK_TSInputBuffer * buffer);
/**
   * @brief
   * This function will discard input data buffer.
   * This function is the same usage as Player_DiscardInputBuffer except handle difference.
   *
   * @param[in]	demux		    handle of demux to discard input buffer
   * @param[in]	buffer                  define input buffer viraddr, size, filled_size, islast
   *
   * @retval		RTK_OK		    DiscardInputBuffer has successfully processed
   * @retval		RTK_Fail 		    parameters error occurred or doesn't get buffer first
   *
   */
	virtual RTK_Error Demux_DiscardInputBuffer(RTK_Demux demux, RTK_TSInputBuffer * buffer);

 /**
   * @brief
   * This function add PES filter and parse the PES data through callback function.
   *
   * @param[in] demux			handle of demux to add PES filter
   * @param[in] pattern 		defines filter pid, filter id and callback function
   *
   * @retval		RTK_OK			AddPESFilter has successfully processed
   * @retval		RTK_Fail			parameters error occurred or add filter failed
   *
   */
	virtual RTK_Error Demux_AddPESFilter(RTK_Demux demux,  RTK_PESFilterPattern * pattern);
/**
   * @brief
   * removes the PESFilter, then it should not parse the specific PES data
   *
   * @param[in] demux			  handle of demux to deal with
   * @param[in] u8FilterID		  Filter ID only support 0x00 or 0x01 now
   *
   * @retval		RTK_OK			PES filter has been stopped parsing with specific filter_handle successfully
   * @retval		RTK_Fail			parameters error occurred or ts filter remove failed
   *
   */
	virtual RTK_Error Demux_RemovePESFilter(RTK_Demux demux,UINT8_T u8FilterID);

/**
   * @brief
   * This function will add subtitle to the player.
   *
   * @param[in] demux				 handle of demux to deal with
   * @param[in] ptrSubtitleTrackInfo  the subtitle information	
   *
   * @retval		RTK_OK					 add subtitle to the player ok
   * @retval		RTK_Fail					 add subtitle to the player failed
   *
   */
	virtual RTK_Error Demux_SetSubtitle(RTK_Demux demux, RTK_SubtitleConfig ptrSubtitleTrackInfo);



#endif

//RTKHAL 2.0 remove EMM demux.
#if 0
/**
   * @brief
   * This function will modifies pid of PCR. During playing, if audio and video are out of sync, 
   * Demux_ChangePCR is used to  modify PCR PID for the demux with specific handle, 
   * which is good for correcting audio/video 	out-sync problems.
   * This function is the same usage as Player_ChangePCR except handle difference.
   *
   * @param[in]	demux		 	handle of demux to deal with
   * @param[in]	pcr_pid                   pcr pid which the demux will change to
   *
   * @retval		RTK_OK			the PCR has changed successfully
   * @retval		RTK_Fail 			parameters error occurred or PCR change failed
   *
   */
	virtual RTK_Error Demux_ChangePCR(RTK_Demux demux, uint16_t pcr_pid);
/**
   * @brief
   * This function will change audio pid, audio codec type, then play back the audio.
   * This function is the same usage as Player_ChangeAudio except handle difference.
   *
   * @param[in]	demux		     handle of demux to change audio configuration
   * @param[in]	audio                   pointer to a prepared RTK_AudioConfig object
   *
   * @retval		RTK_OK		     change audio and play successfully
   * @retval		RTK_Fail 		     parameter input error such as player or audio equals NULL
   *
   */
	virtual RTK_Error Demux_ChangeAudio(RTK_Demux demux, RTK_AudioConfig * audio);
/**
   * @brief
   * This function will change video pid, video codec type, then play back the video.
   * This function is the same usage as Player_ChangeVideo except handle difference.
   *
   * @param[in]	demux		     handle of demux to change video configuration
   * @param[in]	audio                   pointer to a prepared RTK_AudioConfig object
   *
   * @retval		RTK_OK		     change audio and play successfully
   * @retval		RTK_Fail 		     parameter input error such as player or audio equals NULL
   *
   */
	virtual RTK_Error Demux_ChangeVideo(RTK_Demux demux, RTK_VideoConfig * video);

	//demux feed data functions
/**
   * @brief
   * This function will apply a buffer from share memory to reserve TS data. 
   * If successfully, these information be will returned in the parameter buffer, this parameter is clear to recognize buffer features.
   *
   * @param[in]	demux		     handle of demux to get buffer
   * @param[out]	buffer                  get the viraddr, size, filled_size, islast when return input buffer
   *
   * @retval		RTK_OK		    got the buffer information successfully
   * @retval		RTK_Fail 		    parameters error occurred or share buff has full
   *
   */
	virtual RTK_Error Demux_GetInputBuffer(RTK_Demux demux, RTK_TSInputBuffer * buffer);
/**
   * @brief
   * This function will feed some TS data used for playing or parsed in input_buffer.
   * This function is the same usage as Player_FeedInputBuffer except handle difference.
   *
   * @param[in]	demux		     handle of demux to feed buffer
   * @param[in]	buffer                  define input buffer viraddr, size, filled_size, islast
   *
   * @retval		RTK_OK		    FeedInputBuffer has successfully processed
   * @retval		RTK_Fail 		    parameters error occurred or file size not packet align
   *
   */
	virtual RTK_Error Demux_FeedInputBuffer(RTK_Demux demux, RTK_TSInputBuffer * buffer);
/**
   * @brief
   * This function will discard input data buffer.
   * This function is the same usage as Player_DiscardInputBuffer except handle difference.
   *
   * @param[in]	demux		    handle of demux to discard input buffer
   * @param[in]	buffer                  define input buffer viraddr, size, filled_size, islast
   *
   * @retval		RTK_OK		    DiscardInputBuffer has successfully processed
   * @retval		RTK_Fail 		    parameters error occurred or doesn't get buffer first
   *
   */
	virtual RTK_Error Demux_DiscardInputBuffer(RTK_Demux demux, RTK_TSInputBuffer * buffer);

/**
   * @brief
   * The function will create EmmDemux and do initial works. 
   * The main jobs of EmmDemux are to receive and parse the EMM data.
   *
   * @param[out]	emmdemux	     return the EmmDemux handle created by this function
   * @param[in]	config                  the configs used when creating the Player
   *
   * @retval		RTK_OK		    Creat has successfully processed
   * @retval		RTK_Fail 		    parameters error occurred or Create emmdemux return failed
   *
   */
	virtual RTK_Error EmmDemux_Create(RTK_EmmDemux *emmdemux, RTK_EmmDemuxConfig *config);
/**
   * @brief
   * This function destories the emmdemux handler and frees related memories.
   *
   * @param[in]	emmdemux		the emmdemux handler
   *
   * @retval		RTK_OK			the emmdemux is destoried  successfully.
   * @retval		RTK_Fail 			parameters error occurred or the emmdemux doesn't stop.
   *
   */
	virtual RTK_Error EmmDemux_Destroy(RTK_EmmDemux emmdemux);
/**
   * @brief
   * This function will apply a buffer from share memory to reserve TS data. 
   * If successfully, these information be will returned in the parameter buffer, this parameter is clear to recognize buffer features.
   *
   * @param[in]	emmdemux	    handle of emmdemux to get buffer
   * @param[out]	buffer                  get the viraddr, size, filled_size, islast when return input buffer
   *
   * @retval		RTK_OK		    got the buffer information successfully
   * @retval		RTK_Fail 		    parameters error occurred or share buff has full
   *
   */	
	virtual RTK_Error EmmDemux_GetInputBuffer(RTK_EmmDemux emmdemux, RTK_TSInputBuffer * buffer);
/** @brief
   * This function will feed some TS data used for playing or parsed in input_buffer.
   * This function is the same usage as Player_FeedInputBuffer except handle difference.
   *
   * @param[in]	emmdemux         handle of emmdemux to feed buffer
   * @param[in]	buffer                  define input buffer viraddr, size, filled_size, islast
   *
   * @retval		RTK_OK		    FeedInputBuffer has successfully processed
   * @retval		RTK_Fail 		    parameters error occurred or file size not packet align
   *
   */
	virtual RTK_Error EmmDemux_FeedInputBuffer(RTK_EmmDemux emmdemux, RTK_TSInputBuffer * buffer);
/**
   * @brief
   * This function will discard input data buffer.
   * This function is the same usage as Player_DiscardInputBuffer except handle difference.
   *
   * @param[in]	emmdemux	    handle of emmdemux to discard input buffer
   * @param[in]	buffer                  define input buffer viraddr, size, filled_size, islast
   *
   * @retval		RTK_OK		    DiscardInputBuffer has successfully processed
   * @retval		RTK_Fail 		    parameters error occurred or doesn't get buffer first
   *
   */
	virtual RTK_Error EmmDemux_DiscardInputBuffer(RTK_EmmDemux emmdemux, RTK_TSInputBuffer * buffer);
/**
   * @brief
   * This function opens and sets sectionFilter by specific pid, add the SectionFilter into SI engine with the demux handle.
   * This function is the same usage as Player_AddSectionFilter except handle difference.
   *
   * @param[in]      emmdemux		handle of emmdemux to add section filter
   * @param[out]      filter_handle            handle of sectionFilter to specify
   * @param[in]      pattern                    defines filter pid, mask, mode, compose type, monitor mode, 
   *                                                        user param and a function pointer to SI section filter callback.
   *
   * @retval		RTK_OK			AddSectionFilter has successfully processed
   * @retval		RTK_Fail 			parameters error occurred or filter open failed
   *
   */
	virtual RTK_Error EmmDemux_AddSectionFilter(RTK_EmmDemux emmdemux, RTK_SectionFilter * filter_handle, RTK_SectionFilterMask * pattern);
/**
   * @brief
   * This function will stop the sectionFilter, then it should not parse the specific sections.
   * This function asks for stopping filter the sections with specific filter_handle. 
   * When sectionFilter stops, it will not receive related sections.
   * This function is the same usage as Player_RemoveSectionFilter except handle difference.
   *
   *
   * @param[in]	emmdemux		 handle of emmdemux to add section filter
   * @param[in]       filter_handle            handle of sectionFilter to stop
   *
   * @retval		RTK_OK			RemoveSectionFilter has successfully processed.
   * @retval		RTK_Fail 			parameters error occurred or filter open failed.
   *
   */
	virtual RTK_Error EmmDemux_RemoveSectionFilter(RTK_EmmDemux emmdemux, RTK_SectionFilter filter_handle);
#endif

/**
   * @brief
   * This function will do the initial work to start detect the frequence.
   *
   *
   * @retval		RTK_OK			initial the tuner successfully.
   * @retval		RTK_Fail		initial the tuner failed.
   *
   */
	virtual RTK_Error TunerInit(void);

/**
   * @brief
   * This function will reset tuner/demod and it's related module(TP and IIC interface).
   *
   *
   * @retval		RTK_OK			initial the tuner successfully.
   * @retval		RTK_Fail 		initial the tuner failed.
   *
   */
	virtual RTK_Error TunerUnInit(void);

/**
   * @brief
   * This function will scan the frequence with the configuration parameter such as tuner id, tuner type and  detailed parameter related to the type.
   *
   * @param[in] 	ucTunerID 		tuner instance ID
   * @param[in] 	param			the objet to the the RTK_TunerParam_t struct
   *
   * @retval		RTK_OK			scan the frequence successfully with the given information
   * @retval		RTK_Fail		parameters error occurred or scan the frequence failed.
   *
   */
	virtual RTK_Error TunerScan(uint8_t ucTunerID, RTK_TunerParam_t param);

/**
   * @brief
   * This function will release the target tuner channel 
   *
   * @param[in] 	ucTunerID 		tuner instance ID
   *
   * @retval		RTK_OK
   * @retval		RTK_Fail
   *
   */
	virtual RTK_Error TunerRelease(uint8_t ucTunerID);
/**
   * @brief
   * This function will get the tuner lock status with specified tuner instance id.
   *
   * @param[in]		ucTunerID		tuner instance ID
   *
   * @retval		RTK_OK			the tuner is locked.
   * @retval		RTK_Fail		the tuner is unlocked.
   *
   */
	virtual RTK_Error TunerIsLock(uint8_t ucTunerID);

/**
   * @brief
   * This function will get the detail information of tuner with specified tuner instance ID.
   *
   * @param[in] 	ucTunerID 		tuner ID
   * @param[out]	info			the detailed return infomation object
   *
   * @retval		RTK_OK			get tuner information has successfully processed..
   * @retval		RTK_Fail		parameters error occurred or get tuner information failed.
   *
   */
	virtual RTK_Error TunerGetInfo(uint8_t ucTunerID, RTK_TunerInfo * info);

/**
   * @brief 
   * This function will get the tuner chip id.
   *
   * @param[in]    tunerid               tuner ID
   * @param[out]    *pDevicechipid        tuner chipid
   * @retval        RTK_OK            the tuner chipid is got.
   * @retval        RTK_Fail             the tuner chipid get fail.
   *
   */
    virtual RTK_Error Tuner_GetDeviceChipid(uint32_t uiTunerID,uint8_t * pDevicechipid);

/**
   * @brief 
   * This function can be used to control LNB.
   *
   * @param[in]	   tunerid         tuner ID
   * @param[in]    eLNBMode        target LNB mode
   * @retval       RTK_OK          success.
   * @retval       RTK_Fail        fail.
   *
   */
    virtual RTK_Error Tuner_SetLNBMode(uint32_t uiTunerID, enum RTK_LNB_MODE eLNBMode);

/**
   * @brief 
   * This function can be used to get current LNB status.
   *
   * @param[in]	   tunerid         tuner ID
   * @param[out]   ptLNBStatus     current LNB status
   * @retval       RTK_OK          success.
   * @retval       RTK_Fail        fail.
   *
   */
    virtual RTK_Error Tuner_GetLNBStatus(uint32_t uiTunerID, RTK_LNB_STATUS_t* ptLNBStatus);

/**
   * @brief 
   * This function can be used to send DiSEqC tone burst.
   *
   * @param[in]	   tunerid         tuner ID
   * @param[in]    eTone           tone burst type
   * @retval       RTK_OK          success.
   * @retval       RTK_Fail        fail.
   *
   */
    virtual RTK_Error Tuner_SendDiSEqCToneBurst(uint32_t uiTunerID, enum RTK_TONE_BURST eTone);

/**
   * @brief 
   * This function can be used to enable or disable 22KHz continues output.
   *
   * @param[in]	   tunerid         tuner ID
   * @param[in]    eEnable         enable or disable
   * @retval       RTK_OK          success.
   * @retval       RTK_Fail        fail.
   *
   */
    virtual RTK_Error Tuner_Set22KContinues(uint32_t uiTunerID, RTK_Bool eEnable);

/**
   * @brief 
   * This function can be used to send DiSEqC command.
   *
   * @param[in]	   tunerid         tuner ID
   * @param[in]    tCmdMsg         DiSEqC command message struct
   * @retval       RTK_OK          success.
   * @retval       RTK_Fail        fail.
   *
   */
    virtual RTK_Error Tuner_SendDiSEqCCmd(uint32_t uiTunerID, RTK_DISEQC_MSG_t tCmdMsg);

/**
   * @brief 
   * This function can be used to get DiSEqC reply message.
   *
   * @param[in]    tunerid         tuner ID
   * @param[in]    ptReplyMsg      DiSEqC replay message struct
   * @retval       RTK_OK          success.
   * @retval       RTK_Fail        fail.
   *
   */
    virtual RTK_Error Tuner_GetDiSEqCReply(uint32_t uiTunerID, RTK_DISEQC_MSG_t* ptReplyMsg);

/**
   * @brief
   * This function is to set tuner total channel num.
   *
   * @param[in]     ucTunerIndex         tuner index
   * @param[in]     ucTotalTunerNum    tuner total channelnum
   *
   * @retval        RTK_OK            set  successfully
   * @retval        RTK_Fail            set failed.
   *
   */
 virtual RTK_Error TunerSetChannelNum(uint8_t ucTunerIndex,uint8_t ucTotalTunerNum);

/**
   * @brief
   * This function will create a display window.
   *
   * @param[in]	win_cfg		a instance of RTK_WinConfig structures,
   *                                                mainly specify window either main or pip window,its start point x and y position,
   *                                                and its width and height
   *
   * @retval		RTK_OK	        create the display window successfully
   * @retval		RTK_Fail 		 create the display window failed or paramerer error occured
   *
   */
	virtual RTK_Error CreateSurface(RTK_WinConfig win_cfg);
/**
   * @brief
   * This function will destroy a display window that has been created.
   *
   * @param[in]	plane		 a instance of RTK_WinPlane enumeration type,must the same as win_cfg->plane created
   *
   * @retval		RTK_OK	        destroy the display window successfully
   * @retval		RTK_Fail 		 destroy the display window failed or paramerer error occured
   *
   */
	virtual RTK_Error DestorySurface(RTK_WinPlane plane);

/**
   * @brief
   * Start DVR record.
   *
   * @param[in]		player		 	handle of player
   *
   * @retval		RTK_OK			DVR successfully stared.
   * @retval		RTK_Fail 			DVR start failed.
   *
   */
	virtual RTK_Error Player_DVRStart(RTK_Player player);
/**
   * @brief
   * Stop DVR record.
   *
   * @param[in]		player		 	handle of player
   *
   * @retval		RTK_OK			DVR successfully stoped.
   * @retval		RTK_Fail 			DVR stop failed.
   *
   */
	virtual RTK_Error Player_DVRStop(RTK_Player player);

/**
   * @brief
   * This function will set spdif output mode.
   *
   * @param[in]	outputMode	     specified spdif output mode,such as off,raw and lpcm mode.
   * 
   * @retval		RTK_OK		    set the spdif output mode successfully.
   * @retval		RTK_Fail 		    set the spdif output mode failed
   *
   */
	virtual RTK_Error System_Set_Spdif_Output_Mode(Hal_Spdif_Mode_e outputMode);

/**
   * @brief
   * This function will get spdif output mode.
   *
   * @param[out]	outputMode	    return spdif output mode,such as off,raw and lpcm mode.
   * 
   * @retval		RTK_OK		    get the spdif output mode successfully.
   * @retval		RTK_Fail 		    get the spdif output mode failed
   *
   */
	virtual RTK_Error System_Get_Spdif_Output_Mode(Hal_Spdif_Mode_e *outputMode);

/**
   * @brief
   * This function will set hdmi output mode.
   *
   * @param[in]	e_mode	    specified hdmi output mode,such as off,raw and lpcm mode.
   * 
   * @retval		RTK_OK		    set the spdif output mode successfully.
   * @retval		RTK_Fail 		    set the spdif output mode failed
   *
   */
	virtual RTK_Error System_Set_Hdmi_Output_Mode(Hal_Hdmi_Mode_e outputMode);


/**
   * @brief
   * This function will get hdmi output mode.
   *
   * @param[out]	outputMode	    return hdmi output mode,such as raw, lpcm and auto mode.
   * 
   * @retval		RTK_OK		    get the hdmi output mode successfully.
   * @retval		RTK_Fail 		    get the hdmi output mode failed
   *
   */
	virtual RTK_Error System_Get_Hdmi_Output_Mode(Hal_Hdmi_Mode_e *outputMode);

/**
   * @brief
   * This function will set the audio output mode.
   *
   * @param[in]	ch_output_mode	    a enumeration type value to indicate audio output mode.
   *
   * @retval		RTK_OK		           Set Audio OutputMode successfully.
   * @retval		RTK_ERR 		    Set Audio OutputMode is not support.
   *
   */
	virtual RTK_Error System_SetAudioOutputMode(Hal_Audio_Output_Mode_e ch_output_mode);
 /**
   * @brief
   * This function add PES filter and parse the PES data through callback function.
   *
   * @param[in] player			handle of player to add PES filter
   * @param[in] pattern 		defines filter pid, filter id and callback function
   *
   * @retval		RTK_OK			AddPESFilter has successfully processed
   * @retval		RTK_Fail			parameters error occurred or add filter failed
   *
   */
	virtual RTK_Error Player_AddPESFilter(RTK_Player player,  RTK_PESFilterPattern * pattern);
/**
   * @brief
   * removes the PESFilter, then it should not parse the specific PES data
   *
   * @param[in] player			  handle of player to deal with
   * @param[in] u8FilterID		  Filter ID only support 0x00 or 0x01 now
   *
   * @retval		RTK_OK			PES filter has been stopped parsing with specific filter_handle successfully
   * @retval		RTK_Fail			parameters error occurred or ts filter remove failed
   *
   */
	virtual RTK_Error Player_RemovePESFilter(RTK_Player player,UINT8_T u8FilterID);

/**
   * @brief
   * This function will add subtitle to the player.
   *
   * @param[in] player				 handle of player to deal with
   * @param[in] ptrSubtitleTrackInfo  the subtitle information	
   *
   * @retval		RTK_OK					 add subtitle to the player ok
   * @retval		RTK_Fail					 add subtitle to the player failed
   *
   */
 	virtual RTK_Error Player_SetSubtitle(RTK_Player player, RTK_SubtitleConfig ptrSubtitleTrackInfo);


/**
   * @brief
   * This function will remove subtitle from the player.
   *
   * @param[in] player				 handle of player to deal with
   *
   * @retval		RTK_OK					 add subtitle to the player ok
   * @retval		RTK_Fail					 add subtitle to the player failed
   *
   */
 	virtual RTK_Error Player_RemoveSubtitle(RTK_Player player);

/**
	* @brief
	* This function will set ErrorConcealLevel of a specific player
	*
	* @param[in] 	player			handle of player 
	* @param[in] 	level			conceallLevel want to be set, Value from 0 to 4096
	* @param[in] 	delayTime		after delay time(in milliseconds)will apply error concelment level,
	*								0 means run immediately 
	*
	* @retval		RTK_OK			set the ErrorConcealLevel successfully.
	* @retval		RTK_Fail			rtk_porting_av_set_error_conceal_level return error or parameters
	*								error occurred.
	*
	*/
 	virtual RTK_Error Player_SetErrorConcealLevel(RTK_Player player, uint32_t level, int32_t delayTime);

/**
   * @brief
   * This function used to enable SLOW AV SYNC.
   *
   * @param[in]	player		    handle of player to deal with
   * @param[in]	enable       true or false
   *
   * @retval		RTK_OK		    enable successfully
   * @retval		RTK_Fail 		    parameters error occurred or Player_SetSlowAvSync return error
   *
   */
	//virtual RTK_Error Player_SetSlowAvSync(RTK_Player player, RTK_Bool enable);
	
		virtual RTK_Error Player_GetScrambleStatus(RTK_Player player, RTK_Bool* isScramble);

	/**
	* @brief
	* Set video show mode when video changed. And this mode will force inactive when RTK_Player_Stop called.
	*
	* @param[in]		player		handle of player to deal with
	* @param[in]		mode		video show mode.
	*
	* @retval		RTK_OK			success.
	* @retval		RTK_Fail		failed.
	*
	*/
	virtual RTK_Error Player_SetShowMode_OnVideoChanged(RTK_Player player, RTK_VideoShowMode_OnChanged_e mode);

	/**
	* @brief
	* This function provide a methord to config DVB resource.
	*
	* @param[in]	config	     DVB resource config
	* 
	* @retval		RTK_OK	     finalize success.            
	* @retval		RTK_Fail 	 finalize failed.   
	*
	*/
	virtual RTK_Error DvbResourceConfig(RTK_DvbResourceConfig_t *pConfig);

	/**
	* @brief
	* Set Secure Meida Path enable or disable.
	*
	* @param[in]	channel_id	     DVB channel id, it normally mean the decoder id.
	* @param[in]	enable	     	 when enable is RTK_TRUE, it mean SMP enabled.
	*
	* 
	* @retval		RTK_OK	     success.            
	* @retval		RTK_Fail 	 failed.   
	*
	*/
	virtual RTK_Error DvbSetSMP(uint32_t channel_id, RTK_Bool enable);

	/**
	* @brief
	* Get Secure Meida Path enable or disable.
	*
	* @param[in]	channel_id	  DVB channel id, it normally mean the decoder id.
	* 
	* @retval		RTK_TRUE	  SMP is enabled.            
	* @retval		RTK_FALSE 	 SMP is disabled.   
	*
	*/
	virtual RTK_Bool DvbGetSMP(uint32_t channel_id);


	/**
	** @brief
	** Audio behind or early video output
	**
	** @param[in]		delayInMs			delay ms (+3000 ~ -3000)
	**
	**
	*/
	virtual void SetHdmiOutVSDBDelay(int delayInMs);

	virtual void SetAudioHdmiDelay(char enable,int delayInMs);	

	/**
	** @brief
	** enable ccDecoder in order to create cc thread in unittest to get ccdata and deal cc data through player callback
	**
	** @param[in]		player		 	handle of player to deal with
	**
	** @retval		RTK_OK			enable transporting cc data  to player callback  successfully.
	** @retval		RTK_Fail 			parameters error such as player equals NULL
	**
	*/
	virtual RTK_Error Player_EnableCCDecoder(RTK_Player player);

	/**
	** @brief
	** disable ccDecoder in order to destroy cc thread , notice DvdPlayer to stop read and transport ccdata out.
	**
	** @param[in]		player		 	handle of player to deal with
	**
	** @retval		RTK_OK			disable  transporting cc data successfully.
	** @retval		RTK_Fail 			parameters error such as player equals NULL
	**
	*/
	virtual RTK_Error Player_DisableCCDecoder(RTK_Player player);
	/**
	** @brief
	** This function control ccdata whether to show with special channel number.
	** When bEnable is true, it will create playback cc thread, false will disable playback cc thread.
	** Playback cc thread will check the index and whether ccdata is changed to decide show or not.
	** If bEnable is true and index is correct, playback cc thread will be created and the cc can be shown. 
	** If bEnable is true and index is error, playback cc thread will be created and the cc cann't be shown. 
	** If bEnable is false,playback cc thread will be destroyed and the cc cann't be shown,whether index is correct.
	**
	** @param[in]		player		 	handle of player to deal with
	** @param[in]		bEnable		 	true to enable show,false to disable show
	** @param[in]		index		 	begin with 0, lower than ccStreamNum
	**
	** @retval		RTK_OK			to set cc show status successfully.
	** @retval		RTK_Fail 			parameters error such as player equals NULL
	**
	*/
	virtual RTK_Error Player_SetCCPlayback(RTK_Player player,bool bEnable,int index);
	/**
	** @brief
	** get cc data 
	**
	** @param[in]		player		 	handle of player to deal with
	**
	** @retval		RTK_OK			get cc data successfully.
	** @retval		RTK_Fail 			parameters error such as player equals NULL
	**
	*/
	virtual RTK_Error Player_GetCCData(RTK_Player player,unsigned char** pCCData,unsigned int* pLen);
	/**
	** @brief
	** This function to get cc information,which include stream number,ccType index and ccType String.
	** Here eight cc streams are supported,when you use this function,you can get how many cc stream in the video.
	** This functio is use to tell you whether there are ccdata according to info->ccStreamNum.
	** If info->ccStreamNum is 0,there is no cc data.
	** If info->ccStreamNum is larger than 0,you can use Player_SetCCPlayback to show it.
	** you can get ccType string from  info->ccTypeString[i],i is from 0 to ccStreamNum-1
	**
	** @param[in]		player		handle of player to deal with
	** @param[out]		info		 	pointer to struct RTK_ClosedCaption_Param with cc information
	**
	** @retval		RTK_OK			get cc information successfully.
	** @retval		RTK_Fail 			get cc information fail
	**
	*/
	virtual RTK_Error Player_GetCCInfo(RTK_Player player,RTK_ClosedCaption_Param_t* info);


	
	virtual RTK_Error Player_GetTPBufferStatus(RTK_Player player, uint32_t* pFreeSize, uint32_t *pDataSize);
	/**
	* @brief
	*  Feed DMA buffer into player.
	*
	* @param[in]		player		 	handle of player 
	* @param[in]		phyaddr 		DMA buffer physical address.
	* @param[in]		size 			feed size.
	* @param[in]		timeout 		wait timeout(millisecond)	
	* @param[in]		is_last 		imply EOS
	*
	* @retval		RTK_OK				successfully processed
	* @retval		RTK_Fail 			fail or timeout.
	*
	*/
	virtual RTK_Error Player_FeedDMABuffer(RTK_Player player, uint32_t phyaddr, uint32_t size, uint32_t timeout, bool is_last);

	/**
	* @brief
	*  add AV event callback
	*
	* @param[in]		player			handle of player 
	* @param[in]		av_event		av event
	* @param[in]		callback		av event callback
	*
	* @retval		RTK_OK				successfully processed
	* @retval		RTK_Fail			fail or timeout.
	*
	*/
	virtual RTK_Error Player_AddAvEventCallback(RTK_Player player, RTK_HAL_AV_EVENT av_event, RTK_HAL_AV_EVENT_FUNC callback);

	/**
	* @brief
	*  remove AV event callback
	*
	* @param[in]		player			handle of player 
	* @param[in]		av_event		av event
	* @param[in]		callback		av event callback
	*
	* @retval		RTK_OK				successfully processed
	* @retval		RTK_Fail			fail or timeout.
	*
	*/
	virtual RTK_Error Player_RemoveAvEventCallback(RTK_Player player, RTK_HAL_AV_EVENT av_event, RTK_HAL_AV_EVENT_FUNC callback);
	
    // 
    
    //extend functions

	/**
	* @brief
	* Set frequency for multi channel-tuner, this function only for FCC.
	*
	* @param[in]		tunerid		       tuner id
	* @param[in]		MultiCH_param      multich tuner paramters.
	*
	* @retval		RTK_OK			
	* @retval		RTK_Fail 		
	*
	*/
	virtual RTK_Error Tuner_Multich_SetFreq(int tunerid, RTKMultiCH_Param_t MultiCH_param);

    
	/**
	* @brief
	* Get tuner signal info in multi channel mode. For FCC V2
	*
	* @param[in]		tunerid			tuner ID
	* @param[in]		ucProgID		program ID
	* @param[out]		pInfo			result tuner info
	*
	* @retval		RTK_OK			
	* @retval		RTK_Fail		
	*
	*/
	virtual RTK_Error Tuner_Multich_GetSignalInfo(int tunerid, UINT8_T ucProgID, RTK_TunerInfo * pInfo);

    /**
	* @brief
	* Set PID filters for multich tp, this function only for FCC. 
	*
	* @param[in]		player		       player handle
	* @param[in]		MultiCHPID_param   the PID parameters to set into multich tp.
	* @param[in]		channel_count      indicate the count of multi channel PID parameter.
	*
	* @retval		RTK_OK			
	* @retval		RTK_Fail 		
	*
	*/
	virtual RTK_Error Player_SetMultiPIDFilters(RTK_Player player, RTKMultich_Pids_t* MultiCHPID_param,int channel_count);

	/**
	* @brief
	* Fast change video and audio, this function only for FCC. 
	*
	* @param[in]		player		player handle
	* @param[in]		pVideo   	a pointer of video config.
	* @param[in]		pAudio      a pointer of audio config.
	*
	* @retval		RTK_OK			
	* @retval		RTK_Fail 		
	*
	*/
	virtual RTK_Error Player_ChangeChannel(RTK_Player player, RTK_VideoConfig *pVideo, RTK_AudioConfig *pAudio);

	/**
	* @brief
	* enable or disable keepLastFrame for Player_ChangeChannel. 
	*
	* @param[in]		player		player handle
	* @param[in]		enable   	state of keepLastFrame.
	*
	* @retval		RTK_OK			
	* @retval		RTK_Fail 		
	*
	*/
    virtual RTK_Error Player_EnableKeepLastFrameForChangeChannel(RTK_Player player, RTK_Bool enable);

	/**
	** @brief
	** changes multi-channel's program(source)
	**
	** @param[in]		player			handle of player to deal with
	** @param[in]		program_id		program id
	**
	** @retval		RTK_OK				change program id successfully.
	** @retval		RTK_Fail			parameters error such as player or audio equals NULL
	**
	*/
	virtual RTK_Error Player_ChangeProgram(RTK_Player player, UINT8_T program_id);
	virtual RTK_Error Source_SetRecKey(RTK_Source source, const char* pRecKey,unsigned int length);
	virtual RTK_Error Player_SetSource(RTK_Player player, RTK_Source source);
	virtual RTK_Error Player_GetAudioTrackInfo(RTK_Player player, RTK_Audio_TrackInfo * ptrAudioTrackInfo);
	virtual RTK_Error Player_GetVideoTrackInfo(RTK_Player player, RTK_Video_TrackInfo* ptsVideoTrackInfo);
	
	virtual RTK_Error Demux_SetSource(RTK_Demux demux, RTK_Source source);
	virtual RTK_Error Source_Create(RTK_Source *source, RTK_SourceParam_t param);
	virtual RTK_Error Source_Destroy(RTK_Source source);
	virtual RTK_Error Source_SetRfDemod(RTK_Source source, RTK_RfDemodParam_t param);
	virtual RTK_Error Source_PresetPidFilter(RTK_Source source, uint16_t* pids, unsigned char numPID);
	virtual RTK_Error Source_GetIsLocked(RTK_Source source);
	virtual RTK_Error Source_GetSignalInfo(RTK_Source source, RTK_TunerInfo* info);
	virtual RTK_Error Source_GetInputBuffer(RTK_Source source, RTK_TSInputBuffer * buffer);
	virtual RTK_Error Source_FeedInputBuffer(RTK_Source source, RTK_TSInputBuffer * buffer);
	virtual RTK_Error Source_DiscardInputBuffer(RTK_Source source, RTK_TSInputBuffer * buffer);
	virtual RTK_Error Source_Flush(RTK_Source source);
	virtual RTK_Error Source_SetTsidSidx(RTK_Source source, unsigned int tsid_sidx);
        virtual RTK_Error Source_SetTsidSid_Ex(RTK_Source source, unsigned int tsid_sidx, unsigned char numPID, unsigned short* pPIDs);

	virtual RTK_Error Source_SetKeyUpdateMode(unsigned int tsid_sidx,unsigned int is_encrypt,Hal_Key_UpdateMode_e update_mode);
	virtual RTK_Error Source_AddSectionFilter(RTK_Source source, RTK_SectionFilter * filter_handle, RTK_SectionFilterMask * pattern);
	virtual RTK_Error Source_RemoveSectionFilter(RTK_Source source, RTK_SectionFilter filter_handle);
	virtual RTK_Error Source_RemoveAllSectionFilters(RTK_Source source);
	virtual RTK_Error Source_GetIsIdle(RTK_Source source, RTK_Bool* is_idle);
	virtual RTK_Error Player_GetDecodeStatus(RTK_Player player,uint32_t* pGoodDecodedFrmNum, uint32_t* pErrorDecodedFrmNum);
	virtual RTK_Bool System_VOisUsing(RTK_WinPlane vo_plane);
	virtual RTK_Error Player_SendDisplayInfoToHdmiSer(RTK_Player player,RTK_WinPlane vo_plane);
	virtual RTK_Error mCleanup_AV(int channel_id);

    ///////

public:
	RTK_Error m_halInitSuccess;
	
protected:
	
	RTK_Bool m_AvInit;
	pthread_mutex_t m_rtk_hal_mutex[RTK_MAX_DTVFLOW_NO];
	RTK_ShareFd	m_ionmem_sharefd[RTK_MAX_DTVFLOW_NO];
	//RTK_Bool m_ChannelInUse[RTK_MAX_DTVFLOW_NO]; 
	//HAL_TYPE m_ChannelHalType[RTK_MAX_DTVFLOW_NO];

	RTK_Bool m_BufInUse[RTK_MAX_DTVFLOW_NO];

	native_handle_t *m_SidebandHandle[RTK_MAX_DTVFLOW_NO]; 
	
	int m_dtvflow_num;
	RTK_DvbResourceConfig_t m_dvbResourceConfig;

//#ifdef CLIENT_DTVSOURCE
	pthread_mutex_t mtp_mutex;
	Rtk_MtpBuf_t mtp_buf[MTP_BUF_COUNT];
	//std::vector<int> m_channels={-1,-1};
	std::vector<int> m_channels;

	bool m_dtvSource_isInit; 

	RHAL_ChannelStatus_t m_channelStatus[RTK_MAX_DTVFLOW_NO];
	RHAL_SourceDmxHandleInfo_t m_sourceDmxHandle[RTK_MAX_DTVSOURCE_MUM];
	pthread_mutex_t m_sourceDmxHandle_mutex;

	virtual RTK_Error mMtpBufInit(void);
	virtual RTK_Error mMtpBufUninit(void);
	virtual ION_DATA* mMtpBufLend(Source_Handle source);
	virtual ION_DATA* mMtpBufPreLend(void);
	virtual RTK_Error mMtpBufSetSource(ION_DATA* p, Source_Handle source, Rtk_Porting_Source_Type_e type);
	virtual ION_DATA* mMtpBufFind(RTK_Source source, unsigned char* virtual_addr);
	virtual RTK_Error mMtpBufReturn(ION_DATA* p);
//#endif

/**
   * @brief
   * This function will initial AV tuner porting and other parameters.
   *
   * @param[in]       input                     an object with the RTK_InputConfig type
   *
   * @retval		RTK_OK			init tuner successfully.
   * @retval		RTK_Fail 			parameters error occurred or init tuner failed.
   *
   */
	virtual RTK_Error mOnceInit_AV(RTK_InputConfig input, int channel=-1, HAL_TYPE hal_type=HAL_TYPE_DEMUX);
	/**
   * @brief
   * This function will destroy the dtv flow and uninitial smartcard, av and display windows.
   *
   */
	virtual void mCleanup_AV(void);

	/**
   * @brief
   * This function will apply a buffer from share memory to reserve TS data. 
   * If successfully, these information be will returned in the parameter buffer, this parameter is clear to recognize buffer features.
   *
   * @param[in]	handle		     handle to get buffer
   * @param[out]	buffer                  get the viraddr, size, filled_size, islast when return input buffer
   * @param[in]	type		            the tv flow handle type,HAL_TYPE_PLAYER or HAL_TYPE_DEMUX or HAL_TYPE_EMMDEMUX
   *
   * @retval		RTK_OK		    got the buffer information successfully
   * @retval		RTK_Fail 		    parameters error occurred or share buff has full
   *
   */
	virtual RTK_Error mGetInputBuffer(HAL_HANDLE_S *handle, RTK_TSInputBuffer *buffer,  HAL_TYPE type);
/**
   * @brief
   * This function will feed some TS data used for playing or parsed in input_buffer.
   *
   * @param[in]	handle		     handle to feed buffer
   * @param[in]	buffer		     define input buffer viraddr, size, filled_size, islast
   * @param[in]	type                     the tv flow handle type,HAL_TYPE_PLAYER or HAL_TYPE_DEMUX or HAL_TYPE_EMMDEMUX
   *
   * @retval		RTK_OK		    FeedInputBuffer has successfully processed
   * @retval		RTK_Fail 		    parameters error occurred or file size not packet align
   *
   */
	virtual RTK_Error mFeedInputBuffer(HAL_HANDLE_S *handle, RTK_TSInputBuffer * buffer,  HAL_TYPE type);
/**
   * @brief
   * This function will discard input data buffer.
   *
   * @param[in]	handle		     handle to discard buffer
   * @param[in]	buffer		     define the data in the input buffer viraddr, size, filled_size, islast
   * @param[in]	type                     the tv flow handle type,HAL_TYPE_PLAYER or HAL_TYPE_DEMUX or HAL_TYPE_EMMDEMUX
   *
   * @retval		RTK_OK		    DiscardInputBuffer has successfully processed
   * @retval		RTK_Fail 		    parameters error occurred or discard buffer failed
   *
   */
	virtual RTK_Error mDiscardInputBuffer(HAL_HANDLE_S* handle, RTK_TSInputBuffer * buffer,  HAL_TYPE type);
/**
   * @brief
   * This function will prepare the section process callback function with parameter pointer.
   *
   * @param[in]	param		     an pointer that will be transfered to strucrt HAL_HANDLE_S type
   *
   * @retval		NULL		    parameters error occurred or process end 
   *
   */
	virtual void* mSection_Ready_Thread(void* param);

/**
   * @brief
   * This function will prepare the pes process callback function with parameter pointer.
   *
   * @param[in] param			 an pointer that will be transfered to strucrt HAL_HANDLE_S type
   *
   * @retval		NULL			parameters error occurred or process end 
   *
   */
	virtual void* mPes_Ready_Thread(void* param);
	/**
    * @brief
   * This function will prepare cc process callback function with parameter pointer.
   *
   * @param[in] param			 an pointer that will be transfered to strucrt HAL_HANDLE_S type
   *
   * @retval		NULL			parameters error occurred or process end 
   *
   */
	virtual void* mCC_Ready_Thread(void* param);
/**
   * @brief
   * This function will record the process callback thread with parameter pointer.
   *
   * @param[in]	param		     an pointer that will be transfered to strucrt HAL_HANDLE_S type
   *
   * @retval		NULL		    parameters error occurred or process end 
   *
   */
	virtual void* mThreadRecord(void* param);
/**
   * @brief
   * This function will create a HAL_HANDLE_S handle with  input parameter.
   *
   * @param[in]	input			an pointer that will be transfered to strucrt HAL_HANDLE_S type
   * @param[in]	rec				an parameter that will be transfered to strucrt RTK_RecConfig type
   * @param[out]	p_hal_handle		an pointer to a HAL_HANDLE_S to get after processed
   * @param[in]	type				the tv flow handle type,HAL_TYPE_PLAYER or HAL_TYPE_DEMUX or HAL_TYPE_EMMDEMUX
   *								It is reserved parameter,and it is not used now.
   *
   * @retval		RTK_OK			handle is  created successfully 
   * @retval		RTK_Fail			parameters error occurred or handle is not created successfully 
   *
   */
	RTK_Error mCreate(RTK_InputConfig input, RTK_RecConfig rec, HAL_HANDLE_S **p_hal_handle, HAL_TYPE type,int channel);
/**
   * @brief
   * This function will destory the HAL_HANDLE_S handle.
   *
   * @param[in]	handle		     an pointer that will be transfered to strucrt HAL_HANDLE_S type
   *
   * @retval		RTK_OK		    handle is  destoried successfully 
   * @retval		RTK_Fail		    parameters error occurred or destory handle failed 
   *
   */
	RTK_Error mDestroy(HAL_HANDLE_S* handle);
/**
   * @brief
   * This function will execute the play callback event.
   *
   * @param[in]	data		           the higher 16 bits indicates the tv flow channel,
   *                                                    the lower 16 bits indicates the callback event, and limit to enumeration type RTK_AV_CALLBACK_EVENT_e
   *
   */
	void mPlayback_Callback(INT32_T data);
/**
   * @brief
   * This function will execute the first audio frame callback event.
   *
   * @param[in]	data		           the higher 16 bits indicates the tv flow channel,
   *                                                    the lower 16 bits indicates the first audio frame callback event, and limit to enumeration type RTK_AV_CALLBACK_EVENT_e
   *
   */
	void mFirstAudioFrame_Callback(int data);
/**
   * @brief
   * This function will execute the play callback event.Note this function is static type.
   *
   * @param[in]	data		           the higher 16 bits indicates the tv flow channel,
   *                                                    the lower 16 bits indicates the callback event, and limit to enumeration type RTK_AV_CALLBACK_EVENT_e
   *
   */	
	static void mPlayback_Callback_Entry(INT32_T data);
/**
   * @brief
   * This function will execute the first audio frame callback event. Note this function is static type.
   *
   * @param[in]	data		           the higher 16 bits indicates the tv flow channel,
   *                                                    the lower 16 bits indicates the first audio frame callback event, and limit to enumeration type RTK_AV_CALLBACK_EVENT_e
   *
   */
	static void mFirstAudioFrame_Callback_Entry(int data);
/**
   * @brief
   * This function will prepare the section process callback function with parameter pointer.Note this function is static type.
   *
   * @param[in]	param		     an pointer that will be transfered to strucrt HAL_HANDLE_S type
   *
   * @retval		NULL		    parameters error occurred or process end 
   *
   */
	static void* mThreadSectionReadyEntry(void* param);
/**
   * @brief
   * This function will prepare the pes process callback function with parameter pointer.Note this function is static type.
   *
   * @param[in]	param		     an pointer that will be transfered to strucrt HAL_HANDLE_S type
   *
   * @retval		NULL		    parameters error occurred or process end 
   *
   */
	static void* mThreadPesReadyEntry(void* param);
/**
   * @brief
   * This function will record the process callback thread with parameter pointer. Note this function is static type.
   *
   * @param[in]	param		     an pointer that will be transfered to strucrt HAL_HANDLE_S type
   *
   * @retval		NULL		    parameters error occurred or process end 
   *
   */
	static void* mThreadRecordEntry(void* param);
	
    	RTK_Error mAllocate_Channel_Index(HAL_TYPE channel_type, int &channel_index);
	bool mForceUseChannelIndex(HAL_TYPE channel_type, int channelId);

	int mFindSourceDmxIndex_bySource(Source_Handle source_handle);
	int mGetSourceDmxIndex(void);
	void mPutSourceDmxIndex(int index);
	/**
   * @brief
   * This function will prepare the cc process callback function with parameter pointer.Note this function is static type.
   *
   * @param[in]	param		     an pointer that will be transfered to strucrt HAL_HANDLE_S type
   *
   * @retval		NULL		    parameters error occurred or process end 
   *
   */
	static void* mThreadCCEnableEntry(void* param);
	//save cc data
	bool addCCData(HAL_HANDLE_S *handle,unsigned char *pUnit,unsigned int len);
	//fetch out cc data
	bool readCCData(HAL_HANDLE_S *handle,unsigned char **pUnit,unsigned int* pLen);     	
	RTK_Error mCreateSourceSectionFilter(RTK_Source source);
	RTK_Error mDestroySourceSectionFilter(RTK_Source source);
        	
};

/**
   *@defgroup RHAL_Transfer
   *@ingroup RTK_HAL
   *@brief this group mainly refer to transfer av codec or modulation
   *@{
   */
/**
   * @brief
   * This function will transfor the outer codec type to record codec name.
   *
   * @param[in]  codec                        A/V codec type, reference to the enumerating type RTK_Codec for detail
   *
   * @retval	   RTK_REC_CODEC_e  return the record codec type, including RTK_REC_CODEC_MPEG2,
   *                                                      RTK_REC_CODEC_H264,	RTK_REC_CODEC_AUDIO, RTK_REC_CODEC_H265
   */
RTK_REC_CODEC_e RHAL_Transfer_AV_Codec(RTK_Codec codec);
/**
   * @brief
   * This function will transfor the outer codec type to video type.
   *
   * @param[in]  codec                        A/V codec type, reference to the enumerating type RTK_Codec for detail
   *
   * @retval	   RTK_VIDEO_CODEC_TYPE_e   return the video codec type, only four type can be processed to specified type, respectively, RTK_REC_CODEC_MPEG1,
   *                                                                    RTK_REC_CODEC_MPEG2, RTK_CODEC_VIDEO_MPEG4_PART2,RTK_CODEC_VIDEO_H264,
   *                                                                    RTK_REC_CODEC_H264,	RTK_REC_CODEC_AUDIO, RTK_REC_CODEC_H265,RTK_CODEC_VIDEO_H265.
   *                                                                    other type are transfered to RTK_VIDEO_CODEC_TYPE_UNKNOWN for default.
   */
RTK_VIDEO_CODEC_TYPE_e RHAL_Transfer_Video_Codec(RTK_Codec codec);
/**
   * @brief
   * This function will transfor the outer codec type to Audio type.
   *
   * @param[in]  codec                             A/V codec type, reference to the enumerating type RTK_Codec for detail
   *
   * @retval RTK_AUDIO_CODEC_TYPE_e   return the audio codec type to specified type.
   *
   */

RTK_AUDIO_CODEC_TYPE_e RHAL_Transfer_Audio_Codec(RTK_Codec codec);
/**
   * @brief
   * This function will transfor the demod QAM type to cable tnr QAM type.
   *
   * @param[in]  type                                demod QAM type, reference to the enumerating type RTK_ModulationType for detail
   *
   * @retval  Rtk_CableTnr_Modulation_e   return the specified cable tnr QAM type.
   *
   */
Rtk_CableTnr_Modulation_e RHAL_Transfer_Modulation(RTK_ModulationType type);
/**
   *RHAL_Transfer
   *@}
   */



#endif 
