/**
    *@copyright 
    * Copyright (c) Realtek, Inc. http://www.realtek.com. All rights reserved.
    * Realtek PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
    * 
    * @file  rtk_hal_base.h
    * @brief  Realtek HAL base interface.\n
    *
    * This file is defined for the customer. Customers can use those data structures and function APIs to impliment\n
    * themselves middleware.
    *
    */

#ifndef _RTK_HAL_BASE_H_
#define _RTK_HAL_BASE_H_


#include "rtk_socket_cfg.h"
#include "rtk_psi.h"  //for psi parse.


#ifdef __cplusplus
extern "C" {
#endif
///<add by changxi_huang
typedef struct DataUnit{
	unsigned char* pData;
	unsigned int length;
	struct DataUnit *next;
}DataUnit_t;
typedef struct Hal_CCData_Info{
	DataUnit_t* pHead;
	DataUnit_t* pTail;
	DataUnit_t* pRead;
}Hal_CCData_t;

typedef enum 
{
	HAL_UPDATE_DIRECTLY, //start descramble when got key 
	HAL_UPDATE_WAIT_FOR_DUAL_KEY, //wait for dual key before start descramble
	HAL_UPDATE_WAIT_FOR_DUAL_KEY_IF_EXIST
} Hal_Key_UpdateMode_e;

///<add end
/**
   * @brief
   *	RTK HAL BASE APIS version.
   *    History:
   *    v1.0: 
   *        --The first release. 
   *    v2.0.0: 
   *        -- Remove TS demux and EMM demux APIs.
   *        -- Add DVR APIs.
   *        -- Add tuner APIs.
   *    v2.1.0: 
   *        -- Add TS demux APIs.
   */
#define RTK_HAL_BASE_APIS_VERSION	"v2.1.0"

#define CLIENT_DTVSOURCE // test for ENABLE_DTVSOURCE
#define UNKNOWN_TSID_SIDX       	(0xFFFFFFFF)

#define RTK_DISEQC_MSG_MAX				(6)

/**
   * @brief
   *	This enumeration type list av type,either audio or video 
   */
typedef enum 
{
	HAL_AV_INVALID_TYPE=0,
	HAL_AV_VIDEO=1,
	HAL_AV_AUDIO=2,
}Hal_AV_Type_e;
/**
   * @brief
   *	This struct type definies the demux audio and video packet and pes data information. 
   */
typedef struct Hal_Demux_Info{
	int64_t videoPacketCount;			/*!< all the video ts packets count*/
	int64_t videoPacketLost;			/*!< all the lost video ts packets count*/
	int64_t videoPacketLostCount;		/*!< not used*/
	int64_t videoPESCount;			/*!< all the video pes packets count*/
	int64_t videoPESError;			/*!< error count of video pes packets*/
	int64_t videoLength;				/*!< video data length*/
	int64_t audioPacketCount;			/*!< all the audio ts packets count*/
	int64_t audioPacketLost;			/*!< all the lost audio ts packets count*/
	int64_t audioPacketLostCount;		/*!< not used*/
	int64_t audioPESCount;			/*!< all the audio pes packets count*/
	int64_t audioPESError;			/*!< error count of audio pes packets*/
	int64_t audioLength;				/*!< audio data length*/
}Hal_Demux_Info_t;

/**
   * @brief
   *	This enumeration type list all SPDIF mode 
   */
typedef enum Hal_Spdif_Mode
{
	HAL_SPDIF_OFF =0x00,
	HAL_SPDIF_RAW=0x01,
	HAL_SPDIF_LPCM=0x02,
}Hal_Spdif_Mode_e;

/**
   * @brief
   *	This enumeration type list all hdmi mode 
   */
typedef enum Hal_Hdmi_Mode
{
	HAL_HDMI_RAW=0x00,
	HAL_HDMI_LPCM=0x01,
	HAL_HDMI_LPCM_MULTI_CH=0x02,
	HAL_HDMI_AUTO=0x03,
}Hal_Hdmi_Mode_e;


/**
   * @brief
   *	This enumeration type lists all audio output mode. 
   */
typedef enum Hal_Audio_Output_Mode
{
	HAL_AUDIO_STEREO =0, 
	HAL_AUDIO_MAIN_BILINGUAL=1, 			/*!<pcm (copy L->R)*/
	HAL_AUDIO_SUB_BILINGUAL=2, 			/*!< pcm (copy R->L)*/
	HAL_AUDIO_FLASH_STEREO=3,
	HAL_AUDIO_FLASH_MAIN_BILINGUAL=4,    /*!<(copy L->R)*/
	HAL_AUDIO_FLASH_SUB_BILINGUAL=5, 	/*!< (copy R->L)*/
	
}Hal_Audio_Output_Mode_e;

/**
   * @brief
   *	This enumeration type lists hal_handler type.
   *    player and demux are used to deal with ts flow, player is totally processed through software code. 
   *    demux deals with the ts flow with the help of hardware.
   *   emmdemux deals with the ES flow.
   */
typedef enum
{
	HAL_TYPE_NONE = 0,				/*!< init value*/
	HAL_TYPE_PLAYER   = 1,			/*!< player type, not use HW PID filter, can not play from tuner*/
	HAL_TYPE_DEMUX    = 2,			/*!< demux type, use HW PID filter*/
	HAL_TYPE_EMMDEMUX = 3,		/*!< not implement*/
	HAL_TYPE_DEMUX_ONLY =4,		/*!< demux only type, use HW PID filter, can not decode anything, only for recoding*/
	HAL_TYPE_ES_PLAYER	= 5,		/*!< es player type, only support decode the raw element data.*/
} HAL_TYPE;
/**
   * @brief
   *	This enumeration type lists input detailed data source.
   */
typedef enum
{
	RTK_INPUT_TUNER = 0,      	/*!< data input from tuner, ARCH V2 and above not used*/
	RTK_INPUT_TUNER_2 = 5,  	/*!< data input from tuner 2, only valid in the dual tuner platform, ARCH V2 and above not used*/
	RTK_INPUT_MEMORY, 		/*!< data input from memory, ARCH V2 and above not used*/
	RTK_INPUT_MEMORY_PIP, 	/*!< data input from memory, but this data for PIP, ARCH V2 and above not used*/
	RTK_INPUT_MEMORY_MD, 		/*!< data input from memory, support secure memory*/
	RTK_INPUT_SOURCE,    		/*!< DTVSource*/
}
RTK_InputType;

typedef enum
{
	RTK_STOP_CHANNEL,
	RTK_STOP_ALL
}
RTK_StopType;

/**
   * @brief
   * Video window type indicates video output types,
   * MAIN or PIP on the window to be managed.
   */
typedef enum
{
    RTK_WIN_MAIN,
    RTK_WIN_PIP,
} RTK_WinPlane;



typedef enum
{
	RTK_OK = 0,  /*!< everything is OK.*/  
	RTK_FILTER_ERROR,  /*!< filter error*/   
	RTK_BUFFER_NOT_AVAILABLE, /*!< buffer is not avaliable*/   
	RTK_Fail,  /*!< failed. */ 
	RTK_WAIT_EOS /*!< wait EOS.*/
}
RTK_Error;

/**
   *@brief
   * RTK bool typedef.
   */
typedef enum
{
	RTK_FALSE, /*!< false*/   
	RTK_TRUE, /*!< true*/   
}
RTK_Bool;

 /**
    *@brief 
    * this struct defines Video window config, 
    * Include video plane, window position,window size.  
    */
typedef struct
{
	RTK_WinPlane	 plane;		/*!< specified plnae to create surface*/  	
	int32_t pos_x;			/*!< win size : x pos*/
	int32_t pos_y;			/*!< win size : y pos*/
	int32_t width;			/*!< win size :  width*/
	int32_t height;			/*!< win size :  height*/
}RTK_WinConfig;
 /**
    * @brief
    *  This enum value define aspect ratio
    */
 typedef enum 
 {
	 RTK_RATIO_4_3=0,	 /// width:height=4:3    	 
	 RTK_RATIO_16_9=1,	 /// width:height=16:9(fixed)   
	 RTK_RATIO_AUTO=2,	 /// width:height=16:9(auto)    
 }RTK_Aspect_Ratio;

/**
    * @brief
    * list Section filter monitoring mode.
    */
typedef enum
{
	RTK_SECTION_MODE_ONESHOT,	///	specified section callback should be called 1 time only.
	RTK_SECTION_MODE_CONTINOUS	///	specified section callback should be called whenever section was received.
}
RTK_SectionMonitoringMode;
/**
    * @brief
    * list Section filter monitoring mode.
    */
typedef enum
{
	//	SDK should not do CRC check operation, when specific section received.
	//	This option is used for sections which does not have CRC. for example TDT/TOT.
	RTK_SECTION_OPTION_NO_CHECK_CRC = (1 << 0),
}
RTK_SectionOptions;

/**
    * @brief 
    * Section filter pattern size.
    */
#define RTK_MAX_SECTION_MASK_LENGTH		(16)	    /// [REALTEK] TODO: Please update value which is supported by Realtek SDK.

/**
    *@brief 
    * This enum value defines call callback event
    */
typedef enum
{
	//	player events.
	RTK_FIRST_IFRAME_DISPLAYED,		///	when first i-frame displayed
	RTK_FRAME_DISPLAYED,			///	whenever a frame displayed
	RTK_FIRST_AUDIO_FRAME_DECODED, 	//	when first audio-frame decoded by audeo codec
	RTK_EOS,						///	end of stream
	RTK_VIDEO_END,
	RTK_AUDIO_END,
	//	TS filter.
	RTK_FILTER_TS_DATA,		///	data -> TS data.
	RTK_FILTER_TS_SECURE_DATA,  //TS data in secure buffer.
	RTK_FILTER_TS_INDEX,	///	data -> RTK_TS_Index
	RTK_FILTER_TS_CTRL,		///	data -> RTK_TS_Ctrl
	RTK_FILTER_TS_STOREINFO,
	RTK_FILTER_TS_PCR,      /// for connect PVR
	RTK_FILTER_TS_TAG,      /// for connect PVR

	RTK_PICTURE_USER_DATA,     /// data->cc
	
	RTK_VIDEO_IFRAME_ERRORBLOCKS,   /// video i-frame error block   
	RTK_PLAYBACK_FATAL_ERROR,		/// video decode fatal error  
	RTK_CLEANUP,
	RTK_VORUN_AFTER_AVSYNC,
	RTK_AVPLAY_VIDEO_LOST,

	//playback error events
	RTK_VIDEO_FRAMING_ERR=20, //framing error 
	RTK_AV_RINGBUF_FULL_ERR,   //decorder ring buffer full

	//max event id.
	RTK_CB_EVENT_MAX=65536,
}
RTK_CallbackEvent;
/**
   *@brief 
   * This enum value defines descramble mode
   */
typedef enum 
{    
    HAL_NO_DESCRAMBLE,        
    HAL_TSP_LEVEL_DESCRAMBLE,
    HAL_PES_LEVEL_DESCRAMBLE,
    HAL_TSD_DESCRAMBLE_MODE_MAX,
}RTK_HAL_DESCRAMBLE_MODE;

/**
   *@brief 
   * This enum value defines descramble algorithm
   */
typedef enum 
{
    // DES    
    HAL_DES_ECB_KEEP_CLEAR,    
    HAL_DES_ECB,    
    HAL_DES_CBC_KEEP_CLEAR,    
    HAL_DES_CBC,
    HAL_DES_MARS_CBC,          
    
    HAL_TDES_ECB_KEEP_CLEAR,
    HAL_TDES_ECB,
    HAL_TDES_CBC_KEEP_CLEAR,
    HAL_TDES_CBC,
    HAL_TDES_MARS_CBC,
    
    // AES
    HAL_AES_ECB_KEEP_CLEAR,
    HAL_AES_ECB,
    HAL_AES_CBC_KEEP_CLEAR,
    HAL_AES_CBC,
    HAL_AES_CBC_CTS,
    
    HAL_HDCP2_AES_CTR,
    HAL_TSD_DESCRAMBLE_ALGO_MAX,
}RTK_HAL_DESCRAMBLE_ALGO;
/**
   *@brief 
   * This enum value defines section callback status, either timeout or is called
   */
typedef enum 
{
	HAL_SECTION_FILTER_TIMEOUT,           ///0x00  
	HAL_SECTION_FILTER_HIT      		  ///0x01
}RTK_HAL_SECTION_CB_STATUS;


///define void*  with RTK_Player
typedef void * RTK_Player;
///define void*  with RTK_Demux
typedef void * RTK_Demux;
///define void*  with RTK_SectionFilter
typedef void * RTK_SectionFilter;
///define void*  with RTK_TSFilter
typedef void * RTK_TSFilter;
///define Function pointer for tv flow event callback with RTK_Callback
typedef void (*RTK_Callback)(RTK_CallbackEvent event, HAL_TYPE type, void * user_param, void * data, uint32_t data_size);
///define Function pointer for section filter callback with RTK_Callback
typedef void (*RTK_SectionFilterCallback)(const RTK_SectionFilter handle, const unsigned char *section, const unsigned int length, void * userParam);
///define another Function pointer for section filter callback with RTK_Callback,which can return callback status
 typedef void (*RTK_SectionFilterCallback_01)(const RTK_SectionFilter handle, RTK_HAL_SECTION_CB_STATUS enStatus,const unsigned char *section, const unsigned int length, void * userParam);
 ///define Function pointer for PES filter callback with RTK_Callback
 typedef void (*RTK_PESFilterCallback)(uint32_t filterID , void * pes, uint32_t length, void * userParam);

typedef void * RTK_Source;


/**
   *@brief 
   * This enum value defines demod QAM mode type
   */
 typedef enum
{
	RTK_DEMOD_UNKNOWN = -1,
	RTK_DEMOD_16QAM,
	RTK_DEMOD_32QAM,
	RTK_DEMOD_64QAM,
	RTK_DEMOD_128QAM,
	RTK_DEMOD_256QAM,
}
RTK_ModulationType;

/**
   * @brief
   *  This enum value define aspect ratio in video spec.
   */
typedef enum
{
	RTK_VSPEC_ASPECT_RATIO_Forbidden,		// forbidden
	RTK_VSPEC_ASPECT_RATIO_SquareSample,	// 1.0 (square sample)
	RTK_VSPEC_ASPECT_RATIO_4_3,				// 4 : 3
	RTK_VSPEC_ASPECT_RATIO_16_9,			// 16 : 9
	RTK_VSPEC_ASPECT_RATIO_221_1,			// 2.21 : 1
	RTK_VSPEC_ASPECT_RATIO_Reserved,
} RTK_VIDEO_SPEC_ASPECT_RATIO_e;

/**
** @brief
**	This enum define av event
*/
typedef enum
{
	RTK_HAL_VIDEO_STOPPED_EVT = 0,    /*!< video stop */
	RTK_HAL_AV_EVENT_COUNT
}RTK_HAL_AV_EVENT;

/**
** @brief
**	This structure define av event message from server
*/
typedef struct
{
	unsigned int     av_handle;
	RTK_HAL_AV_EVENT av_event;
	unsigned int     data;
}RTK_HAL_AV_EVENT_MSG;

/**
** @brief 
** 	define av event callback function
*/
typedef void (*RTK_HAL_AV_EVENT_FUNC)(RTK_HAL_AV_EVENT_MSG msg);

/**
   *@brief 
   * this struct defines Video track information in ts flow, 
   * Include video pid,type, size and so on.  
   */
typedef struct
{
	int	pid;
	long type;
	long hor_size;
	long ver_size;
	long frame_rate;
	long bit_rate;
	long isProg;
	long isMVC;
	long isHDR;
	long isCompressBufferMode;
	long offsetTable_ysize;
	long offsetTable_csize;
	RTK_VIDEO_SPEC_ASPECT_RATIO_e aspect_ratio;
} RTK_Video_TrackInfo;

/**
   *@brief 
   * This struct defines audio track information, 
   * Include video pid,type, samplate rate and so on.  
   */
typedef struct
{
	long idx;
	long type;
	long channelCount;
	long bps; 
	long sampleRate;
	long pid;
	long langCode;
} RTK_Audio_TrackInfo;

/**
   *@brief 
   * This struct defines  pts info, 
   * Include video pts and audio pts.  
   */
typedef struct
{
	uint64_t presentVideoPts;
	uint64_t presentAudioPts;
	uint64_t demuxSubtitlePts;
} RTK_PTSInfo;

/**
   *@brief 
   * This enum value defines tuner type
   */
typedef enum RTK_Tuner_Type
{
	RTK_TUNER_TYPE_DVBC    = 1,		   /*!< DVBC function tuner*/   
	RTK_TUNER_TYPE_ABS     = 2,		   /*!< ABS function tuner*/   
	RTK_TUNER_TYPE_DTMB    = 3,	       /*!< DTMB function tuner*/   
	RTK_TUNER_TYPE_DVB_T   = 4,        /*!< DVBT function tuner*/   
	RTK_TUNER_TYPE_DVB_S   = 5,
	RTK_TUNER_TYPE_DVB_S2  = 6,
	RTK_TUNER_TYPE_DVB_T2  = 7,
} RTK_Tuner_Type_e;

/**
** @brief
**  Type of CAS
*/
typedef enum RTK_CAS_Type
{
    RTK_CAS_TYPE_NONE           = 0, //RT
    RTK_CAS_TYPE_VMX            = 1, //RT
    RTK_CAS_TYPE_WVCAS          = 2, //RT
    RTK_CAS_TYPE_NAGRA_TKL      = 3, //RT
    RTK_CAS_TYPE_NAGRA_CONAX    = 4, //RT
    RTK_CAS_TYPE_NAGRA_NOCS     = 5, //NV
    RTK_CAS_TYPE_NDS            = 6, //RT
    RTK_CAS_TYPE_CTI            = 7, //RT
    RTK_CAS_TYPE_CHINADIGITAL   = 8, //RT
    RTK_CAS_TYPE_PRIVATE_REC    = 9, //RT
    RTK_CAS_TYPE_3RDPARTY_CA           = 10, //RT
    RTK_CAS_TYPE_MAXIMUM        = 11,
} RTK_CAS_Type_e;

/**
** @brief
** CAS parameters.
*/
typedef struct RTK_CAS_Param
{
    RTK_CAS_Type_e              type;      /*!<CAS type*/
    unsigned long               tsid_sidx; /*!<NAGRA tsid or VMX service index or ... */
} RTK_CAS_Param_t;

typedef enum RTK_Source_Type
{
    RTK_SOURCE_TYPE_UNKNOWN = 0,
    RTK_SOURCE_TYPE_MTP     = 1,
    RTK_SOURCE_TYPE_MCP     = 2,
    RTK_SOURCE_TYPE_MD      = 3,
    RTK_SOURCE_TYPE_DVBC    = 4,
    RTK_SOURCE_TYPE_ABS     = 5,
    RTK_SOURCE_TYPE_DTMB    = 6,
    RTK_SOURCE_TYPE_DVB_T   = 7,
    RTK_SOURCE_TYPE_DVB_S   = 8,
    RTK_SOURCE_TYPE_DVB_S2  = 9,
    RTK_SOURCE_TYPE_DVB_T2  =10,
    RTK_SOURCE_TYPE_TP      =11,
    RTK_SOURCE_TYPE_MAXIMUM =12,
} RTK_Source_Type_e;

typedef enum RTK_Tuner_Bandwidth
{
    RTK_TUNER_BANDWIDTH_6M    = 1,
    RTK_TUNER_BANDWIDTH_7M     = 2,
    RTK_TUNER_BANDWIDTH_8M    = 3,
	 RTK_TUNER_BANDWIDTH_5M    = 4,
} RTK_Tuner_Bandwidth_e;

typedef enum RTK_DVBC_ANNEX_TYPE
{
    RTK_DVBC_ANNEX_B    = 1,
    RTK_DVBC_ANNEX_A    = 2,
} RTK_DVBC_ANNEX_TYPE_e;


/**
   *@brief 
   * This enum value defines DVBC standard system mode 
   */
typedef enum RTK_CableTnr_Mod
{
	RTK_CABLETNR_16QAM       = 0x01,		///16 -QAM   
	RTK_CABLETNR_32QAM       = 0x02,		///32 -QAM   
	RTK_CABLETNR_64QAM       = 0x03,		///64 -QAM   
	RTK_CABLETNR_128QAM      = 0x04,		///128-QAM   
	RTK_CABLETNR_256QAM      = 0x05,		///256-QAM 
} RTK_CableTnr_Mod_e;

/**
   *@brief 
   * This enum value defines DVBS standard system mode 
   */
typedef enum RTK_LNB_MODE
{    
    RTK_LNBMODE_OFF  ,
    RTK_LNBMODE_13V  ,
    RTK_LNBMODE_18V  ,
    RTK_LNBMODE_13V_WITH_22K_BURST ,
    RTK_LNBMODE_18V_WITH_22K_BURST ,
    RTK_LNBMODE_MAX,
}RTK_LNB_MODE_e;

/**
   *@brief 
   * This enum value defines DVBS standard system mode 
   */
typedef enum RTK_TONE_BURST
{    
    RTK_TONE_BURST_A,
    RTK_TONE_BURST_B,
}RTK_TONE_BURST_e;

typedef struct RTK_LNB_STATUS
{    
    unsigned short   output_disable    : 1;
    unsigned short   over_current      : 1;
    unsigned short   power_not_good    : 1;
    unsigned short   thermal_shut_down : 1;
    unsigned short   vin_under_voltage : 1;
    unsigned short   cable_disconnected: 1;
    unsigned short   tone_detected     : 1;
}RTK_LNB_STATUS_t;

typedef struct RTK_DISEQC_MSG
{    
    unsigned char ucMsgBuf[RTK_DISEQC_MSG_MAX];
	unsigned char ucMsgLen;
}RTK_DISEQC_MSG_t;

/**
   *@brief 
   * This struct defines DVBC standard system parameter, 
   * Including frequence,rate and mode type.  
   */
typedef struct RTK_Tnr_DVBC_Param
{
	unsigned int                 u32frequency;		///frequency(MHz)
	unsigned int                 u32symbol_rate;	///symbol rate(sym/s)
	RTK_CableTnr_Mod_e           modulation;		///Cable signal modulation 
	RTK_DVBC_ANNEX_TYPE_e        annex_type;        ///annex type
} RTK_Tnr_DVBC_Param_t;

/**
   *@brief 
   * This struct defines DTMB standard system parameter, 
   * Including frequence.  
   */
typedef struct RTK_Tnr_DTMB_Param
{
	unsigned int                        u32frequency;	///frequency(MHz)   
} RTK_Tnr_DTMB_Param_t;

/**
   *@brief 
   * This struct defines ABS standard system parameter, 
   * Including frequence and rate.  
   */
typedef struct RTK_Tnr_ABS_Param
{
	unsigned int                             u32frequency;		///<frequency point(MHz)
	unsigned int                             u32symbol_rate;	///<symbol rate(sym/s)
	//  Rtk_Polarization_e                  polarization;		    ///<polarization
} RTK_Tnr_ABS_Param_t;

/**
   *@brief 
   * This struct defines DVBT standard system parameter, 
   * Include frequence.  
   */
typedef struct RTK_Tnr_DVBT_Param
{
	unsigned int  				u32frequency;		///frequency(MHz)
}RTK_Tnr_DVBT_Param_t;

/**
   *@brief 
   * This struct defines DVBS standard system parameter, 
   * Including frequence, rate and signal mode type.
   */
typedef struct RTK_Tnr_DVBS_Param
{
	unsigned int                 u32frequency;		///frequency(MHz)
	unsigned int                 u32symbol_rate;	///symbol rate(sym/s)
	RTK_LNB_MODE_e           lnbmode;		///Cable signal modulation 
} RTK_Tnr_DVBS_Param_t;

/**
** @brief
** paramters of memory (MTP, MCP, MD)
*/
typedef struct RTK_MMT_Param
{
       uint8_t    flag;
}RTK_Memory_Param_t;

/**
   *@brief 
   * It defines Tuner Parameter structures,
   * Include standard type, and respective parametric structures.
   */
typedef struct
{
	RTK_Tuner_Type_e            type;				/*!< tuner type*/  
	RTK_Tuner_Bandwidth_e    bandwidth;			/*!< bandwidth*/
	union
	{
		RTK_Tnr_DVBC_Param_t   dvb_c_param;    /*!< European-standard cable tuner parameters*/ 
		RTK_Tnr_DVBT_Param_t   dvb_t_param;
		RTK_Tnr_DTMB_Param_t  dtmb_t_param;	  /*!< terrestrial tuner parameters*/
		RTK_Tnr_ABS_Param_t   abs_s_param;	  /*!< satellite tuner parameters*/  
		RTK_Tnr_DVBS_Param_t   dvb_s_param;
	} tnr_param;
} RTK_TunerParam_t;

typedef struct
{
	RTK_Source_Type_e          type;
	RTK_Tuner_Bandwidth_e      bandwidth;
	union
	{
		RTK_Tnr_DVBC_Param_t   dvb_c_param;	// European-standard cable tuner parameters
		RTK_Tnr_DVBT_Param_t   dvb_t_param;
		RTK_Tnr_DTMB_Param_t   dtmb_t_param; // terrestrial tuner parameters
		RTK_Tnr_ABS_Param_t    abs_s_param; // satellite tuner parameters>
		RTK_Tnr_DVBS_Param_t   dvb_s_param;
	} tnr_param;
} RTK_RfDemodParam_t;





typedef struct
{
	RTK_Source_Type_e          type;				/*!< source type*/
	RTK_Tuner_Bandwidth_e      bandwidth;			/*!< bandwidth*/
	RTK_CAS_Param_t            cas_param;			/*!< CAS type and paramters*/
	union
	{
		RTK_Tnr_DVBC_Param_t   dvb_c_param;	 /*!< European-standard cable tuner parameters*/
		RTK_Tnr_DVBT_Param_t   dvb_t_param;
		RTK_Tnr_DTMB_Param_t   dtmb_t_param; /*!< terrestrial tuner parameters*/
		RTK_Tnr_ABS_Param_t    abs_s_param; /*!< satellite tuner parameters */
		RTK_Tnr_DVBS_Param_t   dvb_s_param; 
		RTK_Memory_Param_t     mm_param;
	} tnr_param;
	RTK_Bool                   isSMP;			/*!< secure media path option*/
	unsigned char              TunerID; 		/*!< tuner id, indicate this source using witch tuner.*/
} RTK_SourceParam_t;

/**
   *@brief 
   * It defines Input tv flow Parameter structures,
   * Including standard type, time mark, encrypt flag and differnt source type parametric structures.
   */
typedef struct
{
	RTK_InputType type;			/*!< specify tuner or memory input*/
	RTK_Bool	enableTimeShift; 	/*!< not used now...*/
	union
	{
		RTK_TunerParam_t tuner;   /*!< tuner parameters*/
		RTK_SOCKET_t iptv;		/*!< iptv parameters, not used*/
		struct
		{
			char* filePath;
			int32_t filePathLen;
		}memory;				/*!< memory, local file, not used*/
		struct
		{
			char *filePath; 			/*!< file path including file name*/ 
			int32_t filePathLen;		/*!< the file path size*/
			char *storeInfoFilePath;	/*!< the address pointer of buffer to store the file*/
			int32_t storeInfofileLen;	/*!< the length of file to be stored*/
		}playback;				/*!< local ts file, not used*/ 
	};
	RTK_Bool isEncrypt; 			/*!< indicate whether the data is encrypted*/
	RTK_Bool isUseAssignedChannelId;	/*!< If false, the channel id will be auto allocate. 
									If true, assignedChannelId will be referred as channel id.*/
	int assignedChannelId;		/*!< user assigned channel id*/
	RTK_Bool disableSignalHandle;/* if true,don't use signal handle. 
									but if signal is already initialized and set to true is useless*/
}
RTK_InputConfig;

/**
    *@brief 
    *It defines TS tream input buffer structures,
    *
    * User can use this data structure to feed data into the RTKHAL.
    * 
    */
typedef struct
{
	void * memory;			/*!< [IN/OUT]memory pointer. RTKHAL set this pointer value and return when user call RTK_Demux/Player_GetInputBuffer().
							 see RTK_Demux_GetInputBuffer, RTK_Player_GetInputBuffer*/   
	void * begin;
	void * end;							 
	uint32_t size;			      /*!< memory size in free state. RTKHAL set this value when user call RTK_Player_Demux/GetInputBuffer().
							see RTK_Demux_GetInputBuffer, RTK_Player_GetInputBuffer*/

	uint32_t filled_size;	      /*!< indicate how much data filled.
							User copy MPEG TS data to memory pointer which was got using RTK_Player_Demux/GetInputBuffer(),
							and set this copied bytes value to this value. Must be small or equal than size field. */

	RTK_Bool isLast;		     /*!< Indicate last feed of data.
						     It may helpful for RTKHAL to calculate EOS/BOS event. For example set true when feed last block of a file. */  
	RTK_Bool isFeedAsync;   /// feed this buffer in non-cached way
	RTK_Bool isFrameEnd;
}
RTK_TSInputBuffer;


/**
   *@brief 
   * For I-frame-only mode. Tell dvdplayer I frame's size and start address for further manipulation
   *This structure should be consistent with structure IframeOffset_t in rtk_porting_av.h
   */
typedef struct
{
    unsigned int data_phyaddr;	/*!< the physical address when user feed data into input buffer. Now user should set 0.*/
    unsigned int data_size;		/*!< all the data size copy to input buffer*/
    unsigned int iframe_offset;	/*!< the iframe offset in the input data*/
    unsigned int iframe_size;	/*!< the iframe size in the input data*/
}
RTK_IframeOffset_t;


/**
   *@brief 
   * This enum value defines A/V codec types.
   */
typedef enum
{
	RTK_CODEC_NONE              = 0,

	///	audio.
	RTK_CODEC_AUDIO_MPEG1,
	RTK_CODEC_AUDIO_MPEG2,
	
	RTK_CODEC_AUDIO_AAC,
	RTK_CODEC_AUDIO_AACPLUS,

	RTK_CODEC_AUDIO_AC3,
	RTK_CODEC_AUDIO_AC3_TRUE_HD,
	RTK_CODEC_AUDIO_AC3_PLUS,
	RTK_CODEC_AUDIO_EAC3,

	RTK_CODEC_AUDIO_DTS,
	RTK_CODEC_AUDIO_DTS_HD,
	RTK_CODEC_AUDIO_DTS_HD_MA,

	RTK_CODEC_AUDIO_LPCM,
	RTK_CODEC_AUDIO_MP3,
	RTK_CODEC_AUDIO_MP4,
	
	///	video. offset 100
	RTK_CODEC_VIDEO_MPEG1     = 100,
	RTK_CODEC_VIDEO_MPEG2,
	RTK_CODEC_VIDEO_MPEG4_PART2,

	RTK_CODEC_VIDEO_H264,
	RTK_CODEC_VIDEO_H265,

}
RTK_Codec;


/**
    *@brief 
    * This enum value defines HDCP version,
    * including HDCP14 and HDCP22.
    */
typedef enum
{
	RTK_HDCP_VER_NULL = 0,
	RTK_HDCP_VER_HDCP14 = 14,
	RTK_HDCP_VER_HDCP22 = 22,
}RTK_HDCP_VERSION_e;


/**
    *@brief 
    * This enum value defines recording mode.
    * We support ts mode and block mode.
    */
typedef enum
{
	RTK_REC_MODE_TS = 0,
	RTK_REC_MODE_BLOCK,
}RTK_REC_MODE;


/**
    *@brief 
    * This enum value defines encryption method for recording,
    * including our own MCP method and melon DSC method.
    */
typedef enum
{
	RTK_REC_METHOD_PRIVATE = 0,
	RTK_REC_METHOD_DSC,
	RTK_REC_METHOD_CONNECTPVR,
	RTK_REC_METHOD_TKL,
}RTK_REC_METHOD;


/**
    *@brief 
    * It defines the Recording info structure.
    */
typedef struct
{
	RTK_REC_MODE	mode;
	RTK_REC_METHOD	enc_method;
}RTK_RecConfig;


/**
    *@brief 
    * It defines the Audio codec info structure.
    *
    *Note:
    * 		Audio/Video codec input may be invalid.
    *
    * 		for example:
    * 		when playing a channel.
    * 		we cannot know A/V pid & codec information until PMT received.
    */
typedef struct
{
	uint16_t	pid;					/*!< pid of audio stream. can be invalid (0)*/
	RTK_Codec 	codec;			/*!< Audio coded type, can be invalid(RTK_CODEC_NONE). at first.*/
	RTK_Bool	encrypted;		      /*!< indicate pid stream is encrypted (PVR recorded stream)*/
	RTK_Bool	enable_raw_output_mode;	/*!< enable/disable RAW output mode (AAC, AC3 raw output mode, NOT PCM output mode)*/
	RTK_Bool	focused;				/*!< enable focuse the audio config or not*/
}
RTK_AudioConfig;

/**
    *@brief
    * It defines the Video codec info structure.
    * including pid,codec,outplane and whether is encrypted.
    */
typedef struct
{
	uint16_t	pid;			/*!< pid of video stream. can be invalid (0).*/
	RTK_Codec 	codec;	/*!< can be RTK_CODEC_NONE at initialize time.*/
	RTK_WinPlane	output;	/*!< video output plane for main or PIP*/
	RTK_Bool	encrypted;	/*!< indicate pid stream is encrypted (PVR recorded stream)*/
}
RTK_VideoConfig;
/**
   *@brief 
   * This enum value defines dvb subtitle types.
   */
typedef enum RTKAP_SUBTITLE_TYPE
{
	RTKTYPE_DVB_SUBTITLE = 0,///DVB subtitle
	RTKTYPE_TT_SUBTITLE, ///external subtitle
	RTKTYPE_DVB_HARD_HEARING,	///DVB subtitle
}RTKAP_SUBTITLE_TYPE_e;
/**
    *@brief
    * It defines the subtitle config info structure.
    * including pid,language code,subtitle type and whether to show.
    */
typedef struct
{
		int32_t 					idx;	
		uint16_t					pid;			/*!< subtitle pid*/				
		int32_t					langcode; 	/*!< language code*/ 		  
		RTKAP_SUBTITLE_TYPE_e    subtype;	/*!< the subtitle type, RTKTYPE_TT_SUBTITLE or RTKTYPE_DVB_HARD_HEARING*/
		bool        					bshow;	/*!< show flag*/			
}
RTK_SubtitleConfig;


/**
    *@brief
    * It defines the section filter structure.
    * including pid,monitoring_mode,callback and so on.
    */
typedef struct
{
	uint16_t pid;					/*!< pid */
	uint32_t options;				/*!< control CRC mode at bit0, if bit0 is RTK_SECTION_OPTION_NO_CHECK_CRC, CRC will not check*/
	RTK_SectionMonitoringMode monitoring_mode; /*!< monitior mode, oneshot or continous*/

	    /*
		Description
		
		inclusion mask (mask) - for all inclusion mask bits that are masked
		(set to 0xff) ALL of the corresponding bits in the
		message must match the bits specified by the coefficient in order to
		be included. If one bit differs, it will be excluded.

		exclusion mask (mode) - for all exclusion mask bits that are masked
		(set to 0xff) at least one of the corresponding bits
		in the message must NOT match the bits specified
		by the coefficient in order to be included. If all of the bits
		match, it will be excluded.
	    */
	uint8_t mask[RTK_MAX_SECTION_MASK_LENGTH];	/*!< inclusion mask for comp[]. See Description.*/
	uint8_t mode[RTK_MAX_SECTION_MASK_LENGTH];	/*!< exclusion mask for comp[]. See Description.*/
	uint8_t comp[RTK_MAX_SECTION_MASK_LENGTH];	/*!<  bits to be included or excluded, depending on mask[] and mode[].*/
	
	RTK_SectionFilterCallback callback;			/*!< section callback function*/

	void * section_user_param;			/*!< user section callback parameter (should be delivered via RTK_SectionData.section_user_param when callback)*/
	uint32_t u32byWaitTimeoutMs;			/*!< unit is millisecond*/
	RTK_SectionFilterCallback_01 callback_01;	/*!< section callback function, support timeout*/
}
RTK_SectionFilterMask;

/**
    *@brief
    * It defines the section data structure.
    * including filter_handle, section data pointer ,section data size and so on.
    */
typedef struct
{
	RTK_SectionFilter filter_handle;	///	filter ha/ndle which was created before
	void * section_user_param;			//	user section callback parameter

	void * section_data;				///	section data pointer. which have received section
	uint32_t section_data_size;			///	section data size
}
RTK_SectionData;

/**
    *@brief
    * It defines the TS  filter structure.
    * including pid to be filtered out, whether encrypted packet to be filtered out ,PTS and codec to return.
    */
typedef struct
{
	uint16_t 	pid;		/*!< pid to capture from input TS*/
	RTK_Bool 	encrypt;	 /*!< captured packets encrypted or not. (AV pids for recording)*/
	RTK_Bool 	isAnchor;	/*!< if true, use this pid to get PTS, Frame type*/	
	RTK_Codec 	codec;		/*!< specify audio/video codec. RTK_CODEC_NONE if pid contains NOT AV data (for ex: PAT, PMT, ...)*/
	RTK_Bool	isTag;		/*!< if true, will callback pid/offset*/
}
RTK_TSFilterPattern;
/**
    *@brief
    * It defines the callback function structure.
    * including callback function entry pointer and the data size the data will deal with.
    */
typedef struct
{
	void * user_param;		///	callback user_param
	RTK_Callback function;

	///	indicates when callback called: sdk calls client whenever collected data size over following values.
	///	must be multiple of 188
	uint32_t TSDataCallbackSize;
	uint32_t indexDataCallbackSize;
	uint32_t ctrlDataCallbackSize;
}
RTK_CallbackConfig;

#define RTK_SUPPORT_MAX_STREAMS (8) ///the maximun streams to support

/**
    *@brief
    * It defines the PMT info structure.
    */
typedef struct Rtk_PMT_INFO
{
	uint16_t serviceId;
	uint16_t index;
	uint16_t pawEcmPid[RTK_SUPPORT_MAX_STREAMS];
	uint16_t pawStreamPid[RTK_SUPPORT_MAX_STREAMS];
	uint8_t  serviceIdx;
}Rtk_PMT_INFO_t;

/**
    *@brief
    * It defines the descramble structure information
    */
typedef struct
{
	RTK_Bool isEncrypt;/// if true,encrypted data will be descrambled
	Descrambler_Algorithm_ID_e algo;///the algorithm used to descrambled
	uint16_t emm_pid;///the ems data pid to descramble
	Rtk_PMT_INFO_t pmtInfo;///the pmt info struct return after descrambling
}
RTK_DescrambleInfo;


/**
   * @brief tuner info.
   *
   */
typedef struct
{
	uint32_t number;			/*!< tuner number */  
	RTK_Bool locked;			/*!< specify whether tuner locked */   
	uint32_t strength;			/*!< tuner strength (0 ~ 100 the bigger the better) */   
	uint32_t quality;			/*!< tuner quality (0 ~ 100 the bigger the better) */   
	float snr;					/*!< snr value (dB) */  
	float ber;					/*!< bit error ratio before error correction (0.0 ~ 1.0) */	
	uint32_t agc;				/*!< agc value */   
	int32_t powerLevel;			/*!< power level (x10 dBuV) */   
	float packetError;			/*!< packet error */   
	float mer;					/*!< mer value */   
	float lber;					/*!< bit error ratio after error correction(reserved, not all demod supported) (0.0 ~ 1.0) */
	float temperature;     	/*!< tuner or demod chip temperature in degree centigrade*/
}
RTK_TunerInfo;

/**
   * @brief audio and vedio sync mode.
   *
   */
typedef enum
{
    RTK_AVSYNC_AUDIO_MASTER_AUTO_AF,       /*!< auto no skip mode, but use audio master first at the start*/
    RTK_AVSYNC_VIDEO_ONLY,
    RTK_AVSYNC_SYSTEM_MASTER_BY_PCR,
    RTK_AVSYNC_SYSTEM_MASTER, 			/*!<av free fun*/
    RTK_AVSYNC_AUDIO_MASTER_AUTO,
    RTK_AVSYNC_AUDIO_MASTER_AUTO_SKIP,
    RTK_AVSYNC_SLOW,
    RTK_AVSYNC_AUDIO_ONLY,
    RTK_AVSYNC_INIT_MODE		
}
RTK_AVsyncMode;
/**
   *@brief Player Config
   * 
   * Player is used for live, live with timeshift, pvr record file.
   */
typedef struct
{
	RTK_CallbackConfig callback;   	/*!< the callback function config when create player*/
	RTK_InputConfig input;			/*!< input type*/
	RTK_VideoConfig video;		/*!< video config*/
	RTK_AudioConfig audio;		/*!< audio config*/
	uint16_t	pcr_pid;				/*!< pid of pcr.*/
	RTK_DescrambleInfo descrambler;	/*!< descrambler config*/
	RTK_SubtitleConfig subtitle;			/*!< subtitle config*/
	RTK_RecConfig rec;				/*!< record config*/
	RTK_AVsyncMode sync_mode;          /* !avsync_mode config*/
}
RTK_PlayerConfig;

/**
    *@brief
    * It defines the demux configure structure.
    */
typedef struct
{
	RTK_CallbackConfig callback;	/*!< the callback function to execute when demux */
	RTK_InputConfig input;			/*!< the input data source config, including type, data source struct, and encrypted flag*/
	RTK_VideoConfig video;	      /*!< the video pid,codec mask information used to parsed the video data*/
	RTK_AudioConfig audio;	     /*!< the audio pid,codec mask information used to parsed the audio data*/
	uint16_t	pcr_pid;			     /*!< pid of pcr.*/
	RTK_DescrambleInfo descrambler;	/*!< descrambler information or config*/
	RTK_SubtitleConfig subtitle;		/*!< the subtitle pid and type*/
	RTK_RecConfig rec;			/*!< recode config*/
	RTK_AVsyncMode sync_mode;          /* !avsync_mode config*/
}
RTK_DemuxConfig;

 /**
    * @brief frame type
    *
    */
typedef enum
{
	RTK_FRAME_TYPE_I,  /// I frame
	RTK_FRAME_TYPE_B, /// B frame
	RTK_FRAME_TYPE_P, /// P frame
}
RTK_FrameType;



/**
   * @brief TS data index
   */
typedef struct
{
	uint64_t offset;			/// captured data offset, will be used as saved file offset.
	RTK_FrameType frame_type;  /// frame type
}
RTK_TS_Index;

/**
   * @brief TS control
   */
typedef struct
{
	uint64_t offset;			/// captured data offset, will be used as saved file offset.
	uint64_t pts;				/// pts value
}
RTK_TS_Ctrl;

/**
   * @brief PES filter pattern.
   *
   */
typedef struct
{
	uint16_t pid;				/*!< pid to capture pes from input TS.*/
	uint8_t u8filterID;	/*!< filterID , only support 0x00 or 0x01.*/
	void * userParam;		/*!< callback user parameter*/
	RTK_PESFilterCallback callback; /*!< callback function pointer*/
}
RTK_PESFilterPattern;

/**
* @brief Video show mode on video changed.
*
*/
typedef enum
{
	RTK_VO_AUTOSHOW_INACTIVE=0, /*!< video audoshow on video changed inactive */
	RTK_VOBLACK_AND_AUTOSHOW_AF_DECODED=1, /*!< video show black on video changed and auto show after video decode*/
	RTK_VOFRAME_AND_AUTOSHOW_AF_DECODED=2,	/*!< video show last frame on video changed and auto show after video decode*/
	RTK_VOSHOW_AF_DECODED_IFRAME=3,	/*!<  show video and run vo after video decode ok ,it is suitable when stop with keepLastFrame =true */
}RTK_VideoShowMode_OnChanged_e;


/**
* @brief define the max TP number.
*
*/
#define HWTP_MAX_NUM	(2)


/**
* @brief Config of the DVB resource.
*
*/
typedef struct  
{
	int demux_usenum;	/*!< the using number of demux*/
	int player_usenum;	/*!< the using number of player*/
	int playerWithTP_usenum;  /*!< the using number of player with TP*/
	int esplayer_usenum;	/*!< the using number of es player(this type player only can play elemet steam)*/
}RTK_DvbResourceConfig_t;

typedef struct
{
	RTK_Codec codec; /*!< video or auido codec*/
	int64_t pts;     /*!< pts value, -1 mean no pts*/
	struct
	{
		uint8_t* pData;  /*!< element stream buffer start address*/
		uint32_t secPhyAddr;
	};
	uint32_t length;   /*!< element stream length*/
	uint8_t isSecureBuf; /*!< input buffer is secure or normal*/
	uint8_t isDstSecureBuf; /*!< dst buffer is secure or normal*/
	union
	{
		struct 
		{
			int sampleRate; /*!< audio sampleRate, only valid when codec is audio*/
			int channelNum; /*!< audio channel number, only valid when codec is audio*/
		}audio_configs;
	};
}RTK_EsInputBuf_t;
/**
* @brief Config of Closed Caption Parameters.
*
*/
typedef struct RTK_ClosedCaption_Param
{     
       int 	ccStreamNum;   				/*!<the number of closed Caption channel */
       unsigned int ccType[8];			/*!< the type id of closed Caption*/
       char *ccTypeString[8];     			/*!<the type string of closed Caption*/
}RTK_ClosedCaption_Param_t;


/**
    *@brief
    * It defines the audio, video and tuner parameters structure of channel.
    */
typedef struct CHParam
{
       int   program_id;
       int   freq;///tuner frequence
       int   vedio_pid;
       int   audio_pid;
       int   qam;
       int   lnb_mode;
       int  symbol_rate;
} CHParam_t;

#define   MAXMULTI_CHANNEL_NUM   8 ///define the maximum programm each channel

/**
*@brief
* It defines the channel data parameters structure,
* including the number of channel and data structure of each channel.
*/
typedef struct RtkMultiCH_Param
{
       int   Channel_count;
       CHParam_t    MultiCH[MAXMULTI_CHANNEL_NUM];
	   RTK_Tuner_Type_e tunerType;
} RTKMultiCH_Param_t;


#define PIDSMAXNUM_IN_ONE_PROGRAM	(2)  ///define the maximum of the pid in one program

typedef struct RTKMultich_Pids
{
	int8_t prog_id;///the program pid associate with the video and audio pid 
	uint8_t pid_num;///the number of pid in each channel
	uint16_t pid[PIDSMAXNUM_IN_ONE_PROGRAM];///the pid value array of each channel
	int codec[PIDSMAXNUM_IN_ONE_PROGRAM];///the codec value array of each channel
}RTKMultich_Pids_t;

typedef enum
{
	RHAL_AV_TRICK_IONLY,
	RHAL_AV_TRICK_FULLTS,	
}RTK_HAL_avTrickMode_e;

/**
   *@defgroup RTK_HAL
   *@{
   */
 /**
   *end defgroup RTK_HAL
   *@}
   */

/**
   *@defgroup RTK_Global
   *@ingroup RTK_HAL
   *@{
   */



/**
* @brief
* This function initialize all the resource used in RTK HAL.
*
* @retval	   RTK_OK	   initialise success.			   
* @retval	   RTK_Fail    initialise failed.				 
*
*/
#if (RTKPLAYER_ARCH_VER >= RTKPLAYER_ARCH_V1)

RTK_Error RTK_initialize(void);



/**
* @brief
* This function finalize all the resource used in RTK HAL.
*
* @retval		RTK_OK	     finalize success.            
* @retval		RTK_Fail 	 finalize failed.   
*
*/
RTK_Error RTK_finalize(void);


/**
* @brief
* This function provide a methord to config DVB resource.
*
* @param[in]	pConfig	     DVB resource config
* 
* @retval		RTK_OK	     finalize success.            
* @retval		RTK_Fail 	 finalize failed.   
*
*/
RTK_Error RTK_DvbResourceConfig(RTK_DvbResourceConfig_t *pConfig);


/**
* @brief
* Set Secure Meida Path enable or disable.
*
* @param[in]    channel_id		DVB channel id, it normally mean the decoder id.
* @param[in]    enable			when enable is RTK_TRUE, it mean SMP enabled.
*
* 
* @retval	   RTK_OK		success.			
* @retval	   RTK_Fail 	failed.   
*
*/
RTK_Error RTK_DvbSetSMP(uint32_t channel_id, RTK_Bool enable);

/**
* @brief
* Get Secure Meida Path enable or disable.
*
* @param[in]    channel_id	 DVB channel id, it normally mean the decoder id.
* 
* @retval	   RTK_TRUE 	 SMP is enabled.			
* @retval	   RTK_FALSE	SMP is disabled.   
*
*/
RTK_Bool RTK_DvbGetSMP(uint32_t channel_id);



   /**
   *end defgroup RTK_Global
   *@}
   */
   
/**
   *@defgroup RTK_Player
   *@ingroup RTK_HAL
   *@{
   */


/**
   * @brief
   * creates player and does initial works for this player
   *
   * @param[out]		player		 return the Player handle created by this function
   * @param[in]		config	        the configs used when creating the player
   * @param[in]		type			mainly refer to HAL_TYPE_PLAYER,HAL_TYPE_DEMUX,HAL_TYPE_EMMDEMUX
   *                                                      HAL_TYPE_PLAYER deal TS flow by software without demux process,
   *                                                      HAL_TYPE_DEMUX deal TS flow by both hardware and software without demux process
   *                                                      HAL_TYPE_EMMDEMUX deal TS flow by both hardware and software with demux process 
   *
   * @retval		RTK_OK			 RTK_Player_Creat has successfully processed.
   * @retval		RTK_Fail 			 parameters error occurred or _xCreate return failed.
   *
   */
RTK_Error RTK_Player_Create(RTK_Player * player, RTK_PlayerConfig * config, HAL_TYPE type);
/**
   * @brief
   * destroys the player
   *
   * @param[in]		player		 	the player handle
   *
   * @retval		RTK_OK			 RTK_Player_Destroy has successfully processed
   * @retval		RTK_Fail 			 parameters error occurred or _xDestroy return failed
   *
   */
RTK_Error RTK_Player_Destroy(RTK_Player player);



/**
   * @brief
   * starts player
   *
   * @param[in]		player		 	the player handle
   *
   * @retval		RTK_OK			the player starts playing successfully.
   * @retval		RTK_Fail 			parameters error occurred or the player doesn't start to play.
   *
   */
RTK_Error RTK_Player_Start(RTK_Player player);

/**
   * @brief
   * stop player
   *
   * @param[in]		player		 	the player handle
   * @param[in]	keepLastFrame	if true the last frame will be show when stop,if false nothing will be shown when stop
   *
   * @retval		RTK_OK			 RTK_Player_Stop has successfully processed
   * @retval		RTK_Fail 			 parameters error occurred or the player doesn't stop to play.
   *
   */
RTK_Error RTK_Player_Stop(RTK_Player player, RTK_Bool keepLastFrame);


/**
   * @brief
   * pauses the player
   *
   * @param[in]		player		 	 the Player handle
   *
   * @retval		RTK_OK			 RTK_Player_Pause has successfully processed
   * @retval		RTK_Fail 			 parameters error occurred or the player doesn't stop to play
   *
   */
RTK_Error RTK_Player_Pause(RTK_Player player);

/**
   * @brief
   *resumes the Player
   *
   * @param[in]		player		 	the Player handle
   *
   * @retval		RTK_OK			 the player replays again successfully
   * @retval		RTK_Fail 			 parameters error occurred or the player doesn't replay.
   *
   */
RTK_Error RTK_Player_Resume(RTK_Player player);



/**
   * @brief
   *Drops all previously fed TS data from internal buffer which is not yet processed
   *
   * @param[in]		player		 	the Player handle
   *
   * @retval		RTK_OK			 RTK_Player_DropData has successfully processed
   * @retval		RTK_Fail 			 Dvbapp instance does not exist.
   *
   */
RTK_Error RTK_Player_DropData(RTK_Player player);

/**
   * @brief
   * modifies pid of PCR
   *
   * @param[in]		player		 	handle of player to deal with
   * @param[in]		pcr_pid		 	pcr pid which the player will change to p
   *
   * @retval		RTK_OK			the PCR has changed successfully.
   * @retval		RTK_Fail 			parameters error occurred or PCR change failed.
   *
   */

RTK_Error RTK_Player_ChangePCR(RTK_Player player, uint16_t pcr_pid);

/**
   * @brief
   * changes audio pid, audio codec type, then play back the audio
   *
   * @param[in]		player		 	handle of player to change audio configuration
   * @param[in]		audio		 	pointer to a prepared RTK_AudioConfig object
   *
   * @retval		RTK_OK			change audio and play successfully.
   * @retval		RTK_Fail 			parameter input error such as player or audio equals NULL.
   *
   */
RTK_Error RTK_Player_ChangeAudio(RTK_Player player, RTK_AudioConfig * audio);

/** @brief
   * @brief
   * changes each track audio PID, codec type, then plays back the multi-audio
   *
   * @param[in]           player                  handle of player to change audio configuration
   * @param[in]           audio                   pointer to an Array of RTK_AudioConfig object
   * @param[in]           count                   the amount number of audio track
   *
   * @retval              RTK_OK                  change audio and play successfully.
   * @retval              RTK_Fail                        parameter input error such as player or audio equals NULL.
   *
   */
RTK_Error RTK_Player_ChangeAudioEx(RTK_Player player, RTK_AudioConfig   **audio, uint8_t count) ;


/**
   * @brief
   * changes video PID, video codec type, then play back the audio
   *
   * @param[in]		player		 	handle of player to deal with	
   * @param[in]		video 	 		defines video PID, codec type, a flag indicate PID stream 
   *									is encrypted or not, Video window type
   *
   * @retval		RTK_OK			change video and play successfully.
   * @retval		RTK_Fail 			parameters error such as player or audio equals NULL
   *
   */
RTK_Error RTK_Player_ChangeVideo(RTK_Player player, RTK_VideoConfig * video);



/**
   * @brief
   * sets AV_sync_mode of a specific player
   *
   * @param[in]		player		 	handle of player 
   * @param[in]		AvSyncMode 		avsync type
   *
   * @retval		RTK_OK			set the avsync mode successfully.
   * @retval		RTK_Fail 			parameters error occurred or rtk_porting_av_set_AV_sync_mode 
   *								return error.
   *
   */
RTK_Error RTK_Player_SetAVSyncMode(RTK_Player player, RTK_AVsyncMode AvSyncMode);

/**
   * @brief
   * sets the speed of play
   *
   * @param[in]		player		 	handle of player to deal with	
   * @param[in]		speed 	 		multiplier factor of change
   *
   * @retval		RTK_OK			the play speed has changed successfully.
   * @retval		RTK_Fail 			parameters error occurred or the play speed change failed.
   *
   */
RTK_Error RTK_Player_SetPlaySpeed(RTK_Player player, float speed);

/**
   * @brief
   * gets play speed of current
   *
   * @param[in]		player		 	handle of player to deal with	
   * @param[out]		speed 	 		get the multiplier factor of current paly speed 
   *
   * @retval		RTK_OK			the play speed has got successfully
   * @retval		RTK_Fail 			parameters error occurred or play speed get failed.
   *
   */
RTK_Error RTK_Player_GetPlaySpeed(RTK_Player player, float * speed);

/**
   * @brief
   * sets the player with specific handle video plane visible or not
   *
   * @param[in]		player		 	handle of player to deal with
   * @param[in]		mute 			defines set video mute or not, RTK_FALSE means 
   *								set video plane  visible, RTK_TRUE means not.  
   *
   * @retval		RTK_OK			set the specific player video plane visible mute successfully.
   * @retval		RTK_Fail 			parameters error occurred or set mute failed.
   *
   */
RTK_Error RTK_Player_SetVideoMute(RTK_Player player, RTK_Bool mute);

/**
   * @brief
   * gets a specific player video visible status
   *
   * @param[in]		player		 	handle of player to deal with
   * @param[out]		mute 			the value of video visible status currently	
   *
   * @retval		RTK_OK			get the specific player video visible status successfully.
   * @retval		RTK_Fail 			parameters error occurred or got video visible status failed.
   *
   */
RTK_Error RTK_Player_GetVideoMute(RTK_Player player, RTK_Bool * mute);

/**
   * @brief
   * sets the player with specific handle audio mute.
   *
   * NOT system wide mute, just for player. (used while trick playing)
   *
   * @param[in]	player		 	handle of player to deal with
   * @param[in]	mute 			defines set audio mute or not, RTK_FALSE 
   *								means not , RTK_TRUE means set mute
   *
   * @retval		RTK_OK			set the specific player mute successfully.
   * @retval		RTK_ERR 		parameters error occurred or set mute failed.
   *
   */
RTK_Error RTK_Player_SetAudioMute(RTK_Player player, RTK_Bool mute);

/**
   * @brief
   * gets a specific player audio mute status
   *
   * @param[in]	player		 	handle of player to deal with
   * @param[out]	mute      		        get the value of audio mute status currently,
   *								RTK_FALSE means not mute, RTK_TRUE means mute 
   *
   * @retval		RTK_OK			get the specific player mute status successfully.
   * @retval		RTK_ERR 			parameters error occurred or got mute status failed. 
   *
   */
RTK_Error RTK_Player_GetAudioMute(RTK_Player player, RTK_Bool * mute);

/**
   * @brief
   * This function will transfor the outer codec type to video type.
   *
   * @param[in]  codec                                     A/V codec type, reference to the enumerating type RTK_Codec for detail
   *
   * @retval	   RTK_VIDEO_CODEC_TYPE_e   return the video codec type, only four type can be processed to specified type.
   *                                                                    
   */
RTK_Codec RTK_TS_Transfer_Video_Codec(unsigned short codec);

/**
   * @brief
   * This function will transfor the outer codec type to Audio type.
   *
   * @param[in]  codec                                      A/V codec type, reference to the enumerating type RTK_Codec for detail
   *
   * @retval         RTK_AUDIO_CODEC_TYPE_e   return the audio codec type to specified type.
   *
   */
RTK_Codec RTK_TS_Transfer_Audio_Codec(unsigned short codec);

/**
    * @brief
    * update vo plane after player started.
    *
    * @param[in]		player		 	the player
    * @param[in]		plane			the plane
    *
    * @retval		RTK_OK			 success.
    * @retval		RTK_Fail 		 failed.
    *
*/
RTK_Error RTK_Player_Update_VO_Plane(RTK_Player player, RTK_WinPlane plane);

/**
   * @brief
   * gets the current displaying frame's pts
   *
   * @param[in]		player		 	handle of player to deal with	
   * @param[out]		*_pts 	 		get the current displaying frame's pts
   *
   * @retval		RTK_OK			the current displaying PTS has got successfully
   * @retval		RTK_Fail 			parameters error occurred or PTS get failed
   *
   */
RTK_Error RTK_Player_GetCurrentDisplayingPTS(RTK_Player player, uint64_t * _pts);

/**
   * @brief
   * gets the current displaying frame's video and audio pts
   *
   * @param[in]		player		handle of player to deal with	
   * @param[out]		*pts_info 	get the current displaying frame's video and audio pts
   *
   * @retval		RTK_OK			the current displaying PTS has got successfully
   * @retval		RTK_Fail 		parameters error occurred or PTS get failed
   *
   */

RTK_Error RTK_Player_GetCurrentDisplayingPTSInfo(RTK_Player player, RTK_PTSInfo * pts_info);

/**
   * @brief
   * send I frame info to dvdplayer when it's in I-frame-only mode
   *
   * @param[in]		player 		handle of player to deal with
   * @param[in]		info 			object composed of frame info
   *
   * @retval		RTK_OK			send info successfully.
   * @retval		RTK_Fail 			parameters error occurred or sending info failed.
   *
   */
RTK_Error RTK_Player_SendIFrameInfo(RTK_Player player, RTK_IframeOffset_t info);


/**
* @brief
* Set video show mode when video changed. And this mode will force inactive when RTK_Player_Stop called.
*
* @param[in]		player 		handle of player to deal with
* @param[in]		mode 		video show mode.
*
* @retval		RTK_OK			success.
* @retval		RTK_Fail 		failed.
*
*/
RTK_Error RTK_Player_SetShowMode_OnVideoChanged(RTK_Player player, RTK_VideoShowMode_OnChanged_e mode);


   
/*
 * In-band section filter for PSI, SI
 */
 
/**
   * @brief
   * opens and sets sectionFilter by specific pid, add the SectionFilter into SI engine with the player handle
   *
   * @param[in]		player		 	handle of player to add section filter
   * @param[out]		filter_handle 	 	handle of filter to specify
   * @param[in]		pattern			defines filter pid, mask, mode, compose type, monitor mode, 
   *								user param and a function pointer to SI section filter callback
   *
   * @retval		RTK_OK			_xAddSectionFilter has successfully processed
   * @retval		RTK_Fail 			parameters error occurred or filter open failed.
   *
   */
RTK_Error RTK_Player_AddSectionFilter(RTK_Player player, RTK_SectionFilter * filter_handle, RTK_SectionFilterMask * pattern);

/**
   * @brief
   * stops the sectionFilter, then it should not parse the specific sections
   *
   * @param[in]		player		 	handle of player 
   * @param[in]		filter_handle 	 	handle of filter to stop
   *
   * @retval		RTK_OK			section filter has been stopped parsing with 
   *								specific filter_handle successfully.
   * @retval		RTK_Fail 			parameters error occurred or SI engine PID 
   *								filter remove failed
   *
   */
RTK_Error RTK_Player_RemoveSectionFilter(RTK_Player player, RTK_SectionFilter filter_handle);

/**
   * @brief
   * stops all sectionFilters player added, then sectionFilters will not work 
   *
   * @param[in]		player		 	handle of player 
   *
   * @retval		RTK_OK			all section filters have been stopped parsing successfully.
   * @retval		RTK_Fail 			parameters error occurred or SI engine PID filter remove failed.
   *
   */
RTK_Error RTK_Player_RemoveAllSectionFilters(RTK_Player player);


  /**
  ** @brief
  ** enable the sectionFilter status
  **
  ** @param[in] 	  player		  handle of player 
  ** @param[in] 	  filter_handle  handle of filter to enable
  ** @param[in] 	  enable		  enable or disable section fitler
  **
  ** @retval	  RTK_OK		  section filter has been enable or disable success
  ** @retval	  RTK_Fail		  parameters error occurred or SI engine PID 
  **							  filter enable or disable failed
  **
  */
  RTK_Error RTK_Player_ControlSectionFilter(RTK_Player player, RTK_SectionFilter filter_handle,RTK_Bool enable);



/*
  * Memory input
  */
 /**
   * @brief
   *applys a buffer from share memory for reserving TS data
   *
   *if apply successfully ,return the buffer information back, this information includes the pointer of share  
   *memory, the total size of memory, the used size ,and a flag to judge whether there is memory left
   *
   * @param[in]		player		 	handle of player to deal with
   * @param[out]		buffer 			define input buffer viraddr, size, filled_size, Islast
   *
   * @retval		RTK_OK			got the buffer information successfully
   * @retval		RTK_Fail 			parameters error occurred or share buff has full
   *
   */
RTK_Error RTK_Player_GetInputBuffer(RTK_Player player, RTK_TSInputBuffer * buffer);

/**
   * @brief
   *Get input info buffer. This buffer can store some decrypt control info when playback the encrypted stream.
   *
   * The info buffer format is defined by CAS type or decrypt control info. And this buffer should be returned to CAS.
   * CAS will use it when feed TS data if it need process some private decrypt info.
   *
   * @param[in]		player		 	handle of player to deal with
   * @param[out]		ppInfoBuf 		return the info buffer address.
   * @param[out]		pInfoBufSize		return the info buffer size.
   *
   * @retval		RTK_OK			got the buffer successfully
   * @retval		RTK_Fail 			parameters error occurred
   *
   */
RTK_Error RTK_Player_GetInputInfoBuffer(RTK_Player player, unsigned char **ppInfoBuf, unsigned int *pInfoBufSize);

/**
   * @brief
   *Reset Decrypt offset.
   * 
   * When playback block mode encrypt stream, the decrypt offset is using for some especial CAS, and this API
   * support to reset this offset.
   *
   * @param[in]		player		 	handle of player to deal with
   *
   * @retval		RTK_OK			successfully
   * @retval		RTK_Fail 			parameters error occurred
   *
   */
RTK_Error RTK_Player_DecryptResetOffset(RTK_Player player, uint64_t ofs);

/**
   * @brief
   * Wait the data in the input buffer being read out. 
   *
   * @param[in]		player		 	handle of player to deal with
   * @param[in]		timeOut_MS 		wait timeout, unit is millisecond.
   *
   * @retval		RTK_OK			successfully
   * @retval		RTK_Fail 			parameters error occurred
   *
   */
RTK_Error RTK_Player_WaitDataGone(RTK_Player player, int32_t timeOut_MS);


/**
   * @brief
   * Redirect the input of the Player. 
   * 
   * This function can support to redirect input and let Demux as the input. If this function called, 
   * The data got from Demux will force transfer to this player. 
   *
   * @param[in]		player		 			handle of player to deal with
   * @param[in]		isEnable 					indicate whether the redirect enabled.
   * @param[in]		redirect_from 				the source handle for redirect 			
   *
   * @retval		RTK_OK			successfully
   * @retval		RTK_Fail 			parameters error occurred
   *
   */
RTK_Error RTK_Player_InputRedirect(RTK_Player player, bool isEnable, RTK_Demux redirect_from);

/**
   * @brief
   *feeds some TS data used for playing in input_buffer
   *
   * @param[in]		player		 	handle of player to reserve play data and share memory information
   * @param[in]		buffer 			define input buffer viraddr, size, filled_size, islast
   *
   * @retval		RTK_OK			_xFeedInputBuffer has successfully processed.
   * @retval		RTK_Fail 			parameters error occurred or file size not packet align
   *
   */
RTK_Error RTK_Player_FeedInputBuffer(RTK_Player player, RTK_TSInputBuffer * buffer);

/**
   * @brief
   * discards input buffer
   *
   * @param[in]		player		 	handle of player 
   * @param[in]		buffer 			define input buffer viraddr, size, filled_size, islast
   *
   * @retval		RTK_OK			_xDiscardInputBuffer has successfully processed
   * @retval		RTK_Fail 			parameters error occurred or doesn't get buffer first
   *
   */
RTK_Error RTK_Player_DiscardInputBuffer(RTK_Player player, RTK_TSInputBuffer * buffer);

/**
   * @brief
   * Set input buffer align.
   *
   * @param[in]		player		 	handle of player. 
   * @param[in]		align 			byte align.(if set zero, align not required)
   *
   * @retval		RTK_OK			successful
   * @retval		RTK_Fail 			failed
   *
   */
RTK_Error RTK_Player_SetInputBufferAlign(RTK_Player player, uint32_t align);

/**
   * @brief
   * Feed element stream into es buffer.
   *
   * @param[in]		player		 	handle of player 
   * @param[in]		buffer 			element stream input buffer.
   * @param[in]		timeout 		feed timeout.(millisecond)
   *
   * @retval		RTK_OK			_xDiscardInputBuffer has successfully processed
   * @retval		RTK_Fail 			parameters error occurred or doesn't get buffer first
   *
*/
RTK_Error RTK_Player_FeedEsInputBuffer(RTK_Player player, RTK_EsInputBuf_t *buffer, int32_t timeout);

/**
   * @brief
   * Get the channel ID of the player.
   *
   * @param[in]		player		 	handle of player 
   * @param[out]	p_channelId		return the channel ID of this player.
   *
   * @retval		RTK_OK			successfully
   * @retval		RTK_Fail 			parameters error occurred
   *
*/
RTK_Error RTK_Player_GetChannelId(RTK_Player player,unsigned int *p_channelId);

/**
   * @brief
   *  set AV event callback
   *
   * @param[in]		player		 	handle of player 
   * @param[in]		av_event 		av event
   * @param[in]		callback        av event callback
   *
   * @retval		RTK_OK				successfully processed
   * @retval		RTK_Fail 			fail or timeout.
   *
*/
RTK_Error RTK_Player_AddAvEventCallback(RTK_Player player, RTK_HAL_AV_EVENT av_event, RTK_HAL_AV_EVENT_FUNC callback);

/**
   * @brief
   *  Enable or disable secure play share memory.
   *
   *  This function should be called once player was created. When player destroy, 
   *  the secure share memory will be force released.
   *
   * @param[in]		player		 	handle of player 
   * @param[in]		en 				enable or disable flag
   *
   * @retval		RTK_OK				successfully processed
   * @retval		RTK_Fail 				fail.
   *
*/
RTK_Error RTK_Player_EnableSecurePlayShareMem(RTK_Player player, RTK_Bool en);


/**
   * @brief 
   *	Set Trick play work mode.
   *
   * @param	[in] player  	handle of player.
   * @param 	[in] trickMode	enum value, only I mode or full TS mode defined by RTK_HAL_avTrickMode_e.
   * 
   * @retval	RTK_OK			Success
   * @retval	RTK_ERR			Failed
   * 
*/
RTK_Error RTK_Player_SetTrickMode(RTK_Player player, RTK_HAL_avTrickMode_e trickMode);

/**
   * @brief
   * Get Ts data input bitrate which calculated with PCR info.
   *
   * @param[in]		player		 	handle of player to deal with
   * @param[out]	pullBitrate 	pointer of the variable witch will store the result.
   *
   * @retval		RTK_OK			success.
   * @retval		RTK_ERR 		failed. 
   *
   */
RTK_Error RTK_Player_GetTsInputBitrate(RTK_Player player, uint64_t* pullBitrate);

/**
   *end defgroup RTK_Player
   *@}
   */


/**
   *@defgroup RTK_Demux
   *@ingroup RTK_HAL
   *@{
   */
   
/**
** @brief
** creates Demux for recording
**
** @param[out]		demux			return the Demux handle created by this function
** @param[in]		demux_config 	the configs used when creating the Demux
**
** @retval		RTK_OK			Demux created successfully
** @retval		RTK_Fail 			Demux created fail
**
*/
RTK_Error RTK_Demux_Create(RTK_Demux * demux, RTK_DemuxConfig * demux_config);

/**
** @brief
** destroys the Demux
**
** @param[in]		demux 	the Demux handle, this handle must be same with the created one
**
** @retval		RTK_OK			RTK_ Demux_Destroy has successfully processed
** @retval		RTK_Fail 			parameters error occurred or _xDestroy return failed.
**
*/
RTK_Error RTK_Demux_Destroy(RTK_Demux demux);

/**
** @brief
** lets Demux running
**
** @param[in]		demux 	the Demux handle
**
** @retval		RTK_OK			RTK_ Demux_Start has successfully processed.
** @retval		RTK_Fail 			parameters error occurred or started the Demux failed.
**
*/
RTK_Error RTK_Demux_Start(RTK_Demux demux);

/**
** @brief
** stops Demux
**
** @param[in]		demux 	the Demux handle
**
** @retval		RTK_OK			RTK_ Demux_Stop has successfully processed.
** @retval		RTK_Fail 			parameters error occurred or stopped the Demux failed.
**
*/
RTK_Error RTK_Demux_Stop(RTK_Demux demux);

/**
   * @brief
   * pauses the demux
   *
   * @param[in]		demux		the demux handle
   *
   * @retval		RTK_OK			 successfully processed
   * @retval		RTK_Fail 			 parameters error occurred
   *
   */
RTK_Error RTK_Demux_Pause(RTK_Demux demux);

/**
   * @brief
   *resumes the demux
   *
   * @param[in]		demux		 	the demux handle
   *
   * @retval		RTK_OK			 successfully
   * @retval		RTK_Fail 			 parameters error occurred.
   *
   */
RTK_Error RTK_Demux_Resume(RTK_Demux demux);
   
/**
** @brief
** opens and sets sectionFilter by specific pid, add the SectionFilter into SI engine with the demux handle
**
** @param[in]		demux		 handle of demux to add section filter
** @param[out]		filter_handle 	 handle of filter to specify
** @param[in]		pattern		 defines filter pattern,include pid, mask, mode, compose type, monitor mode,and so on
**
** @retval		RTK_OK			add success
** @retval		RTK_Fail 			add fail
**
*/
RTK_Error RTK_Demux_AddSectionFilter(RTK_Demux demux, RTK_SectionFilter * filter_handle, RTK_SectionFilterMask * pattern);

/**
** @brief
** stops the sectionFilter, then it should not parse the specific sections
**
** @param[in]		demux		 handle of demux 
** @param[in]		filter_handle 	 handle of filter to stop
**
** @retval		RTK_OK			remove success
** @retval		RTK_Fail 			remove fail
**
*/
RTK_Error RTK_Demux_RemoveSectionFilter(RTK_Demux demux, RTK_SectionFilter filter_handle);

/**
** @brief
** stops all sectionFilters demux added, then sectionFilters will not work 
**
** @param[in]		demux		 handle of demux to deal with
**
** @retval		RTK_OK			remove all success
** @retval		RTK_Fail 			remove all fail
**
*/
RTK_Error RTK_Demux_RemoveAllSectionFilters(RTK_Demux demux);

/*
 * TS filter
 * 		- These functions can be called multiple times.
 */
 
/**
** @brief
** adds TSFilter for demux
**
** @param[in]		demux		 	handle of demux to add 
** @param[in]		pattern 	 		defines filter pattern ,include pid, encrypted or not, A/V codec types, get PTS and Frame type or not
** @param[out]		filter_handle		handle of filter to specify
**
** @retval		RTK_OK			add success
** @retval		RTK_Fail 			add fail
**
*/
RTK_Error RTK_Demux_AddTSFilter(RTK_Demux demux, RTK_TSFilterPattern * pattern, RTK_TSFilter * filter_handle);

/**
** @brief
** removes the TSFilter, then it should not parse the specific TS data
**
** @param[in]		demux		 	handle of player to deal with
** @param[in]		filter_handle		handle of TsFilter to stop
**
** @retval		RTK_OK			remove success
** @retval		RTK_Fail 			remove fail
**
*/
RTK_Error RTK_Demux_RemoveTSFilter(RTK_Demux demux, RTK_TSFilter filter_handle);

/**
** @brief
** stops all TSFilter demux added, then TSFilter will not work 
**
** @param[in]		demux		 	handle of player to deal with
**
** @retval		RTK_OK			remove all success
** @retval		RTK_Fail 			remove all fail
**
*/
RTK_Error RTK_Demux_RemoveAllTSFilters(RTK_Demux demux);


/**
** @brief
** Set the ts data buffer type of the TS Filter callback.
**
** This function must be called before RTK_Demux_Start and RTK_Demux_AddTSFilter.
**
** @param[in]		demux		 	handle of player to deal with
** @param[in]		isSecureBuf		secure buffer flag.

**
** @retval		RTK_OK			success
** @retval		RTK_Fail 			fail
**
*/
RTK_Error RTK_Demux_EnableTSFilterSecurebuf(RTK_Demux demux, RTK_Bool isSecureBuf);	


/**
   * @brief
   * Reset TS filter parser.
   *
   * After this operation, the TS Filter Parser will be reset to the init state. It mean all the 
   * TS callback buffer will be reset, include pts, frames...
   *
   * @param[in]      demux		 	handle of player to deal with
   *
   * @retval		RTK_OK			success
   * @retval		RTK_Fail 			fail
   *
   */
RTK_Error RTK_Demux_ResetTSFilterParser(RTK_Demux demux);

/*
 * Memory input
 */
 
/**
** @brief
** gets a buffer from share memory to reserve TS data
**
** @param[in]		demux		 	handle of player to deal with
** @param[out]		buffer			define input buffer viraddr, size, filled_size, islast
**
** @retval		RTK_OK			get success
** @retval		RTK_Fail 			get fail
**
*/
RTK_Error RTK_Demux_GetInputBuffer(RTK_Demux demux, RTK_TSInputBuffer * buffer);

/**
** @brief
** feeds TS data to input buffer for play
**
** @param[in]		demux		 	handle of player to reserve play data and share memory information
** @param[out]		buffer			define input buffer viraddr, size, filled_size, islast
**
** @retval		RTK_OK			feed success
** @retval		RTK_Fail 			feed fail
**
*/
RTK_Error RTK_Demux_FeedInputBuffer(RTK_Demux demux, RTK_TSInputBuffer * buffer);

/**
** @brief
** discards input buffer
**
** @param[in]		demux		 	handle of player to deal with
** @param[out]		buffer			define input buffer viraddr, size, filled_size, islast
**
** @retval		RTK_OK			 discard success
** @retval		RTK_Fail 			 discard fail
**
*/
RTK_Error RTK_Demux_DiscardInputBuffer(RTK_Demux demux, RTK_TSInputBuffer * buffer);
/*
 * In-band PES filter for DVB subtitle
 */

/**
 ** @brief
 ** This function add PES filter and parse the PES data through callback function.
 **
 ** @param[in] demux		   handle of demux to add PES filter
 ** @param[in] pattern		   defines filter pid,filterID, callback function
 **
 ** @retval 	   RTK_OK		   AddPESFilter has successfully processed
 ** @retval 	   RTK_Fail 		   parameters error occurred or add filter failed
 **
 */
RTK_Error RTK_Demux_AddPESFilter(RTK_Demux demux, RTK_PESFilterPattern * pattern);

/**
 * * @brief
 ** removes the PESFilter, then it should not parse the specific PES data
 **
 ** @param[in]	demux		 	  handle of demux to deal with
 ** @param[in] u8FilterID         only support filter ID 0x00 or x001 now
 **
 ** @retval		RTK_OK			PES filter has been stopped parsing with specific filter_handle successfully
 ** @retval		RTK_Fail 			parameters error occurred or ts filter remove failed
 **
 */
RTK_Error RTK_Demux_RemovePESFilter(RTK_Demux demux, uint8_t u8FilterID);

/**
* * @brief
 ** This function will add subtitle to the player.
 **
 ** @param[in]	demux		         handle of demux to deal with
 ** @param[in]	ptrSubtitleTrackInfo  the subtitle information  
 **
 ** @retval		RTK_OK	                 add subtitle to the player ok
 ** @retval		RTK_Fail 	                 add subtitle to the player failed
 **
 */
RTK_Error RTK_Demux_SetSubtitle(RTK_Demux demux, RTK_SubtitleConfig ptrSubtitleTrackInfo);



/**
   *end defgroup RTK_Demux_Buffer
   *@}
   */ 


/**
   *@ingroup RTK_Player
   *@{
   */

/**
* @brief
*  Get TP buffer status.
*
* @param[in]		player		 	handle of player 
* @param[out]		pFreeSize 		buffer free size.
* @param[out]		pDataSize 		the size of data in buffer.
*
* @retval		RTK_OK				successfully processed
* @retval		RTK_Fail 			fail.
*
*/
RTK_Error RTK_Player_GetTPBufferStatus(RTK_Player player, uint32_t* pFreeSize, uint32_t *pDataSize);

/**
* @brief
*  Feed DMA buffer into player.
*
* @param[in]		player		 	handle of player 
* @param[in]		phyaddr 		DMA buffer physical address.
* @param[in]		size 			feed size.
* @param[in]		timeout 		wait timeout(millisecond)	
* @param[in]		is_last 		imply EOS
*
* @retval		RTK_OK				successfully processed
* @retval		RTK_Fail 			fail or timeout.
*
*/
RTK_Error RTK_Player_FeedDMABuffer(RTK_Player player, uint32_t phyaddr, uint32_t size, uint32_t timeout, bool is_last);



/**
   *end ingroup RTK_Player
   *@}
   */
  
/**
   *@defgroup RTK_Tuner
   *@ingroup RTK_HAL
   *@{
   */
/**
   * @brief
   * This function will get the tuner lock status with specified tuner instance ID.
   *
   * @param[in] 	ucTunerID		tuner instance ID
   *
   * @retval		RTK_OK			the tuner is locked.
   * @retval		RTK_Fail		the tuner is unlocked.
   *
   */
RTK_Error RTK_Tuner_IsLocked(uint8_t ucTunerID);

/**
   * @brief
   * This function will do the initial work to start detect the frequence.   *
   *
   * @retval		RTK_OK			initial the tuner successfully.
   * @retval		RTK_Fail		initial the tuner failed.
   *
   */
RTK_Error RTK_Tuner_Init(void);

/**
   * @brief
   * This function will reset tuner/demod and it's related module(TP and IIC interface).
   *
   *
   * @retval		RTK_OK			initial the tuner successfully.
   * @retval		RTK_Fail 		initial the tuner failed.
   *
   */
RTK_Error RTK_Tuner_UnInit(void);

/**
   * @brief
   * get tuner information
   *
   * @param[in]		ucTunerID		tuner instance ID
   * @param[out]	info			signal information
   *
   * @retval		RTK_OK			get success
   * @retval		RTK_Fail		get fail
   */
RTK_Error RTK_Tuner_GetInfo(uint8_t ucTunerID, RTK_TunerInfo *info);

/**
* @brief
* This function will get the tuner chip id.
*
* @param[in]    uiTunerID               tuner ID
* @param[out]    *pDevicechipid        tuner chipid
* @retval        RTK_OK            the tuner chipid is got.
* @retval        RTK_Fail             the tuner chipid get fail.
*
*/
RTK_Error RTK_Tuner_GetDeviceChipid(uint32_t uiTunerID,uint8_t * pDevicechipid);


/**   
   *end ingroup RTK_Tuner
   *@}
   */

/**
   *@defgroup RTK_Global
   *@ingroup RTK_HAL
   *@{
   */
   
/**
   * @brief
   * This function will create a display window.
   *
   * @param[in]	win_cfg		a instance of RTK_WinConfig structures,
   *                                                mainly specify window either main or pip window,its start point x and y position,
   *                                                and its width and height
   *
   * @retval		RTK_OK	        create the display window successfully
   * @retval		RTK_Fail 		 create the display window failed or paramerer error occured
   *
   */
RTK_Error RTK_CreateSurface(RTK_WinConfig win_cfg);
/**
   * @brief
   * This function will destroy a display window that has been created.
   *
   * @param[in]	plane		 a instance of RTK_WinPlane enumeration type,must the same as win_cfg->plane created
   *
   * @retval		RTK_OK	        destroy the display window successfully
   * @retval		RTK_Fail 		 destroy the display window failed or paramerer error occured
   *
   */
RTK_Error RTK_DestorySurface(RTK_WinPlane plane);


/**   
   *end ingroup RTK_Global
   *@}
   */


/**
   *@ingroup RTK_Player
   *@{
   */
#ifdef ANDROID
/**
   * @brief
   * This function will create the side band handle about dvdplayer.
   *
   * @param[in]	player		    handle of player to deal with
   *
   * @retval		handle		    handle of  the side band
   *
   */
void* RTK_Player_GetSidebandHandle(RTK_Player player);
#endif


  
/**
   * @brief
   * Start DVR record.
   *
   * @param[in]		player		 	handle of player
   *
   * @retval		RTK_OK			DVR successfully stared.
   * @retval		RTK_Fail 			DVR start failed.
   *
   */
RTK_Error RTK_Player_DVRStart(RTK_Player player);

/**
   * @brief
   * Stop DVR record.
   *
   * @param[in]		player		 	handle of player
   *
   * @retval		RTK_OK			DVR successfully stoped.
   * @retval		RTK_Fail 			DVR stop failed.
   *
   */
RTK_Error RTK_Player_DVRStop(RTK_Player player);


/**
   * @brief
   * adds TSFilter for demux
   *
   * @param[in]		player		 	handle of player to add 
   * @param[in]		pattern 	 		defines filter pattern ,include pid, encrypted or not, A/V codec types, get PTS and Frame type or not
   * @param[out]		filter_handle		handle of filter to specify
   *
   * @retval		RTK_OK			add success
   * @retval		RTK_Fail 			add fail
   *
   */
RTK_Error RTK_Player_AddTSFilter(RTK_Player player, RTK_TSFilterPattern * pattern, RTK_TSFilter * filter_handle);

/**
   * @brief
   * removes the TSFilter, then it should not parse the specific TS data
   *
   * @param[in]		player		 	handle of player to deal with
   * @param[in]		filter_handle		handle of TsFilter to stop
   *
   * @retval		RTK_OK			remove success
   * @retval		RTK_Fail 			remove fail
   *
   */
RTK_Error RTK_Player_RemoveTSFilter(RTK_Player player, RTK_TSFilter filter_handle);

/**
   * @brief
   * stops all TSFilter demux added, then TSFilter will not work 
   *
   * @param[in]      player		 	handle of player to deal with
   *
   * @retval		RTK_OK			remove all success
   * @retval		RTK_Fail 			remove all fail
   *
   */
RTK_Error RTK_Player_RemoveAllTSFilters(RTK_Player player);

/**
   * @brief
   * Reset TS filter.
   *
   * After this operation, the TS Filter Parser will be reset to the init state. It mean all the 
   * TS callback buffer will be reset, include pts, frames...  
   *
   * @param[in]      player		 	handle of player to deal with
   *
   * @retval		RTK_OK			success
   * @retval		RTK_Fail 			fail
   *
   */
RTK_Error RTK_Player_ResetTSFilterParser(RTK_Player player);

/**
   *end ingroup RTK_Player
   *@}
   */

/**
   *@ingroup  RTK_Global
   *@{
   */

 /**
   * @brief
   *  set audio volume;          
   * @param[in]	i32volume	the volume value to set, range:0~31    
   *                                                 0=> Manimum , 31=> is Maxmum
   * @retval		void
   */

 void RTK_audio_SetVolume(uint32_t i32volume);
  /**
   * @brief
   *  get audio volume;          
   * @param[out]	i32volume	the volume value to set, range:0~31    
   *                                                 0=> Manimum , 31=> is Maxmum
   * @retval		void
   */

void RTK_audio_GetVolume(uint32_t* i32volume);

/**
   * @brief
   * This function will set spdif output mode.
   *
   * @param[in]	outputMode	     specified spdif output mode,such as off,raw and lpcm mode.
   * 
   * @retval		RTK_OK		    set the spdif output mode successfully.
   * @retval		RTK_Fail 		    set the spdif output mode failed
   *
   */
RTK_Error RTK_Set_Spdif_Output_Mode(Hal_Spdif_Mode_e outputMode);

/**
   * @brief
   * This function will get spdif output mode.
   *
   * @param[out]	outputMode	    return spdif output mode,such as off,raw and lpcm mode.
   * 
   * @retval		RTK_OK		    get the spdif output mode successfully.
   * @retval		RTK_Fail 		    get the spdif output mode failed
   *
   */
RTK_Error RTK_Get_Spdif_Output_Mode(Hal_Spdif_Mode_e* outputMode);

/**
   * @brief
   * This function will set hdmi output mode.
   *
   * @param[in]	outputMode	    specified hdmi output mode,such as off,raw and lpcm mode.
   * 
   * @retval		RTK_OK		    set the spdif output mode successfully.
   * @retval		RTK_Fail 		    set the spdif output mode failed
   *
   */
RTK_Error RTK_Set_Hdmi_Output_Mode(Hal_Hdmi_Mode_e outputMode);

/**
   * @brief
   * This function will get hdmi output mode.
   *
   * @param[out]	outputMode	    return hdmi output mode,such as raw, lpcm and auto mode.
   * 
   * @retval		RTK_OK		    get the hdmi output mode successfully.
   * @retval		RTK_Fail 		    get the hdmi output mode failed
   *
   */
RTK_Error RTK_Get_Hdmi_Output_Mode(Hal_Hdmi_Mode_e* outputMode);

/**
   * @brief
   * This function will set the audio output mode.
   *
   * @param[in]	ch_output_mode	    a enumeration type value to indicate audio output mode.
   *
   * @retval		RTK_OK		           Set Audio OutputMode successfully.
   * @retval		RTK_ERR 		    Set Audio OutputMode is not support.
   *
   */
RTK_Error RTK_SetAudioOutputMode(Hal_Audio_Output_Mode_e ch_output_mode);

/**
   *end ingroup RTK_Global
   *@}
   */


/**
   *@ingroup  RTK_Player
   *@{
   */

/*
  * In-band PES filter for DVB subtitle
  */
 /**
   * @brief
   * This function add PES filter and parse the PES data through callback function.
   *
   * @param[in]	player		 	handle of player to add PES filter
   * @param[in]	pattern			defines filter pid,filterID, callback function
   *
   * @retval		RTK_OK			AddPESFilter has successfully processed
   * @retval		RTK_Fail 			parameters error occurred or add filter failed
   *
   */

RTK_Error  RTK_Player_AddPESFilter(RTK_Player player, RTK_PESFilterPattern * pattern);
/**
   * @brief
   * removes the PESFilter, then it should not parse the specific PES data
   *
   * @param[in]	player		 	  handle of player to deal with
   * @param[in] u8FilterID         only support filter ID 0x00 or x001 now
   *
   * @retval		RTK_OK			PES filter has been stopped parsing with specific filter_handle successfully
   * @retval		RTK_Fail 			parameters error occurred or ts filter remove failed
   *
   */
RTK_Error RTK_Player_RemovePESFilter(RTK_Player player, uint8_t u8FilterID);


/**
   * @brief
   * This function will add subtitle to the player.
   *
   * @param[in]	player		         handle of player to deal with
   * @param[in]	ptrSubtitleTrackInfo  the subtitle information  
   *
   * @retval		RTK_OK	                 add subtitle to the player ok
   * @retval		RTK_Fail 	                 add subtitle to the player failed
   *
   */
RTK_Error RTK_Player_SetSubtitle(RTK_Player player, RTK_SubtitleConfig ptrSubtitleTrackInfo);


/**
   * @brief
   * This function will remove subtitle from the player.
   *
   * @param[in]	player		         handle of player to deal with
   *
   * @retval		RTK_OK	                 add subtitle to the player ok
   * @retval		RTK_Fail 	                 add subtitle to the player failed
   *
   */
RTK_Error RTK_Player_RemoveSubtitle(RTK_Player player);

/**
   * @brief
   * This function will set ErrorConcealLevel of a specific player
   *
   * @param[in] 	player			handle of player 
   * @param[in] 	level			conceallLevel want to be set, Value from 0 to 4096
   * @param[in] 	delayTime		after delay time(in milliseconds)will apply error concelment level,
   *								0 means run immediately 
   *
   * @retval		RTK_OK			set the ErrorConcealLevel successfully.
   * @retval		RTK_Fail			rtk_porting_av_set_error_conceal_level return error or parameters
   *								error occurred.
   *
   */

RTK_Error RTK_Player_SetErrorConcealLevel(RTK_Player player, uint32_t level, uint32_t delayTime);
  
/**
   * @brief
   * sets SLOW AV SYNC of a specific player
   *
   * @param[in]		player		 	handle of player 
   * @param[in]		enable 		true or false
   *
   * @retval		RTK_OK			enable successfully.
   * @retval		RTK_Fail 			parameters error occurred or RTK_Player_SetSlowAvSync 
   *								return error.
   *
   */
//RTK_Error RTK_Player_SetSlowAvSync(RTK_Player player, RTK_Bool enable);

RTK_Error RTK_Player_GetScrambleStatus(RTK_Player player, RTK_Bool* isScramble);
/**
** @brief
** enable ccDecoder in order to create cc thread in unittest to get ccdata and deal cc data through player callback
**
** @param[in]		player		 	handle of player to deal with
**
** @retval		RTK_OK			enable transporting cc data  to player callback  successfully.
** @retval		RTK_Fail 			parameters error such as player equals NULL
**
*/

RTK_Error RTK_Player_EnableCCDecoder(RTK_Player player);
/**
** @brief
** get ccdata from player. 
**
** @param[in]		player		 	handle of player to deal with
** @param[out]		pCCData		 	handle of cc data
** @param[out]		pLen		 	length of cc data -- note:0 means no data while pCCData not NULL
**
** @retval		RTK_OK			
** @retval		RTK_Fail 	    
**
*/

RTK_Error RTK_Player_GetCCData(RTK_Player player,unsigned char** pCCData,unsigned int* pLen);

/**
** @brief
** disable ccDecoder in order to destroy cc thread , notice DvdPlayer to stop read and transport ccdata out.
**
** @param[in]		player		 	handle of player to deal with
**
** @retval		RTK_OK			disable  transporting cc data successfully.
** @retval		RTK_Fail 			parameters error such as player equals NULL
**
*/
RTK_Error RTK_Player_DisableCCDecoder(RTK_Player player);
/**
** @brief
** This function control ccdata whether to show with special channel number.
** When bEnable is true, it will create playback cc thread, false will disable playback cc thread.
** Playback cc thread will check the index and whether ccdata is changed to decide show or not.
** If bEnable is true and index is correct, playback cc thread will be created and the cc can be shown. 
** If bEnable is true and index is error, playback cc thread will be created and the cc cann't be shown. 
** If bEnable is false,playback cc thread will be destroyed and the cc cann't be shown,whether index is correct.
**
** @param[in]		player		 	handle of player to deal with
** @param[in]		bEnable		 	true to enable show,false to disable show
** @param[in]		index		 	begin with 0, lower than ccStreamNum
**
** @retval		RTK_OK			to set cc show status successfully.
** @retval		RTK_Fail 			parameters error such as player equals NULL
**
*/
RTK_Error RTK_Player_SetCCPlayback(RTK_Player player,bool bEnable,int index);


/**
   *end ingroup RTK_Player
   *@}
   */

/**
   *@ingroup   RTK_Source
   *@{
   */

//
#if (RTKPLAYER_ARCH_VER == RTKPLAYER_ARCH_V1)  
/**
* @brief
* Set frequency for mulitch-tuner.
*
* @param[in]		tunerid 		   tuner id
* @param[in]		MultiCH_param	   multich tuner paramters.
*
* @retval		RTK_OK			
* @retval		RTK_Fail		
*
*/
RTK_Error RTK_Tuner_Multich_SetFreq(int tunerid, RTKMultiCH_Param_t MultiCH_param);

/**
* @brief
* Get tuner signal info in muilt channel mode. For FCC V2
*
* @param[in]		tunerid			tuner ID
* @param[in]		ucProgID		program ID
* @param[out]		pInfo			result tuner info
*
* @retval		RTK_OK			
* @retval		RTK_Fail		
*
*/
RTK_Error RTK_Tuner_Multich_GetSignalInfo(int tunerid, unsigned char ucProgID, RTK_TunerInfo * pInfo);
   /**  
   *ingroup  RTK_Source
   *@}
   */


/**
   *@ingroup  RTK_Player
   *@{
   */
   
   
/**
* @brief
* Set PID filters for multich tp, this function only for FCC. 
*
* @param[in]		player			   tuner id
* @param[in]		MultiCHPID_param   the PID parameters to set into multich tp.
* @param[in]		channel_count	   indicate the count of mulitch PID parameter.
*
* @retval		RTK_OK			
* @retval		RTK_Fail		
*
*/	
RTK_Error RTK_Player_SetMultiPIDFilters(RTK_Player player, RTKMultich_Pids_t* MultiCHPID_param,int channel_count);

/**
** @brief
** changes multi-channel's program(source)
**
** @param[in]		player		 	handle of player to deal with
** @param[in]		program_id 	 	program id
**
** @retval		RTK_OK				change program id successfully.
** @retval		RTK_Fail 			parameters error such as player or audio equals NULL
**
*/
RTK_Error RTK_Player_ChangeProgram(RTK_Player player, uint8_t program_id);
#endif //(RTKPLAYER_ARCH_VER == RTKPLAYER_ARCH_V1)


/**
* @brief
* enable or disable keepLastFrame for Player_ChangeChannel. 
*
* @param[in]		player		player handle
* @param[in]		enable   	state of keepLastFrame.
*
* @retval		RTK_OK			
* @retval		RTK_Fail 		
*
*/
RTK_Error RTK_Player_EnableKeepLastFrameForChangeChannel(RTK_Player player, RTK_Bool enable);

/**
* @brief
* Fast change video and audio, this function only for FCC. 
*
* @param[in]		player		player handle
* @param[in]		pVideo		a pointer of video config.
* @param[in]		pAudio		a pointer of audio config.
*
* @retval		RTK_OK			
* @retval		RTK_Fail		
*
*/
RTK_Error RTK_Player_ChangeChannel(RTK_Player player, RTK_VideoConfig *pVideo, RTK_AudioConfig *pAudio);

/**
** @brief
** Get the transport steam packets infomation after demux.
**
** @param[in]		player		 	handle of player
** @param[out]		pDemuxInfo 	 	A poiner for returned demux info.
**
** @retval		RTK_OK			Get demux info success.
** @retval		RTK_Fail 			Failed.
**
*/
RTK_Error RTK_Player_GetPacketDemuxInfo(RTK_Player player, Hal_Demux_Info_t *pDemuxInfo);

   /**  
   *ingroup  RTK_Player
   *@}
   */

/**
*ingroup RTK_Source
*@}
*/


/**
*@ingroup RTK_Player
*@{
*/

/**
** @brief
** This function to get cc information,such as stream number,ccType index and ccType String.
** Here eight cc streams are supported,you can get how many cc stream in the video with this function.
** This functio can tell you whether there are ccdata according to info->ccStreamNum.
** If info->ccStreamNum is 0,there is no cc data.
** If info->ccStreamNum is larger than 0,you can use Player_SetCCPlayback to show it.
** you can get ccType string from  info->ccTypeString[i],i is from 0 to ccStreamNum-1
**
** @param[in]		player		handle of player to deal with
** @param[out]		info		 	pointer to struct RTK_ClosedCaption_Param with cc information
**
** @retval		RTK_OK			get cc information successfully.
** @retval		RTK_Fail 			get cc information fail
**
*/
RTK_Error RTK_Player_GetCCInfo(RTK_Player player,RTK_ClosedCaption_Param_t* info);

/**
*ingroup RTK_Player
*@}
*/


/**
*@ingroup RTK_Global
*@{
*/
/**
** @brief
** Audio behind or early video output
**
** @param[in]		delayInMs			delay ms (+3000 ~ -3000)
**
**
*/
void RTK_SetHdmiOutVSDBDelay(int delayInMs);


void RTK_SetAudioHdmiDelay(bool isEnable,int delayInMs);


RTK_Error RTK_Player_GetDecodeStatus(RTK_Player player,uint32_t* pGoodDecodedFrmNum, uint32_t* pErrorDecodedFrmNum);

/**
** @brief
** get audio track info by player.
**
** @param[in]      player     				player handle
** @param[out]    ptrAudioTrackInfo 		the poit of struct RTK_Audio_TrackInfo with audio track info
**
** @retval         RTK_OK
** @retval         RTK_Fail
*/
RTK_Error RTK_Player_GetAudioTrackInfo(RTK_Player player, RTK_Audio_TrackInfo * ptrAudioTrackInfo);

/**
* @brief
* This function will get video track information in ts flow.
*
* @param[in]   player                          handle of player to deal with
* @param[out]  ptsVideoTrackInfo       the video information return 
*
* @retval              RTK_OK                          get video track information ok
* @retval              RTK_Fail                        get video track information failed
*
*/
RTK_Error RTK_Player_GetVideoTrackInfo(RTK_Player player, RTK_Video_TrackInfo* ptsVideoTrackInfo);

#ifdef ANDROID
/**
** @brief
** This function will sets the video surface of the upper player.
** 
** @param[in]	plane_id		The plane id to deal with, MAIN or PIP on the window to be managed.
** @param[in]	native_window	The pointer to a ANativeWindow object. If it is NULL, it will unset the video surface.
   **
** @retval		RTK_OK			Set Video Surface successfully.
** @retval		RTK_Fail		Error occurred.
**
**/
RTK_Error RTK_SetVideoSurfaceEx(RTK_WinPlane plane_id, void* native_window);
#endif

/**
*ingroup RTK_Global
*@}
*/

/**
   *@ingroup  RTK_Player
   *@{
   */
   
int RTK_Player_GetVersion(char version[]);
#endif //(RTKPLAYER_ARCH_VER >= RTKPLAYER_ARCH_V1)

#if (RTKPLAYER_ARCH_VER >= RTKPLAYER_ARCH_V2)  
/**
** @brief
** set record encrypt key
**
** @param[in]      source     source handle
** @param[in]      pRecKey    encrypt key content
** @param[in]      length     encrypt key length
**
** @retval         RTK_OK
** @retval         RTK_Fail
*/
RTK_Error RTK_Source_SetRecKey(RTK_Source source, const char* pRecKey,unsigned int length);

/**
** @brief
** Connect Source to Demux.
**
** @param[in]      demux     demux handle
** @param[in]      source     source handle. if NULL, it means disconnect
**
** @retval         RTK_OK
** @retval         RTK_Fail
*/
RTK_Error RTK_Demux_SetSource(RTK_Demux demux, RTK_Source source);

/**
** @brief
** Connect Source to Player.
**
** @param[in]      player     player handle
** @param[in]      source     source handle. if NULL, it means disconnect
**
** @retval         RTK_OK
** @retval         RTK_Fail
*/
RTK_Error RTK_Player_SetSource(RTK_Player player, RTK_Source source);

/**
*@ingroup RTK_Source
*@{
*/

/**
** @brief
** Create source from Frontend or others and does initial works.
**
** @param[out]     source    return the Source handle created by this function.
** @param[in]      param     the param used when creating the source.
**
** @retval         RTK_OK
** @retval         RTK_Fail  parameters error or memory alloc failed.
*/
RTK_Error RTK_Source_Create(RTK_Source *source, RTK_SourceParam_t param);

/**
** @brief
** Destroy an opened source from Frontend or others.
**
** @param[in]      source    source handle.
**
** @retval         RTK_OK
** @retval         RTK_Fail
*/
RTK_Error RTK_Source_Destroy(RTK_Source source);
#if (RTKPLAYER_ARCH_VER == RTKPLAYER_ARCH_V2)
/**
** @brief
** Change RF/Demod setting of tuner based source
**
** @param[in]      source    source handle.
** @param[in]      param     RF Demod Param
**
** @retval         RTK_OK
** @retval         RTK_Fail
*/
RTK_Error RTK_Source_SetRfDemod(RTK_Source source, RTK_RfDemodParam_t param);

/**
** @brief
** Get tuner lock status.
**
** @param[in]      source    source handle.
**
** @retval         RTK_OK    the tuner is locked.
** @retval         RTK_Fail  the tuner is unlocked or not tuner source.
*/
RTK_Error RTK_Source_GetIsLocked(RTK_Source source);

/**
** @brief
** Get tuner information.
**
** @param[in]      source    source handle.
** @param[out]     info      signal information.
**
** @retval         RTK_OK    get success.
** @retval         RTK_Fail  get fail or not tuner source.
*/
RTK_Error RTK_Source_GetSignalInfo(RTK_Source source, RTK_TunerInfo * info);
#endif //(RTKPLAYER_ARCH_VER == RTKPLAYER_ARCH_V2)

/**
** @brief
** Preset pid filter for a source before a player/demuxer use it.
** After connect to Player/Demuxer, it will union with player/demuxer pids.
** Those pids will work until call this function again or destory source.
**
** @param[in]      source    source handle.
** @param[in]      pids      pid array.
** @param[in]      numPID    number of pids.
**
** @retval         RTK_OK
** @retval         RTK_Fail
*/
RTK_Error RTK_Source_PresetPidFilter(RTK_Source source, uint16_t* pids, unsigned char numPID);

RTK_Bool RTK_System_VOisUsing(RTK_WinPlane vo_plane);

/**
** @brief
** Lend a buffer from source for Feeding TS data (NOT for tuner source).
**
** If lend success, the information in buffer includes the pointer of share
** memory, the total size of memory, the used size ,and some other flags.
**
** @param[in]      source    source handle.
** @param[out]     buffer    lend buffer info.
**
** @retval         RTK_OK    got the buffer success
** @retval         RTK_Fail  parameters error or buf full or NOT right source type
**/
RTK_Error RTK_Source_GetInputBuffer(RTK_Source source, RTK_TSInputBuffer * buffer);

/**
** @brief
** Feeds TS data (NOT for tuner source).
**
** The buffer must get from RTK_Source_GetInputBuffer().
** If feed success, it will auto retrun the lend buffer,
** else need call RTK_Source_DiscardInputBuffer().
**
** @param[in]       source    source handle.
** @param[in]       buffer    feed buffer info.
**
** @retval          RTK_OK
** @retval          RTK_Fail  size not packet align or full or NOT right source type
**/
RTK_Error RTK_Source_FeedInputBuffer(RTK_Source source, RTK_TSInputBuffer * buffer);

/**
** @brief
** Discards lent input buffer
**
** @param[in]       source    source handle.
** @param[in]       buffer    want to return buffer.
**
** @retval          RTK_OK
** @retval          RTK_Fail
**/
RTK_Error RTK_Source_DiscardInputBuffer(RTK_Source source, RTK_TSInputBuffer * buffer);

/**
** @brief
** 	Flush data in source. 
**
** @param[in]       source    source handle.
**
** @retval          RTK_OK
** @retval          RTK_Fail
**/
RTK_Error RTK_Source_Flush(RTK_Source source);

/**
** @brief
** set source tsid/sidx
**
** @param[in]      source     source handle.
** @param[in]      tsid_sidx  NAGRA tsid or VMX service index or ...
**
** @retval         RTK_OK
** @retval         RTK_Fail
*/
RTK_Error RTK_Source_SetTsidSidx(RTK_Source source, unsigned int tsid_sidx);
RTK_Error RTK_Source_SetTsidSid_Ex(RTK_Source source, unsigned int tsid_sidx, unsigned char numPID, unsigned short* pPIDs);


/**
* @brief
* set key update mode wheather to wait dual key or not before start descramble
*
* @param[in]      tsid_sidx  NAGRA tsid or VMX service index or ...
*@param[in]      is_encrypt For decrypt or encrypte session
*@param[in]      update_mode  key update mode , directly or wait for dual key 
*
* @retval		 RTK_OK 		set key update mode  successfully.
* @retval		 RTK_Fail		set key update mode failed.
*
**/
RTK_Error RTK_Source_SetKeyUpdateMode(unsigned int tsid_sidx,unsigned int is_encrypt,Hal_Key_UpdateMode_e update_mode);


/**
   * @brief
   * opens and sets sectionFilter by specific pid, add the SectionFilter into SI engine with the source handle
   *
   * @param[in]		source		 	handle of source to add section filter
   * @param[out]		filter_handle 	 	handle of filter to specify
   * @param[in]		pattern			defines filter pid, mask, mode, compose type, monitor mode, 
   *								user param and a function pointer to SI section filter callback
   *
   * @retval		RTK_OK			_xAddSectionFilter has successfully processed
   * @retval		RTK_Fail 			parameters error occurred or filter open failed.
   *
   */
RTK_Error RTK_Source_AddSectionFilter(RTK_Source source, RTK_SectionFilter * filter_handle, RTK_SectionFilterMask * pattern);

/**
   * @brief
   * stops the sectionFilter, then it should not parse the specific sections
   *
   * @param[in]		source		 	handle of source 
   * @param[in]		filter_handle 	 	handle of filter to stop
   *
   * @retval		RTK_OK			section filter has been stopped parsing with 
   *								specific filter_handle successfully.
   * @retval		RTK_Fail 			parameters error occurred or SI engine PID 
   *								filter remove failed
   *
   */
RTK_Error RTK_Source_RemoveSectionFilter(RTK_Source source, RTK_SectionFilter filter_handle);

/**
   * @brief
   * stops all sectionFilters source added, then sectionFilters will not work 
   *
   * @param[in]		source		 	handle of source 
   *
   * @retval		RTK_OK			all section filters have been stopped parsing successfully.
   * @retval		RTK_Fail 			parameters error occurred or SI engine PID filter remove failed.
   *
   */
RTK_Error RTK_Source_RemoveAllSectionFilters(RTK_Source source);

/**
** @brief
** get source idle status
**
** @param[in]      source     source handle.
** @param[out]     is_idle    source is idle or not.
**
** @retval         RTK_OK
** @retval         RTK_Fail
*/
RTK_Error RTK_Source_GetIsIdle(RTK_Source source, RTK_Bool* is_idle);

#endif //(RTKPLAYER_ARCH_VER >= RTKPLAYER_ARCH_V2)

#if (RTKPLAYER_ARCH_VER >= RTKPLAYER_ARCH_V3) 
/**
   * @brief
   * This function will scan the frequence with the configuration parameter such as tuner id, tuner type and  detailed parameter related to the type.
   *
   * @param[in]		ucTunerID		tuner instance ID
   * @param[in]		param			the objet to the the RTK_TunerParam_t struct
   *
   * @retval		RTK_OK			scan the frequence successfully with the given information
   * @retval		RTK_Fail		parameters error occurred or scan the frequence failed.
   *
   */
RTK_Error RTK_Tuner_Scan(uint8_t ucTunerID, RTK_TunerParam_t param);

/**
   * @brief
   * This function will release the target tuner channel.
   *
   * @param[in]		ucTunerID		tuner instance ID
   *
   * @retval		RTK_OK
   * @retval		RTK_Fail
   *
   */
RTK_Error RTK_Tuner_Release(uint8_t ucTunerID);

/**
* @brief 
* This function can be used to control LNB.
*
* @param[in]	tunerid         tuner ID
* @param[in]    eLNBMode        target LNB mode
* @retval       RTK_OK          success.
* @retval       RTK_Fail        fail.
*
*/
RTK_Error RTK_Tuner_SetLNBMode(uint32_t uiTunerID, enum RTK_LNB_MODE eLNBMode);

/**
* @brief 
* This function can be used to get current LNB status.
*
* @param[in]	tunerid         tuner ID
* @param[out]   ptLNBStatus     current LNB status
* @retval       RTK_OK          success.
* @retval       RTK_Fail        fail.
*
*/
RTK_Error RTK_Tuner_GetLNBStatus(uint32_t uiTunerID, RTK_LNB_STATUS_t* ptLNBStatus);

/**
* @brief 
* This function can be used to enable or disable 22KHz continues output.
*
* @param[in]	tunerid         tuner ID
* @param[in]    eEnable       	enable or disable
* @retval       RTK_OK          success.
* @retval       RTK_Fail        fail.
*
*/
RTK_Error RTK_Tuner_Set22KContinues(uint32_t uiTunerID, RTK_Bool eEnable);

/**
* @brief 
* This function can be used to send DiSEqC tone burst.
*
* @param[in]	tunerid         tuner ID
* @param[in]    eTone        	tone burst type
* @retval       RTK_OK          success.
* @retval       RTK_Fail        fail.
*
*/
RTK_Error RTK_Tuner_SendDiSEqCToneBurst(uint32_t uiTunerID, enum RTK_TONE_BURST eTone);

/**
* @brief 
* This function can be used to send DiSEqC command.
*
* @param[in]	tunerid         tuner ID
* @param[in]    tCmdMsg         DiSEqC command message struct
* @retval       RTK_OK          success.
* @retval       RTK_Fail        fail.
*
*/
RTK_Error RTK_Tuner_SendDiSEqCCmd(uint32_t uiTunerID, RTK_DISEQC_MSG_t tCmdMsg);
/**
* @brief 
* This function can be used to get DiSEqC reply message.
*
* @param[in]	tunerid         tuner ID
* @param[in]    ptReplyMsg      DiSEqC replay message struct
* @retval       RTK_OK          success.
* @retval       RTK_Fail        fail.
*
*/
RTK_Error RTK_Tuner_GetDiSEqCReply(uint32_t uiTunerID, RTK_DISEQC_MSG_t* ptReplyMsg);
#endif //(RTKPLAYER_ARCH_VER >= RTKPLAYER_ARCH_V3)

#if (RTKPLAYER_ARCH_VER <= RTKPLAYER_ARCH_V2)  
RTK_Error RTK_Set_TunerScanning(uint32_t uiTunerID, RTK_TunerParam_t param);
#endif //(RTKPLAYER_ARCH_VER <= RTKPLAYER_ARCH_V2)

/**
   * @brief
   * This function is to set tuner total channel num.
   *
   * @param[in]     ucTunerIndex         tuner index
   * @param[in]     ucTotalTunerNum    tuner total channelnum
   *
   * @retval        RTK_OK            set  successfully
   * @retval        RTK_Fail            set failed.
   *
   */
RTK_Error RTK_Tuner_SetChannelNum(uint8_t ucTunerindex, uint8_t totalchannelnum);

/**
*ingroup RTK_Player
*@}
*/

#ifdef __cplusplus
}
#endif

#endif /* _RTK_HAL_BASE_H_    */
